#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/asm_linkage.h>

struct itimerval value;
int itimer_real = 0;

void alarm() {
    itimer_real += 1;
    signal( SIGALRM, alarm );				/* System V requires handler be reset */
    setitimer(ITIMER_REAL, &value, 0);
}

static inline void uCopy( void *src, void *dst, unsigned int len ) {
    memcpy( dst, src, len );
} /* uCopy */

static inline void *uReadStackPointer( void ) {
    void *addr;

    asm volatile ( "mov %%sp, %0" : "=r" (addr) );
    return( addr );
} /* uReadStackPointer */

static inline void uWriteStackPointer( void *addr ) {
    asm volatile ( "mov %0, %%sp" : : "r" (addr) : "%sp" );
} /* uWriteStackPointer */

static inline void *uReadFramePointer( void ) {
    void *addr;

    asm volatile ( "mov %%fp, %0" : "=r" (addr) );
    return( addr );
} /* uReadFramePointer */

static inline void uWriteFramePointer( void *addr ) {
    asm volatile ( "mov %0, %%fp" : : "r" (addr) : "%fp" );
} /* uWriteFramePointer */

static inline void *uReadReturnAddress( void ) {
    void *addr;

    asm volatile ( "mov %%i7, %0" : "=r" (addr) );
    return( addr );
} /* uReadReturnAddress */

static inline void uWriteReturnAddress( void *addr ) {
    asm volatile ( "mov %0, %%i7" : : "r" (addr) : "%i7" );
} /* uWriteReturnAddress */

static inline void uCallUsingStack( void (*begin)() ) {
    asm volatile ( "mov %0, %%l0" : : "r" (begin) : "%l0" );
    asm volatile ( "ld [%%sp+0x44], %%o0" : : : "%o0" );
    asm volatile ( "ld [%%sp+0x48], %%o1" : : : "%o1" );
    asm volatile ( "ld [%%sp+0x4c], %%o2" : : : "%o2" );
    asm volatile ( "ld [%%sp+0x50], %%o3" : : : "%o3" );
    asm volatile ( "ld [%%sp+0x54], %%o4" : : : "%o4" );
    asm volatile ( "call %l0, 0" );
    asm volatile ( "ld [%%sp+0x58], %%o5" : : : "%o5" );
} /* uCallUsingStack */

#ifndef max
#define max( a, b ) ( ( a ) > ( b ) ? ( a ) : ( b ) )
#endif /* max */

#define U_CEILING( V, A ) ( ( (V) + (A) - 1 ) & ~ ( (A) - 1 ) )
#define U_FLOOR( V, A ) ( (V) & ~ ( (A) - 1 ) )

struct uCxtBase {					/* base type for all context areas */
    int size;
    void (*saver)( void * );
    void (*restorer)( void * );
    struct uCxtBase *link;
    char cxtarea[1];
};

typedef struct uStackD {
    void *base;						/* pointer to base of stack */
    void *limit;					/* pointer to limit of stack */
    void *sp;						/* current stack pointer */
    void *fp;						/* current frame pointer */
    void *pc;						/* return address */
} *uStack;

typedef struct uTaskD {
    int state;
    int save;
    void *start;
    struct uCxtBase *cxtlist;
    struct uStackD StackData;				/* stack information */
    void (*begin)();					/* pointer to subroutine started as a coroutine */
} *uTask;

#define U_TASK() (struct uTaskD){ 0, 0, NULL, NULL, { 0, 0, 0, 0, 0 }, NULL }

uTask uWorkTask, t1, t2;
struct uTaskD mainTCB = U_TASK();

#define __U_STACK_GROWS__DOWN__ 1			/* stack grows down */

uTask uAllocTask( int space ) {
    void *start;
    uTask task;

    space = U_CEILING( space, sizeof(double) ) + sizeof(double); /* move value up to multiple of sizeof(double) */
    start = malloc( space + sizeof(struct uTaskD) );

#if __U_STACK_GROWS__DOWN__				/* stack grows down */
    task = (void *) U_FLOOR( (int) start + space, sizeof(double) ); /* move value down to multiple of sizeof(double) */
    *task = U_TASK();
    task->start = start;				/* free address */
    task->StackData.base = (void *) ( (int) task - sizeof( double ) ); /* prime stack for first push */
    task->StackData.limit = start;
#else							/* stack grows up */
    task = start;
    *task = U_TASK();
    task->start = start;				/* free address */
    task->StackData.base = (void *) U_CEILING( (int)start + sizeof(struct uTaskD), sizeof(double) );
    task->StackData.limit = start + space + sizeof(struct uTaskD) - sizeof(int);
#endif /* __U_STACK_GROWS__DOWN__ */

    return( task );
} /* uAllocTask */

static inline void uMakeFrameUsingStack( uStack stack, void *buf, int len ) {
    stack->fp = stack->base - WINDOWSIZE;
    stack->sp = stack->fp - U_CEILING( MINFRAME + (len - ARGPUSHSIZE > 0 ? len - ARGPUSHSIZE : 0 ), sizeof( double ) );
    uCopy( buf, stack->sp + ARGPUSH, len );
} /* uMakeFrameUsingStack */

static inline void uPushFixedRegs( void ) {
    asm volatile ( "ta 3" );
} /* uPushFixedRegs */

static inline void uPopFixedRegs( void ) {
} /* uPopFixedRegs */

static inline void uPushFloatRegs( void ) {
} /* uPushFloatRegs */

static inline void uPopFloatRegs( void ) {
} /* uPopFloatRegs */

static inline void uSave( void ) {
    uStack stack = &(uWorkTask->StackData);

    stack->pc = uReadReturnAddress();			/* save the return address */

    if ( uWorkTask->cxtlist != NULL ) {			/* if user contexts to save... */
	register struct uCxtBase *p;

	for ( p = uWorkTask->cxtlist; p != NULL; p = p->link ) {
	    p->saver( &(p->cxtarea) );
	} /* for */
    } /* if */

    /* Nothing but inline routine calls past this point */

    uPushFixedRegs();					/* push the fixed point registers onto the task stack */

    if ( uWorkTask->save == 1 ) {			/* if performing a floating point save... */
	uPushFloatRegs();				/* push the floating point registers on the stack also */
    } /* if */

    stack->fp = uReadFramePointer();			/* save the current frame pointer */
    stack->sp = uReadStackPointer();			/* save the current stack pointer */
} /* uSave */

static inline void uRestore( void ) {
    uStack stack = &(uWorkTask->StackData);

    uWriteStackPointer( stack->sp );			/* restore the stack pointer */
    uWriteFramePointer( stack->fp );			/* restore the frame pointer */
    uWriteReturnAddress( stack->pc );			/* restore the return address */
    uPopFixedRegs();					/* pop the fixed point registers from the stack */
} /* uRestore */

static inline void uCallTask( uTask task ) {
    uStack stack= &(task->StackData);

    uWriteStackPointer( stack->sp );			/* install the new task stack pointer */
    uWriteFramePointer( stack->fp );			/* install the new task frame pointer */

    uCallUsingStack( task->begin );			/* call the task, arguments are already on the stack */
} /* uCallTask */

void uExecuteTask( uTask );

void uDie( void* msg, int len ) {
    static count = 0;

    if ( count == 0 ) {
	count += 1;
	uExecuteTask( t2 );
    } else {
	uExecuteTask( &mainTCB );
    }
} /* uDie */

void uExecuteTask( uTask task ) {
    uSave();

    uWorkTask = task;

    if ( task->state & 1 ) {
	task->state &= ~1;
	uCallTask( task );
	uDie( NULL, 0 );
    } /* if */

    uRestore();
} /* uExecuteTask */

uTask uDoEmit( void *cluster, long space, void (*begin)(), long arglen, void *argadr ) {
    uTask task;
    uStack stack;

    task = uAllocTask( space );
    task->state = 1;
    task->cxtlist = NULL;
    task->begin = begin;				/* task begins execution at this address */
    stack = &(task->StackData);

    uMakeFrameUsingStack( stack, argadr, arglen );

    return( task );
} /* uDoEmit */

uTask uEmit( void (*begin)(), ... ) {
    va_list argadr;
    uTask task;

    va_start( argadr, begin );
    task = uDoEmit( 0, 16000, (void (*)())begin, 64, argadr );
    va_end( argadr );

    return( task );					/* return a reference to the new task */
} /* uEmit */

#define NoOfTimes 50

void task1( int a, int b, int c, int d, int e, int f, int g, int h, int i, int j ) {
    int x;

    printf( "task1, stack:0x%x %d, %d, %d, %d, %d, %d, %d, %d, %d, %d\n", uReadStackPointer(), a, b, c, d, e, f, g, h, i, j );

    for ( x = 1; x <= NoOfTimes; x += 1 ) {
	printf( "task1, uWorkTask:0x%x, stack:0x%x, x:%d\n", uWorkTask, uReadStackPointer(), x );
	uExecuteTask( t2 );
    } /* for */
}

void task2( int a, int b, int c, int d, int e, int f, int g, int h, int i, int j ) {
    int x;

    printf( "task2, stack:0x%x %d, %d, %d, %d, %d, %d, %d, %d, %d, %d\n", uReadStackPointer(), a, b, c, d, e, f, g, h, i, j );
    printf( "t1, state:%d, save:%d, start:0x%x\n", t1->state, t1->save, t1->start );

    for ( x = 1; x <= NoOfTimes; x += 1 ) {
	printf( "task2, uWorkTask:0x%x, stack:0x%x, x:%d\n", uWorkTask, uReadStackPointer(), x );
	uExecuteTask( t1 );
    } /* for */
}

uAbsorb( uTask t ) {
    printf( "uAbsorb, t:0x%x\n", t );
}

int main() {
    t1 = uEmit( task1, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 );
    t2 = uEmit( task2, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 );

    value.it_value.tv_sec = 0;
    value.it_value.tv_usec = 10;
    value.it_interval.tv_sec = 0;
    value.it_interval.tv_usec = 0;

    signal( SIGALRM, alarm );
    setitimer( ITIMER_REAL, &value, 0 );

    printf( "main, t1:0x%x, t2:0x%x\n", t1, t2 );
    uWorkTask = &mainTCB;
    
    printf( "main, stack:0x%x\n", uReadStackPointer() );
    uExecuteTask( t1 );
    uAbsorb( t1 );
    uAbsorb( t2 );
    printf( "main, stack:0x%x, itimer_real:%d\n", uReadStackPointer(), itimer_real );
}

/* Local Variables: */
/* compile-command: "gcc -O cxtsw.c" */
/* End: */
