/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.4.3, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uSampler.c -- General asynchronous sampling facility.  See the documentation for usage.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:29:57 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Mon Jan 20 22:10:28 1992
 * Update Count     : 32
 */

#include <uSystem.h>
#include <uSampler.h>
#include <uUnix.h>
#include <usclkc.h>

static void Sampler( uSampler sample ) {
    usclk_t startime, currtime, zerotime = 0;
    int i, j, first;
    uSample *sp;

    uP( &(sample->start) );				/* wait for the gun */
    first = 1;
    for ( i=0, sp=sample->Samples; i<sample->NoOfSamples; i+=1, (int)(sp)+=sizeof(uSample)+sample->NoOfTargets*sizeof(int) ) {
	startime = getusclk();
	for ( ;; ) {					/* spin until the next sampler clock tick */
	    currtime = getusclk();
	  if ( currtime - startime >= sample->Frequency ) break;
	} /* for */
	for ( j = 0; j < sample->NoOfTargets; j += 1 ) { /* store the target values into a sample */
	    sp->TargetVal[j] = *(sample->Targets[j]);
	} /* for */
	if ( first ) {					/* first time, store the start time as time zero */
	    zerotime = startime;
	    first = 0;
	} /* if */
	sp->time = currtime - zerotime;			/* store the current time */
    } /* for */
    uDie( NULL, 0 );
} /* Sampler */

#include <stdarg.h>

uSampler uCreateSampler( int NoOfSamples, int Frequency, int *targets, ... ) {
    va_list args;
    int NoOfTargets, i, *next;
    uSampler sample;
    
    usclk_init();					/* map microsecond clock into process address space */

    for ( NoOfTargets = 1, va_start( args, targets );; NoOfTargets += 1 ) { /* count number of targets in the argument list */
	next = va_arg( args, int * );
      if ( next == 0 ) break;
    } /* for */

    sample = uMalloc( sizeof(struct uSamplerStruct) + NoOfTargets * sizeof(int *) );
    
    sample->NoOfTargets = NoOfTargets;
    sample->Targets[0] = targets;			/* store the required target */
    for ( i = 1, va_start( args, targets ); i < sample->NoOfTargets ; i += 1 ) { /* store the optional targets */
	sample->Targets[i] = va_arg( args, int * );
    } /* for */
    sample->NoOfSamples = NoOfSamples;
    sample->Frequency = Frequency;
    sample->start = U_SEMAPHORE( 0 );			/* sampler will block until explicitly started */
    sample->Cluster = uCreateCluster( 1, 0 );		/* create the sampler cluster with no time slicing */
    sample->Samples = uMalloc( NoOfSamples * ( sizeof(uSample) + sample->NoOfTargets * sizeof(int) ) );
    sample->task = uLongEmit( sample->Cluster, 512, Sampler, uGetArgLen(), sample ); /* create sampler task */
    va_end( args );
    return sample;
} /* CreateSampler */

void uStartSampler( uSampler sample ) {
    uV( &(sample->start) );				/* ... and they're off, BANG! */
} /* uStartSampler */

void uDestroySampler( uSampler sample, uStream DumpFile ) {
    int i, j;
    uSample *sp;

    uAbsorb( sample->task, NULL, 0 );			/* wait for sampler to finish */
    uDestroyCluster( sample->Cluster );			/* destroy the monitor cluster */

    for ( i = 0; i < sample->NoOfTargets; i += 1 ) {	/* print out each target value separately */
	for ( j=0, sp=sample->Samples; j<sample->NoOfSamples; j+=1, (int)(sp)+=sizeof(uSample)+sample->NoOfTargets*sizeof(int) ) {
	    uFprintf( DumpFile, "%d %d\n", sp->time, sp->TargetVal[i] );
	} /* for */
	uFprintf( DumpFile, "\n" );			/* data set separator */
    } /* for */

    uFree( sample->Samples );				/* free the sample array */
    uFree( sample );					/* free the sampler administration information */
} /* uDestroySampler */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
