/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.4.3, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uFile.c -- I/O routines.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:29:03 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Tue Jan  4 19:05:52 1994
 * Update Count     : 459
 */

/* change the following to a 1 if you want the debugging code included */

/* #define __DEBUG_IO__ */

#define __U_KERNEL__

#include <uUnix.h>
#include <uSystem.h>
#include <uFile.h>

#if defined( __ibm__ )
#define U_FNDELAY FNONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EWOULDBLOCK
#elif defined( __sgi__ )
#define U_FNDELAY FNONBLK				/* POSIX non-blocking */
#define U_EWOULDBLK EWOULDBLOCK
#elif defined( __hp__ )
#define U_FNDELAY O_NONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#elif defined( __sun__ ) && defined( __svr4__ )
#define U_FNDELAY O_NONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#else
#define U_FNDELAY FNDELAY
#define U_EWOULDBLK EWOULDBLOCK
#endif

/*
 * This type is used to indicate whether and when a file should be set into
 * non-blocking mode.
 *
 * U_FNDELAY_NEVER:  it never blocks (e.g. file access) so don't bother with FNDELAY
 * U_FNDELAY_DEMAND: it blocks, but setting it to non-blocking mode at open is too
 *                      dangerous beause an interrupt can leave the file in non-blocking
 *                      mode after the application terminates (e.g. stdin).
 *			To mitigate the problem, set it to non-blocking mode on each syscall.
 * U_FNDELAY_ALWAYS: it blocks.  Set it to non-blocking mode when the file is opened.
 *			It is reset to the original mode when the file is closed.
 */

typedef enum { U_FNDELAY_NEVER, U_FNDELAY_DEMAND, U_FNDELAY_ALWAYS } uBlockVal;

typedef struct uFileD {
    uSemaphore mutex;
    int file;
    uCluster cluster;
    uBlockVal block;
} uFileD;

static char data;

static inline void uTouch( char *addr ) {
    /*
     * Used to reference shared memory so that the page tables for
     * separate address spaces will be updated properly.
     * Apparently, the I/O system does not check to see whether the
     * pages are a part of the shared memory before it raises
     * a segmentation fault.
     */

    if ( addr != NULL ) {
	data = *addr;
    } /* if */
} /* uTouch */

void uFileEnter( uFile file ) {
    uMigrate( file->cluster );    
    uP( &(file->mutex) );
} /* uFileEnter */

void uFileLeave( uFile file, uCluster prevClus ) {
    uV( &(file->mutex) );
    uMigrate( prevClus );
} /* uFileLeave */

int uFileFd( uFile file ) {
    return file->file;
} /* uFileFd */

static inline uFile uCreateFile() {
    uFile file;

    file = uMalloc( sizeof( uFileD ) );
    file->mutex = U_SEMAPHORE( 1 );
    file->cluster = uCreateCluster( 1, 0 );

    return file;
} /* uCreateFile */

static inline void uDestroyFile( uFile file ) {
    uDestroyCluster( file->cluster );
    uFree( file );
} /* uDestroyFile */

static inline uBlockVal uDoesFileBlock( uFile file ) {
    /*
     * Determine the type of the file, so that we know whether to block on access or not.
     */

    struct stat buf;

    if ( fstat( file->file, &buf ) < 0 ) {		/* can't stat the file...  block by default */
	return U_TRUE;
    } /* if */
    	
    /*
     * examine the "type" bits
     * 
     * File access may block if file type is:
     * 	S_IFIFO  : named pipe
     * 	S_IFCHR  : character special file (tty/pty)
     * 	S_IFSOCK : socket
     */

    if ( ( buf.st_mode & S_IFCHR ) == S_IFCHR ) {	/* if a tty, always on demand (only has bearing on input) */
	return U_FNDELAY_DEMAND;
#ifdef S_IFIFO
    } else if ( ( buf.st_mode & S_IFIFO ) == S_IFIFO ) {
	return U_FNDELAY_ALWAYS;
#endif
    } else if ( ( buf.st_mode & S_IFSOCK ) == S_IFSOCK ) {
	return U_FNDELAY_ALWAYS;
    } else {
	return U_FNDELAY_NEVER;
    } /* if */

} /* uDoesFileBlock */

static inline void uSetBlockFlag( uFile file ) {

    int flags;
    
    flags = fcntl( file->file, F_GETFL );
    if ( flags == -1 ) {
	file->block = U_FNDELAY_NEVER;
	return;
    } /* if */

    flags = fcntl( file->file, F_SETFL, flags | U_FNDELAY );
    if ( flags == -1 ) {
	file->block = U_FNDELAY_NEVER;
	return;
    } /* if */

} /* uSetBlockFlag */

static inline void uClearBlockFlag( uFile file ) {

    int flags;
    
    flags = fcntl( file->file, F_GETFL );
    if ( flags == -1 ) {
	file->block = U_FNDELAY_NEVER;
	return;
    } /* if */

    flags = fcntl( file->file, F_SETFL, flags & ~ U_FNDELAY );
    if ( flags == -1 ) {
	file->block = U_FNDELAY_NEVER;
	return;
    } /* if */

} /* uClearBlockFlag */

static inline void uSetFileAccess( uFile file ) {
    /*
     * if the file is a blocking file, set the fcntl bits to allow non blocking access
     */
    if ( file->block == U_FNDELAY_ALWAYS ) uSetBlockFlag( file );
} /* uSetFileAccess */

static inline void uClearFileAccess( uFile file ) {
    /*
     * if the file is a blocking file, set the fcntl bits to what they originally were
     */
    if ( file->block == U_FNDELAY_ALWAYS ) uClearBlockFlag( file );
} /* uClearFileAccess */

static int uCheckFile( int delay ) {

    int found;
    struct timeval timeout = { delay, 0 };
    uCluster cluster = uThisCluster();
    uProcessor processor = uThisProcessor();
    fd_set rfds = cluster->rfds;			/* make local copy of fd sets because select operation destroys */
    fd_set wfds = cluster->wfds;			/* the sets and the io operation pending information is lost */
    fd_set efds = cluster->efds;
    
    processor->state = U_IDLE;				/* change this processor's state to idle */
    cluster->ready.idle = U_TRUE;			/* set the idle flag so that the processor gets woken up */

#ifdef __DEBUG_IO__
    for ( found = 0; found < FD_SETSIZE; found += 1 ) {
	if ( FD_ISSET( found, &(cluster->rfds) ) ) fprintf( stderr, "(0x%x) rfds %d is set\n", uThisTask(), found );
	if ( FD_ISSET( found, &(cluster->wfds) ) ) fprintf( stderr, "(0x%x) wfds %d is set\n", uThisTask(), found );
	if ( FD_ISSET( found, &(cluster->efds) ) ) fprintf( stderr, "(0x%x) efds %d is set\n", uThisTask(), found );
    } /* for */
#endif
    
#ifdef __DEBUG_IO__
    fprintf( stderr, "(0x%x) before select\n", uThisTask() );
#endif
    found = select( FD_SETSIZE, &rfds, &wfds, &efds, &timeout );
#ifdef __DEBUG_IO__
    fprintf( stderr, "(0x%x) after select, found:%d, errno:%d\n", uThisTask(), found, errno );
#endif
   
    processor->state = U_BUSY;				/* set processor state back to busy */
    cluster->ready.idle = U_FALSE;			/* set the cluster idle back to active */

    if ( found > 0 ) {					/* io is ready */
	int fd;
	
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is available, clearing masks and releasing blocked tasks\n", uThisTask() );
#endif
	uP( &(cluster->mutex) );			/* acquire the cluster */
	
	cluster->iopoller = 0;				/* nobody is the first io task */

	/* check to see which io operations were ready. */
	/* wake the tasks for those io operations which */
	/* are ready.  no need to clear the bits because */
	/* they are only local bits set by the select operation */

	for ( fd = 0; fd < FD_SETSIZE; fd += 1 ) {

	    if ( FD_ISSET( fd, &rfds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } else if ( FD_ISSET( fd, &wfds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } else if ( FD_ISSET( fd, &efds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } /* if */

	} /* for */

	/* must ensure that at least one task that does not */
	/* have io ready must be woken up so that at least */
	/* one task will resume the responsibilities of the poller.  */
	/* it may be the case that all tasks that wake up for io */
	/* will go onto do other things, and nobody will check for */
	/* io for those tasks that are still sleeping. */
	
	for ( fd = 0; fd < FD_SETSIZE; fd += 1 ) {

	    if ( cluster->iocount[fd] != 0 ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

		break;
		
	    } /* if */

	} /* for */
	
	uV( &(cluster->mutex) );			/* release the cluster */
	
	return 1;					/* tell the caller io is ready */
	
    } else if ( found == 0 ) {
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is not available\n", uThisTask() );
#endif
	return 0;					/* tell the caller io is not ready */
    } else {
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is not available, errno:%d\n", uThisTask(), errno );
#endif
	if ( errno != EINTR ) {
	    uAbort( "uCheckFile(): internal error!" );
	} /* exit */
	return 0;					/* tell the caller io is not ready */
    } /* if */

} /* uCheckFile */

static void uSelectFile( uFile file, fd_set *set ) {

    uCluster cluster = uThisCluster();

    uV( &(file->mutex) );				/* release the file */
    uP( &(cluster->mutex) );				/* acquire the cluster */

    FD_SET( file->file, set );				/* register interest in this event */

    if ( cluster->iopoller == 0 ) {			/* if this is the first io task */
	cluster->iopoller = uThisTask();		/* remember who the first io task is */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) is the first io task\n", uThisTask() );
#endif
	for ( ;; ) {
#ifdef __DEBUG_IO__
	    fprintf( stderr, "(0x%x) adding %u to set 0x%x\n", uThisTask(), file->file, *set );
#endif
	    uV( &(cluster->mutex) );			/* release the cluster */
	    if ( uReadyTasks() == 0 ) {			/* if this is the only task running */
#ifdef __DEBUG_IO__
		fprintf( stderr, "(0x%x) no ready tasks, blocking\n", uThisTask() );
#endif
		if ( uCheckFile( 1 ) ) break;		/* wait for io, if it is ready, try io again */
	    } else {
#ifdef __DEBUG_IO__
		fprintf( stderr, "(0x%x) ready tasks, polling\n", uThisTask() );
#endif
		if ( uCheckFile( 0 ) ) break;		/* check for io, if it is ready, try io again */
	    } /* if */
#ifdef __DEBUG_IO__
	    fprintf( stderr, "(0x%x) no io ready, yielding\n", uThisTask() );
#endif
	    uYield();					/* give up the processor */
	    uP( &(cluster->mutex) );			/* acquire the cluster */
	} /* for */
    } else {						/* this is not the first io task */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) is blocking for io\n", uThisTask() );
#endif
	cluster->iocount[file->file] += 1;		/* remember that someone is blocked */
	uV( &(cluster->mutex) );			/* release the cluster */
	uP( &(cluster->ioqueue[file->file]) );		/* wait for some io */
    } /* if */
    uP( &(file->mutex) );				/* acquire the file */
    
} /* uSelectFile */

static inline void uInterEnable( int previous ) {
    /*
     * restores async interventions to the given state
     */

    uTask me = uThisTask();

    if ( previous ) {
	me->state &= ~U_DISABLE_INTER;
	if ( me->inters && me->icheckneeded ) {
	    /*
	     * Pending intervention(s) which haven't been checked; possibly deliver one.
	     */
	    uAsyncInterDeliver();
	} /* if */
    } /* if */
} /* uInterEnable */

static inline int uInterDisable( void ) {
    /*
     * Disables async interventions, returning old state.
     */

    uTask me = uThisTask();
    int enabled = (me->state & U_DISABLE_INTER) == 0;

    me->state |= U_DISABLE_INTER;
    return enabled;
} /* uInterDisable */

#ifndef __U_UNIXRC__
static volatile void uOpenError( uFile file, char *path, int flags, int mode ) {
    uOpenExMsg msg;
    uException* except;

    msg.base.errno = errno;
    msg.path = path;
    msg.perms = 0;					/* not used for uOpen  */
    msg.flags = flags;
    msg.mode = mode;

    switch( msg.base.errno ) {
      case EEXIST:
	msg.base.msg = "uOpen: filename exists on create.\n";
	except = &uOpenEx;
	break;
      case EPERM:
	msg.base.msg = "uOpen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.base.msg = "uOpen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.base.msg = "uOpen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.base.msg = "uOpen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.base.msg = "uOpen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.base.msg = "uOpen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.base.msg = "uOpen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.base.msg = "uOpen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.base.msg = "uOpen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.base.msg = "uOpen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.base.msg = "uOpen: special file already opened with exclusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.base.msg = "uOpen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.base.msg = "uOpen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.base.msg = "uOpen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.base.msg = "uOpen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.base.msg = "uOpen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.base.msg = "uOpen: unknown error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uOpenError */
#endif

uFile uOpen( char *path, int flags, int mode ) {
    uFile file;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    file = uCreateFile();				/* create new cluster */
    uFileEnter( file );					/* migrate to I/O cluster */

    file->file = open( path, flags, mode );		/* open the file */

    if ( file->file < 0 ) {				/* operation failed */
	uFileLeave( file, prevClus );			/* migrate back to previous cluster */
	uDestroyFile( file );				/* clean up */
	uInterEnable( istat );				/* re-enable interventions */
#ifdef __U_UNIXRC__
	return NULL;
#else
	uOpenError( file, path, flags, mode );		/* does not return */
#endif
    } /* exit */

    file->block = uDoesFileBlock( file );
    uSetFileAccess( file );
    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return file;
} /* uOpen */

#ifndef __U_UNIXRC__
static volatile void uReadError( uFile file ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uRead: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uRead: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uRead: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uRead: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uRead: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( file, except, &msg, sizeof(msg) );
} /* uReadError */
#endif

int uRead( uFile file, void *buf, int len ) {
    int rlen, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( file );					/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    for ( ;; ) {
	if ( file->block == U_FNDELAY_DEMAND ) uSetBlockFlag( file );
	rlen = read( file->file, buf, len );		/* do a nonblocking read */
	terrno = errno;
	if ( file->block == U_FNDELAY_DEMAND ) uClearBlockFlag( file );
	errno = terrno;

	if ( rlen != -1 ) break;			/* if the read was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( file, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uReadError( file );				/* does not return */
#endif
	} /* exit */

	uSelectFile( file, &(file->cluster->rfds) );	/* read failed to complete */

    } /* for */

    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */

#ifndef __U_UNIXRC__
    if ( rlen == 0 ) {
	uEofExMsg msg;

	msg.msg = "uRead: End of File\n";
	uRaise( file, &uEofEx, &msg, sizeof( msg ) );	/* does not return */
    } /* exit */
#endif
    return rlen;
} /* uRead */

#ifndef __U_UNIXRC__
static volatile void uWriteError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uWrite: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uWrite: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uWrite: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uWrite: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uWrite: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uWrite: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uWrite: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( file, except, &msg, sizeof(msg) );
} /* uWriteError */
#endif

int uWrite( uFile file, void *buf, int len ) {
    int wlen, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( file );					/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    for ( ;; ) {
	if ( file->block == U_FNDELAY_DEMAND ) uSetBlockFlag( file );
	wlen = write( file->file, buf, len );		/* do a nonblocking write */
	terrno = errno;
	if ( file->block == U_FNDELAY_DEMAND ) uClearBlockFlag( file );
	errno = terrno;

        if ( wlen != -1 ) break;			/* if the write was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( file, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uWriteError( file );			/* does not return */
#endif
	} /* exit */

	uSelectFile( file, &(file->cluster->wfds) ); /* write failed to complete */

    } /* for */

    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return wlen;
} /* uWrite */

#ifndef __U_UNIXRC__
static volatile void uLseekError( uFile file ) {
    uIOErrorExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uLseek: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EINVAL:
	msg.msg = "uLseek: bad parameter\n";
	except = &uIOErrorEx;
	break;
      case EPIPE:
	msg.msg = "uLseek: attempting to seek on a socket.\n";
	except = &uBadFileEx;
      default:
	msg.msg = "uLseek: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */    
    uRaise( file, except, &msg, sizeof(msg) );
} /* uLseekError */
#endif

off_t uLseek( uFile file, off_t offset, int whence ) {
    off_t pos;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( file );					/* migrate to I/O cluster */

    pos = lseek( file->file, offset, whence );

    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( pos == -1 ) {					/* operation failed */
#ifndef __U_UNIXRC__
	uLseekError( file );				/* does not return */
#endif
    } /* if */
    return pos;
} /* uLseek */

#ifndef __U_UNIXRC__
static volatile void uFsyncError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFsync: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFsync: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFsync: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFsync: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EINVAL:
	msg.msg = "uFsync: fsyncing a socket.\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFsync: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFsync: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( file, except, &msg, sizeof(msg) );
} /* uFsyncError */
#endif

int uFsync( uFile file ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( file );					/* migrate to I/O cluster */

    code = fsync( file->file );

    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {					/* operation failed */
#ifndef __U_UNIXRC__
	uFsyncError( file );				/* does not return */
#endif
    } /* if */
    return code;
} /* uFsync */

#ifndef __U_UNIXRC__
static volatile void uCloseError( uFile file ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uClose: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      default:
	msg.msg = "uClose: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */    
    uRaise( file, except, &msg, sizeof(msg) );
} /* uCloseError */
#endif

int uClose( uFile file ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( file );					/* migrate to I/O cluster */
    uClearFileAccess( file );				/* reset blocking if set */

    code = close( file->file );

    uFileLeave( file, prevClus );			/* migrate back to previous cluster */
    uDestroyFile( file );				/* clean up */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {					/* operation failed */
#ifndef __U_UNIXRC__
	uCloseError( file );				/* does not return */
#endif
    } /* exit */
    return code;
} /* uClose */

#ifndef __U_UNIXRC__
static volatile void uSocketError( uFile sock, int af, int type, int protocol ) {
    uCreateSockExMsg msg;
    uException *except;
    int len = sizeof(msg);

    msg.base.errno = errno;
    msg.af = af;
    msg.type = type;
    msg.protocol = protocol;

    switch( msg.base.errno ) {
      case EACCES:
	msg.base.msg = "uSocket: Can't create socket of given type or protocol\n";
	except = &uCreateSockEx;
	break;
      case EMFILE:
	msg.base.msg = "uSocket: this process has too many files open\n";
	except = &uCreateSockEx;
	break;
      case ENFILE:
	msg.base.msg = "uSocket: too many files open in the system\n";
	except = &uCreateSockEx;
	break;
      case EPROTONOSUPPORT:
	msg.base.msg = "uSocket: bad protocol\n";
	except = &uCreateSockEx;
	break;
      case ENOBUFS:
	msg.base.msg = "uSocket: no buffer space\n";
	except = &uNoBufsEx;
	len = sizeof(uNoBufsExMsg);
	break;
      default:
	msg.base.msg = "uSocket: unknown creation error\n";
	except = &uCreateSockEx;
	break;
    } /* switch */    
    uRaise( NULL, except, &msg, len );
} /* uSocketError */
#endif

uFile uSocket( int af, int type, int protocol ) {
    uFile sock;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    sock = uCreateFile();				/* create new cluster */
    uFileEnter( sock );					/* migrate to I/O cluster */

    sock->file = socket( af, type, protocol );		/* create socket descriptor on socket cluster */

    if ( sock->file == -1 ) {				/* operation failed */
	uFileLeave( sock, prevClus );			/* migrate back to previous cluster */
	uDestroyFile( sock );				/* clean up */
	uInterEnable( istat );				/* re-enable interventions */
#ifdef __U_UNIXRC__
	return NULL;
#else
	uSocketError( sock, af, type, protocol );	/* does not return */
#endif
    } /* exit */

    if ( af == AF_UNIX ) {				/* only UNIX internal protocol can have operations restarted */
	sock->block = U_FNDELAY_ALWAYS;
    } else if ( af == AF_INET ) {			/* non-blocking sockets are not available on all vendor OS */
	sock->block = uDoesFileBlock( sock );
    } else {
	sock->block = U_FNDELAY_NEVER;
    } /* if */

    uSetFileAccess( sock );

    uFileLeave( sock, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */

    return sock;
} /* uSocket */

#ifndef __U_UNIXRC__
static volatile void uBindError( uFile socket, char *name, int namelen ) {
    uBadSockAddressExMsg msg;
    uException *except;
    int len = sizeof(msg);

    msg.base.errno = errno;
    msg.name = name;
    msg.namelen = namelen;

    switch( msg.base.errno ) {
      case EACCES:
	msg.base.msg = "uBind: Address protected, no permission\n";
	except = &uBadSockAddressEx;
	break;
      case EFAULT:
	msg.base.msg = "uBind: addr parm not in address space\n";
	except = &uBadSockAddressEx;
	break;
      case EBADF:
	msg.base.msg = "uBind: bad file descriptor\n";
	except = &uBadFileEx;
	len = sizeof(uBadFileExMsg);
	break;
      case EINVAL:
	msg.base.msg = "uBind: this socket is already bound\n";
	except = &uBadSockAddressEx;
	break;
      case ENOTSOCK:
	msg.base.msg = "uBind: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	len = sizeof(uNotSockExMsg);
	break;
      case EADDRNOTAVAIL:
	msg.base.msg = "uBind: cannot use specified address\n";
	except = &uBadSockAddressEx;
	break;
      case EADDRINUSE:
	msg.base.msg = "uBind: specified address is already in use\n";
	except = &uBadSockAddressEx;
	break;
      default:
	msg.base.msg = "uBind: unknown socket error\n";
	except = &uBadSockAddressEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, len );
} /* uBindError */
#endif

int uBind( uFile socket, void *name, int namelen ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    code = bind( socket->file, name, namelen );

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {					/* operation failed */
#ifndef __U_UNIXRC__
	uBindError( socket, name, namelen );		/* does not return */
#endif
    } /* if */
    return code;
} /* uBind */

#ifndef __U_UNIXRC__
static volatile void uConnectError( uFile socket, char *name, int namelen ) {
    uBadSockAddressExMsg msg;
    uException *except;
    int len = sizeof(msg);

    msg.base.errno = errno;
    msg.name = name;
    msg.namelen = namelen;

    switch( msg.base.errno ) {
      case EFAULT:
	msg.base.msg = "uConnect: addr parm not in address space\n";
	except = &uBadSockAddressEx;
	break;
      case EBADF:
	msg.base.msg = "uConnect: bad file descriptor\n";
	except = &uBadFileEx;
	len = sizeof(uBadFileExMsg);
	break;
      case ENOTSOCK:
	msg.base.msg = "uConnect: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	len = sizeof(uNotSockExMsg);
	break;
      case EADDRNOTAVAIL:
	msg.base.msg = "uConnect: cannot use specified address\n";
	except = &uConnFailedEx;
	break;
      case EADDRINUSE:
	msg.base.msg = "uConnect: specified address is already in use\n";
	except = &uConnFailedEx;
	break;
      case EAFNOSUPPORT:
	msg.base.msg = "uConnect: no support for this address family\n";
	except = &uConnFailedEx;
	break;
      case EISCONN:
	msg.base.msg = "uConnect: socket already connected\n";
	except = &uConnFailedEx;
	break;
      case ETIMEDOUT:
	msg.base.msg = "uConnect: connection timeout\n";
	except = &uConnFailedEx;
	break;
      case ECONNREFUSED:
	msg.base.msg = "uConnect: connection rejected\n";
	except = &uConnFailedEx;
	break;
      case ENETUNREACH:
	msg.base.msg = "uConnect: network unreachable\n";
	except = &uConnFailedEx;
	break;
      case U_EWOULDBLK:
	msg.base.msg = "uConnect: non-blocking socket, no connection yet\n";
	except = &uConnFailedEx;
	break;
      default:
	msg.base.msg = "uConnect: unknown socket error\n";
	except = &uConnFailedEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, len );
} /* uConnectError */
#endif

int uConnect( uFile socket, void *name, int namelen ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = connect( socket->file, name, namelen );	/* attempt a nonblocking connect */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the connect was successful, break */
#if defined( __sun__ ) && defined( __svr4__ )
	/* connect works differently on Solaris from all the other BSD socket routines. */
	if ( errno == EINPROGRESS ) {
	    uSelectFile( socket, &(socket->cluster->wfds) ); /* select on write file desc. set */
	    break;
	} /* if */
#endif
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uConnectError( socket, name, namelen );	/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->wfds) ); /* select on write file desc. set */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uConnect */

#ifndef __U_UNIXRC__
static volatile void uListenError( uFile socket ) {
    uSocketErrorExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EOPNOTSUPP:
	msg.msg = "uListen: incorrect type of socket\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uListen: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uListen: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uListen: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uListenError */
#endif

int uListen( uFile socket, int log ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    code = listen( socket->file, log );

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {					/* operation failed */
#ifndef __U_UNIXRC__
	uListenError( socket );				/* does not return */
#endif
    } /* if */
    return code;
} /* uListen */

#ifndef __U_UNIXRC__
static volatile void uAcceptError( uFile socket ) {
    uSocketErrorExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EMFILE:
	msg.msg = "uAccept: Too many files open for this process\n";
	except = &uSocketErrorEx;
	break;
      case ENFILE:
	msg.msg = "uAccept: Too many file open in the system\n";
	except = &uSocketErrorEx;
	break;
      case EFAULT:
	msg.msg = "uAccept: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      case EOPNOTSUPP:
	msg.msg = "uAccept: Socket is not SOCK_STREAM, or incorrect type\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uAccept: Bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOBUFS:
	msg.msg = "uAccept: no buffer space\n";
	except = &uNoBufsEx;
	break;
      case ENOTSOCK:
	msg.msg = "uAccept: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uAccept: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uAcceptError */
#endif

uFile uAccept( uFile socket, void *addr, int *addrlen ) {
    uFile connect;
    int tmpaddrlen = 0, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus1;
    uCluster prevClus2;

    prevClus1 = uThisCluster();
    uFileEnter( socket );
    connect = uCreateFile();				/* create new cluster so UNIX process is child of socket */
    prevClus2 = uThisCluster();
    uFileEnter( connect );

    if ( addrlen ) {					/* save *addrlen, as it gets set to 0 if multiple attempt occur */
	tmpaddrlen = *addrlen;
    } /* if */

    for ( ;; ) {
	if ( addrlen )	{				/* nonblocking retry might have scromped *addrlen */
	    *addrlen = tmpaddrlen;
	} /* if */

	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	connect->file = accept( socket->file, addr, addrlen ); /* attempt a nonblocking accept */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( connect->file != -1 ) break;		/* if the accept was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
	    uFileLeave( connect, prevClus2 );		/* migrate back to previous cluster */
	    uDestroyFile( connect );			/* clean up */
	    uFileLeave( socket, prevClus1 );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
#ifdef __U_UNIXRC__
	    return NULL;
#else
	    uAcceptError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(connect->cluster->rfds) ); /* accept failed to complete */

    } /* for */
    
    connect->block = socket->block;			/* inherit socket blocking attributes */
    
    uSetFileAccess( connect );
    uFileLeave( connect, prevClus2 );			/* migrate back to previous cluster */
    uFileLeave( socket, prevClus1 );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */

    return connect;
} /* uAccept */

#ifndef __U_UNIXRC__
static volatile void uSsendError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uSsend: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uSsend: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EMSGSIZE:
	msg.msg = "uSsend: message to large for atomic send\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uSsend: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uSsendError */
#endif

int uSsend( uFile socket, void *buf, int len, int flags ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = send( socket->file, buf, len, flags );	/* do a non blocking send */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the send was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uSsendError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->wfds) ); /* write failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uSsend */

#ifndef __U_UNIXRC__
static volatile void uSendtoError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uSendto: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uSendto: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EMSGSIZE:
	msg.msg = "uSendto: message to large for atomic send\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uSendto: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uSendtoError */
#endif

int uSendto( uFile socket, void *buf, int len, int flags, void *to, int tolen ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = sendto( socket->file, buf, len, flags, to, tolen ); /* do a non blocking send */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the send was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uSendtoError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->wfds) ); /* write failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uSendto */

#ifndef __U_UNIXRC__
static volatile void uSendmsgError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uSendmsg: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uSendmsg: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EMSGSIZE:
	msg.msg = "uSendmsg: message to large for atomic send\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uSendmsg: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uSendmsgError */
#endif

int uSendmsg( uFile socket, void *msg, int flags ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = sendmsg( socket->file, msg, flags );	/* do a non blocking send */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the send was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uSendmsgError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->wfds) ); /* write failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uSendmsg */

#ifndef __U_UNIXRC__
static volatile void uRecvError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uRecv: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uRecv: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EFAULT:
	msg.msg = "uRecv: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      default:
	msg.msg = "uRecv: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uRecvError */
#endif

int uRecv( uFile socket, void *buf, int len, int flags ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = recv( socket->file, buf, len, flags );	/* do a non blocking receive */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the receive was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uRecvError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->rfds) ); /* receive failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uRecv */

#ifndef __U_UNIXRC__
static volatile void uRecvfromError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uRecvfrom: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uRecvfrom: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EFAULT:
	msg.msg = "uRecvfrom: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      default:
	msg.msg = "uRecvfrom: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uRecvfromError */
#endif

int uRecvfrom( uFile socket, void *buf, int len, int flags, void *from, int *fromlen ) {
    int code, tmpfromlen = 0, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */
    if ( len != 0 ) uTouch( buf + len - 1 );		/* touch end of buffer to ensure PT update */

    if ( fromlen ) {					/* save *addrlen, as it gets set to 0 if multiple attempt occur */
	tmpfromlen = *fromlen;
    } /* if */

    for ( ;; ) {
	if ( fromlen ) {				/* nonblocking retry might have scromped *fromlen */
	    *fromlen = tmpfromlen;
	} /* if */

	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = recvfrom( socket->file, buf, len, flags, from, fromlen ); /* do a non blocking receive */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the receive was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uRecvfromError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->rfds) ); /* receive failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uRecvfrom */

#ifndef __U_UNIXRC__
static volatile void uRecvmsgError( uFile socket ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EBADF:
	msg.msg = "uRecvmsg: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOTSOCK:
	msg.msg = "uRecvmsg: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      case EFAULT:
	msg.msg = "uRecvmsg: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      default:
	msg.msg = "uRecvmsg: unknown error\n";
	except = &uReadWriteEx;
	break;
    }
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uRecvmsgError */
#endif

int uRecvmsg( uFile socket, void *msg, int flags ) {
    int code, terrno;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    for ( ;; ) {
	if ( socket->block == U_FNDELAY_DEMAND ) uSetBlockFlag( socket );
	code = recvmsg( socket->file, msg, flags );	/* do a non blocking receive */
	terrno = errno;
	if ( socket->block == U_FNDELAY_DEMAND ) uClearBlockFlag( socket );
	errno = terrno;

	if ( code != -1 ) break;			/* if the receive was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uFileLeave( socket, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uRecvmsgError( socket );			/* does not return */
#endif
	} /* exit */

	uSelectFile( socket, &(socket->cluster->rfds) ); /* receive failed to complete */

    } /* for */

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uRecvmsg */

#ifndef __U_UNIXRC__
static volatile void uGetSockNameError( uFile socket ) {
    uSocketErrorExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EFAULT:
	msg.msg = "uGetsockname: Address parameter is not writable or not in address space\n";
	except = &uSocketErrorEx;
	break;
      case EBADF:
	msg.msg = "uGetsockname: Bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOBUFS:
	msg.msg = "uGetsockname: no buffer space\n";
	except = &uNoBufsEx;
	break;
      case ENOTSOCK:
	msg.msg = "uGetsockname: this file descriptor is not a socket\n";
	except = &uNotSockEx;
	break;
      default:
	msg.msg = "uGetsockname: unknown socket error\n";
	except = &uSocketErrorEx;
	break;
    } /* switch */    
    uRaise( socket, except, &msg, sizeof(msg) );
} /* uGetSockNameError */
#endif

int uGetsockname( uFile socket, void *name, int *namelen ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uFileEnter( socket );				/* migrate to I/O cluster */

    code = getsockname( socket->file, name, namelen );

    uFileLeave( socket, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {
#ifndef __U_UNIXRC__
	uGetSockNameError( socket );			/* does not return */
#endif
    } /* if */
    return code;
} /* uGetsockname */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
