/*                               -*- Mode: C -*- 
 * uInter.c -- Provides the uSystem with Interventions
 * uSystem Version 4.4.3, Copyright (C) Hamish Macdonald 1991
 * Author          : Hamish Macdonald
 * Created On      : Fri Apr 12 09:30:10 1991
 * Last Modified By: Peter A. Buhr
 * Last Modified On: Fri Jul 17 16:40:56 1992
 * Update Count    : 125
 */

#define __U_KERNEL__

#include "uUnix.h"
#include "uSystem.h"
#include "uMachine.i"
#include "uMalloc.i"

static inline uStackBlock uInterBlock( uInterName *inamep ) {

    /*
     * This function returns a pointer to the top level intervention
     * block for the given intervention, or NULL if there is none
     */

    uCoroutine me = uThisCoroutine();
    uStackBlock block = me->eistack;

    while( block && (block->type != uIsInterBlock || block->u.uIPart.name != inamep) ) {
	block = block->prev;
    } /* while */

    return block;

} /* uInterBlock */

void *uInterFunc( uInterName *inamep ) {
    
    /*
     * This function returns a pointer to the function pointer for
     * the top level intervention block for the given intervention, or
     * raises an exception if there is no block for the intervention.
     */
    
    uStackBlock block;
    uBadInterExMsg msg;
    
    block = uInterBlock( inamep );

    if ( !block ) {
	msg.msg = "uInterCall: No intervention set for intervention.\n";
	msg.iname = *inamep;
	msg.iptr = inamep;
	uRaise( (void*)inamep, &uBadInterEx, &msg, sizeof( msg ) );
    } /* if */
    
    return block->u.uIPart.fp;
    
} /* uInterFunc */

void uSetupInter( uStackBlock block ) {
    
    /*
     * This routine puts the given intervention block at the top of the
     * coroutines handler stack.
     */
    
    uCoroutine me = uThisCoroutine();
    
    block->prev = me->eistack;
    me->eistack = block;

    /* check for async interventions */
    if( me->inters ) {
	uAsyncInterDeliver();
    } /* if */
    
} /* uSetupInter */

void uDeleteInter( uStackBlock block ) {
    
    /*
     * This routine removes the given intervention block from the
     * top of the coroutine's handler stack.
     */
    
    uWorkCoroutine->eistack = block->prev;
    
} /* uDeleteInter */    

/*
 * The following routines are for Asynchronous Interventions
 */

/*
 * Enable remote interventions for this task.
 */
void uAsyncInterEnable( void ) {

    uWorkTask->state &= ~U_DISABLE_INTER;
    
    if( uWorkTask->inters && uWorkTask->icheckneeded ) {
	/* Pending intervention.  execute it */
	uAsyncInterDeliver();
    } /* if */

} /* uAsyncInterEnable */

/*
 * Disable async interventions for for this task.
 */
void uAsyncInterDisable( void ) {

    uWorkTask->state |= U_DISABLE_INTER;

} /* uAsyncInterDisable */


void uAsyncInterCallFunc( uTask task, uInterName *inamep, void *msg, int len ) {
    /*
     * This function causes an intervention routine in the given task to be called.
     * The intervention is that given by the inamep parameter.  The parameters for
     * the intervention routine are msg and len.
     */

    uQdInter desc = NULL;
    int space;

    /*
     * If the given task is the current task, raise an exception.
     */
    if( task == uWorkTask ) {
	/* uRaise( NULL, uBadInterTask, NULL, 0 ); */
	uAbort( "bad task in async intervention\n" );
    } /* if */

    /*
     * Allocate an intervention queue descriptor and enough space to store the message.
     */
    if( msg && len > 0 ) {
	space = U_CEILING( len, sizeof(double) ) + sizeof(double); /* multiplier of sizeof(double) */
    } else {
	space = 0;
    }
    desc = uLowMalloc( space + sizeof(*desc) );
    if( !desc ) {
	/* need to raise exception ? */
	uAbort( "uAsyncInterCall(0x%x,0x%x,0x%x,%d): Memory allocation failed.  Called from 0x%x\n", task,
	       inamep, msg, len, uReadReturnAddress() );
    } /* if */

    *desc = U_QDINTER();
    desc->pinter = inamep;
    desc->interlen = len;
    if( !msg || len <= 0 ) {
	desc->interdata = NULL;
	desc->interlen = 0;
    } else {
	desc->interdata = desc + U_CEILING( sizeof(*desc), sizeof(double) );
	uCopy( msg, desc->interdata, len );
	desc->interlen = len;
    } /* if */
    
    /*
     * Grab task intervention data lock,
     * stuff in the queued intervention
     * descriptor and release the lock.
     */

    uAcquireLock( &(task->interlock) );
    desc->next = task->inters;
    task->inters = desc;
    task->icheckneeded = U_TRUE;
    uReleaseLock( &(task->interlock) );
	
#if 0
    /*
     * This will only expedite intervention delivery, and
     * gives an undesirable overhead in both user and system time.
     */
#ifdef __U_MULTI__
    /*
     * If the task is not on the current cluster, (or maybe regardless),
     * send a SIGALRM bomb to the appropriate cluster.
     * This may be lost.  Async interventions do not have a guaranteed
     * delivery time.  The task may even have migrated away from the cluster.
     */
    
    uSignalCluster( task->cluster, SIGALRM );
#endif
#endif

} /* uAsyncInterCallFunc */

void uAsyncInterDeliver( void ) {

    /*
     * This function will look for an intervention to deliver
     * for the current task.
     */

    uTask me = uThisTask();
    uStackBlockD block;					/* uStackBlockD to save the data */
    uStackBlock bp = me->eistack;
    uQdInter inter, *last;

    /*
     * Grab task intervention data lock
     *
     * This is a teeny bit scary, since the
     * lock is held for a fair bit of time,
     * as a search is made for a handler for 
     * each intervention in the queue (bag?).
     */
    uAcquireLock( &(me->interlock) );
    
    inter = me->inters;
    last = &me->inters;

    while( inter ) {
	bp = uInterBlock( inter->pinter );

	if( bp ) {
	    *last = inter->next;
	    break;
	} /* if */

	inter = inter->next;
	last = &inter->next;
    } /* while */

    /*
     * At this point, inter is either NULL
     * or we have unlinked an intervention
     * to be delivered.
     */

    /*
     * In any case, we have checked the
     * pending interventions so we reset
     * the icheckneeded flag.
     */
    me->icheckneeded = U_FALSE;

    /* release the lock */
    uReleaseLock( &(me->interlock) );
    
    if( !inter ) {
	/*
	 * no intervention routines were
	 * found for any of the pending interventions.
	 */
	return;
    } /* if */

    /* Setup the data block and link it */
    block.type = uIsDataBlock;
    block.prev = me->eistack;
    block.u.data = inter;				/* data to be freed */
    me->eistack = &block;

    /* call the intervention routine */
    (*(uAsyncInterRoutine)bp->u.uIPart.fp)( inter->interdata, inter->interlen );

    /* unlink the stack block */
    me->eistack = block.prev;

    /* free the data */
    uFree( block.u.data );

} /* uAsyncInterDeliver */
    
/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
