/*                               -*- Mode: C -*- 
 *
 * uSystem Version 4.4.3, Copyright (C) Hamish Macdonald 1990
 *
 * uExcept.c -- uSystem exception handling routines.
 *
 * Author          : Hamish Macdonald
 * Created On      : Mon Jun 11 15:15:38 1990
 * Last Modified By: Peter A. Buhr
 * Last Modified On: Sat Feb 20 11:10:37 1993
 * Update Count    : 133
 */

#define  __U_KERNEL__

#include "uUnix.h"
#include "uSystem.h"
#include "uMachine.i"
#include "uQueue.i"
#include "uMalloc.i"

uException uAny = U_EXCEPTION( U_NULL );	                        /* The "catch-all" exception */

void uExceptSave( register uStackBlock block ) {

    /*
     * This function saves the context of a region in which exception
     * handlers are active.  Called by the uExcept macro.
     */

    register uCoroutine cor = uWorkCoroutine;

    block->type = uIsExceptBlock;
    block->prev = cor->eistack;
    cor->eistack = block;
    block->u.uEPart.exception = NULL;
    block->u.uEPart.data = NULL;
    block->u.uEPart.len = 0;
    block->u.uEPart.return_addr = NULL;
    block->u.uEPart.reraise = U_FALSE;
} /* uExceptSave */

void uExceptEnd( register uStackBlock block ) {

    /*
     * This function does the cleanup required at the end of a
     * region in which exception handlers are active.  It is 
     * called if no exceptions were raised, or when a handler 
     * has completed execution.
     */

    if( block->u.uEPart.data ) {
	uFree( block->u.uEPart.data );
	block->u.uEPart.data = NULL;
    } /* if */

    uWorkCoroutine->eistack = block->prev;
} /* uExceptEnd */

int uExceptCheck( uStackBlock block, void* obj, uException *excp, void *dataptr ) {

    /*
     * This function is called by the uWhen macro to check if the user supplied
     * exception pointer in the uWhen macro matches the exception raised.  If so,
     * the data pointer is copied into the user's pointer and a 1 is returned.
     * Otherwise, 0 is returned.
     *
     * The "obj" parameter must either match the "object" pointer in the exception block or
     * be NULL.
     *
     * An exception matches if the exception pointed to by the "block" parameter
     * (the exception actually raised) either is equal to the "excp" parameter, or
     * the "excp" parameter is equal to one of the addresses in the parent chain
     * of the exception pointed to by the "block" parameter.   If the passed
     * exception address is the address of uAny, then it is a match.
     */

    if( obj && obj != block->u.uEPart.object )
	return 0;

    if( excp != &uAny ) {
	register uException *e = block->u.uEPart.exception;

	while( e && e != excp) {
	    e = e->parent;
	} /* while */

	if( !e || e != excp ) {
	    return 0;
	} /* if */
    } /* if */

    if( !dataptr && block->u.uEPart.data ) {		/* Error if there is data but no data pointer supplied */
	uAbort( "uWhen(0x%x,0x%x,0x%x) : NULL DATA POINTER PROVIDED (0x%x,%d)\nException raised from 0x%x\n",
	       obj, excp, dataptr, block->u.uEPart.data,
	       block->u.uEPart.len, block->u.uEPart.return_addr );
    } /* if */

    if( block->u.uEPart.data ) {			/* if there is data, copy the data pointer */
	(*(void**)dataptr) = block->u.uEPart.data;
    } else {					  /* no data, but the user may have supplied pointers */
	if( dataptr ) {				  /* if there is a data pointer, set the data ptr to NULL */
	    (*(void**)dataptr) = NULL;
	} /* if */
    } /* if */

    return 1;
} /* uExceptCheck */

volatile void uRaise( register void* obj,
	    register uException* exception,
	    register void* data,
	    register int len ) {

    /*
     * This function causes an exception to be raised in the current
     * coroutine.  The address of the associated data is given by the
     * first argument, the address of the exception to be raised is given
     * by the second argument, the data to be passed to the handler in the
     * third argument, and the length of the data in the fourth argument.
     */

    register uCoroutine cor = uWorkCoroutine;
    register uStackBlock block = cor->eistack;
    register uStackBlock iblock = block;		/* Initial exception block */

    /* Determine the top level exception block (skip the intervention blocks) */
    while( iblock && iblock->type != uIsExceptBlock ) {
	/* free data if it is a uIsDataBlock */
	if( iblock->type == uIsDataBlock && iblock->u.data ) {
	    uFree( iblock->u.data );
	    iblock->u.data = NULL;
	} /* if */

	iblock = iblock->prev;
    } /* while */

    /*
     * Search for an exception block which
     * has a NULL exception field.
     * If the exception field is non-NULL,
     * it means that the exception block is
     * already handling an exception.
     */
    while( block && (block->type == uIsDataBlock ||
		     block->type == uIsInterBlock || block->u.uEPart.exception) ) {
	switch ( block->type ) {
	  case uIsExceptBlock:
	    /*
	     * If the data pointer is non-null and this is NOT a reraise,
	     * free it, since we are bypassing an active exception block (i.e. we
	     * are executing in a handler for that block) and the exception
	     * data must be freed.
	     */
	    if( block->u.uEPart.data && !block->u.uEPart.reraise ) {
		uFree( block->u.uEPart.data );
		block->u.uEPart.data = NULL;
	    } /* if */
	    break;
	  case uIsDataBlock:
	    if( block->u.data ) {
		uFree( block->u.data );
		block->u.data = NULL;
	    } /* if */
	    break;
	  default:
	    break;
	} /* switch */

	block = block->prev;
    } /* while */

    cor->eistack = block;

    if( block ) {
	/*
	 * Set up the exception address, object address, 
	 * data pointer and length in the
	 * exception block.
	 */
	block->u.uEPart.exception = exception;
	block->u.uEPart.object = obj;

	/*
	 * If iblock (the first exception block on the stack) is not the same as block
	 * and the reraise field of iblock is non-zero, then this is a reraise done by the
	 * uExceptEnd macro.  The data pointer is copied from the old block to the new block,
	 * and no new memory is allocated.  The return address (where uRaise was called from) is
	 * also propagated from the exception block.
	 */
	if( block != iblock && iblock && iblock->u.uEPart.reraise ) {
	    block->u.uEPart.data = data;
	    block->u.uEPart.len = len;
	    block->u.uEPart.return_addr = iblock->u.uEPart.return_addr;
	} else {
	    /*
	     * Must allocate data area for exception data (but only if passed data is non-null).
	     */
	    if( data && len > 0 ) {
		block->u.uEPart.data = uLowMalloc( len );
		if( block->u.uEPart.data ) {
		    uCopy( data, block->u.uEPart.data, len );
		} else {
		    /* need to raise exception ? */
		    uAbort( "uRaise(0x%x,0x%x,0x%x,%d): Memory allocation failed.  Called from 0x%x\n", obj,
			   exception, data, len, uReadReturnAddress() );
		} /* if */
	    } else {
		block->u.uEPart.data = NULL;
	    }
	    block->u.uEPart.len = len;
	    block->u.uEPart.return_addr = uReadReturnAddress();
	}

	_longjmp( block->u.uEPart.context, 1 );
    }

    /*
     * no exception handler for this exception
     *
     * Print out custom message if a predefined uSystem exception,
     * or generic one if an unknown exception.
     */
    if( !uBadException( exception, data ) ) {
	/*
	 * The return address used is the one from the initial exception block
	 * if this was a reraise by the uExceptEnd macro.
	 */
	uAbort( "uRaise(0x%x,0x%x,0x%x,%d): NO EXCEPTION HANDLER.  Called from 0x%x\n", obj, exception, data, len,
	       (iblock && iblock->u.uEPart.reraise) ? iblock->u.uEPart.return_addr : uReadReturnAddress() );
    } /* if */
} /* uRaise */

int uBadException( uException* except, void *data ) {

    /*
     * This function returns U_FALSE if the exception
     * is not a uSystem predefined exception
     */

    /*
     * Check if it is one of the uSystem defined exceptions.  If it is, then the message will be displayed and execution aborted.
     */
    if( except == &uSystemEx ||
       except == &uSystemEx ||
       except == &uCreationEx ||
       except == &uActiveTasksEx ) {
	uSystemExMsg msg = (uSystemExMsg)data;

	uAbort( msg );
    } else if ( except == &uDataCommEx ) {
	uDataCommExMsg *msg = (uDataCommExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uDataCommEx( 0x%x, %d, 0x%x, %d ) : ERROR IN DATA COMMUNICATION.\n",
	       msg->sbuf, msg->slen, msg->rbuf, msg->rlen );
    } else if ( except == &uDataCopyEx ) {
	uDataCopyExMsg *msg = (uDataCopyExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uDataCopyEx( 0x%x, %d, 0x%x, %d ) : ERROR IN DATA COPYING FROM DATA SENDER TO DATA RECEIVER.\n",
	       msg->sbuf, msg->slen, msg->rbuf, msg->rlen );
    } else if ( except == &uSendMsgTooLongEx ) {
	uSendMsgTooLongExMsg *msg = (uSendMsgTooLongExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uSendMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM SENDING TASK: %s.\n",
	       uGetName(msg->receiver), msg->base.sbuf, msg->base.slen, msg->base.rbuf, msg->base.rlen,
	       uGetName(msg->sender) );
    } else if ( except == &uAbsorbAreaTooShortEx ) {
	uAbsorbAreaTooShortExMsg *msg = (uAbsorbAreaTooShortExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uAbsorbAreaTooShortEx(%s, 0x%x, %d, 0x%x, %d) : ABSORB MESSAGE BUFFER IS TOO SHORT TO RECEIVE MESSAGE FROM ABSORBING TASK: %s.\n",
	       uGetName(msg->dier), msg->base.sbuf, msg->base.slen, msg->base.rbuf, msg->base.rlen,
	       uGetName(msg->absorber) );
    } else if ( except == &uForwardMsgTooLongEx ) {
	uForwardMsgTooLongExMsg *msg = (uForwardMsgTooLongExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uForwardMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE TOO LONG TO BE FORWARDED FROM TASK: %s BY FORWARDING TASK: %s.\n",
	       uGetName(msg->receiver), msg->base.sbuf, msg->base.slen, msg->base.rbuf, msg->base.rlen,
	       uGetName(msg->sender), uGetName(msg->forwarder) );
    } else if ( except == &uReplyAreaTooShortEx ) {
	uReplyAreaTooShortExMsg *msg = (uReplyAreaTooShortExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uReplyAreaTooShortEx(%s, 0x%x, %d) : REPLY MESSAGE BUFFER IS TOO SHORT TO TAKE MESSAGE FROM REPLYING TASK: %s.\n",
	       uGetName(msg->sender), msg->base.rbuf, msg->base.rlen, uGetName(msg->replier) );
    } else if ( except == &uSuspendMsgTooLongEx ) {
	uSuspendMsgTooLongExMsg *msg = (uSuspendMsgTooLongExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uSuspendMsgTooLongEx(0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM SUSPENDING COROUTINE: %s.\n",
	       msg->base.sbuf, msg->base.slen, msg->base.rbuf, msg->base.rlen, uGetName(msg->restarter) );
    } else if ( except == &uResumeMsgTooLongEx ) {
	uResumeMsgTooLongExMsg *msg = (uResumeMsgTooLongExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uResumeMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM RESUMING COROUTINE: %s.\n",
	       uGetName(msg->resumed), msg->base.sbuf, msg->base.slen, msg->base.rbuf, msg->base.rlen, uGetName(msg->restarter) );
    } else if ( except == &uSyncFailEx ) {
	uSyncFailExMsg *msg = (uSyncFailExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uSyncFailEx : SYNCHRONIZATION FAILED BETWEEN TASK %s and TASK %s.\n",
	       uGetName(msg->sender), uGetName(msg->receiver) );
    } else if ( except == &uNotReplyBlockedEx ) {
	uNotReplyBlockedExMsg *msg = (uNotReplyBlockedExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uNotReplyBlockedEx(%s, 0x%x, %d) : REPLYING TASK: %s HAS NOT RECEIVED A MESSAGE FROM SENDER %s TASK OR IT HAS ALREADY REPLIED TO THAT TASK.\n",
	       uGetName(msg->receiver), msg->base.sbuf, msg->base.slen, uGetName(msg->receiver), uGetName(msg->sender) );
    } else if ( except == &uInvalidForwardEx ) {
	uInvalidForwardExMsg *msg = (uInvalidForwardExMsg*)data;

	uFputs( msg->base.base.msg, uStderr );
	uAbort( "uInvalidForwardEx(%s, 0x%x, %d, %s) : FORWARDING TASK: %s DID NOT RECEIVE A MESSAGE FROM SPECIFIED RECEIVER TASK OR IT HAS ALREADY FORWARDED THE MESSAGE.\n",
	       uGetName(msg->base.receiver), msg->base.base.sbuf, msg->base.base.slen, uGetName(msg->base.sender), uGetName(msg->forwarder) );
    } else if ( except == &uCreateClusterEx ) {
	uCreateClusterExMsg *msg = (uCreateClusterExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uCreateClusterEx(%ld, %ld, %ld, %ld, %ld) : CANNOT CREATE CLUSTER.\n",
	       msg->cv.Processors, msg->cv.TimeSlice, msg->cv.Spin, msg->cv.StackSize, msg->cv.ArgLen );
    } else if ( except == &uEmitEx ) {
	uEmitExMsg *msg = (uEmitExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uEmitEx(0x%x, %ld, 0x%x, %ld) : CANNOT EMIT TASK.\n", msg->cluster, msg->space, msg->begin,
	       msg->arglen );
    } else if ( except == &uCocallEx ) {
	uCocallExMsg *msg = (uCocallExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uCocallEx(0x%x, %d, %ld, 0x%x, %ld) : CANNOT COCALL COROUTINE.\n", msg->rbuf, msg->rlen,
	       msg->space, msg->begin, msg->arglen );
    } else if ( except == &uCreateProcessorEx ) {
	uCreateProcessorExMsg *msg = (uCreateProcessorExMsg*)data;

	uFputs( msg->msg, uStderr );
	uAbort( "uCreateProcessorEx() : COULDN'T CREATE PROCESSORS.  CLUSTER CURRENTLY HAS %d PROCESSORS.\n", msg->num_proc );
    } else if ( except == &uBadCoroutineEx ) {
	uBadCoroutineExMsg *msg = (uBadCoroutineExMsg*)data;

	uFputs( msg->base.msg, uStderr );
	uAbort( "uBadCoroutineEx(%s, 0x%x, %d, 0x%x, %d) : ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY TASK: %s.\n    OWNER TASK OF COROUTINE TO BE RESUMED: %s.\n",
	       uGetName(msg->restarter), msg->base.rbuf, msg->base.rlen, msg->base.sbuf, msg->base.slen,
	       uGetName(msg->thistask), uGetName(msg->restarter->owner));
    } else if ( except == &uBadInterEx ) {
	uBadInterExMsg* msgp = (uBadInterExMsg*)data;

	uFputs( msgp->msg, uStderr );
	uAbort( "uBadInterEx( %s, 0x%0x )\n", msgp->iname, msgp->iptr );
    } else if ( except == &uOutOfMemoryEx ) {
	uOutOfMemoryExMsg* msgp = (uOutOfMemoryExMsg*)data;

	uFputs( msgp->msg, uStderr );
	uAbort( "uOutOfMemoryEx( %d )\n", msgp->size );
    } else if ( except == &uNoExtendEx ) {
	uNoExtendExMsg* msgp = (uNoExtendExMsg*)data;

	uFputs( msgp->base.msg, uStderr );
	uAbort( "uNoExtendEx( %d, 0x%x)\n", msgp->base.size, msgp->addr );
    } else if( except == &uIOEx ) {
	uIOExMsg* msgp = (uIOExMsg*)data;

	uAbort( msgp->msg );
    } else if ( except == &uEofEx ) {
	uEofExMsg* msgp = (uEofExMsg*)data;

	uAbort( msgp->msg );
    } else if ( except == &uIOErrorEx ||
	       except ==    &uSocketErrorEx ||
	       except ==      &uNotSockEx ||
	       except ==      &uNoBufsEx ||
	       except ==    &uReadWriteEx ||
	       except ==      &uIOFailedEx ||
	       except ==      &uNoSpaceEx ||
	       except ==      &uBadFileEx ) {
	uIOErrorExMsg* msgp = (uIOErrorExMsg*)data;

	uFprintf( uStderr, "IO error with Unix errno %d\n", msgp->errno );
	uAbort( msgp->msg );
    } else if ( except == &uCreateSockEx ) {
	uCreateSockExMsg* msgp = (uCreateSockExMsg*)data;

	uFputs( msgp->base.msg, uStderr );
	uAbort( "uCreateSockEx( %d, %d, %d ) with Unix errno %d\n", msgp->af, msgp->type, msgp->protocol,
	       msgp->base.errno );
    } else if ( except == &uBadSockAddressEx ) {
	uBadSockAddressExMsg* msgp = (uBadSockAddressExMsg*)data;

	uFprintf( uStderr, "uBadSockAddressEx( 0x%x, %d )\n", msgp->name, msgp->namelen );
	uAbort( msgp->base.msg );
    } if ( except == &uConnFailedEx ) {
	uConnFailedExMsg* msgp = (uConnFailedExMsg*)data;

	uFprintf( uStderr, "uConnFailedEx( 0x%x, %d )\n", msgp->name, msgp->namelen );
	uAbort( msgp->base.msg );
    } else if ( except == &uOpenEx ||
	       except ==    &uOpenIOFailedEx ||
	       except ==    &uOpenNoSpaceEx ||
	       except ==    &uBadPathEx ||
	       except ==    &uNoPermsEx ||
	       except ==    &uNoFilesEx ||
	       except ==    &uBadParmEx ) {
	uOpenExMsg* msgp = (uOpenExMsg*)data;

	uFprintf( uStderr, "%sIO error with Unix errno %d\n", msgp->base.msg, msgp->base.errno );
	uAbort( "uOpenEx( %s, %s, %d, %d )\n", msgp->path, msgp->perms ? msgp->perms : "(nil)", msgp->flags, msgp->mode );
    } /* if */

    return U_FALSE;
} /* uBadException */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
