/*
  Producer-consumer problem with a bounded buffer
  
  Two tasks communicate through a unidirectional queue of finite
  length. Two semaphores are used to ensure that should the queue
  fill, the producer will wait until some free queue element appears,
  and should the queue empty, the consumer will wait until a queue
  element appears.
  */

#include <uSystem.h>

int abs( int );
long random( void );

/* shared variables for process communication */

#define QueueSize 10

struct shrqueue {
    int front;						/* position of front of queue */
    int back;						/* position of back of queue */
    int queue[QueueSize];				/* queue of integers */
    uSemaphore full, empty;				/* communication semaphores */
    uSemaphore ilock, rlock;				/* lock queue semaphore */
}; /* shrqueue */

void qinsert( struct shrqueue *q, int elem ) {
    uP( &(q->empty) );					/* wait if queue is full */

    uP( &(q->ilock) );					/* lock queue modification */
    q->queue[q->back] = elem;				/* insert element in queue */
    q->back = ( q->back + 1 ) % QueueSize;
    uV( &(q->ilock) );

    uV( &(q->full) );					/* signal consumer */
} /* qinsert */

int qremove( struct shrqueue *q ) {
    int elem;
    
    uP( &(q->full) );					/* wait for producer */

    uP( &(q->rlock) );					/* lock queue modification */
    elem = q->queue[q->front];				/* remove element from queue */
    q->front = ( q->front + 1 ) % QueueSize;
    uV( &(q->rlock) );

    uV( &(q->empty) );					/* signal producer about empty queue space */
    return( elem );
} /* qremove */

void Producer( struct shrqueue *q ) {
    int i, N;
    
    for ( i = 1; i <= 50; i += 1 ) {			/* produce a bunch of items */
        N = ( abs( random( ) ) % 100 + 1 ) + N;		/* generate random product */
        uPrintf( " Producer:%x, value:%d\n", uThisTask( ), N );
        qinsert( q, N );				/* insert at back of queue */
    } /* for */
    N = -1;						/* terminate with negative value */
    uPrintf( " Producer:%x, value:%d\n", uThisTask( ), N );
    qinsert( q, N );
    uDie( NULL, 0 );
} /* Producer */

void Consumer( struct shrqueue *q ) {
    int N;
    
    /* consume until a negative element appears */
    
    do {
        N = qremove( q );				/* remove from front of queue */
        uPrintf( "Consumer :%x, value:%d\n", uThisTask( ), N );
    } while ( N > 0 );
    uDie( NULL, 0 );
} /* Consumer */

void uMain( ) {
    uTask t0, t1;
    struct shrqueue queue;

    queue.full = U_SEMAPHORE( 0 );
    queue.empty = U_SEMAPHORE( QueueSize );
    queue.ilock = U_SEMAPHORE( 1 );
    queue.rlock = U_SEMAPHORE( 1 );
    queue.front = queue.back = 0;
    
    t0 = uEmit( Producer, &queue );			/* start producer */
    t1 = uEmit( Consumer, &queue );			/* start consumer */
    
    uAbsorb( t0, NULL, 0 );				/* wait for completion of process */
    uAbsorb( t1, NULL, 0 );				/* wait for completion of process */
    uPrintf( "successful completion\n" );
} /* uMain */
