/* Monitor Program Example 5 */
/* Copyright (c) 1986 P. A. Buhr */

/*
	Dining Philosophers Problem

	Philosophers are eating around a dining table; however, the table is
	set so that there is only one chop stick between each philosopher. Since
	two chop sticks are necessary to eat, not all philosophers can eat at the
	same time. Fortunately, a philosopher is not always eating, he may be
	just hungry or thinking. Hence, a philosopher can be eating when the
	philosophers on the left and right are not eating.
*/

#include <uMonitor.h>

#define NoOfPhil 5
#define NoOfChips 100

uMonitor( PriorityAutomaticSignal ) {
#   define THINKING 0
#   define HUNGRY 1
#   define EATING 2

    uLocal int RightOf( int me ) {
	return( ( me + 1 ) % NoOfPhil );
    } /* RightOf */

    uLocal int LeftOf( int me ) {
	return( ( me != 0 ) ? me - 1 : NoOfPhil - 1 );
    } /* LeftOf */

    int state[NoOfPhil] = { THINKING, THINKING, THINKING, THINKING, THINKING };

    uLocal void TestBeside( int me ) {
	if ( state[LeftOf( me )] != EATING && state[me] == HUNGRY && state[RightOf( me )] != EATING ) {
	    state[me] = EATING;
	} /* if */
    } /* TestBeside */

    uEntry void pickup( int me ) {
	state[me] = HUNGRY;
	TestBeside( me );
	uWaitUntil state[me] == EATING;
    } /* pickup */

    uEntry void putdown( int me ) {
	state[me] = THINKING;
	TestBeside( LeftOf( me ) );
	TestBeside( RightOf( me ) );
    } /* putdown */
}

void philosopher( int i, int chips ) {
    int bite;
    
    for ( ;; ) {
	pickup( i );					/* pick up both chop sticks */
	bite = random(  ) % 5 + 1;			/* take a bite */
    if ( bite >= chips ) {				/* finished eating ? */
	    uPrintf( "philosopher%d finished eating the last %d chips\n", i, chips );
	    putdown( i );				/* don't forget to put down the chop sticks before leaving */
	    break;
	} /* exit */
	chips -= bite;					/* reduce chips */
	uPrintf( "philosopher%d is eating %d chips leaving %d chips\n", i, bite, chips );
	putdown( i );					/* put down both chop sticks */
    } /* for */
    uDie( NULL, 0 );
} /* philosopher */

void uMain( ) {
    int i;
    uTask phil[NoOfPhil];

    for ( i = 0; i < NoOfPhil; i += 1 ) {
	phil[i] = uEmit( philosopher, i, NoOfChips );
    } /* for */

    for ( i = 0; i < NoOfPhil; i += 1 ) {
	uAbsorb( phil[i], NULL, 0 );
    } /* for */
    uPrintf( "successful completion\n" );
} /* uMain */
