//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Robert Denda 1997
// 
// uProfiler.cc -- 
// 
// Author           : Robert Denda
// Created On       : Tue Jul 16 16:45:16 1996
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Nov 17 06:37:17 1998
// Update Count     : 463
// 

#ifndef __U_PROFILER_H__
#define __U_PROFILER_H__

#include <uC++.h>

#include "uProfileTaskSplr.h"
#include "uExecutionMonitor.h"

#define _U_BOOT_TASK_PROFILE_STACK_SIZE 1024		// should be more than enough !!!

class uProfilerStartWidget;
class uProfileAnalyze;
class uSymbolTable;
uTask uProfilerClock;
class uTaskSampler;
class uCallGraphInfo;

extern "C" unsigned int uFunctionPrologue( register char*, register unsigned short *, register unsigned short *);
extern "C" void uFunctionEpilogue();


//######################### uProfileEvent #########################


class uProfileEvent : public uSeqable {
    friend uTask uProfiler;				// only profiler can access member routines
    friend class uProfileMonitorSet;
    
    uExecutionMonitorNode &monitor;
    uTime time;
  public:    
    uProfileEvent( uExecutionMonitorNode &monitor) : monitor ( monitor ) {
	time = uThisProcessor().uGetClock().uGetTime();
    } // uProfileEvent
}; // uProfileEvent


//######################### uProfileMonitorSet #########################


class uProfileMonitorSet : private uSequence<uProfileEvent> {
  public:

    uProfileMonitorSet() {
    } // uProfileMonitorSet:uProfileMonitorSet

    virtual ~uProfileMonitorSet() {
    } // uProfileMonitorSet:~uProfileMonitorSet

    void OrderedInsert( uProfileEvent *np ) {
	uProfileEvent *lp;
	for ( lp = uHead(); lp != 0 && lp->time < np->time; lp = uSucc( lp ) ); // insert in ascending order by event time
	uInsertBef( np, lp );
    } // uProfileMonitorSet::OrderedInsert

    uProfileEvent *Head() {
	return uHead();
    } // uProfileMonitorSet::Head

    uProfileEvent *Succ( uProfileEvent *e ) {
	return uSucc( e );
    } // uProfileMonitorSet::Succ

    void Reorder( uProfileEvent *np ) {
	if ( uTail() != uHead() ) {			// if there is only one monitor in list then don't reorder
	    uRemove( np );
	    OrderedInsert( np );
	} // if
    } // uProfileMonitorSet::Reorder
}; // uProfileMonitorSet


//######################### uProfiler ##########################


uTask uProfiler {
    friend uTask uProcessorTask;
    friend uCoroutine uProcessorKernel;			           
    friend class uProfilerBoot;				           
    friend class uExecutionMonitor;
    friend class uProfileTaskSampler;			// access: addFuncCallInfo
    friend class uProfileAnalyze;			// access: symbolTable, InfoList
    friend class uProfilerStartWidget;
    friend class uCallGraphAnalyze;
    friend class uAnalyzeFuncCallTableWidget;		// access: analyzer 
    friend class uAnalyzeFuncCallTable;			// access: analyzer
    friend class uCGAnalyzeWidget;
    friend class uSPAnalyze;
    friend class uSPAnalyzeWidget;
    friend class uSPClusterAnalyze;
    friend class uSPClusterAnalyzeWidget;
    friend class uRUAnalyze;
    friend class uRUAnalyzeWidget;
    friend class uRUProcessorAnalyze;
    friend class uRUProcessorAnalyzeWidget;
    friend unsigned int uFunctionPrologue( register char*, register unsigned short *, register unsigned short *);
    friend void uFunctionEpilogue();
    friend class uTaskProfileInfo;
    friend uTask uProfilerClock;			           // access: cluster
#ifdef __sparc__
    friend void uMcount();
#else
    friend void mcount();
#endif __sparc__    
    friend class uBaseTask;				           // access: uProfilerInstance
    friend class uBaseCoroutine;			           // access: uProfilerInstance
    friend class uCoroutineConstructor;			           // access: uProfilerInstance
    friend class uTaskConstructor;			           // access: uProfilerInstance
    friend class uSerialConstructor;			           // access: uProfilerInstance
    friend class uCondition;					   // access: uProfilerInstance                    
    friend class uSigHandlerModule;				   // access: uProfilerInstance
    friend class uSerial;					   // access: uProfilerInstance
    friend class uSerialDestructor;				   // access: uProfilerInstance    
    friend class uSerialMember;					   // access: uProfilerInstance
    
    static uProfiler *uProfilerInstance;		           // pointer to profiler
    static bool uProfilerBootDone;				   // flag indicating if uProfiler boot process is complete
    static unsigned int uCurrentTaskNum;			   // number of currently executing tasks				  
    static unsigned int uProfileInHeapExtendCount;		   // counter needed to check if inside serial member
    
    unsigned int currentTaskNum;			           // number of currently profiled tasks 
    char            *filename;				           // name of executable
    
    uProfileMonitorSet     monitorSet;				   // list of poll events for samplers 
    uSequence<uProfileTaskSampler> samplerList;			   // list of profiling data structures

    uCluster        cluster;				           // own cluster
    uProcessor      processor;                                     // own processor
    uProfileAnalyze *analyzer;                                     // profile analyzer, invoked at the end of profiling
    uProfilerClock  *alarmClock;			           // alarm clock for profiler
    uSymbolTable    *symbolTable;                                  // symbol table of the executable

    uLock           lock;
    bool            finish;
    
    // communication 

    uCallGraphInfo        *RemovableTaskcallGraphInfo;
    const uBaseTask       *taskToProcess;
    const uBaseCoroutine  *coroutineToProcess;
    const uProcessor      *processorToProcess;
    const uCluster        *cluster1;
    const uCluster        *cluster2;
    const uSerial         *serial;
    const uCondition      *condition;
    unsigned int          RegisterFunctionEntryCurrentFunction;
    unsigned int          RegisterFunctionEntryCallingFunction;
    unsigned int          firstArgument;
	
    // monitoring modules

    mutable uSequence<uExecutionMonitorNode> &MonitorList;	// list of all selected profiling monitors 
    mutable uSequence<uExecutionMonitorNode> &RegisterTaskMonitorList;  // list of profiling monitors to be informed upon task registration
    mutable uSequence<uExecutionMonitorNode> &DeregisterTaskMonitorList; // list of profiling monitors to be informed upon task deregistration
    mutable uSequence<uExecutionMonitorNode> &RegisterProcessorMonitorList; // list of profiling monitors to be informed upon processor registr.
    mutable uSequence<uExecutionMonitorNode> &DeregisterProcessorMonitorList;  // list of profiling monitors to be informed upon processor dereg.
    mutable uSequence<uExecutionMonitorNode> &MigrateTaskMonitorList;          // list of profiling monitors to be informed upon task migration
    mutable uSequence<uExecutionMonitorNode> &MigrateProcessorMonitorList;     // list of profiling monitors to be informed upon processor migration
    mutable uSequence<uExecutionMonitorNode> &RegisterFunctionEntryMonitorList;// list of profiling monitors to be informed upon routine entry
    mutable uSequence<uExecutionMonitorNode> &RegisterFunctionExitMonitorList;// list of profiling monitors to be informed upon routine exit
    mutable uSequence<uExecutionMonitorNode> &RegisterMutexFunctionEntryTryMonitorList;// list of profiling monitors to be informed upon mutex routine entry petition
    mutable uSequence<uExecutionMonitorNode> &RegisterMutexFunctionEntryDoneMonitorList;// list of profiling monitors to be informed upon mutex routine entry 
    mutable uSequence<uExecutionMonitorNode> &RegisterMutexFunctionExitMonitorList;// list of profiling monitors to be informed upon mutex routine exit
    mutable uSequence<uExecutionMonitorNode> &RegisterMonitorMonitorList;// list of profiling monitors to be informed upon monitor creation
    mutable uSequence<uExecutionMonitorNode> &DeregisterMonitorMonitorList;// list of profiling monitors to be informed upon monitor creation    
    mutable uSequence<uExecutionMonitorNode> &RegisterAcceptStartMonitorList;// list of profiling monitors to be informed upon accept start
    mutable uSequence<uExecutionMonitorNode> &RegisterAcceptEndMonitorList;// list of profiling monitors to be informed upon accept end
    mutable uSequence<uExecutionMonitorNode> &RegisterSuspendMonitorList;// list of profiling monitors to be informed upon coroutine suspend
    mutable uSequence<uExecutionMonitorNode> &RegisterResumeMonitorList;// list of profiling monitors to be informed upon coroutine resume
    mutable uSequence<uExecutionMonitorNode> &RegisterCoroutineMonitorList;// list of profiling monitors to be informed upon coroutine creation
    mutable uSequence<uExecutionMonitorNode> &DeregisterCoroutineMonitorList;// list of profiling monitors to be informed upon coroutine deletion
    mutable uSequence<uExecutionMonitorNode> &RegisterWaitMonitorList;// list of profiling monitors to be informed upon wait on condition
    mutable uSequence<uExecutionMonitorNode> &RegisterSignalMonitorList;// list of profiling monitors to be informed upon signal on condition
    mutable uSequence<uExecutionMonitorNode> &PollMonitorList;		       // list of profiling monitors to be periodically polled
    
    void main();

    void AddToMonitorList( uExecutionMonitor *mon ) const;

    uMutex void RegisterTaskMutex( uBaseTask &task );
    uMutex void DeregisterTaskMutex( uBaseTask &task );
    uMutex void RegisterProcessorMutex( const uProcessor &processor );
    uMutex void DeregisterProcessorMutex( const uProcessor &processor );
    uMutex void MigrateTaskMutex( const uBaseTask &task, const uCluster &fromCluster, const uCluster &toCluster );
    uMutex void MigrateProcessorMutex( const uProcessor &processor, const uCluster &fromCluster, const uCluster &toCluster );
    uMutex void RegisterFunctionEntryMutex( uBaseTask *task, unsigned int pcCallingFunction, unsigned int pcCurrentFunction, unsigned int firstArgument );
    uMutex void RegisterFunctionExitMutex( uBaseTask *task );
    uMutex void RegisterMutexFunctionEntryTryMutex( const uSerial &serial, const uBaseTask &task );
    uMutex void RegisterMutexFunctionEntryDoneMutex( const uSerial &serial, const uBaseTask &task );    
    uMutex void RegisterMutexFunctionExitMutex( const uSerial &serial, const uBaseTask &task );
    uMutex void RegisterMonitorMutex( const uSerial &serial, const uBaseTask &task );
    uMutex void DeregisterMonitorMutex( const uSerial &serial, const uBaseTask &task );    
    uMutex void RegisterAcceptStartMutex( const uSerial &serial, const uBaseTask &task );
    uMutex void RegisterAcceptEndMutex( const uSerial &serial, const uBaseTask &task );
    uMutex void RegisterSuspendMutex( const uBaseCoroutine &coroutine, const uBaseTask &task );
    uMutex void RegisterResumeMutex( const uBaseCoroutine &coroutine, const uBaseTask &task );
    uMutex void RegisterWaitMutex( const uCondition &condition, const uBaseTask &task );
    uMutex void RegisterSignalMutex( const uCondition &condition, const uBaseTask &task );    
    uMutex void RegisterCoroutineMutex( const uBaseCoroutine &coroutine );
    uMutex void DeregisterCoroutineMutex( const uBaseCoroutine &coroutine );

    uMutex void Finish();

    void RegisterTaskNotifyRequest( uExecutionMonitor * const mon ) const;
    void DeregisterTaskNotifyRequest( uExecutionMonitor * const mon ) const;
    void RegisterProcessorNotifyRequest( uExecutionMonitor * const mon ) const;
    void DeregisterProcessorNotifyRequest( uExecutionMonitor * const mon ) const;
    void MigrateTaskNotifyRequest( uExecutionMonitor * const mon ) const;
    void MigrateProcessorNotifyRequest( uExecutionMonitor * const mon ) const;
    void RegisterFunctionEntryNotifyRequest( uExecutionMonitor * const mon ) const ;
    void RegisterFunctionExitNotifyRequest( uExecutionMonitor * const mon ) const ;    
    void RegisterMutexFunctionEntryTryNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterMutexFunctionEntryDoneNotifyRequest( uExecutionMonitor *const mon ) const;    
    void RegisterMutexFunctionExitNotifyRequest( uExecutionMonitor *const mon ) const;    
    void RegisterMonitorNotifyRequest( uExecutionMonitor *const mon ) const;
    void DeregisterMonitorNotifyRequest( uExecutionMonitor *const mon ) const;
    void PollNotifyRequest( uExecutionMonitor * const mon ) const;
    void RegisterAcceptStartNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterAcceptEndNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterSuspendNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterResumeNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterWaitNotifyRequest( uExecutionMonitor *const mon ) const;
    void RegisterSignalNotifyRequest( uExecutionMonitor *const mon ) const;    
    void RegisterCoroutineNotifyRequest( uExecutionMonitor * const mon ) const;
    void DeregisterCoroutineNotifyRequest( uExecutionMonitor * const mon ) const;
  public:
    uProfiler( char *filename );                                   // constructor
    ~uProfiler();                                                  // destructor

    uNoMutex const uSymbolTable *GetSymbolTable() const { return symbolTable; } // doesn't have to be mutex
    uNoMutex const uProfileAnalyze *GetAnalyzer() const { return analyzer; }
    
    void WakeUp();
    void Synchronize();

    uNoMutex uCluster &GetCluster() const { return const_cast<uCluster &>( cluster ); }

    uNoMutex void RegisterTask( uBaseTask &task ) {
	if ( ! task.uProfileTaskSamplerInstance ) {
	    task.uProfileTaskSamplerInstance = new uProfileTaskSampler( *this, task );
	    lock.uAcquire();
	    samplerList.uAdd( task.uProfileTaskSamplerInstance );
	    lock.uRelease();
	} // if

	if ( !RegisterTaskMonitorList.uEmpty() ) {
	    RegisterTaskMutex( task );
	} // if
	lock.uAcquire();
	currentTaskNum += 1;
	lock.uRelease();
    } // uProfiler::RegisterTask

    uNoMutex void DeregisterTask( uBaseTask &task ) {
	if ( !DeregisterTaskMonitorList.uEmpty() ) {
	    DeregisterTaskMutex( task );
	} // if
	lock.uAcquire();
	currentTaskNum -= 1;
	if ( currentTaskNum == 1 ) {			// only the boot task is executing
	    finish = true;
	    Finish();
	} // if
	lock.uRelease();
    } // uProfiler::DeregisterTask
    
    uNoMutex void RegisterProcessor( const uProcessor &processor ) {
	if ( !RegisterProcessorMonitorList.uEmpty() ) {
	    RegisterProcessorMutex( processor );
	} // if
    } // uProfiler::RegisterProcessor
    
    uNoMutex void DeregisterProcessor( const uProcessor &processor ) {
	if ( !DeregisterProcessorMonitorList.uEmpty() ) {
	    DeregisterProcessorMutex( processor );
	} // if
    } // uProfiler::DeregisterProcessor
    
    uNoMutex void MigrateTask( const uBaseTask &task, const uCluster &fromCluster, const uCluster &toCluster ) {
	if ( !MigrateTaskMonitorList.uEmpty() ) {
	    MigrateTaskMutex( task, fromCluster, toCluster );
	} // if
    } // uProfiler::MigrateTask
    
    uNoMutex void MigrateProcessor( const uProcessor &processor, const uCluster &fromCluster, const uCluster &toCluster ) {
	if ( !MigrateProcessorMonitorList.uEmpty() ) {
	    MigrateProcessorMutex( processor, fromCluster, toCluster );
	} // if
    } // uProfiler::MigrateProcessor
    
    uNoMutex void RegisterFunctionEntry( uBaseTask *task, unsigned int pcCallingFunction, unsigned int pcCurrentFunction, unsigned int firstArgument ) {
	if ( !RegisterFunctionEntryMonitorList.uEmpty() ) {
	    RegisterFunctionEntryMutex( task, pcCallingFunction, pcCurrentFunction, firstArgument );
	} // if
    } // uProfiler::RegisterFunctionEntry

    uNoMutex void RegisterFunctionExit( uBaseTask *task ) {
	if ( !RegisterFunctionExitMonitorList.uEmpty() ) {
	    RegisterFunctionExitMutex( task );
	} // if
    } // uProfiler::RegisterFunctionEntry

    uNoMutex void RegisterMutexFunctionEntryTry( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( &task == this ) return;			// do not profile uProfiler
	if ( uKernelModule::uDisableExactProfilingCnt != 0 ) return;
	if ( !RegisterMutexFunctionEntryTryMonitorList.uEmpty() ) {
	    RegisterMutexFunctionEntryTryMutex( serial, task );
	} // if
    } // uProfiler::RegisterMutexFunctionEntryTry

    uNoMutex void RegisterMutexFunctionEntryDone( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( &task == this ) return;			// do not profile uProfiler
	if ( uKernelModule::uDisableExactProfilingCnt != 0 ) return;
	if ( !RegisterMutexFunctionEntryDoneMonitorList.uEmpty() ) {
	    RegisterMutexFunctionEntryDoneMutex( serial, task );
	} // if
    } // uProfiler::RegisterMutexFunctionEntryDone

    uNoMutex void RegisterMutexFunctionExit( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( &task == this ) return;			// do not profile uProfiler
	if ( uKernelModule::uDisableExactProfilingCnt != 0 ) return;
	if ( !RegisterMutexFunctionExitMonitorList.uEmpty() ) {
	    RegisterMutexFunctionExitMutex( serial, task );
	} // if
    } // uProfiler::RegisterMutexFunctionExit

    uNoMutex void RegisterMonitor( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( !RegisterMonitorMonitorList.uEmpty() ) {
	    RegisterMonitorMutex( serial, task );
	} // if
    } // uProfiler::RegisterMonitor

    uNoMutex void DeregisterMonitor( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( !DeregisterMonitorMonitorList.uEmpty() ) {
	    DeregisterMonitorMutex( serial, task );
	} // if
    } // uProfiler::DeregisterMonitor
   
    uNoMutex void RegisterAcceptStart( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( !RegisterAcceptStartMonitorList.uEmpty() ) {
	    RegisterAcceptStartMutex( serial, task );
	} // if
    } // uProfiler::RegisterAcceptStart

    uNoMutex void RegisterAcceptEnd( const uSerial &serial, const uBaseTask &task ) {
	if ( &serial == &uSerialInstance ) return;	// do not profile calls to uProfiler's serial 
	if ( !RegisterAcceptEndMonitorList.uEmpty() ) {
	    RegisterAcceptEndMutex( serial, task );
	} // if
    } // uProfiler::RegisterAcceptEnd

    uNoMutex void RegisterSuspend( const uBaseCoroutine &coroutine, const uBaseTask &task ) {
	if ( !RegisterSuspendMonitorList.uEmpty() ) {
	    RegisterSuspendMutex( coroutine, task );
	} // if
    } // uProfiler::RegisterSuspend

    uNoMutex void RegisterResume( const uBaseCoroutine &coroutine, const uBaseTask &task ) {
	if ( !RegisterResumeMonitorList.uEmpty() ) {
	    RegisterResumeMutex( coroutine, task );
	} // if
    } // uProfiler::RegisterResume

    uNoMutex void RegisterWait( const uCondition &condition, const uBaseTask &task ) {
	if ( !RegisterWaitMonitorList.uEmpty() ) {
	    RegisterWaitMutex( condition, task );
	} // if
    } // uProfiler::RegisterWait

    uNoMutex void RegisterSignal( const uCondition &condition, const uBaseTask &task ) {
	if ( !RegisterSignalMonitorList.uEmpty() ) {
	    RegisterSignalMutex( condition, task );
	} // if
    } // uProfiler::RegisterSignal

    uNoMutex void RegisterCoroutine( const uBaseCoroutine &coroutine ) {
	if ( &coroutine == this ) return;		// do not profile profiler
	if ( !RegisterCoroutineMonitorList.uEmpty() ) {
	    RegisterCoroutineMutex( coroutine );
	} // if
    } // uProfiler::RegisterCoroutine

    uNoMutex void DeregisterCoroutine( const uBaseCoroutine &coroutine ) {
	if ( &coroutine == this ) return;		// do not profile profiler
	if ( !DeregisterCoroutineMonitorList.uEmpty() ) {
	    DeregisterCoroutineMutex( coroutine );
	} // if
    } // uProfiler::DeregisterCoroutine

    // routine pointers
    static void (* uProfiler_RegisterTask)(uProfiler *, const uBaseTask &);	// routine pointer to member routine in uProfiler
    static void (* uProfiler_DeregisterTask)(uProfiler *, const uBaseTask &);
    static void (* uProfiler_RegisterProcessor)(uProfiler *, const uProcessor &);
    static void (* uProfiler_DeregisterProcessor)(uProfiler *, const uProcessor &);
    static void (* uProfiler_RegisterTaskMigrate)(uProfiler *, const uBaseTask &, const uCluster &, const uCluster &);
    static void (* uProfiler_RegisterProcessorMigrate)(uProfiler *, const uProcessor &, const uCluster &, const uCluster &);
    static void (* uProfiler_RegisterFunctionEntry)(uProfiler *, uBaseTask *, unsigned int, unsigned int, unsigned int );
    static void (* uProfiler_RegisterFunctionExit)(uProfiler *, uBaseTask *);
    static void (* uProfiler_RegisterMutexFunctionEntryTry)(uProfiler *, const uSerial &, const uBaseTask &);
    static void (* uProfiler_RegisterMutexFunctionEntryDone)(uProfiler *, const uSerial &, const uBaseTask &);    
    static void (* uProfiler_RegisterMutexFunctionExit)(uProfiler *, const uSerial &, const uBaseTask &);
    static void (* uProfiler_RegisterMonitor)(uProfiler *, const uSerial &, const uBaseTask &);
    static void (* uProfiler_DeregisterMonitor)(uProfiler *, const uSerial &, const uBaseTask &);
    static void (* uProfiler_RegisterAcceptStart)(uProfiler *, const uSerial &, const uBaseTask &);
    static void (* uProfiler_RegisterAcceptEnd)(uProfiler *, const uSerial &, const uBaseTask &);    
    static void (* uProfiler_RegisterWait)(uProfiler *, const uCondition &, const uBaseTask &);
    static void (* uProfiler_RegisterSignal)(uProfiler *, const uCondition &, const uBaseTask &);
    static void (* uProfiler_RegisterResume)(uProfiler *, const uBaseCoroutine &, const uBaseTask &);
    static void (* uProfiler_RegisterSuspend)(uProfiler *, const uBaseCoroutine &, const uBaseTask &);    
    static void (* uProfiler_RegisterCoroutine)(uProfiler *, const uBaseCoroutine &);
    static void (* uProfiler_DeregisterCoroutine)(uProfiler *, const uBaseCoroutine &);

    static void (* uProfiler_RegisterCluster)(uProfiler *, const uCluster &);
    static void (* uProfiler_DeregisterCluster)(uProfiler *, const uCluster &);
}; // uProfiler


#endif __U_PROFILER_H__
