//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uSocket.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:06:26 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun May 23 16:23:04 1999
// Update Count     : 602
// 


#define __U_KERNEL__
#include <uC++.h>
//#include <uDebug.h>
#include <uSocket.h>


#include <string.h>					// strerror, memset
#include <errno.h>
#include <unistd.h>
#include <sys/param.h>					// MAXHOSTNAMELEN on BSD


#if ! defined( __svr4__ ) && defined( __sun__ )
extern "C" int soccket(int, int, int);			// missing from socket.h
extern "C" int accept(int, struct sockaddr *, int *);	// missing from socket.h
extern "C" int bind(int, struct sockaddr *,int);	// missing from socket.h
extern "C" int listen(int, int);			// missing from socket.h
extern "C" int connect(int, struct sockaddr *, int);	// missing from socket.h
extern "C" int send(int, char *, int, int);		// missing from socket.h
extern "C" int sendto(int, char *, int, int, struct sockaddr *, int); // missing from socket.h
extern "C" int recv(int, char *, int, int);		// missing from socket.h
extern "C" int recvfrom(int, char *, int, int, struct sockaddr *, int *); // missing from socket.h
extern "C" int getsockname(int, struct sockaddr *, int *); // missing from socket.h
extern "C" int getpeername(int, struct sockaddr *, int *); // missing from socket.h
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __aix__ ) && defined( __ibm__ )
// missing from older versions of AIX (< 4.0)
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __hpux__ ) && defined( __hp__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __ultrix__ ) && defined( __dec__ )
extern "C" int socket(int, int, int);			// missing from socket.h
extern "C" int accept(int, struct sockaddr *, int *);	// missing from socket.h
extern "C" int bind(int, struct sockaddr *,int);	// missing from socket.h
extern "C" int listen(int, int);			// missing from socket.h
extern "C" int connect(int, struct sockaddr *, int);	// missing from socket.h
extern "C" int getsockname(int, struct sockaddr *, int *); // missing from socket.h
extern "C" int getpeername(int, struct sockaddr *, int *); // missing from socket.h
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __gizmo__ )
extern "C" char *strerror(int n);
#endif


//######################### uSocket #########################


void uSocket::uCreateSocket( int domain, int type, int protocol ) {
    uIOlocnWrapper wrapper( locn );

    for ( ;; ) {
	access.fd = ::socket( domain, type, protocol );
      if ( access.fd != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( access.fd == -1 ) {
        // The exception should be caught by constructor in
	// uSocketServer or uSocketClient and throw a different exception
	// indicating the failure to create a server or a client.  However, it
	// cannot be implemented because the initialization list in a
	// constructor cannot be encapsulated in a try block.  Actually, there
	// is a feature in c++ allowing catch clause to handle exception
	// propoagated out of the initialization list but g++ does not have it
	// yet.

        uThrow uOpenFailure( *this, domain, type, protocol, "unable to open socket" );
    } // if

    // Assuming sockets are non-blocking.
    access.poll.uSetStatus( uAlwaysPoll );
    access.poll.uSetPollFlag( access.fd );
} // uSocket::uCreateSocket


uSocket::uSocket( int domain, int type, int protocol ) : locn( "uSocket" ), domain( domain ), type( type ), protocol( protocol ) {
    uCreateSocket( domain, type, protocol );
} // uSocket::uSocket


uSocket::uSocket( uIOCluster &cluster, int domain, int type, int protocol ) : locn( cluster ), domain( domain ), type( type ), protocol( protocol) {
    uCreateSocket( domain, type, protocol );
} // uSocket::uSocket


uSocket::~uSocket() {
    uIOlocnWrapper wrapper( locn );
    int code;

    for ( ;; ) {
	code = ::close( access.fd );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        uThrow uCloseFailure( *this, "unable to close socket" );
    } // if
} // uSocket::~uSocket


//######################### uSocketIO #########################


int uSocketIO::me_send( char *buf, int len, int flags ) {
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::send( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::me_send

uSocketIO::uSocketIO( uIOlocn &io, uIOaccess &acc, sockaddr *saddr, int saddrlen ) : uFileIO( io, acc ), saddr( saddr ), saddrlen( saddrlen ) {
} // uSocketIO::uSocketIO

int uSocketIO::getsockname( sockaddr *name, int *len ) {
    uIOlocnWrapper wrapper( *locn );

    return ::getsockname( access.fd, name,
#if defined( __aix__ ) && defined( __ibm__ )		// TEMPORARY
			 (size_t *)
#endif
    			 len );
} // uSocketIO::getsockname

int uSocketIO::getpeername( sockaddr *name, int *len ) {
    uIOlocnWrapper wrapper( *locn );

    return ::getpeername( access.fd, name,
#if defined( __aix__ ) && defined( __ibm__ )		// TEMPORARY
			 (size_t *)
#endif
    			 len );
} // uSocketIO::getpeername

int uSocketIO::send( char *buf, int len, int flags, uDuration *timeout ) {
    uIOlocnWrapper wrapper( *locn );
    int code;

    for ( ;; ) {
	code = me_send( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uIOCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		locn->uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uWriteTimeout( *sa, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uWriteTimeout( *sc, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uWriteTimeout( *sc, buf, len, flags, NULL, 0, timeout, "timeout during socket send" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.send: internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uWriteFailure( *sa, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uWriteFailure( *sc, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uWriteFailure( *sc, buf, len, flags, NULL, 0, timeout, "socket send fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.send: internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::send

int uSocketIO::me_sendto( char *buf, int len, int flags, void *to, int tolen ) {
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::sendto( access.fd, buf, len, flags, (sockaddr *)to, tolen );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::me_sendto

int uSocketIO::sendto( char *buf, int len, int flags, uDuration *timeout ) {
    return sendto( buf, len, saddr, saddrlen, flags, timeout );
} // uSocketIO::sendto

int uSocketIO::sendto( char *buf, int len, void *to, int tolen, int flags, uDuration *timeout ) {
    uIOlocnWrapper wrapper( *locn );
    int code;

    for ( ;; ) {
	code = me_sendto( buf, len, flags, to, tolen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uIOCluster::uWriteSelect;
	    if ( timeout == NULL ) {
		locn->uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uWriteTimeout( *sa, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uWriteTimeout( *sc, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uWriteTimeout( *sc, buf, len, flags, to, tolen, timeout, "timeout during socket sendto" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.sendto: internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uWriteFailure( *sa, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uWriteFailure( *sc, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uWriteFailure( *sc, buf, len, flags, to, tolen, timeout, "socket sendto fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.sendto: internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::sendto

int uSocketIO::me_recv( char *buf, int len, int flags ) {
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recv( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::me_recv

int uSocketIO::recv( char *buf, int len, int flags, uDuration *timeout ) {
    uIOlocnWrapper wrapper( *locn );
    int code;

    for ( ;; ) {
	code = me_recv( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uIOCluster::uReadSelect;
	    if ( timeout == NULL ) {
		locn->uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *sc, buf, len, flags, NULL, NULL, timeout, "timeout during socket recv" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recv: internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *sc, buf, len, flags, NULL, NULL, timeout, "socket recv fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recv: internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recv

int uSocketIO::me_recvfrom( char *buf, int len, int flags, void *from, int *fromlen ) {
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recvfrom( access.fd, buf, len, flags, (sockaddr *)from,
#if defined( __aix__ ) && defined( __ibm__ )		// TEMPORARY
			  (size_t *)
#endif
    			  fromlen );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::me_recvfrom

int uSocketIO::recvfrom( char *buf, int len, int flags, uDuration *timeout ) {
    return recvfrom( buf, len, saddr, &saddrlen, flags, timeout );
} // uSocketIO::recvfrom

int uSocketIO::recvfrom( char *buf, int len, void *from, int *fromlen, int flags, uDuration *timeout ) {
    uIOlocnWrapper wrapper( *locn );
    int code, tmp;

    if ( fromlen != 0 ) {				// save *fromlen, as it is set to 0 after each attempt
	tmp = *fromlen;
    }; // if

    for ( ;; ) {
	if ( fromlen != 0 ) *fromlen = tmp;		// reset *fromlen on each attempt
	code = me_recvfrom( buf, len, flags, from, fromlen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uIOCluster::uReadSelect;
	    if ( timeout == NULL ) {
		locn->uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *sc, buf, len, flags, from, fromlen, timeout, "timeout during socket recvfrom" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recvfrom: internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *sc, buf, len, flags, from, fromlen, timeout, "socket recvfrom fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recvfrom: internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recvfrom

int uSocketIO::me_recvmsg( msghdr *msg, int flags ) {
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uSetPollFlag( access.fd );
    int code = ::recvmsg( access.fd, msg, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.uGetStatus() == uPollOnDemand ) access.poll.uClearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::me_recvmsg

int uSocketIO::recvmsg( msghdr *msg, int flags, uDuration *timeout ) {
    uIOlocnWrapper wrapper( *locn );
    int code;

    for ( ;; ) {
	code = me_recvmsg( msg, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uIOCluster::uReadSelect;
	    if ( timeout == NULL ) {
		locn->uSelect( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( access.fd, mask, &t ) == 0 ) { // timeout ?
		    if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
			uThrow uSocketAccept::uReadTimeout( *sa, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
			uThrow uSocketClient::uReadTimeout( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
			uThrow uSocketServer::uReadTimeout( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "timeout during socket recvmsg" );
		    } else {
			uAbort( "(uSocketIO &)0x%p.recvmsg: internal error, invalid runtime type", this );
		    } // if
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	if ( uSocketAccept *sa = dynamic_cast<uSocketAccept *>(this) ) {
	    uThrow uSocketAccept::uReadFailure( *sa, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else if ( uSocketClient *sc = dynamic_cast<uSocketClient *>(this) ) {
	    uThrow uSocketClient::uReadFailure( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else if ( uSocketServer *sc = dynamic_cast<uSocketServer *>(this) ) {
	    uThrow uSocketServer::uReadFailure( *sc, (const char *)msg, 0, flags, NULL, NULL, timeout, "socket recvmsg fails" );
	} else {
	    uAbort( "(uSocketIO &)0x%p.recvmsg: internal error, invalid runtime type", this );
	} // if
    } // if

    return code;
} // uSocketIO::recvmsg


//######################### uSocketServer #########################


void uSocketServer::acceptor() {
    acceptorCnt += 1;
} // uSocketServer::acceptor


void uSocketServer::unacceptor() {
    acceptorCnt -= 1;
} // uSocketServer::unacceptor


int uSocketServer::me_accept( void *adr, int *len ) {
    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
    int fd = ::accept( socket.access.fd, (sockaddr *)adr,
#if defined( __aix__ ) && defined( __ibm__ )		// TEMPORARY
		      (size_t *)
#endif
    		      len );
    int terrno = errno;					// preserve errno across calls
    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
    errno = terrno;
    return fd;
} // uSocketServer::me_accept


void uSocketServer::uCreateSocketServer1( const char *name, int type, int protocol, int backlog ) {
    uIOlocnWrapper wrapper( socket.locn );
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 attempting binding to name:%s\n", this, name );
#endif __U_DEBUG_H__

    ((sockaddr_un *)saddr)->sun_family = AF_UNIX;
    strncpy( ((sockaddr_un *)saddr)->sun_path, name, sizeof(((sockaddr_un *)saddr)->sun_path) );
    saddrlen = sizeof(sockaddr_un::sun_family) + strlen(((sockaddr_un *)saddr)->sun_path);

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        uThrow uOpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 binding to name:%s\n", this, name );
#endif __U_DEBUG_H__
} // uSockeServer::uCreateSocketServer1


uSocketServer::uSocketServer( const char *name, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketServer1( name, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::uSocketServer( const char *name, uIOCluster &cluster, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( cluster, AF_UNIX, type, protocol ) {
    uCreateSocketServer1( name, type, protocol, backlog );
} // uSocketServer::uSocketServer


void uSocketServer::uCreateSocketServer2( unsigned short port, int type, int protocol, int backlog ) {
    uIOlocnWrapper wrapper( socket.locn );
    char name[MAXHOSTNAMELEN+1];
    hostent hp;
    int code;

    if ( uGethostname( name, sizeof(name) ) == -1 ) {	// get host name for this system
        uThrow uOpenFailure( *this, port, "no host name", AF_INET, type, protocol, backlog, "host name lookup failure" );
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 attempting binding to port:%d, name:%s\n", this, port, name );
#endif __U_DEBUG_H__

    if ( uGethostbyname( name, &hp ) == -1 ) {		// get internet address for host name
        uThrow uOpenFailure( *this, port, name, AF_INET, type, protocol, backlog, "internet name lookup failure" );
    } // if

    ((sockaddr_in *)saddr)->sin_family = AF_INET;
    ((sockaddr_in *)saddr)->sin_port = htons( port );
#ifdef __svr4__
    memcpy( &(((sockaddr_in *)saddr)->sin_addr), hp.h_addr, hp.h_length );
    memset( &(((sockaddr_in *)saddr)->sin_zero), '\0', sizeof(((sockaddr_in *)saddr)->sin_zero) );
#else
    bcopy( hp.h_addr, (char *)&(((sockaddr_in *)saddr)->sin_addr), hp.h_length );
    bzero( (char *)&(((sockaddr_in *)saddr)->sin_zero), sizeof(((sockaddr_in *)saddr)->sin_zero) );
#endif
    saddrlen = sizeof(sockaddr_in);

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        uThrow uOpenFailure( *this, port, name, AF_INET, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, port, name, AF_INET, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 binding to port:%d, name:%s\n", this, port, name );
#endif __U_DEBUG_H__
} // uSocketServer::uCreateSocketServer2


uSocketServer::uSocketServer( unsigned short port, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer2( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::uSocketServer( unsigned short port, uIOCluster &cluster, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketServer2( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


void uSocketServer::uCreateSocketServer3( unsigned short *port, int type, int protocol, int backlog ) {
    uCreateSocketServer2( 0, type, protocol, backlog );	// 0 port number => select an used port

    getsockname( saddr, &saddrlen );			// insert unsed port number into address ("bind" does not do it)
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer3 binding to port:%d\n", this, ntohs( ((sockaddr_in *)saddr)->sin_port ) );
#endif __U_DEBUG_H__
    *port = ntohs( ((sockaddr_in *)saddr)->sin_port );
} // uSocketServer::uCreateSocketServer3


uSocketServer::uSocketServer( unsigned short *port, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer3( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::uSocketServer( unsigned short *port, uIOCluster &cluster, int type, int protocol, int backlog ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketServer3( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::~uSocketServer() {
    if ( saddr->sa_family == AF_UNIX ) {
	delete (sockaddr_un *)saddr;
    } else {
	delete (sockaddr_in *)saddr;
    } // if

#ifdef __U_DEBUG__
    if ( acceptorCnt != 0 ) {
	uThrow uCloseFailure( *this, acceptorCnt, "closing socket server with outstanding acceptor(s)" );
    } // if
#endif __U_DEBUG__
} // uSocketServer::~uSocketServer


//######################### uSocketAccept #########################


const uSocket &uSocketAccept::sock() const {
    return socketserver.socket;
} // uSocketAccept::sock


void uSocketAccept::uCreateSocketAcceptor( uDuration *timeout, void *adr, int *len ) {
    uIOlocnWrapper wrapper( socketserver.socket.locn ); // step onto the server cluster
    {
	locn = new uIOlocn( "uSocketAccept" );		// create processor => child of socket processor
	uIOlocnWrapper wrapper( *locn );		// now step onto the acceptor cluster
	int tmp;

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept before accept\n", this );
#endif __U_DEBUG_H__

	if ( len != 0 ) {				// save *len, as it is set to 0 after each attempt
	    tmp = *len;
	} // if
	for ( ;; ) {
	    if ( len != 0 ) *len = tmp;			// reset *len on each attempt
	    access.fd = socketserver.me_accept( adr, len );
	  if ( access.fd != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uIOCluster::uReadSelect;
		if ( timeout == (uDuration *)0 ) {
		    socketserver.socket.locn.uSelect( socketserver.socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( socketserver.socket.locn.uSelect( socketserver.socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			// Cannot throw timeout event here because the cluster
			// is wrong for the necessary cleanup.
			goto fini1;
		    } // if
		} // if
	    } // if
	} // for
	if ( access.fd == -1 ) {
	    // Cannot throw timeout event here because the cluster is wrong for
	    // the necessary cleanup.
	    goto fini2;
	} // if
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept after accept fd:%d\n", this, access.fd );
#endif __U_DEBUG_H__

	// On some UNIX systems the file descriptor created by accept inherits
	// the non-blocking characteristic from the base socket; on other
	// system this does not seem to occur, so explicitly set the file
	// descriptor to non-blocking.

	access.poll.uSetStatus( uAlwaysPoll );
	access.poll.uSetPollFlag( access.fd );
    }
    socketserver.acceptor();
    return;

    // Back on the server cluster because wrapper for acceptor cluster is
    // destroyed so it is now possible to delete IO cluster for acceptor.
  fini1:
    delete locn;					// delete performed on server cluster
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept timeout fd:%d\n", this, access.fd );
#endif __U_DEBUG_H__
    uThrow uOpenTimeout( *this, timeout, adr, len, "timeout during accept" );
  fini2:
    delete locn;					// delete performed on server cluster
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept accept failure fd:%d\n", this, access.fd );
#endif __U_DEBUG_H__
    uThrow uOpenFailure( *this, timeout, adr, len, "unable to accept connection on socket server" );
} // uSocketAccept::uCreateSocketAcceptor


uSocketAccept::uSocketAccept( uSocketServer &s, void *adr, int *len ) :
	uSocketIO( *(uIOlocn *)0, access, s.saddr, s.saddrlen ), socketserver( s ) {
    uCreateSocketAcceptor( (uDuration *)0, adr, len );
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, uDuration *timeout, void *adr, int *len ) :
	uSocketIO( *(uIOlocn *)0, access, s.saddr, s.saddrlen ), socketserver( s ) {
    uCreateSocketAcceptor( timeout, adr, len );
} // uSocketAccept::uSocketAccept


uSocketAccept::~uSocketAccept() {
    uIOlocnWrapper wrapper( socketserver.socket.locn ); // step onto the server cluster
    int code;

    socketserver.unacceptor();
    {
	uIOlocnWrapper wrapper( *locn );		// step onto the acceptor cluster

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.~uSocketAccept, before close fd:%d\n", this, access.fd );
#endif __U_DEBUG_H__

	for ( ;; ) {
	    code = ::close( access.fd );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.~uSocketAccept, after close fd:%d\n", this, access.fd );
#endif __U_DEBUG_H__
    }
    delete locn;					// delete performed on server cluster
    if ( code == -1 ) {
	uThrow uCloseFailure( *this, "unable to close socket acceptor" );
    } // if
} // uSocketAccept::~uSocketAccept


//######################### uSocketClient #########################


void uSocketClient::uCreateSocketClient1( const char *name, uDuration *timeout, int type, int protocol ) {
    uIOlocnWrapper wrapper( socket.locn );
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 attempting connection to name:%s\n", this, name );
#endif __U_DEBUG_H__

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strncpy( ((sockaddr_un *)saddr)->sun_path, name, sizeof(((sockaddr_un *)saddr)->sun_path) );
	saddrlen = sizeof(sockaddr_un::sun_family) + strlen(((sockaddr_un *)saddr)->sun_path);

	for ( ;; ) {
	    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
	    code = ::connect( socket.access.fd, saddr, saddrlen );
	    int terrno = errno;				// preserve errno across calls
	    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
	    errno = terrno;
	  if ( code != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uIOCluster::uReadSelect;
		if ( timeout == (uDuration *)0 ) {
		    locn->uSelect( socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( locn->uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			uThrow uOpenTimeout( *this, 0, name, locn->cluster(), timeout, AF_UNIX, type, protocol, "timeout during connect" );
		    } // if
		} // if
	    } // if
	} // for
	// Alternate approach for waiting for a non-blocking connect.
	if ( code == -1 && errno == EINPROGRESS ) {
	    int mask = uIOCluster::uWriteSelect;
	    if ( timeout == (uDuration *)0 ) {
		locn->uSelect( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    uThrow uOpenTimeout( *this, 0, name, locn->cluster(), timeout, AF_UNIX, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    code = 0;
	} // if

	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, locn->cluster(), timeout, AF_UNIX, type, protocol, "unable to connect to socket" );
	} // if
    } else {
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	// Create temporary file for the client to communicate through.
	char *tmpnm = tempnam( NULL, "uC++" );
	strncpy( ((sockaddr_un *)saddr)->sun_path, tmpnm, sizeof(((sockaddr_un *)saddr)->sun_path) );
	delete tmpnm;
	saddrlen = sizeof(sockaddr_un::sun_family) + strlen(((sockaddr_un *)saddr)->sun_path);

	for ( ;; ) {
	    code = ::bind( socket.access.fd, saddr, saddrlen );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, 0, name, locn->cluster(), timeout, AF_UNIX, type, protocol, "unable to bind name to socket" );
	} // if
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 connection to name:%s\n", this, name );
#endif __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient1


uSocketClient::uSocketClient( const char *name, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( const char *name, uIOCluster &cluster, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( cluster, AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( const char *name, uIOCluster &cluster, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_un, sizeof(sockaddr_un) ), socket( cluster, AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, timeout, type, protocol );
} // uSocketClient::uSocketClient


void uSocketClient::uCreateSocketClient2( unsigned short port, uDuration *timeout, int type, int protocol ) {
    char name[MAXHOSTNAMELEN+1];

    if ( uGethostname( name, sizeof(name) ) == -1 ) {	// get host name for this system
        uThrow uOpenFailure( *this, port, "no host name", locn->cluster(), timeout, AF_INET, type, protocol, "unable to connect to socket" );
    } // if

    uCreateSocketClient3( port, name, timeout, type, protocol );
} // uSocketClient::uCreateSocketClient2


uSocketClient::uSocketClient( unsigned short port, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, uIOCluster &cluster, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketClient2( port, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, uIOCluster &cluster, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketClient2( port, timeout, type, protocol );
} // uSocketClient::uSocketClient


void uSocketClient::uCreateSocketClient3( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) {
    uIOlocnWrapper wrapper( socket.locn );
    hostent hp;
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient3 attempting connection to port:%d, name:%s\n", this, port, name );
#endif __U_DEBUG_H__

    if ( uGethostbyname( name, &hp ) == -1 ) {		// get internet address for host name
        uThrow uOpenFailure( *this, port, name, locn->cluster(), timeout, AF_INET, type, protocol, "internet name lookup failure" );
    } // if

    ((sockaddr_in *)saddr)->sin_family = AF_INET;
    ((sockaddr_in *)saddr)->sin_port = htons( port );
#ifdef __svr4__
    memcpy( (char *)&(((sockaddr_in *)saddr)->sin_addr), hp.h_addr, hp.h_length );
    memset( &(((sockaddr_in *)saddr)->sin_zero), '\0', sizeof(((sockaddr_in *)saddr)->sin_zero) );
#else
    bcopy( hp.h_addr, (char *)&(((sockaddr_in *)saddr)->sin_addr), hp.h_length );
    bzero( (char *)&(((sockaddr_in *)saddr)->sin_zero), sizeof(((sockaddr_in *)saddr)->sin_zero) );
#endif
    saddrlen = sizeof(sockaddr_in);

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uSetPollFlag( socket.access.fd );
	    code = ::connect( socket.access.fd, saddr, saddrlen );
	    int terrno = errno;				// preserve errno across calls
	    if ( socket.access.poll.uGetStatus() == uPollOnDemand ) socket.access.poll.uClearPollFlag( socket.access.fd );
	    errno = terrno;
	  if ( code != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uIOCluster::uReadSelect;
		if ( timeout == (uDuration *)0 ) {
		    locn->uSelect( socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( locn->uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			uThrow uOpenTimeout( *this, port, name, locn->cluster(), timeout, AF_INET, type, protocol, "timeout during connect" );
		    } // if
		} // if
	    } // if
	} // for
	// Alternate approach for waiting for a non-blocking connect.
	if ( code == -1 && errno == EINPROGRESS ) {
	    int mask = uIOCluster::uWriteSelect;
	    if ( timeout == (uDuration *)0 ) {
		locn->uSelect( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( locn->uSelect( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    uThrow uOpenTimeout( *this, port, name, locn->cluster(), timeout, AF_INET, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    code = 0;
	} // if

	if ( code == -1 ) {
	    uThrow uOpenFailure( *this, port, name, locn->cluster(), timeout, AF_INET, type, protocol, "unable to connect to socket" );
	} // if
    } // if
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient3 connection to port:%d, name:%s\n", this, port, name );
#endif __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient3


uSocketClient::uSocketClient( unsigned short port, const char *name, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient3( port, name, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient3( port, name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, uIOCluster &cluster, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketClient3( port, name, (uDuration *)0, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, uIOCluster &cluster, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.locn, socket.access, (sockaddr *)new sockaddr_in, sizeof(sockaddr_in) ), socket( cluster, AF_INET, type, protocol ) {
    uCreateSocketClient3( port, name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::~uSocketClient() {
    if ( saddr->sa_family == AF_UNIX ) {
	if ( socket.type == SOCK_DGRAM ) {
	    // Remove temporary file created for communication.
	    int code = unlink( ((sockaddr_un *)saddr)->sun_path );
	    if ( code == -1 ) {
		uThrow uCloseFailure( *this, "unlink failed for temporary pipe" );
	    } // if
	} // if
	delete (sockaddr_un *)saddr;
    } else {
	delete (sockaddr_in *)saddr;
    } // if
} // uSocketClient::uSocketClient


//######################### uSocket (cont) #########################


uSocket::uFailure::uFailure( const uSocket &socket, const char *const msg ) : uIOFailure( msg ), _socket( socket ) {}

const uSocket &uSocket::uFailure::socket() const { return _socket; }

void uSocket::uFailure::uDefaultTerminate() const {
    uAbort( "(uSocket &)0x%p, %.256s.", &socket(), uMsg() );
} // uSocket::uFailure::uDefaultTerminate


uSocket::uOpenFailure::uOpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocket::uFailure( socket, msg ), domain( domain ), type( type ), protocol( protocol ) {}

void uSocket::uOpenFailure::uDefaultTerminate() const {
    uAbort( "(uSocket &)0x%p.uSocket( domain:%d, type:%d, protocol:%d ), %.256s.\nError(%d): %s.",
	   &socket(), domain, type, protocol, uMsg(), errNo(), strerror( errNo() ) );
} // uSocket::uOpenFailure::uDefaultTerminate


uSocket::uCloseFailure::uCloseFailure( const uSocket &socket, const char *const msg ) :
	uSocket::uFailure( socket, msg ) {}

void uSocket::uCloseFailure::uDefaultTerminate() const {
    uAbort( "(uSocket &)0x%p.~uSocket(), %.256s %d.\nError(%d): %s.",
	   &socket(), socket().access.fd, uMsg(), errNo(), strerror( errNo() ) );
} // uSocket::uCloseFailure::uDefaultTerminate


uInitEvent(uSocket::uFailure);
uInitEvent(uSocket::uOpenFailure);
uInitEvent(uSocket::uCloseFailure);


//######################### uSocketServer (cont) #########################


uSocketServer::uFailure::uFailure( const uSocketServer &server, const char *const msg ) : uSocket::uFailure( server.socket, msg ), _server( server ) {
} // uSocketServer::uFailure::uFailure

const uSocketServer &uSocketServer::uFailure::server() const {
    return _server;
} // uSocketServer::uFailure::server

int uSocketServer::uFailure::fileDescriptor() const {
    return server().access.fd;
} // uSocketServer::uFailure::fileDescriptor

void uSocketServer::uFailure::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p, %.256s.\nError(%d): %s.",
	   &server(), uMsg(), errNo(), strerror( errNo() ) );
} // uSocketServer::uFailure::uDefaultTerminate


uSocketServer::uOpenFailure::uOpenFailure( const uSocketServer &server, unsigned short port, const char *name, int domain, int type, int protocol, int backlog, const char *const msg ) :
	uSocketServer::uFailure( server, msg ), port( port ), domain( domain ), type( type ), protocol( protocol ), backlog( backlog ) {
    strcpy( _name, name );				// copy file name
} // uSocketServer::uOpenFailure::uOpenFailure

const char *const uSocketServer::uOpenFailure::name() const { return _name; }

void uSocketServer::uOpenFailure::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.uSocketServer( port:%d, name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d, backlog:%d ), %.256s.\nError(%d): %s.",
	   &server(), port, name(), &uThisCluster(), domain, type, protocol, backlog, uMsg(), errNo(), strerror( errNo() ) );
} // uSocketServer::uOpenFailure::uDefaultTerminate


uSocketServer::uCloseFailure::uCloseFailure( const uSocketServer &server, int acceptorCnt, const char *const msg ) :
	uSocketServer::uFailure( server, msg ), acceptorCnt( acceptorCnt ) {
} // uSocketServer::uCloseFailure::uCloseFailure

void uSocketServer::uCloseFailure::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.~uSocketServer(), %.256s, %d acceptor(s) outstanding.",
	   &server(), uMsg(), acceptorCnt );
} // uSocketServer::uCloseFailure::uDefaultTerminate


uSocketServer::uReadFailure::uReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketServer::uReadFailure::uReadFailure

void uSocketServer::uReadFailure::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &server(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::uReadFailure::uDefaultTerminate


uSocketServer::uReadTimeout::uReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketServer::uReadTimeout::uReadTimeout

void uSocketServer::uReadTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &server(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketServer::uReadTimeout::uDefaultTerminate


uSocketServer::uWriteFailure::uWriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketServer::uWriteFailure::uWriteFailure

void uSocketServer::uWriteFailure::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &server(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::uWriteFailure::uDefaultTerminate


uSocketServer::uWriteTimeout::uWriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketServer::uWriteTimeout::uWriteTimeout

void uSocketServer::uWriteTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &server(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketServer::uWriteTimeout::uDefaultTerminate


uInitEvent(uSocketServer::uFailure);
uInitEvent(uSocketServer::uOpenFailure);
uInitEvent(uSocketServer::uCloseFailure);
uInitEvent(uSocketServer::uReadFailure);
uInitEvent(uSocketServer::uReadTimeout);
uInitEvent(uSocketServer::uWriteFailure);
uInitEvent(uSocketServer::uWriteTimeout);


//######################### uSocketAccept (cont) #########################


uSocketAccept::uFailure::uFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocket::uFailure( acceptor.sock(), msg ), _acceptor( acceptor ) {}

const uSocketAccept &uSocketAccept::uFailure::acceptor() const {
    return _acceptor;
} // uSocketAccept::uFailure::acceptor

const uSocketServer &uSocketAccept::uFailure::server() const {
    return _acceptor.socketserver;
} // uSocketAccept::uFailure::server

int uSocketAccept::uFailure::fileDescriptor() const {
    return acceptor().access.fd;
} // uSocketAccept::uFailure::fileDescriptor

void uSocketAccept::uFailure::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p( server:0x%p ), %.256s.", &acceptor(), &server(), uMsg() );
} // uSocketAccept::uFailure::uDefaultTerminate


uSocketAccept::uOpenFailure::uOpenFailure( const uSocketAccept &acceptor, const class uDuration *timeout, const void *adr, const int *len, const char *const msg ) :
	uSocketAccept::uFailure( acceptor, msg ), timeout( timeout ), adr( adr ), len( len ) {}

void uSocketAccept::uOpenFailure::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s, error(%d): %s",
	   &acceptor(), &server(), timeout, adr, len, uMsg(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uOpenFailure::uDefaultTerminate


uSocketAccept::uOpenTimeout::uOpenTimeout( const uSocketAccept &acceptor, const class uDuration *timeout, const void *adr, const int *len, const char *const msg ) :
	uSocketAccept::uOpenFailure( acceptor, timeout, adr, len, msg ) {}

void uSocketAccept::uOpenTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s",
	   &acceptor(), &server(), timeout, adr, len, uMsg() );
} // uSocketAccept::uOpenTimeout::uDefaultTerminate


uSocketAccept::uCloseFailure::uCloseFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocketAccept::uFailure( acceptor, msg ) {}

void uSocketAccept::uCloseFailure::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.~uSocketAccept(), %.256s.\nError(%d): %s.",
	   &acceptor(), uMsg(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uCloseFailure::uDefaultTerminate


uSocketAccept::uReadFailure::uReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketAccept::uReadFailure::uReadFailure

void uSocketAccept::uReadFailure::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &acceptor(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uReadFailure::uDefaultTerminate


uSocketAccept::uReadTimeout::uReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketAccept::uReadTimeout::uReadTimeout

void uSocketAccept::uReadTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &acceptor(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketAccept::uReadTimeout::uDefaultTerminate


uSocketAccept::uWriteFailure::uWriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketAccept::uWriteFailure::uWriteFailure

void uSocketAccept::uWriteFailure::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &acceptor(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::uWriteFailure::uDefaultTerminate


uSocketAccept::uWriteTimeout::uWriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketAccept::uWriteTimeout::uWriteTimeout

void uSocketAccept::uWriteTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &acceptor(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketAccept::uWriteTimeout::uDefaultTerminate


uInitEvent(uSocketAccept::uFailure);
uInitEvent(uSocketAccept::uOpenFailure);
uInitEvent(uSocketAccept::uOpenTimeout);
uInitEvent(uSocketAccept::uCloseFailure);
uInitEvent(uSocketAccept::uReadFailure);
uInitEvent(uSocketAccept::uReadTimeout);
uInitEvent(uSocketAccept::uWriteFailure);
uInitEvent(uSocketAccept::uWriteTimeout);


//######################### uSocketClient (cont) #########################


uSocketClient::uFailure::uFailure( const uSocketClient &client, const char *const msg ) : uSocket::uFailure( client.socket, msg ), _client( client ) {
} // uSocketClient::uFailure::uFailure

const uSocketClient &uSocketClient::uFailure::client() const {
    return _client;
} // uSocketClient::uFailure::client

int uSocketClient::uFailure::fileDescriptor() const {
    return client().access.fd;
} // uSocketClient::uFailure::fileDescriptor

void uSocketClient::uFailure::uDefaultTerminate() const {
	uAbort( "(uSocketClient &)0x%p, %.256s.",
	       &client(), uMsg() );
} // uSocketClient::uFailure::uDefaultTerminate


uSocketClient::uOpenFailure::uOpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, const uIOCluster &cluster, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::uFailure( client, msg ), port( port ), cluster( cluster ), timeout( timeout ), domain( domain ), type( type ), protocol( protocol ) {
    strcpy( _name, name );				// copy file name
} // uSocketClient::uOpenFailure::uOpenFailure

const char *const uSocketClient::uOpenFailure::name() const { return _name; }

void uSocketClient::uOpenFailure::uDefaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ): %.256s.\nError(%d): %s.",
	       &client(), name(), &uThisCluster(), timeout, domain, type, protocol, uMsg(), errNo(), strerror( errNo() ) );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ): %.256s.\nError(%d): %s.",
	       &client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, uMsg(), errNo(), strerror( errNo() ) );
    } // if
} // uSocketClient::uOpenFailure::uDefaultTerminate


uSocketClient::uOpenTimeout::uOpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, const uIOCluster &cluster, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::uOpenFailure( client, port, name, cluster, timeout, domain, type, protocol, msg ) {
} // uSocketClient::uOpenTimeout::uOpenTimeout

void uSocketClient::uOpenTimeout::uDefaultTerminate()const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d ): %.256s.",
	       &client(), name(), &uThisCluster(), domain, type, protocol, uMsg() );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ): %.256s.",
	       &client(), port, name(), &uThisCluster(), domain, type, protocol, uMsg() );
    } // if
} // uSocketClient::uOpenTimeout::uDefaultTerminate


uSocketClient::uCloseFailure::uCloseFailure( const uSocketClient &client, const char *const msg ) :
	uSocketClient::uFailure( client, msg ) {}

void uSocketClient::uCloseFailure::uDefaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.~uSocketClient(), %.256s.\nError(%d): %s.",
	   &client(), uMsg(), errNo(), strerror( errNo() ) );
} // uSocketClient::uCloseFailure::uDefaultTerminate


uSocketClient::uReadFailure::uReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketClient::uReadFailure::uReadFailure

void uSocketClient::uReadFailure::uDefaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &client(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::uReadFailure::uDefaultTerminate


uSocketClient::uReadTimeout::uReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const void *from, const int *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketClient::uReadTimeout::uReadTimeout

void uSocketClient::uReadTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &client(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketClient::uReadTimeout::uDefaultTerminate


uSocketClient::uWriteFailure::uWriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uFailure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketClient::uWriteFailure::uWriteFailure

void uSocketClient::uWriteFailure::uDefaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.\nError(%d): %s.",
	   &client(), buf, len, flags, timeout, uMsg(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::uWriteFailure::uDefaultTerminate


uSocketClient::uWriteTimeout::uWriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const void *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::uWriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketClient::uWriteTimeout::uWriteTimeout

void uSocketClient::uWriteTimeout::uDefaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ): %.256s for file descriptor %d.",
	   &client(), buf, len, flags, timeout, uMsg(), fileDescriptor() );
} // uSocketClient::uWriteTimeout::uDefaultTerminate


uInitEvent(uSocketClient::uFailure);
uInitEvent(uSocketClient::uOpenFailure);
uInitEvent(uSocketClient::uOpenTimeout);
uInitEvent(uSocketClient::uCloseFailure);
uInitEvent(uSocketClient::uReadFailure);
uInitEvent(uSocketClient::uReadTimeout);
uInitEvent(uSocketClient::uWriteFailure);
uInitEvent(uSocketClient::uWriteTimeout);


// Local Variables:
// compile-command: "dmake"
// End:
