//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr 1994
// 
// uFile.h -- Nonblocking UNIX I/O library
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 16:38:54 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue May 18 14:24:18 1999
// Update Count     : 88
// 


#ifndef __U_FILE_H__
#define __U_FILE_H__


#include <uIOcntl.h>
#include <uIOEvent.h>


#pragma __U_NOT_USER_CODE__


#include <fcntl.h>					// open flags for users, e.g., O_RDONLY, O_WRONLY, etc.


//######################### uFile #########################


class uFile {
    friend class uFileWrapper;
    friend class uFileAccess;
    friend class uFilebuf;

    char *uName;
    int accessCnt;
    uIOlocn locn;

    void uCreateFile( const char *name );
    uMutex void access();
    uMutex void unaccess();
  public:
    uFile( const char *name );
    uFile( const char *name, uIOCluster &cluster );
    virtual ~uFile();

    const char *uGetName() const;
    void status( struct stat &buf );

    uDualEvent uFailure;
    uDualEvent uTerminateFailure;
    uDualEvent uStatusFailure;
}; // uFile


//######################### uFileIO #########################


class uFileIO {
    uMutex int me_read( char *buf, int len );
    uMutex int me_readv( const struct iovec *iov, int iovcnt );
    uMutex int me_write( char *buf, int len );
    uMutex int me_writev( const struct iovec *iov, int iovcnt );
  protected:
    uIOlocn *locn;
    uIOaccess &access;

    uFileIO( uIOlocn &l, uIOaccess &acc );
    virtual ~uFileIO();
  public:
    int read( char *buf, int len, uDuration *timeout = NULL );
    int readv( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    int write( char *buf, int len, uDuration *timeout = NULL );
    int writev( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
}; // uFileIO


//######################### uFileAccess #########################


class uFileAccess : public uFileIO {		// monitor
    friend class uFilebuf;

    uFile &ufile;
    uIOaccess access;

    uFileAccess( int fd, uFile &f );
  public:
    uFileAccess( uFile &f, int flags, int mode = 0644 );
    virtual ~uFileAccess();

    uMutex off_t lseek( off_t offset, int whence );
    uMutex int fsync();

    uDualEvent uFailure;
    friend uDualEvent uFailure;
    uDualEvent uOpenFailure;
    uDualEvent uCloseFailure;
    uDualEvent uSeekFailure;
    uDualEvent uSyncFailure;
    uDualEvent uReadFailure;
    uDualEvent uReadTimeout;
    uDualEvent uWriteFailure;
    uDualEvent uWriteTimeout;
}; // uFileAccess


//######################### uFile (cont) #########################


uDualEvent uFile::uFailure : public uIOFailure {
    const uFile &f;
    char uName[uAEHMMaxName + sizeof("...")];		// room for "...\0"
  public:
    uFailure( const uFile &f, const char *const msg );
    ~uFailure();
    const uFile &file() const;
    const char *uGetName() const;
    virtual void uDefaultTerminate() const;
}; // uFile::uFailure


uDualEvent uFile::uTerminateFailure : public uFile::uFailure {
    const int accessCnt;
  public:
    uTerminateFailure( const uFile &f, const int accessCnt, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFile::uTerminateFailure


uDualEvent uFile::uStatusFailure : public uFile::uFailure {
    const struct stat &buf;
  public:
    uStatusFailure( const uFile &f, const struct stat &buf, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFile::uStatusFailure


//######################### uFileAccess (cont) #########################


uDualEvent uFileAccess::uFailure : public uFile::uFailure {
    const uFileAccess &fa;
    int fd;
  public:
    uFailure( const uFileAccess &fa, const char *const msg );
    const uFileAccess &fileAccess() const;
    int fileDescriptor() const;
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uFailure


uDualEvent uFileAccess::uOpenFailure : public uFileAccess::uFailure {
    const int flags;
    const int mode;
  public:
    uOpenFailure( uFileAccess &fa, int flags, int mode, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uOpenFailure


uDualEvent uFileAccess::uCloseFailure : public uFileAccess::uFailure {
  public:
    uCloseFailure( uFileAccess &fa, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uCloseFailure


uDualEvent uFileAccess::uSeekFailure : public uFileAccess::uFailure {
    const off_t offset;
    const int whence;
  public:
    uSeekFailure( const uFileAccess &fa, const off_t offset, const int whence, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uSeekFailure


uDualEvent uFileAccess::uSyncFailure : public uFileAccess::uFailure {
  public:
    uSyncFailure( const uFileAccess &fa, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uSyncFailure


uDualEvent uFileAccess::uReadFailure : public uFileAccess::uFailure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    uReadFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uReadFailure


uDualEvent uFileAccess::uReadTimeout : public uFileAccess::uReadFailure {
  public:
    uReadTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::uReadTimeout


uDualEvent uFileAccess::uWriteFailure : public uFileAccess::uFailure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    uWriteFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void uDefaultResume() const;		// handle special case when errno == EIO
    virtual void uDefaultTerminate() const;
}; // uFileAccess::uWriteFailure


uDualEvent uFileAccess::uWriteTimeout : public uFileAccess::uWriteFailure {
  public:
    uWriteTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::uWriteFailure


#pragma __U_USER_CODE__

#endif __U_FILE_H__


// Local Variables:
// compile-command: "dmake"
// End:
