//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uProcessor.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Mon Mar 14 17:39:15 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue May  4 15:48:29 1999
// Update Count     : 1266
// 


#define __U_KERNEL__
#define __U_PROFILE__
#define __U_PROFILEABLE_ONLY__
#ifndef __U_DEBUG__
#define NDEBUG						// turn off assert
#endif


#include <uC++.h>
#include <uProfiler.h>
#include <uProcessor.h>
#include <uAssert.h>
//#include <uDebug.h>

#include <unistd.h>					// prototype: getpid
#include <signal.h>					// prototype: kill, SIGALRM
#include <sys/mman.h>					// prototype: mmap


#if ! defined( __svr4__ ) && defined( __sun__ )
extern "C" int sigsetmask(int);				// missing from signal.h
extern "C" int sigblock(int);				// missing from signal.h
extern "C" int sigpause(int);				// missing from signal.h
#endif

#if defined( __ultrix__ ) && defined( __dec__ )
extern "C" int sigsetmask(int);				// missing from signal.h
extern "C" int sigblock(int);				// missing from signal.h
extern "C" int sigpause(int);				// missing from signal.h
extern "C" void *mmap(caddr_t, int, int, int, int, off_t); // missing from sys/mman.h
#define pid_t int
#endif


#ifndef __U_MULTI__
uEventList	*uProcessor::uEvents			= (uEventList *)0;
uEventNode      *uProcessor::uContextEvent 		= (uEventNode *)0;
uCxtSwtchHndlr  *uProcessor::uContextSwitchHandler      = (uCxtSwtchHndlr *)0;
#endif // ! __U_MULTI__
#ifdef __U_DEBUG__
static const int uMinPreemption = 1000;			// 1 second (milliseconds)
#endif __U_DEBUG__


extern uProcessor &uThisProcessor() {
    return *uKernelModule::uActiveProcessor;
} // uThisProcessor


//######################### uProcessorDL #########################


uProcessorDL::uProcessorDL( uProcessor &w ) : uWho( w ) {}
uProcessor &uProcessorDL::uGet() const { return uWho; }


//######################### uProcessorTask #########################


void uProcessorTask::main() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorTask &)0x%p.main, starting\n", this );
#endif __U_DEBUG_H__
    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );
    uThisProcessor().uPid = getpid();			// set UNIX pid for processor

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif __U_LOCALDEBUGGER_H__

    // Put processor on global processor list. It can only go on the list after
    // uPid is set because uAbortExit traverses the list killing UNIX processes
    // during shutdown.

    uKernelModule::uGlobalProcessorLock->uAcquire();
    uKernelModule::uGlobalProcessors->uAddTail( &(uThisProcessor().uGlobalRef) );
    uKernelModule::uGlobalProcessorLock->uRelease();

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->createKernelThread( uThisProcessor(), uThisCluster() );
#endif __U_LOCALDEBUGGER_H__

    // profiling

    if ( uProfiler::uProfiler_RegisterProcessor	&& uProfileActive ) {
	this->uProfileTaskSamplerInstance->inside_uFunctionPrologue += 1;
	// make sure the currently executing task doesn't profile call
	(*uProfiler::uProfiler_RegisterProcessor)( uProfiler::uProfilerInstance, this->uBound );
	this->uProfileTaskSamplerInstance->inside_uFunctionPrologue -= 1;
    } // if

    // Set the alarm for this process.  Although the signal handlers are
    // inherited by each child process, the alarm setting is not.

    uThisProcessor().uSetContextSwitchEvent( uThisProcessor().uGetPreemption() );

    for ( ;; ) {
	uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );
	uAccept ( ~uProcessorTask ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, ~uProcessorTask\n", this );
#endif __U_DEBUG_H__

	    // profiling

	    if ( uProfiler::uProfiler_DeregisterProcessor && uProfileActive ) {
		this->uProfileTaskSamplerInstance->inside_uFunctionPrologue += 1;
		// make sure the currently executing task doesn't profile call
		(*uProfiler::uProfiler_DeregisterProcessor)( uProfiler::uProfilerInstance, this->uBound );
		this->uProfileTaskSamplerInstance->inside_uFunctionPrologue -= 1;
	    } // if

#if __U_LOCALDEBUGGER_H__
	    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->destroyKernelThread( uThisProcessor() );
#endif __U_LOCALDEBUGGER_H__

	    uActiveProcessorKernel->uTerminated = true;

	    // Remove processor from global processor list. It is removed here
	    // because the next action after this is the termination of the
	    // UNIX process in uProcessorKernel. Therefore, even if the
	    // application is aborted and the process is not on the list of
	    // UNIX processes for this application, the current UNIX process
	    // will terminate itself.  It cannot be removed in uProcessKernel
	    // because the uProcessor storage may be freed already by another
	    // processor.

	    uKernelModule::uGlobalProcessorLock->uAcquire();
	    uKernelModule::uGlobalProcessors->uRemove( &(uThisProcessor().uGlobalRef) );
	    uKernelModule::uGlobalProcessorLock->uRelease();

	    break;
	} uOr uAccept ( uSetPreemption ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, uSetPreemption( %d )\n", this, uPreemption );
#endif __U_DEBUG_H__
	    uThisProcessor().uSetContextSwitchEvent( uPreemption ); // use it to set the alarm for this processor
	    uThisProcessor().uPreemption = uPreemption;
	} uOr uAccept ( uFork ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, uFork\n", this );
#endif __U_DEBUG_H__
	    uActiveProcessorKernel->uForkChild = true;
	    uActiveProcessorKernel->uNewProc = uNewProc;
	} uOr uAccept ( uMmap ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, uMmap\n", this );
#endif __U_DEBUG_H__
#ifdef __gizmo__					// does not have mmap
	    uAbort( "mmap not implemented for gizmo." );
#else
	    void *uAddr = mmap( (caddr_t)uBegin, uSize, uProt, uFlags, uFd, uOff );
	    if ( uAddr == (void *)-1 ) {
		uAbort( ": attempt to uMmap( begin:0x%p, size:%d, prot:0x%p, flags:0x%p, fd:0x%p, off:%d ) but memory mapping failed",
		       uBegin, uSize, uProt, uFlags, uFd, uOff );
	    } // if
#endif __gizmo__
	    uSignal uResult;				// handshake
	    uWait uResponse;
	} uOr uAccept ( uSetCluster ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, uSetCluster\n", this );
#endif __U_DEBUG_H__

#if __U_LOCALDEBUGGER_H__
	    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif __U_LOCALDEBUGGER_H__

	    // Remove the processor from the list of processor that live on
	    // this cluster, and add it to the list of processors that live on
	    // the new cluster. The move has to be done by the processor itself
	    // when it is in a stable state. As well, the processor's task has
	    // to be moved to the new cluster but it does not have to be
	    // migrated there since it is a bound task.

	    uProcessor &uProc = uThisProcessor();	// change processor's notion of which cluster it is executing on
	    uCluster &uPrev = uProc.uGetCluster();
	    uPrev.uProcessorRemove( uProc );
	    uKernelModule::uActiveCluster = uProc.uCurrCluster = uClus;
	    uClus->uProcessorAdd( uProc );
	    uCurrCluster = uClus;			// change task's notion of which cluster it is executing on

#if __U_LOCALDEBUGGER_H__
	    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->migrateKernelThread( uThisProcessor(), *uClus );
#endif __U_LOCALDEBUGGER_H__

	    if ( uProfiler::uProfiler_RegisterProcessorMigrate ) {	// task registered for profiling ?              
		uKernelModule::uActiveTask->uProfileTaskSamplerInstance->inside_uFunctionPrologue += 1;
		// make sure the currently executing task doesn't profile call
		(*uProfiler::uProfiler_RegisterProcessorMigrate)( uProfiler::uProfilerInstance, uThisProcessor(), uPrev, *uClus );
		uKernelModule::uActiveTask->uProfileTaskSamplerInstance->inside_uFunctionPrologue -= 1;
	    } // if

	    uSignal uResult;				// handshake
	    uWait uResponse;
	} // uAccept
    } // for
} // uProcessorTask::main


void uProcessorTask::uSetPreemption( int ms ) {
    uPreemption = ms;
} // uProcessorTask::uSetPreemption


void uProcessorTask::uFork( uProcessor *np ) {
    uNewProc = np;
} // uProcessorTask::uFork


void *uProcessorTask::uMmap( void *begin, int size, int prot, int flags, int fd, int off ) {
    uBegin = begin;					// copy arguments
    uSize = size;
    uProt = prot;
    uFlags = flags;
    uFd = fd;
    uOff = off;
    uWait uResult;					// wait for result
    uSignal uResponse;
    return uAddr;
} // uProcessorTask::uMmap


void uProcessorTask::uSetCluster( uCluster &clus ) {
    uClus = &clus;					// copy arguments
    uWait uResult;					// wait for result
    uSignal uResponse;
} // uProcessor::uSetCluster


uProcessorTask::uProcessorTask( uCluster &uClus, uProcessor &uProc ) : uBaseTask( uClus, uProc ) {
} // uProcessorTask::uProcessorTask


uProcessorTask::~uProcessorTask() {
} // uProcessorTask::~uProcessorTask


//######################### uProcessorKernel #########################


void uProcessorKernel::uFork() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.uFork, uNewProc:0x%p\n", this, uNewProc );
#endif __U_DEBUG_H__
    uForkChild = false;					// turn off fork flag

#ifdef __U_MULTI__
    pid_t uPid = fork();
#else
    pid_t uPid = 0;					// uniprocessor runs on a single UNIX process
#endif __U_MULTI__

    // PUT NO CODE HERE, CHILD NOT READY TO RUN!
    if ( uPid == 0 ) {					// child ?
	// PUT NO CODE HERE, CHILD NOT READY TO RUN!

	uKernelModule::uActiveProcessor = uNewProc;
	uKernelModule::uActiveCluster   = uKernelModule::uActiveProcessor->uCurrCluster;

	// real-time

        uThisProcessor().uProcessorClock = &uActiveProcessorKernel->uKernelClock;
#ifdef __U_MULTI__
	uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt == 1 );

	// real-time

        uKernelClock.uResetClock();

	uKernelModule::uDisableIntSpin = false;
	uKernelModule::uDisableIntSpinCnt = 0;
	uKernelModule::uInKernelRF = 0;
#endif __U_MULTI__

	// CHILD IS NOW READY TO RUN!

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uProcessorKernel &)0x%p.uFork, child started\n", this );
#endif __U_DEBUG_H__
    } // if

    if ( uPid == -1 ) {
	uAbort( "(uProcessor &)0x%p.uFork() : internal error, could not create virtual processor.", this );
    } // if
} // uProcessorKernel::uFork


void uProcessorKernel::uSchedule() {
    uAssert( ! uThisTask().uReadyRef.uListed() );
    uKernelModule::uDisableInterrupts();

    uThisTask().uSetState( uBaseTask::uBlocked );
    uCode = 0;
    uContextSw();					// not uResume because entering kernel
    uThisTask().uSetState( uBaseTask::uRunning );

    uKernelModule::uEnableInterrupts();
} // uProcessorKernel::uSchedule


void uProcessorKernel::uSchedule( uSpinLock *l ) {
    uAssert( ! uThisTask().uReadyRef.uListed() );
    uKernelModule::uDisableInterrupts();

    uThisTask().uSetState( uBaseTask::uBlocked );	
    uCode = 1;
    uPrevLock = l;
    uContextSw();					// not uResume because entering kernel
    uThisTask().uSetState( uBaseTask::uRunning );

    uKernelModule::uEnableInterrupts();
} // uProcessorKernel::uSchedule


void uProcessorKernel::uSchedule( uBaseTask *t ) {
    // SKULLDUGGERY: uBootTask is on ready queue for first entry into the kernel.
    uAssert( &uThisTask() != (uBaseTask *)uKernelModule::uTaskBoot ? ! uThisTask().uReadyRef.uListed() : true );
    uKernelModule::uDisableInterrupts();

    uThisTask().uSetState( uBaseTask::uBlocked );	
    uCode = 2;
    uNextTask = t;
    uContextSw();					// not uResume because entering kernel
    uThisTask().uSetState( uBaseTask::uRunning );

    uKernelModule::uEnableInterrupts();
} // uProcessorKernel::uSchedule


void uProcessorKernel::uSchedule( uSpinLock *l, uBaseTask *t ) {
    uAssert( ! uThisTask().uReadyRef.uListed() );
    uKernelModule::uDisableInterrupts();

    uThisTask().uSetState( uBaseTask::uBlocked );      
    uCode = 3;
    uPrevLock = l;
    uNextTask = t;
    uContextSw();					// not uResume because entering kernel
    uThisTask().uSetState( uBaseTask::uRunning );

    uKernelModule::uEnableInterrupts();
} // uProcessorKernel::uSchedule


inline void uProcessorKernel::uOnBehalfOfUser() {
    switch( uCode ) {
      case 0:
	break;
      case 1:
	uPrevLock->uRelease();
	break;
      case 2:
	uNextTask->uWake();
	break;
      case 3:
	uPrevLock->uRelease();
	uNextTask->uWake();
	break;
      default:
	uAbort( "(uProcessorKernel &)0x%p.uOnBehalfOfUser, internal error uCode:%d", this, uCode );
	break;
    } // switch
} // uProcessorKernel::uOnBehalfOfUser


void uProcessorKernel::uSetTimer( uDuration dur ) {
  if ( dur < 0 ) return;				// if duration is negative, it's invalid

    // For now, write only code for non-posix timer. When posix timer is
    // available use timer_create and timer_settimer.

    timeval conv = dur;
    itimerval it;
    it.it_value = conv;					// fill in the value to the next expiry
    it.it_interval.tv_sec = 0;				// not periodic
    it.it_interval.tv_XSEC = 0;
#ifdef __U_DEBUG_H__
    char buffer[256];
    uDebugPrtBuf( buffer, "uProcessorKernel::uSetTimer, it.it_value.tv_sec:%d, it.it_value.tv_ssec:%d\n", it.it_value.tv_sec, it.it_value.tv_usec );
#endif __U_DEBUG_H__
    uKernelModule::uSettingTimerRF = true;
    setitimer( ITIMER_REAL, &it, (itimerval *)0 );	// set the alarm clock to go off
    uKernelModule::uSettingTimerRF = false;
} // uProcessorKernel::uSetTimer


void uProcessorKernel::uSetTimer( uTime time ) {
  if ( time == 0 ) return;				// if time is zero, it's invalid

    // Since this is private, and only uEventList is a friend, the assumption
    // is made that the time parameter is always in real-time (not virtual
    // time)

#if defined( REALTIME_POSIX )
    timespec curr;
    if ( clocktype < 0 ) type = CLOCK_REALTIME;
    clock_gettime( type, &curr );
#else
    timeval curr;
    GETTIMEOFDAY( &curr );
#endif
    uTime currtime( curr.tv_sec, curr.tv_usec * 1000 );	// convert to nanoseconds

    uDuration dur = time - currtime;
    if ( dur <= 0 ) {					// if duration is negative (it has already past)
	// fill in the value to the next expiry by setting alarm to soonest
	// time an alarm may come
	uSetTimer( uDuration( 0, TIMEGRAN / 1000000L ) );
    } else {
	uSetTimer( dur );
    } // if
} // uProcessorKernel::uSetTimer


void uProcessorKernel::nextProcessor( int &uPrevPreemption, uSeqGen<uProcessorDL> &uGenProc, uProcessorDL *&uCurrProc ) {
    // Get next processor to execute.

    uPrevPreemption = uThisProcessor().uGetPreemption(); // remember preemption value
    if ( ! (uGenProc >> uCurrProc) ) {			// make list appear circular
	uGenProc.uOver( *uKernelModule::uGlobalProcessors ); // restart at beginning of list
	uGenProc >> uCurrProc;
    } // if
    uKernelModule::uActiveProcessor = &(uCurrProc->uGet());
    uKernelModule::uActiveCluster   = uKernelModule::uActiveProcessor->uCurrCluster;

    // The time slice must be reset or some programs will not work.

    if ( uThisProcessor().uGetPreemption() != uPrevPreemption ) {
	uThisProcessor().uSetContextSwitchEvent( uPrevPreemption );
    } // if
} // nextProcessor


void uProcessorKernel::main() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.main, child is born, stack:0x%p\n", this );
#endif __U_DEBUG_H__

    // This initialization code is executed only once as all other instances of
    // processor kernels are cloned by the UNIX fork.

    // Now that the kernel thread is off the UNIX stack and on the processor
    // kernel stack, it is possible to mmap the UNIX stack so that it is shared
    // among the UNIX processors. Sharing the UNIX stack is only necessary to
    // handle the execution of the global constructors and destructors by the
    // boot task because the boot task's thread may be migrated to other
    // processors during this phase.

    uMemoryModule::uStackInit();

#ifndef __U_MULTI__
    // Use a generator because processor nodes are deleted during the list
    // traversal.

    uSeqGen<uProcessorDL> uGenProc( *uKernelModule::uGlobalProcessors );

    // SKULLDUGGERY: The system processor in not on the global list until the
    // processor task runs, so explicitly set the current processor to the
    // system processor.

    uProcessorDL *uCurrProc = &(uKernelModule::uSystemProcessor->uGlobalRef);
    uProcessorDL *uCycleStart = (uProcessorDL *)0;
    int uPrevPreemption;
    bool &uOkToSelect = uIOCluster::NBIO->uOkToSelect;
    uBaseTask *&uIOPoller = uIOCluster::NBIO->uIOPoller;
#endif ! __U_MULTI__

    for ( int spin = 0;; ) {
	// Advance the spin counter now to detect if a task is executed.

	spin += 1;

	if ( ! uThisProcessor().uExternal.uEmpty() ) {	// check processor specific ready queue
	    // Only this processor removes from this ready queue so no other
	    // processor can remove this task after it has been seen.

	    uKernelModule::uActiveTask = uThisProcessor().uCurrTask = &(uThisProcessor().uExternal.uDropHead()->uGet());

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(1) bef: task 0x%p (%.256s) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:0x%p on processor:0x%p, %d,%d,%d,%d,%d,%d\n",
		       this, uThisProcessor().uCurrTask->uCurrCoroutine, uThisProcessor().uCurrTask->uCurrCoroutine->uName,
		       uThisProcessor().uCurrTask->uLimit, uThisProcessor().uCurrTask->uBase, uThisProcessor().uCurrTask->uSP,
		       uThisProcessor().uCurrCluster, &uThisProcessor(),
		       uKernelModule::uDisableInt, uKernelModule::uDisableIntCnt, uKernelModule::uDisableIntSpin, uKernelModule::uDisableIntSpinCnt, uKernelModule::uInKernelRF, uKernelModule::uSettingTimerRF
		);
#endif __U_DEBUG_H__

	    // SKULLDUGGERY: The processor task is part of the kernel, and
	    // therefore, must execute as uninterruptible code. By incrementing
	    // the interrupt counter here, the decrement when the processor
	    // task is scheduled leaves the processor task's execution in the
	    // kernel with regard to interrupts. This assumes the uDisableInt
	    // flag is set already.

	    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );
	    uKernelModule::uDisableInterrupts();
	    uSwitch( uThisProcessor().uCurrTask->uCurrCoroutine, this );
	    uKernelModule::uEnableInterrupts();
	    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(1) aft: task 0x%p (%.256s) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:0x%p on processor:0x%p, %d,%d,%d,%d,%d,%d\n",
		       this, uThisProcessor().uCurrTask->uCurrCoroutine, uThisProcessor().uCurrTask->uCurrCoroutine->uName,
		       uThisProcessor().uCurrTask->uLimit, uThisProcessor().uCurrTask->uBase, uThisProcessor().uCurrTask->uSP,
		       uThisProcessor().uCurrCluster, &uThisProcessor(),
		       uKernelModule::uDisableInt, uKernelModule::uDisableIntCnt, uKernelModule::uDisableIntSpin, uKernelModule::uDisableIntSpinCnt, uKernelModule::uInKernelRF, uKernelModule::uSettingTimerRF
		);
#endif __U_DEBUG_H__

	    spin = 0;					// set number of spins back to zero
	    uOnBehalfOfUser();				// execute code on scheduler stack on behalf of user

	    if ( uForkChild ) {
		// The fork must occur here because only now is execution on
		// the private stack for the processor kernel.

		uFork();
		continue;				// make sure the processor task is started first
	    } // if
	    if ( uTerminated ) {
#ifdef __U_MULTI__
		if ( &uThisProcessor() != uKernelModule::uSystemProcessor ) break;
		// If control reaches here, the boot task must be the only task
		// on the system-cluster ready-queue, and it must be restarted
		// to finish the close down.
#else
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main termination, uCurrProc:0x%p, uThisProcessor:0x%p\n",
			   this, &(uCurrProc->uGet()), &uThisProcessor() );
#endif __U_DEBUG_H__
		// In the uniprocessor case, only terminate the processor kernel
		// when the processor task for the system processor is deleted;
		// otherwise the program stops when the first processor is deleted.

		if ( &uThisProcessor() != uKernelModule::uSystemProcessor ) {
		    uTerminated = false;

		    // Get next processor to execute because the current one
		    // just terminated.

		    nextProcessor( uPrevPreemption, uGenProc, uCurrProc );
		} // if
#endif __U_MULTI__
	    } // if
	} // if

	uBaseTask *uReadyTask = &(uThisProcessor().uCurrCluster->uReadyTaskTryRemove());
	if ( uReadyTask != (uBaseTask *)0 ) {		// ready queue not empty, schedule that task
	    uAssert( ! uReadyTask->uReadyRef.uListed() );
	    uKernelModule::uActiveTask = uThisProcessor().uCurrTask = uReadyTask;

#ifdef __U_MULTI__
	    // It is necessary to touch the memory base before a context switch
	    // from the uC++ kernel to a task. The problem occurs when one
	    // processor tries to execute a task that is not mapped into its
	    // address space. Normally, loading the invalid stack pointer
	    // causes a sigsegv, and the sigsegv handler fixes up the address
	    // space. However, if the invalid stack pointer is loaded and a
	    // sigalrm is delivered before the sigsegv, the sigalrm cannot be
	    // delivered on the stack because it is invalid. The problem is
	    // that control is now in the UNIX kernel, which is trying to
	    // deliver the sigalrm, when the invalid stack pointer is detected;
	    // UNIX does not invoke the sigsegv handler in the user program.
	    // This is the same problem that occurs when an invalid address is
	    // given to an I/O routine.  The I/O routine accesses the pointer
	    // in the kernel, and the kernel does not invoke the user's sigsegv
	    // handler.  The fundamental problem is that UNIX should
	    // recursively invoke the user's sigsegv for invalid address and it
	    // does not.

	    int dummy;
	    dummy = *(unsigned int *)(uMemoryModule::uProgramFileEnd - sizeof(int));
#endif __U_MULTI__

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(2) bef: task 0x%p (%.256s) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:0x%p on processor:0x%p, %d,%d,%d,%d,%d,%d\n",
		       this, uReadyTask->uCurrCoroutine, uReadyTask->uCurrCoroutine->uName,
		       uReadyTask->uCurrCoroutine->uLimit, uReadyTask->uCurrCoroutine->uBase, uReadyTask->uCurrCoroutine->uSP,
		       uThisProcessor().uCurrCluster, &uThisProcessor(),
		       uKernelModule::uDisableInt, uKernelModule::uDisableIntCnt, uKernelModule::uDisableIntSpin, uKernelModule::uDisableIntSpinCnt, uKernelModule::uInKernelRF, uKernelModule::uSettingTimerRF
		);
#endif __U_DEBUG_H__

	    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );
	    uSwitch( uReadyTask->uCurrCoroutine, this );
	    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(2) aft: task 0x%p (%.256s) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:0x%p on processor:0x%p, %d,%d,%d,%d,%d,%d\n",
		       this, uReadyTask->uCurrCoroutine, uReadyTask->uCurrCoroutine->uName,
		       uReadyTask->uLimit, uReadyTask->uBase, uReadyTask->uSP,
		       uThisProcessor().uCurrCluster, &uThisProcessor(),
		       uKernelModule::uDisableInt, uKernelModule::uDisableIntCnt, uKernelModule::uDisableIntSpin, uKernelModule::uDisableIntSpinCnt, uKernelModule::uInKernelRF, uKernelModule::uSettingTimerRF
		);
#endif __U_DEBUG_H__

#ifdef __U_MULTI__
	    spin = 0;					// set number of spins back to zero
#else
	    // Poller task does not count as an executed task, if its last
	    // execution found no I/O and this processor's ready queue is
	    // empty. Check before calling uOnBehalfOfUser, because uIOPoller
	    // may put itself back on the ready queue, which makes the ready
	    // queue appear non-empty.

	    if ( uReadyTask != uIOPoller || uIOCluster::NBIO->uFound != 0 || ! uThisProcessor().uCurrCluster->uReadyTasksEmpty() ) {
		spin = 0;				// set number of spins back to zero
	    } // if
#endif __U_MULTI__
	    uOnBehalfOfUser();				// execute code on scheduler stack on behalf of user
	} // if

#ifdef __U_MULTI__
	if ( spin > uThisProcessor().uGetSpin() ) {
	    // SKULLDUGGERY: During pausing, the processor's uCurrTask should
	    // not point to the last task executed. Otherwise, the visualizer
	    // incorrectly displays the last task executed during pausing. So
	    // uCurrTask is set to the uProcessorTask for this processor.

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, on processor 0x%p nothing to do\n", this, &uThisProcessor() );
#endif __U_DEBUG_H__
	    uKernelModule::uActiveTask = uThisProcessor().uCurrTask = uThisProcessor().uProcTask;
	    uThisProcessor().uCurrCluster->uProcessorPause(); // put processor to sleep
	    spin = 0;					// set number of spins back to zero
	} // if
#else
	// A cycle starts when a processor executes no tasks. If the cycle
	// completes and no task has executed, deadlock has occurred unless
	// there are pending I/O tasks. If there are pending I/O tasks, the I/O
	// poller task pauses the UNIX process at the "select".

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uProcessorKernel &)0x%p.main, uCycleStart:0x%p, uCurrProc:0x%p (%.256s), spin:%d, uReadyTask:0x%p, uIOPoller:0x%p\n",
		  this, uCycleStart, uCurrProc, uCurrProc->uGet().uCurrCluster->uGetName(), spin, uReadyTask, uIOPoller );
#endif __U_DEBUG_H__

	if ( uCycleStart == uCurrProc && spin != 0 ) {
#if __U_LOCALDEBUGGER_H__
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, uLocalDebuggerInstance:0x%p, uIOPoller: 0x%p (%.256s), dispatcher:0x%p, debugger_blocked_tasks:%d, uPendingIO.uHead:0x%p, uPendingIO.uTail:0x%p\n",
		       this, uLocalDebugger::uLocalDebuggerInstance,
		       uIOPoller, uIOPoller != NULL ? uIOPoller->uGetName() : "no I/O poller task",
		       uLocalDebugger::uLocalDebuggerInstance != (uLocalDebugger *)0 ? uLocalDebugger::uLocalDebuggerInstance->dispatcher : NULL,
		       uLocalDebugger::uLocalDebuggerInstance != (uLocalDebugger *)0 ? uLocalDebugger::uLocalDebuggerInstance->debugger_blocked_tasks : 0,
		       uIOCluster::NBIO->uPendingIO.uHead(), uIOCluster::NBIO->uPendingIO.uTail() );
#endif __U_DEBUG_H__
#endif __U_LOCALDEBUGGER_H__
	    if ( uIOPoller != (uBaseTask *)0		// I/O poller task ?
#if __U_LOCALDEBUGGER_H__
		&& (
		    uLocalDebugger::uLocalDebuggerInstance == (uLocalDebugger *)0 || // local debugger initialized ?
		    uIOPoller != (uBaseTask *)uLocalDebugger::uLocalDebuggerInstance->dispatcher || // I/O poller the local debugger reader ?
		    uLocalDebugger::uLocalDebuggerInstance->debugger_blocked_tasks != 0 || // any tasks debugger blocked ?
		    uIOCluster::NBIO->uPendingIO.uHead() != uIOCluster::NBIO->uPendingIO.uTail() // any other tasks waiting for I/O ?
		    )
#endif __U_LOCALDEBUGGER_H__
		) {
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main, poller blocking\n", this );
#endif __U_DEBUG_H__

		uOkToSelect = true;			// tell poller it is ok to call UNIX select, reset in uPollIO
	    } else if ( uThisProcessor().uEvents->uUserEventPresent() ) {  // tasks sleeping, except system task  ?
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main, sleeping with pending events\n", this );
#endif __U_DEBUG_H__
		uThisProcessor().uCurrCluster->uProcessorPause(); // put processor to sleep
#if 0
		// stop generating SIGALRM signals on this processor until woken up

		uThisProcessor().uSetContextSwitchEvent( uDuration( 0, 0 ) ); // turn off context-switching

		// real-time

		uTime nextAlarm;
		nextAlarm = uThisProcessor().uEvents->uNextAlarm();	// find next event time

		// Block any SIGALRM signals from arriving.
#ifdef __svr4__
		sighold( SIGALRM );
#else
		unsigned int mask = sigblock( sigmask( SIGALRM ) );
#endif __svr4__
		uDuration resolution( 0, CLOCKGRAN / ( 1000000000L / TIMEGRAN ) );
		if ( uKernelModule::uInKernelRF && nextAlarm != 0 &&
		     uActiveProcessorKernel->uKernelClock.uGetTime() + resolution >= nextAlarm ) { // roll-forward flag on and valid event in the event-list?
#ifdef __svr4__
		    sigrelse( SIGALRM );
#else
		    sigsetmask( mask );
#endif __svr4__
		    uKernelModule::uRollForward( true ); // make sure to do chores
		} else {
		    // Install the old signal mask and wait for a signal to arrive.
#ifdef __svr4__
		    sigpause( SIGALRM );
		    sigrelse( SIGALRM );
#else
		    sigpause( mask );
		    sigsetmask( mask );
#endif __svr4__
#ifdef __U_DEBUG_H__
		    uDebugPrt( "(uProcessorkernel &)0x%p., after sigpause\n", this );
#endif __U_DEBUG_H__
		    // Just woken up after an alarm but in kernel/library code,
		    // so no actual popping from the event list took place in
		    // the sigalarm handler (i.e., signal alarm handler just
		    // ignored the interrupt).  Therefore, do a roll-forward to
		    // ensure that any necessary events are popped from the
		    // event list.

		    uKernelModule::uRollForward( true );
		} // if

		uThisProcessor().uSetContextSwitchEvent( uThisProcessor().uGetPreemption() ); // reset context-switch
#endif
	    } else {
		// Time slice can occur during processing of deadlock, so
		// control might come here multiple times.

		if ( ! uKernelModule::uDeadlock ) {
		    // System task detects the deadlock and aborts the program.

		    uKernelModule::uDeadlock = true;
		    uEventNode *node = uThisProcessor().uEvents->uNextEvent(); // get the system task's event node
		    node->SigHandler->uHandler();			       // reschedule the system task
		} else {
		    uKernelModule::uDeadlock = false;
		    uAbort( "(uProcessorKernel &)0x%p.main(): internal error, during deadlock shutdown.", this );
		} // if
		spin = 0;				// now loop through processors again to start system task
	    } // if
	} // if

	if ( spin == 0 ) {				// task executed ?
	    uCycleStart = uCurrProc;			// mark new start for cycle
	} else {
	    // SKULLDUGGERY: During spinning, the processor's uCurrTask should
	    // not point to the last task executed. Otherwise, the visualizer
	    // incorrectly displays the last task executed during spinning. So
	    // uCurrTask is set to the uProcessorTask for this processor.

	    uKernelModule::uActiveTask = uThisProcessor().uCurrTask = uThisProcessor().uProcTask;
	} // if

	// Get next processor to execute.

	nextProcessor( uPrevPreemption, uGenProc, uCurrProc );
#endif __U_MULTI__
    } // for

    // ACCESS NO KERNEL DATA STRUCTURES AFTER THIS POINT BECAUSE THEY MAY NO
    // LONGER EXIST.

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.main, exiting\n", this );
#endif __U_DEBUG_H__

    _exit( 0 );						// exit without calling the global destructors.
} // uProcessorKernel::main


uProcessorKernel::uProcessorKernel() : uBaseCoroutine( 1.0 ) {
    uTerminated = false;
    uForkChild = false;

    uThisProcessor().uProcessorClock = &uActiveProcessorKernel->uKernelClock;
    uKernelModule::uInKernelRF = 0;
} // uProcessorKernel::uProcessorKernel

uProcessorKernel::~uProcessorKernel() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.~uProcessorKernel, exiting\n", this );
#endif __U_DEBUG_H__
    uStorage = NULL;					// don't free static processor stack
} // uProcessorKernel::~uProcessorKernel


//######################### uProcessor #########################


void uProcessor::uCreateProcessor( uCluster &uClus, int ms, int spin ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessor &)0x%p.uCreateProcessor, on cluster 0x%p (%s)\n", this, &uClus, uClus.uGetName() );
#endif __U_DEBUG_H__

#ifdef __U_DEBUG__
    if ( ms == 0 ) {					// 0 => infinity, reset to minimum preemption second for local debugger
	ms = uMinPreemption;				// approximate infinity
    } else if ( ms < 0 ) {				// special case for testing, not for public consumption
	ms = 0;
    } // if
#ifdef __U_MULTI__
    uDebugIgnore = false;
#else
    uDebugIgnore = true;
#endif __U_MULTI__
#endif __U_DEBUG__

    uCurrTask = (uBaseTask *)0;
    uCurrCluster = &uClus;
    uPreemption = ms;
    uSpin = spin;

#ifdef __U_MULTI__
    uEvents = new uEventList;
    uContextSwitchHandler = new uCxtSwtchHndlr;
    uContextEvent = new uEventNode( *uContextSwitchHandler );
#endif __U_MULTI__
    uCurrCluster->uProcessorAdd( *this );
    uProcTask = new uProcessorTask( uClus, *this );
} // uProcessor::uCreateProcessor


uProcessor::uProcessor( uCluster &uClus, double ) : uIdleRef( *this ), uProcessorRef( *this ), uGlobalRef( *this ) {
    uCreateProcessor( uClus, uDefaultPreemption(), uDefaultSpin() );
} // uProcessor::uProcessor


uProcessor::uProcessor( int ms, int spin ) : uIdleRef( *this ), uProcessorRef( *this ), uGlobalRef( *this ) {
    uCreateProcessor( uThisCluster(), ms, spin );
    uThisProcessor().uFork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::uProcessor( uCluster &uClus, int ms, int spin ) : uIdleRef( *this ), uProcessorRef( *this ), uGlobalRef( *this ) {
    uCreateProcessor( uClus, ms, spin );
    uThisProcessor().uFork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::~uProcessor() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessor &)0x%p.~uProcessor\n", this );
#endif __U_DEBUG_H__
    delete uProcTask;
    uCurrCluster->uProcessorRemove( *this );
#ifdef __U_MULTI__
    delete uContextEvent;
    delete uContextSwitchHandler;
    delete uEvents;
#endif __U_MULTI__
} // uProcessor::~uProcessor


void uProcessor::uFork( uProcessor *np ) {
    uProcTask->uFork( np );

    // Asynchronous with regard to other processors, synchronous with regard to
    // this processor.

    if ( &uThisProcessor() == this ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uProcessor &)0x%p.uFork, extra yield\n", this );
#endif __U_DEBUG_H__
	uThisTask().uYield();
    } // if
} // uProcessor::uFork


void uProcessor::uSetContextSwitchEvent( uDuration dur ) {
    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt == 1 );

    if ( ! uContextEvent->uListed() && dur != 0 ) {	// first context switch event ?
	uContextEvent->timerT = uActiveProcessorKernel->uKernelClock.uGetTime() + dur;
	uContextEvent->timerD = dur;
	uEvents->uAddEvent( *uContextEvent );
    } else if ( dur > 0  && uContextEvent->timerD != dur ) { // if event is different from previous ? change it
	uEvents->uRemoveEvent( *uContextEvent );
	uContextEvent->timerT = uActiveProcessorKernel->uKernelClock.uGetTime() + dur;
	uContextEvent->timerD = dur;
	uEvents->uAddEvent( *uContextEvent );
    } else if ( dur == 0 && uContextEvent->timerT != 0 ) { // zero duration and current CS is nonzero ?
	uEvents->uRemoveEvent( *uContextEvent );
	uContextEvent->timerT = 0;   
	uContextEvent->timerD = 0;
    } // if
}; // uProcessor::uSetContextSwitchEvent


void uProcessor::uSetContextSwitchEvent( int msecs ) {
    uSetContextSwitchEvent( uDuration( msecs / 1000L, msecs % 1000L * ( TIMEGRAN / 1000L ) ) ); // convert msecs to uDuration type
} // uProcessor::uSetContextSwitchEvent


uClock &uProcessor::uGetClock() const {
    return *uProcessorClock;
} // uProcessor::uGetClock


int uProcessor::uGetPid() const {
    return uPid;
} // uProcessor::uGetPid


uCluster &uProcessor::uGetCluster() const {
    return *uCurrCluster;
} // uProcessor::uGetCluster


uCluster &uProcessor::uSetCluster( uCluster &uClus ) {
  if ( &uClus == &(uThisProcessor().uGetCluster()) ) return uClus; // trivial case

    uCluster &prev = uClus;
    uProcTask->uSetCluster( uClus );			// operation must be done by the processor itself
    return prev;
} // uProcessor::uSetCluster


uBaseTask &uProcessor::uGetTask() const {
    return *uCurrTask;
} // uProcessor::uGetTask


int uProcessor::uSetPreemption( int ms ) {
#ifdef __U_DEBUG__
    if ( ms == 0 ) {					// 0 => infinity, reset to minimum preemption second for local debugger
	ms = uMinPreemption;				// approximate infinity
    } else if ( ms < 0 ) {				// special case for testing, not for public consumption
	ms = 0;
    } // if
#endif __U_DEBUG__
    int prev = uPreemption;
    uProcTask->uSetPreemption( ms );

    // Asynchronous with regard to other processors, synchronous with regard to
    // this processor.

    if ( &uThisProcessor() == this ) {
	uThisTask().uYield();
    } // if
    return prev;
} // uProcessor::uSetPreemption


int uProcessor::uGetPreemption() const {
    return uPreemption;
} // uProcessor::uGetPreemption


int uProcessor::uSetSpin( int spin ) {
    int prev = uSpin;
    uSpin = spin;
    return prev;
} // uProcessor::uSetSpin


int uProcessor::uGetSpin() const {
    return uSpin;
} // uProcessor::uGetSpin


bool uProcessor::uIdle() const {
    return uIdleRef.uListed();
} // uProcessor::uIdle


void *uProcessor::uMmap( void *begin, int size, int prot, int flags, int fd, int off ) {
    return uProcTask->uMmap( begin, size, prot, flags, fd, off );
} // uProcessor::uMmap


// Local Variables: //
// compile-command: "dmake" //
// End: //
