//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uNBIO.cc -- non-blocking IO
// 
// Author           : Peter A. Buhr
// Created On       : Mon Mar  7 13:56:53 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Jun  1 16:46:04 1999
// Update Count     : 372
// 


#define __U_KERNEL__
#ifndef __U_DEBUG__
#define NDEBUG						// turn off assert
#endif

#if defined( __linux__ )
#define _XOPEN_SOURCE
#define _BSD_SOURCE
#endif

#include <uC++.h>
#include <uAssert.h>
//#include <uDebug.h>
#ifdef __U_DEBUG_H__
#include <stdio.h>
#endif __U_DEBUG_H__


#include <string.h>					// strerror
#include <errno.h>
#include <sys/socket.h>
#if defined( __linux__ )
#include <sys/param.h>					// howmany
#endif

#ifndef NBBY
#define NBBY 8
#endif


#if ! defined( __svr4__ ) && defined( __sun__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *); // missing from
#endif

#if defined( __sgi__ )
extern "C" void bzero(void *, int);			// missing from
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *); // missing from
#endif

#if defined( __aix__ ) && defined( __ibm__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from
#endif

#if defined( __hpux__ ) && defined( __hp__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
#endif

#if defined( __ultrix__ ) && defined( __dec__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *); // missing from
#endif

#if defined( __gizmo__ )
extern "C" char *strerror(int n);
#endif


static const int uSelectTimeOut = 1;			// how many seconds may elapse before call to select times out
static const short uBitsPerValue[16] = { 0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4 };

static int uCountBits( fd_mask x ) {
    int cnt;

    for ( cnt = 0; x > 0; x >>= 4 ) {
	cnt += uBitsPerValue[x & 0xf];
    } // for
    return cnt;
} // uCountBits


//######################### uNBIO #########################


void uNBIO::uCheckIO( uDuration delay ) {
    unsigned int i;
    struct timeval timeout = delay;			// convert to timeval for select

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uCheckIO: timeout:%d,%d\n", this, timeout.tv_sec, timeout.tv_usec );
#endif __U_DEBUG_H__

    // Make a local copy of the fd sets because the select operation destroys
    // the sets and the master information is lost.

    fd_set lrfds = uRFDs;
    fd_set lwfds = uWFDs;
    fd_set lefds = uEFDs;

    // This processor is about to become idle. First, interrupts are disabled
    // because the following operations affect some kernel data structures.
    // Second, preemption is turned off because it is now controlled by the
    // "select" timeout.  Third, the idle state is set so that if a task is
    // migrated to the cluster this processor is on, the processor is woken
    // up. Unfortunately, there is race condition between seeing that the
    // processor is idle and the SIGALRM sent to wake it; the signal can be
    // sent and received before the UNIX process actually blocks with the
    // "select". In such a case, the SIGALRM is treated as spurious.
    // Therefore, it is necessary to poll.

    uKernelModule::uDisableInterrupts();

    int uPrevTime = uThisProcessor().uGetPreemption();	// remember previous preemption time
    if ( uPrevTime != 0 ) {				// optimize out UNIX call if possible
	uThisProcessor().uSetContextSwitchEvent( uDuration( 0, 0 ) ); // turn off preemption or it keeps waking the UNIX processor
    } // if

    uThisCluster().uMakeProcessorIdle( uThisProcessor() );

#ifdef __U_DEBUG_H__
//    uDebugAcquire();
//    uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lrfds:", this );
//    for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	uDebugPrt2( "%x ", lrfds.fds_bits[i] );
//    } // for
//    uDebugPrt2( "\n" );
//    uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lwfds:", this );
//    for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	uDebugPrt2( "%x ", lwfds.fds_bits[i] );
//    } // for
//    uDebugPrt2( "\n" );
//    uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lefds:", this );
//    for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	uDebugPrt2( "%x ", lefds.fds_bits[i] );
//    } // for
//    uDebugPrt2( "\n" );
//    uDebugRelease();
#endif __U_DEBUG_H__

    // Check for IO pending on any descriptors.

#if defined( __hpux__ ) && defined( __hp__ )
    uFound = select( FD_SETSIZE, (int *)&lrfds, (int *)&lwfds, (int *)&lefds, &timeout );
#else
    uFound = select( FD_SETSIZE, &lrfds, &lwfds, &lefds, &timeout );
#endif

    uThisCluster().uMakeProcessorActive( uThisProcessor() );

    if ( uPrevTime != 0 ) {				// optimize out UNIX call if possible
	uThisProcessor().uSetContextSwitchEvent( uPrevTime ); // reset processor preemption time
    } // if

    uKernelModule::uEnableInterrupts();			// does necessary roll forward

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uCheckIO: select returns: found %d, timeout:%d,%d\n", this, uFound, timeout.tv_sec, timeout.tv_usec );
#endif __U_DEBUG_H__

    if ( uFound > 0 ) {					// I/O has occurred ?

#ifdef __U_DEBUG_H__
//	uDebugAcquire();
//	uDebugPrt2( "(uNBIO &)0x%p.uCheckIO: found %d pending IO operations\n", this, uFound );
//	uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lrfds:", this );
//	for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	    uDebugPrt2( "%x ", lrfds.fds_bits[i] );
//	} // for
//	uDebugPrt2( "\n" );
//	uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lwfds:", this );
//	for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	    uDebugPrt2( "%x ", lwfds.fds_bits[i] );
//	} // for
//	uDebugPrt2( "\n" );
//	uDebugPrt2( "(uNBIO &)0x%p.uCheckIO, lefds:", this );
//	for ( i = 0; i < howmany(FD_SETSIZE, NFDBITS); i += 1 ) {
//	    uDebugPrt2( "%x ", lefds.fds_bits[i] );
//	} // for
//	uDebugPrt2( "\n" );
//	uDebugRelease();
#endif __U_DEBUG_H__

	// Remove the ready I/O fds from the master lists as these I/O
	// operations have completed.

	for ( i = 0; i < howmany( FD_SETSIZE, NFDBITS ); i += 1 ) {
	    uRFDs.fds_bits[i] ^= lrfds.fds_bits[i];
	    uWFDs.fds_bits[i] ^= lwfds.fds_bits[i];
	    uEFDs.fds_bits[i] ^= lefds.fds_bits[i];
	} // for

	// Check to see which tasks are waiting for ready I/O operations and
	// wake them.

	uNBIOnode *p;
	int cnt;
	for ( uSeqGen<uNBIOnode> gen(uPendingIO); gen >> p; ) {
	    cnt = 0;
	    if ( p->fdType == uNBIOnode::singleFd ) {	// single fd
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uNBIO &)0x%p.uCheckIO: found task waiting on single fd %d with mask 0x%x\n",
			   this, p->smfd.sfd.fd, *p->smfd.sfd.uRWE );
#endif __U_DEBUG_H__
		int temp = 0;
		if ( (*p->smfd.sfd.uRWE & uIOCluster::uReadSelect) && FD_ISSET( p->smfd.sfd.fd, &lrfds ) ) {
		    temp |= uIOCluster::uReadSelect;
		    cnt += 1;
		} // if
		if ( (*p->smfd.sfd.uRWE & uIOCluster::uWriteSelect ) && FD_ISSET( p->smfd.sfd.fd, &lwfds ) ) {
		    temp |= uIOCluster::uWriteSelect;
		    cnt += 1;
		} // if
		if ( (*p->smfd.sfd.uRWE & uIOCluster::uExceptSelect) && FD_ISSET( p->smfd.sfd.fd, &lefds ) ) {
		    temp |= uIOCluster::uExceptSelect;
		    cnt += 1;
		} // if
		if ( cnt != 0 ) {			// reset user mask only if I/O available
		    *p->smfd.sfd.uRWE = temp;
		} // if
	    } else {					// multiple fds
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uNBIO &)0x%p.uCheckIO: found task waiting for multiple fd\n", this );
#endif __U_DEBUG_H__
		fd_set tfds;				// temporary masks to construct result mask
		int tcnt;
		int nbytes = ( p->smfd.mfd.uNFDs + ( NBBY - 1 ) ) / NBBY;

		if ( p->smfd.mfd.uRFDs ) {		// non-zero user mask ?
		    FD_ZERO( &tfds );
		    tcnt = 0;
		    for ( i = 0; i < howmany( p->smfd.mfd.uNFDs, NFDBITS ); i += 1 ) {
			tfds.fds_bits[i] = p->smfd.mfd.uRFDs->fds_bits[i] & lrfds.fds_bits[i];
			tcnt += uCountBits( tfds.fds_bits[i] );
		    } // for
		    if ( tcnt != 0 ) {			// reset user mask only if I/O available
#ifdef __svr4__
			memcpy( p->smfd.mfd.uRFDs, &tfds, nbytes );
#else
			bcopy( &tfds, p->smfd.mfd.uRFDs, nbytes );
#endif
			cnt += tcnt;
		    } // if
		} // if
		if ( p->smfd.mfd.uWFDs ) {		// non-zero user mask ?
		    FD_ZERO( &tfds );
		    tcnt = 0;
		    for ( i = 0; i < howmany( p->smfd.mfd.uNFDs, NFDBITS ); i += 1 ) {
			tfds.fds_bits[i] = p->smfd.mfd.uWFDs->fds_bits[i] & lwfds.fds_bits[i];
			tcnt += uCountBits( tfds.fds_bits[i] );
		    } // for
		    if ( tcnt != 0 ) {			// reset user mask only if I/O available
#ifdef __svr4__
			memcpy( p->smfd.mfd.uWFDs, &tfds, nbytes );
#else
			bcopy( &tfds, p->smfd.mfd.uWFDs, nbytes );
#endif
			cnt += tcnt;
		    } // if
		} // if
		if ( p->smfd.mfd.uEFDs ) {		// non-zero user mask ?
		    FD_ZERO( &tfds );
		    tcnt = 0;
		    for ( i = 0; i < howmany( p->smfd.mfd.uNFDs, NFDBITS ); i += 1 ) {
			tfds.fds_bits[i] = p->smfd.mfd.uEFDs->fds_bits[i] & lefds.fds_bits[i];
			tcnt += uCountBits( tfds.fds_bits[i] );
		    } // for
		    if ( tcnt != 0 ) {			// reset user mask only if I/O available
#ifdef __svr4__
			memcpy( p->smfd.mfd.uEFDs, &tfds, nbytes );
#else
			bcopy( &tfds, p->smfd.mfd.uEFDs, nbytes );
#endif
			cnt += tcnt;
		    } // if
		} // if
	    } // if
	    if ( cnt != 0 || p->uTimedout ) {		// I/O completed for this task or timed out ?
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uNBIO &)0x%p.uCheckIO: removing node 0x%p\n", this, p );
#endif __U_DEBUG_H__
		uPendingIO.uRemove( p );		// remove node from list of waiting tasks
		p->nfds = cnt;				// set return value
		uSignal p->uPending;			// wake up waiting task (empty for IOPoller)
	    } // if
	} // for
    } else if ( uFound == 0 ) {				// time limit expired, no IO is ready
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uNBIO &)0x%p.uCheckIO: time limit expired, errno:%d\n", this, errno );
#endif __U_DEBUG_H__
	// Check for timed out IO

	uNBIOnode *p;
	for ( uSeqGen<uNBIOnode> gen(uPendingIO); gen >> p; ) {
	    if ( p->uTimedout ) {			// timed out waiting for I/O for this task ?
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uNBIO &)0x%p.uCheckIO: removing node 0x%p\n", this, p );
#endif __U_DEBUG_H__
		uPendingIO.uRemove( p );		// remove node from list of waiting tasks
		p->nfds = 0;				// set return value
		uSignal p->uPending;			// wake up waiting task (empty for IOPoller)
	    } // if
	} // for
    } else {
	// Either an EINTR occurred or one of the clients specified a bad file
	// number, and a EBADF was received.  This is handled by waking up all
	// the clients, telling them that IO has occured so that they will
	// retry their IO operation and get the error message at that point.
	// They can recover from their errors at that point more gracefully
	// than can be done here.

	if ( errno == EINTR ) {
	    // probably sigalrm from migrate, do nothing
	} else if ( errno == EBADF ) {
	    // Received an unexpected error, chances are that one of the tasks
	    // has fouled up a call to some IO routine. Wake up all the tasks
	    // that were waiting for IO, allow them to retry their IO call and
	    // hope they catch the error this time.

	    uNBIOnode *p;
	    for ( uSeqGen<uNBIOnode> gen(uPendingIO); gen >> p; ) {
		uPendingIO.uRemove( p );		// remove node from list of waiting tasks
		p->nfds = -1;				// mark the fact that something is wrong
		uSignal p->uPending;			// wake up waiting task (empty for IOPoller)
	    } // for
	} else {
	    uAbort( "(uNBIO &)0x%p.uCheckIO() : internal error, error(%d): %s", this, errno, strerror( errno ) );
	} // if
    } // if
} // uNBIO::uCheckIO


inline bool uNBIO::uCheckPoller( uNBIOnode &uNode ) {
    uPendingIO.uAddTail( &uNode );			// node is removed by IOPoller

    // If this is the first task to register interest, this task is nominated
    // as the IO poller task.

    if ( uIOPoller == (uBaseTask *)0 ) {
	uIOPoller = &uThisTask();			// make this task the poller
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uNBIO &)0x%p.uCheckPoller, set poller task 0x%p (%.256s)\n", this, uIOPoller, uIOPoller->uGetName() );
#endif __U_DEBUG_H__
	return true;
    } else {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uNBIO &)0x%p.uCheckPoller, blocking non-poller I/O task 0x%p (%.256s))\n", this, &uThisTask(), uThisTask().uGetName() );
#endif __U_DEBUG_H__
	// This is not the poller task so add it to the list of tasks waiting
	// for IO events.  It is woken when an IO event is ready or nominated
	// as the poller.

	uWait uNode.uPending;
	if ( uNode.uListed() ) {			// poller ?
	    uAssert( uIOPoller == &uThisTask() );
	    return true;
	} else {
	    return false;
	} // if
    } // if
} // uNBIO::uCheckPoller


bool uNBIO::uPollIO( uNBIOnode &uNode ) {
    // If there are other tasks on the ready queue of this cluster that can
    // still execute, do a nonblocking select call, otherwise do a blocking
    // select call so that the application does not waste cpu cycles.  In
    // either case, return to the caller the number of IO operations that are
    // now possible.  If this number is zero, the caller yields the processor,
    // allowing other tasks to execute and then tries again.  Notice that this
    // delay must occur outside of the mutex members of this monitor, so that
    // other tasks can enter the monitor and register their interest in other
    // IO events.

#ifdef __U_MULTI__
    if ( uThisCluster().uReadyTasksEmpty() ) {
#else
    // uOkToSelect is set in the uniprocessor kernel when *all* clusters have
    // empty ready queues.
    if ( uOkToSelect ) {
#endif __U_MULTI__
	uTime nexttime;
	uDuration timeout;

	nexttime = uThisProcessor().uEvents->uNextAlarm(); // find first non-context-switch event
	timeout = nexttime - uThisProcessor().uGetClock().uGetTime(); // time to sleep

	// The minimum wake up time is always uSelectTimeOut to deal with the
	// problem of a missing SIGALRM due to the race condition with
	// "select".

	if ( 0 < timeout && timeout < uSelectTimeOut ) {
	    uCheckIO( timeout );
	} else {
	    uCheckIO( uSelectTimeOut );
	} // if

	uOkToSelect = false;				// reset select flag
    } else {
	uCheckIO( 0 );
    } // if

    // If the IOPoller's I/O completed, attempt to nominate another waiting
    // task to be the IOPoller.

    if ( ! uNode.uListed() ) {				// IOPoller's node removed ?
	if ( ! uPendingIO.uEmpty() ) {			// any other tasks waiting for IO?
	    // For the uniprocessor kernel, the new poller will appear to have
	    // found some IO because uFound is not cleared until the new poller
	    // makes a call to check for IO.

	    uNBIOnode *p = uPendingIO.uHead();		// node from list of waiting tasks
	    uIOPoller = p->uPendingTask;		// next poller task
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uNBIO &)0x%p.uPollIO, poller 0x%p (%.256s) nominating task 0x%p (%.256s) to be next poller\n",
		       this, &uThisTask(), uThisTask().uGetName(), uIOPoller, uIOPoller->uGetName() );
#endif __U_DEBUG_H__
	    uSignal p->uPending;			// wake up waiting task
	} else {
	    uIOPoller = (uBaseTask *)0;
	} // if
	return false;
    } else {
	return true;
    } // if
} // uNBIO::uPollIO


bool uNBIO::uInitSfd( uNBIOnode &uNode ) {
    // set the appropriate fd bit in the master fd mask
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uReadSelect ) FD_SET( uNode.smfd.sfd.fd, &uRFDs );
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uWriteSelect ) FD_SET( uNode.smfd.sfd.fd, &uWFDs );
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uExceptSelect ) FD_SET( uNode.smfd.sfd.fd, &uEFDs );

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uInitSfd, adding node 0x%p\n", this, &uNode );
#endif __U_DEBUG_H__

    return uCheckPoller( uNode );
} // uNBIO::uInitSfd


bool uNBIO::uInitSfd( uNBIOnode &uNode, uEventNode &uTimeoutEvent, uProcessor &proc ) {
    // set the appropriate fd bit in the master fd mask
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uReadSelect ) FD_SET( uNode.smfd.sfd.fd, &uRFDs );
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uWriteSelect ) FD_SET( uNode.smfd.sfd.fd, &uWFDs );
    if ( *uNode.smfd.sfd.uRWE & uIOCluster::uExceptSelect ) FD_SET( uNode.smfd.sfd.fd, &uEFDs );

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uInitSfd, adding node 0x%p\n", this, &uNode );
#endif __U_DEBUG_H__

    proc.uEvents->uAddEvent( uTimeoutEvent, proc );

    return uCheckPoller( uNode );
} // uNBIO::uInitSfd


uNBIO::uNBIO() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uNBIO\n", this );
#endif __U_DEBUG_H__
    uIOPoller = (uBaseTask *)0;				// no task is first yet
    FD_ZERO( &uRFDs );					// clear the read set
    FD_ZERO( &uWFDs );					// clear the write set
    FD_ZERO( &uEFDs );					// clear the exceptional set
} // uNBIO::uNBIO


uNBIO::~uNBIO() {
} // uNBIO::~uNBIO


int uNBIO::uSelect( int fd, int &rwe, timeval *timeout ) {
#ifdef __U_DEBUG__
    if ( fd < 0 || FD_SETSIZE <= fd ) {
	uAbort( ": attempt to select on file descriptor %d, which exceeds range 0-%d.",
	       fd, FD_SETSIZE - 1 );
    } // if
#endif __U_DEBUG__

    uNBIOnode uNode;
    uNode.nfds = 0;
    uNode.uPendingTask = &uThisTask();
    uNode.fdType = uNBIOnode::singleFd;
    uNode.uTimedout = false;
    uNode.smfd.sfd.fd = fd;
    uNode.smfd.sfd.uRWE = &rwe;

    if ( timeout != NULL ) {				// timeout ?
	if ( timeout->tv_sec == 0 && timeout->tv_usec == 0 ) { // optimization
	    // It is unnecessary to create a timeout event for this trivial
	    // case. Just mark the timeout as having already occurred.

	    uNode.uTimedout = true;
	    if ( uInitSfd( uNode ) ) {			// poller task ?
		while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	    } // if
	} else {
	    uDuration delay( timeout->tv_sec, timeout->tv_usec * 1000 );
	    uTime time = uActiveProcessorKernel->uKernelClock.uGetTime() + delay;
	    uSelectTimeoutHndlr handler( uThisTask(), uNode );

	    uEventNode uTimeoutEvent( uThisTask(), handler, time ); // event node for event list
	    uTimeoutEvent.uExecuteLocked = true;
	    uProcessor &proc = uThisProcessor();	// copy the current processor as it could change during execution

	    if ( uInitSfd( uNode, uTimeoutEvent, proc ) ) { // poller task ?
		while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	    } // if

	    // always doing remove guarantees the node is not deallocated prematurely
	    proc.uEvents->uRemoveEvent( uTimeoutEvent, proc );
	} // if
    } else {
	if ( uInitSfd( uNode ) ) {			// poller task ?
	    while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	} // if
    } // if

    return uNode.nfds;
} // uNBIO::uSelect


bool uNBIO::uInitMfds( fd_mask nfds, uNBIOnode &uNode ) {
    // set the appropriate fd bits in the master fd mask
    for ( unsigned int i = 0; i < howmany( nfds, NFDBITS ); i += 1 ) {
	// mask pointers can be NULL => nothing in that mask
    	if ( uNode.smfd.mfd.uRFDs ) uRFDs.fds_bits[i] |= uNode.smfd.mfd.uRFDs->fds_bits[i];
    	if ( uNode.smfd.mfd.uWFDs ) uWFDs.fds_bits[i] |= uNode.smfd.mfd.uWFDs->fds_bits[i];
    	if ( uNode.smfd.mfd.uEFDs ) uEFDs.fds_bits[i] |= uNode.smfd.mfd.uEFDs->fds_bits[i];
    } // for

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uInitMfds, adding node 0x%p\n", this, &uNode );
#endif __U_DEBUG_H__

    return uCheckPoller( uNode );
} // uNBIO::uInitMfds

     
bool uNBIO::uInitMfds( fd_mask nfds, uNBIOnode &uNode, uEventNode &uTimeoutEvent, uProcessor &proc ) {
    // set the appropriate fd bits in the master fd mask
    for ( unsigned int i = 0; i < howmany( nfds, NFDBITS ); i += 1 ) {
	// mask pointers can be NULL => nothing in that mask
    	if ( uNode.smfd.mfd.uRFDs ) uRFDs.fds_bits[i] |= uNode.smfd.mfd.uRFDs->fds_bits[i];
    	if ( uNode.smfd.mfd.uWFDs ) uWFDs.fds_bits[i] |= uNode.smfd.mfd.uWFDs->fds_bits[i];
    	if ( uNode.smfd.mfd.uEFDs ) uEFDs.fds_bits[i] |= uNode.smfd.mfd.uEFDs->fds_bits[i];
    } // for

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uNBIO &)0x%p.uInitMfds, adding node 0x%p\n", this, &uNode );
#endif __U_DEBUG_H__

    proc.uEvents->uAddEvent( uTimeoutEvent, proc );

    return uCheckPoller( uNode );
} // uNBIO::uInitMfds


int uNBIO::uSelect( fd_mask nfds, fd_set *rfds, fd_set *wfds, fd_set *efds, timeval *timeout ) {
#ifdef __U_DEBUG__
    if ( nfds < 1 || FD_SETSIZE < nfds ) {
	uAbort( ": attempt to select with a file descriptor set size of %d, which exceeds range 0-%d.",
	       nfds, FD_SETSIZE );
    } // if
#endif __U_DEBUG__

    uNBIOnode uNode;
    uNode.nfds = 0;
    uNode.uPendingTask = &uThisTask();
    uNode.fdType = uNBIOnode::multipleFds;
    uNode.uTimedout = false;
    uNode.smfd.mfd.uNFDs = nfds;
    uNode.smfd.mfd.uRFDs = rfds;
    uNode.smfd.mfd.uWFDs = wfds;
    uNode.smfd.mfd.uEFDs = efds;

    if ( timeout != NULL ) {				// timeout ?
	if ( timeout->tv_sec == 0 && timeout->tv_usec == 0 ) { // optimization
	    // It is unnecessary to create a timeout event for this trivial
	    // case. Just mark the timeout as having already occurred.

	    uNode.uTimedout = true;
	    if ( uInitMfds( nfds, uNode ) ) {		// poller task ?
		while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	    } // if
	} else {
	    uDuration delay( timeout->tv_sec, timeout->tv_usec * 1000 );
	    uTime time = uActiveProcessorKernel->uKernelClock.uGetTime() + delay;
	    uSelectTimeoutHndlr handler( uThisTask(), uNode );

	    uEventNode uTimeoutEvent( uThisTask(), handler, time ); // event node for event list
	    uTimeoutEvent.uExecuteLocked = true;
	    uProcessor &proc = uThisProcessor();	// copy the current processor as it could change during execution

	    if ( uInitMfds( nfds, uNode, uTimeoutEvent, proc ) ) { // poller task ?
		while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	    } // if

	    // always doing remove guarantees the node is not deallocated prematurely
	    proc.uEvents->uRemoveEvent( uTimeoutEvent, proc );
	} // if
    } else {
	if ( uInitMfds( nfds, uNode ) ) {		// poller task ?
	    while ( uPollIO( uNode ) ) uThisTask().uYield(); // busy wait
	} // if
    } // if

    return uNode.nfds;
} // uNBIO::uSelect


// Local Variables:
// compile-command: "dmake"
// End:
