//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uMachContext.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Fri Feb 25 15:46:42 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Apr 28 23:13:16 1999
// Update Count     : 292
// 


#define __U_KERNEL__
#include <uC++.h>
#include <stdio.h>					// fprintf
#include <exception>					// access: set_terminate
//#include <uDebug.h>


void uMachContext::uInvokeCoroutine( uBaseCoroutine &This ) { // magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

    uKernelModule::uEnableInterrupts();

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    try {
	This.uCoStarter();				// moved from uCoroutineMain to allow recursion on "main"
	This.main();					// start coroutine's "main" routine
    } catch ( ... ) {
	uRaise uBaseCoroutine::uFailure( This, "Unhandled thrown event propagated to starter" ) uAt *(This.uStart);
    } // try
    This.uCoFinish();
    // CONTROL SHOULD NOT RETURN HERE! 
    uAbort( "uInvokeCoroutine() : internal error." );
} // uMachContext::uInvokeCoroutine


void uMachContext::uInvokeTask( uBaseTask &This ) {	// magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

    set_terminate( uAEHM::uTerminate );			// reset default handler for each task
    errno = 0;						// reset errno for each task
    uKernelModule::uEnableInterrupts();

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    try {
	This.main();					// start task's "main" routine
    } catch( uAEHM::uDualClass &evt ) {
	evt.uDefaultTerminate();
    } catch( uAEHM::uThrowClass &evt ) {
	evt.uDefaultTerminate();
    } catch( ... ) {
    } // try

    terminate();					// call task terminate routine
    uAEHM::uTerminate();				// call abort terminate routine
    // CONTROL SHOULD NOT RETURN HERE!
} // uMachContext::uInvokeTask


void *uMachContext::uRtnAdr( void (*rtn)() ) {
#if defined( __aix__ ) && defined( __ibm__ )
    struct TOC {
	void *rtn;
	void *toc;
    };

    return ((TOC *)rtn)->rtn;
#elif defined( __hpux__ ) && defined( __hp__ )
    struct uLT {
	void *rtn;
	void *LTP;
    };

    if ( (int)rtn & 0x2 ) {
	return ((uLT *)((int)rtn & 0xfffffffd))->rtn;
    } else {
	return (void *)rtn;
    } // if
#else
    return (void *)rtn;
#endif
} // uMachContext::uRtnAdr


unsigned long int uMachContext::uMagic() {
    return 0x12345678;
} // uMachContext::uMagic


void uMachContext::uStartHere( void (*uInvoke)( uMachContext & ) ) {
#if defined( __i386__ )

    struct uFakeStack {
	void *uFixedRegisters[3];			// fixed registers ebx, edi, esi
	void *uReturn;
	void *uDummyReturn;
	void *uArgument;
    };

    // set base and limits of stack

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + (char *)uSize - uAlign());

    uSP = (char *)uBase - sizeof( uFakeStack );

    ((uFakeStack *)uSP)->uArgument = this;		// argument to uInvoke
    ((uFakeStack *)uSP)->uReturn = uRtnAdr( (void (*)())uInvoke );

#elif defined( __m68k__ )

    struct uFakeStack {
	int   uDataReg2to7[6];
	void *uAddrReg2to5[4];
	void *uFP;
	void *uReturn;
	void *uDummyReturn;
	void *uArgument;
    };

    // set base and limits of stack

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uSP = uBase - sizeof( uFakeStack );

    ((uFakeStack *)uSP)->uArgument = this;		// argument to uInvoke
    ((uFakeStack *)uSP)->uReturn = uRtnAdr( (void (*)())uInvoke );
    ((uFakeStack *)uSP)->uFP = NULL;			// terminate FP list

#elif defined( __sparc__ )

    struct uFakeStack {
	void *uLocalRegs[8];
	void *uInRegs[8];
	void *uStructRetAddress;
	void *uCalleeRegArgs[6];
    };

    // set base and limits of stack

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uFP = (void *)((unsigned long int)uBase - SA( MINFRAME ));
    uSP = (void *)((unsigned long int)uFP - SA( MINFRAME ));
    uPC = (void *)((unsigned long int)uRtnAdr( (void (*)())uInvoke ) - 8);

    ((uFakeStack *)uFP)->uInRegs[0] = this;		// argument to uInvoke

#elif defined( __mips__ )

    struct uFakeStack {
	void *uFixedRegisters[10];			// fixed registers r16-r23,r30,r31
    };

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uSP = (void *)((unsigned long int)uBase - sizeof( uFakeStack ));

    uProcVal  = (void (*)())uInvoke;

    // argument to uInvoke appears by magic, see uSwitch
    ((uFakeStack *)uSP)->uFixedRegisters[9] = uRtnAdr( (void (*)())uInvoke );

#elif defined( __alpha__ )

    struct uFakeStack {
	void *uFixedRegisters[8];			// fixed registers s0(r9)-s5(r14),fp(r15),ra(r26)
    };

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uSP = (void *)(uBase - sizeof( uFakeStack ));
    ((uFakeStack *)uSP)->uFixedRegisters[6] = uBase;	// set fp

    uProcVal = (void (*)())uInvoke;

    // argument to uInvoke appears by magic, see uSwitch
    ((uFakeStack *)uSP)->uFixedRegisters[7] = uRtnAdr( (void (*)())uInvoke );

#elif defined( __rs6000__ )

    struct uFakeStack {
	int uNVRegisters[19];				// fixed registers r13-r31
	struct {
	    char *uBackChain;
	    int  uSavedCR;
	    void *uSavedLR;
	    int  uReserved[2];
	    void *uSavedTOC;
	} uLinkArea;
    };

    uLimit = uStorage;
    uBase = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uSP = uBase - sizeof( uFakeStack );

    struct TOC {
	void *rtn;
	void *toc;
    };

    // argument to uInvoke appears by magic, see uSwitch
    ((uFakeStack *)uSP)->uLinkArea.uSavedLR  = uRtnAdr( (void (*)())uInvoke );
    ((uFakeStack *)uSP)->uLinkArea.uSavedTOC = ((TOC *)uInvoke)->toc;

#elif defined( __hppa__ )

    // fields in reverse order because stack grows up

    struct uLinkArea {
	int  uR19SharedLib;
	int  uReserved1;
	void *uReturnLinkSharedLib;
	void *uReturnLink;
	void *uStaticLink;
	int  uCleanUp;
	void *uExtPtr;
	void *uPrevSP;
    };

    struct uFakeStack {
	uLinkArea uFirstLinkArea;			// link area used by starting coroutine/task
	int uPadding[4];				// bring total to multiple of 64
	double uFloatRegisters[10];			// float registers fr12-fr21, used all the time
	int uFixedRegisters[16];			// fixed registers r3-r18
	uLinkArea uDummyLinkArea;			// dummy link area for interrupt routines
    } *uStackBase;

    // stack grows up

    uBase = uStorage;
    uLimit = (void *)((unsigned long int)uStorage + uSize - uAlign());

    uStackBase = (uFakeStack *)(uBase + uAlign());	// leave room for magic word
    uSP = (void *)uStackBase + sizeof(uFakeStack);

    // argument to uInvoke appears by magic, see uSwitch
    uStackBase->uFirstLinkArea.uReturnLink = uRtnAdr( (void (*)())uInvoke ); // return address is starting routine

#else

    unsupported architecture

#endif

    // set magic words on stack

    if ( uStorage != NULL ) {				// boot task ?
	*(unsigned long int *)uLimit = uMagic();
	*(unsigned long int *)uBase = uMagic();
    } // if
} // uMachContext::uStartHere


void uMachContext::uCreateContext( unsigned int stacksize ) {	// used by all constructors
    uSize = uCeiling( stacksize, uAlign() ) + uAlign();
    uLimit = uBase = NULL;
    uExtras.uAllExtras = 0;
} // uMachContext::uCreateContext


uMachContext::uMachContext( double ) {
    // adjust off amount added for magic numbers because the storage size is fixed.
    uCreateContext( sizeof(uKernelModule::uProcessorKernelStack) - uAlign() );
    uStorage = uKernelModule::uProcessorKernelStack;
} // uMachContext::uMachContext


void uMachContext::uSave() {
    uError = errno;
    // doesn't matter what it is set to as it is restored when task restarts
    uTerminateRtn = set_terminate( (terminate_handler)NULL );

    // Any extra work that must occur on this side of a context switch is
    // performed here.

    if ( uExtras.uAllExtras ) {
	uExtraSave();					// to reduce the amount of inlined code, a routine is called.
    } // if

#ifdef __U_DEBUG__
    uVerify();
#endif __U_DEBUG__
} // uMachContext::uSave


void uMachContext::uRestore() {
#ifdef __U_DEBUG__
    uVerify();
#endif __U_DEBUG__

    // Any extra work that must occur on this side of a context switch is
    // performed here.

    if ( uExtras.uAllExtras ) {
	uExtraRestore();				// to reduce the amount of inlined code, a routine is called.
    } // if

    set_terminate( uTerminateRtn );
    errno = uError;
} // uMachContext::uRestore

uMachContext::uMachContext( unsigned int stacksize ) {
    uCreateContext( stacksize );
    uStorage = malloc( uSize );				// use malloc because "new" handles the overflow
    if ( uStorage == NULL ) {				// allocate stack from heap
	uAbort( "attempt to allocate %d bytes of storage for coroutine or task execution-state but insufficient memory available.", stacksize );
    } // if
} // uMachContext::uMachContext


uMachContext::~uMachContext() {
    free( uStorage );
} // uMachContext::~uMachContext


void *uMachContext::uStackPointer() const {
    void *sp;
#if defined( __i386__ )
    asm(" movl %%esp,%0" : "=m" (sp) : );
#elif defined( __m68k__ )
    asm(" movl sp,%0" : "=m" (sp) : );
#elif defined( __sparc__ )
    asm(" st %%sp,%0" : "=m" (sp) : );
#elif defined( __mips__ )
    asm(" sw $sp,%0" : "=m" (sp) : );
#elif defined( __alpha__ )
    asm(" stq $sp,%0" : "=m" (sp) : );
#elif defined( __rs6000__ )
    asm(" st 1,%0" : "=m" (sp) : );
#elif defined( __hppa__ )
    asm(" stw %%r30,%0" : "=m" (sp) : );
#else
    unsupported architecture
#endif
    return sp;
} // uMachContext::uStackPointer


unsigned int uMachContext::uStackSize() const {
    return uSize;
} // uMachContext::uStackSize


ptrdiff_t uMachContext::uStackFree() const {
    return (char *)uStackPointer() - (char *)uLimit;
} // uMachContext::uStackFree


ptrdiff_t uMachContext::uStackUsed() const {
    ptrdiff_t amt = (char *)uBase - (char *)uStackPointer();
    return amt < 0 ? -amt : amt;			// abs
} // uMachContext::uStackUsed


void uMachContext::uVerify() {
    // If executing on an allocated state, which can be detected by looking at
    // the Storage variable, verify that the allocated state has not been
    // corrupted.

    if ( uStorage != NULL ) {
#ifdef __U_STACK_GROWS_UP__
	if ( uStackPointer() > uLimit ) {
	    uAbort( "uVerify() : stack overflow detected: stack pointer 0x%p above limit 0x%p.\n"
		   "Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		   uStackPointer(), uBase );
	else if ( uStackPointer() + 4 * 1024 > uLimit ) {
	    fprintf( stderr, "uVerify() : WARNING within 4K of stack limit: stack pointer 0x%p above limit 0x%p.\n"
		   "Possible cause is allocation of large stack frame(s) and/or deep call stack.\n",
		   uStackPointer(), uBase );
	} else if ( uStackPointer() < uBase ) {
	    uAbort( "uVerify() : stack corruption detected.\n"
		   "Possible cause is corrupted stack frame via overwriting memory." );
#else
	if ( uStackPointer() < uLimit ) {
	    uAbort( "uVerify() : stack overflow detected: stack pointer 0x%p below limit 0x%p.\n"
		   "Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		   uStackPointer(), uLimit );
	} else if ( (char *)uStackPointer() + 4 * 1024 < uLimit ) {
	    fprintf( stderr, "uVerify() : WARNING within 4K of stack limit: stack pointer 0x%p and limit 0x%p.\n"
		   "Possible cause is allocation of large stack frame(s) and/or deep call stack.\n",
		   uStackPointer(), uLimit );
	} else if ( uStackPointer() > uBase ) {
	    uAbort( "uVerify() : stack corruption detected.\n"
		   "Possible cause is corrupted stack frame via overwriting memory." );
#endif __U_STACK_GROWS_UP__
	} else if ( ( *(unsigned int *)uBase != uMagic() ) || ( *(unsigned int *)uLimit != uMagic() ) ) {
	    uAbort( "uVerify() : stack corruption detected.\n"
		   "Possible cause is corrupted stack frame via overwriting memory." );
	} // if
    } // if
} // uMachContext::uVerify


void uMachContext::uExtraSave() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqGen<uContext> gen(uAdditionalContexts); gen >> context; ) {
	context->uSave();
    } // for
} // uMachContext::uExtraSave


void uMachContext::uExtraRestore() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqGen<uContext> gen(uAdditionalContexts); gen >> context; ) {
	context->uRestore();
    } // for
} // uMachContext::uExtraRestore


// Local Variables:
// compile-command: "dmake"
// End:
