//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uHeapLmmm.cc -- Lean Mean Malloc Machine - a runtime configurable replacement
//                 for malloc.
// 
// Author           : Peter A. Buhr
// Created On       : Sat Nov 11 16:07:20 1988
// Last Modified By : Peter A. Buhr
// Last Modified On : Fri Mar 19 17:38:22 1999
// Update Count     : 328
// 

#define __U_KERNEL__
#include <uC++.h>
#include <uHeapLmmm.h>
//#include <uDebug.h>


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>


extern "C" void uDebugPrt( const char fmt[], ... );	// used by heap trace


#if ! defined( __svr4__ ) && defined( __sun__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __aix__ ) && defined( __ibm__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __hpux__ ) && defined( __hp__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __ultrix__ ) && defined( __dec__ )
extern "C" void bcopy(void *, void *, int);		// missing from 
extern "C" void bzero(void *, int);			// missing from 
#endif

#if defined( __gizmo__ )
extern "C" int fprintf(FILE *, const char *, ...);
#endif


static double uHeapStorage[(sizeof(uHeap) + sizeof(double)) / sizeof(double)] = { 0.0 };
bool uHeap::uTraceAlloc = false;
bool uHeap::uStrict = true;


#if defined( __gizmo__ )
// Written by Tom Clegg.
//
// The gizmo sbrk assumes a single stack. When it checks if the stack and heap
// have collided, it compares the current stack pointer with the current heap
// end. This check fails for a uC++ task, whose stack is allocated in the heap
// area. The following routine cheats by managing the end of the heap itself,
// and hence, makes no calls to the sbrk after determining the heap end.
// Therefore, no calls can be made to sbrk in a gizmo application or sbrk and
// this routine will be out of synchronization.

void *gizmo_sbrk( int amount ) {
    static void *top = NULL;
    if ( ! top ) {					// first call ?
	top = sbrk( 0 );				// get the current heap end
    } // if
    void *new_area = top;
    if ( amount > 0 ) {
	(char *)top += amount;
    } // if
    return new_area;
} // gizmo_sbrk

// fake a few things
#define sbrk gizmo_sbrk
#endif


// Based on an idea from:
// Martin Schonert,   Martin.Schoenert@Math.RWTH-Aachen.DE,  +49 241 804551
// Lehrstuhl D fur Mathematik, Templergraben 64, RWTH, D 51 Aachen, Germany

static const unsigned char msbpostab [256] = {
    0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
    6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7 };

static inline size_t ceillog2( size_t i ) {
    size_t pos;

    if ( i >> 16 )
	if ( i >> 24 )
	    pos = 24 + msbpostab[i >> 24];
	else
	    pos = 16 + msbpostab[i >> 16];
    else
	if ( i >> 8 )
	    pos =  8 + msbpostab[i >> 8];
	else
	    pos =  0 + msbpostab[i];
    if ( 1ul << pos < i ) pos += 1;			// ceil
    return pos;
} // ceillog2


bool uTraceHeapOn() {
    bool temp = uHeap::uTraceAlloc;
    uHeap::uTraceAlloc = true;
    return temp;
} // uTraceHeapOn


bool uTraceHeapOff() {
    bool temp = uHeap::uTraceAlloc;
    uHeap::uTraceAlloc = false;
    return temp;
} // uTraceHeapOff


bool uStrictHeapOn() {
    bool temp = uHeap::uStrict;
    uHeap::uStrict = true;
    return temp;
} // uStrictHeapOn


bool uStrictHeapOff() {
    bool temp = uHeap::uStrict;
    uHeap::uStrict = false;
    return temp;
} // uStrictHeapOff


size_t uHeap::uUserSize( void *user ) {
    // Locate hash table entry from block header, which contains the total size
    // of the storage block.

    uStorage *block = (uStorage *)( (char *)user - sizeof(uStorage::uHeader) );

    // Subtract off the header storage at the beginning of the block.

    return block->uHdr.home->blockSize - sizeof(uStorage::uHeader);
} // uHeap::uUserSize


void *uHeap::uExtend( size_t size ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uExtend( %d ), uHeapBegin:0x%p, uHeapEnd:0x%p, uHeapRemaining:0x%p, uStrict:%d\n",
               this, size, uHeapBegin, uHeapEnd, uHeapRemaining, uStrict );
#endif __U_DEBUG_H__
    ptrdiff_t rem = uHeapRemaining - size;
    if ( rem < 0 ) {
	// If the size requested is bigger than the current remaining storage,
	// increase the size of the heap.

	size_t uDefault = uDefaultHeapExpansion();
	size_t uAmount = uCeiling( size > uDefault ? size : uDefault, uAlign() );
	if ( sbrk( uAmount ) == (void *)-1 ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "0x%p = (uHeap &)0x%p.uExtend( %d ), uHeapBegin:0x%p, uHeapEnd:0x%p, uHeapRemaining:0x%p, uStrict:%d\n",
		       NULL, this, size, uHeapBegin, uHeapEnd, uHeapRemaining, uStrict );
#endif __U_DEBUG_H__
	    return NULL;
	} // if
	rem = uHeapRemaining + uAmount - size;
    } // if
    uStorage *block = (uStorage *)uHeapEnd;
    uHeapRemaining = rem;
    uHeapEnd = (char *)uHeapEnd + size;
#ifdef __U_DEBUG_H__
    uDebugPrt( "0x%p = (uHeap &)0x%p.uExtend( %d ), uHeapBegin:0x%p, uHeapEnd:0x%p, uHeapRemaining:0x%p, uStrict:%d\n",
               block, this, size, uHeapBegin, uHeapEnd, uHeapRemaining, uStrict );
#endif __U_DEBUG_H__
    return block;
} // uHeap::uExtend


void *uHeap::uDoMalloc( size_t size ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uDoMalloc( %d )\n", this, size );
#endif __U_DEBUG_H__

    uStorage *block;

    // Look up size in the size list.  Make sure the user request includes
    // space for the header that must be allocated along with the block and is
    // a multiple of the alignment size.

    uSizeHeader &header = headers[ceillog2( size + sizeof(uStorage::uHeader) )]; // optimize out subsequent subscripting
    size_t tsize = header.blockSize;			// total space needed for request

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uDoMalloc, size after lookup:%d\n", this, tsize );
#endif __U_DEBUG_H__
    
    // Spin until the lock is acquired for this particular size of block.

    header.lock.uAcquire();
    if ( header.freeList != (uStorage *)0 ) {
	block = header.freeList;
	header.freeList = block->uHdr.next;
	header.lock.uRelease();
    } else {
	header.lock.uRelease();

	// Freelist for that size was empty, so carve it out of the heap if
	// there's enough left, or get some more and then carve it off.

	uKernelModule::uDisableExactProfilingCnt += 1;
	block = (uStorage *)uExtend( tsize );		// mutual exclusion on call
	uKernelModule::uDisableExactProfilingCnt -= 1;
	if ( block == (uStorage *)0 ) return NULL;
    } // if

    block->uHdr.home = &header;				// pointer back to free list of apropriate size
    void *area = &(block->uData);
#ifdef __U_DEBUG__
    if ( uTraceAlloc ) {
	uDebugPrt( "0x%p = Malloc( %d ) (allocated %d)\n", area, size, tsize );
    } // if
#endif __U_DEBUG__
#ifdef __U_DEBUG_H__
    uDebugPrt( "0x%p = (uHeap &)0x%p.uDoMalloc\n", area, this );
#endif __U_DEBUG_H__
    return area;					// adjust off header to user bytes
} // uHeap::uDoMalloc


void uHeap::uDoFree( void *user ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uDoFree( 0x%p )\n", this, user );
#endif __U_DEBUG_H__
#ifdef __U_DEBUG__
    if ( uTraceAlloc ) {
	uDebugPrt( "Free( 0x%p )\n", user );
    } // if
#endif __U_DEBUG__

  if ( user == NULL ) return;

#ifdef __U_DEBUG__
    if ( user < uHeapBegin || user > uHeapEnd ) {
	if ( uStrict ) {				// TEMPORARY: incorrect frees in motif
	    uAbort( ": attempt to free storage 0x%p outside the current heap range:0x%p to 0x%p.\n"
		   "Possible cause is corrupted pointer.",
		   user, uHeapBegin, uHeapEnd );
	} else {
	    fprintf( stderr, "(UNIX pid:%ld) : attempt to free storage 0x%p outside the current heap range:0x%p to 0x%p.\n"
		    "Possible cause is corrupted pointer.\n",
		    getpid(), user, uHeapBegin, uHeapEnd );
	    return;
	} // if
    } // if
#endif __U_DEBUG__
    uStorage &block = *(uStorage *)( (char *)user - sizeof(uStorage::uHeader) );
#ifdef __U_DEBUG__
    if ( block.uHdr.home < &headers[0] || block.uHdr.home > &headers[31] ) {
	if ( uStrict ) {				// TEMPORARY: incorrect frees in motif
	    uAbort( ": attempt to free storage 0x%p with corrupted header.\n"
		   "Possible cause is duplicate free on same block or overwriting of header information.",
		   user );
	} else {
	    fprintf( stderr, "(UNIX pid:%ld) : attempt to free storage 0x%p with corrupted header.\n"
		    "Possible cause is duplicate free on same block or overwriting of header information.\n",
		    getpid(), user );
	    return;
	} // if
    } // if
#endif __U_DEBUG__
    uSizeHeader &header = *(block.uHdr.home);
#ifdef __U_DEBUG__
    // Set free memory to garbage so subsequent usages might fail.
#ifdef __svr4__
    // Tried setting to '\377' and it caused motif to fail.
    memset( user, '\0', uUserSize( user ) );
#else
    bzero( user, uUserSize( user ) );			// '\377' would be better
#endif __svr4__
#endif __U_DEBUG__

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uDoFree( 0x%p ) block:0x%p header:0x%p\n", this, user, &block, &header );
#endif __U_DEBUG_H__
    header.lock.uAcquire();
    block.uHdr.next = header.freeList;
    header.freeList = &block;
    header.lock.uRelease();

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uDoFree( 0x%p ) returning free block in list 0x%p\n", this, user, header.blockSize );
#endif __U_DEBUG_H__
} // uHeap::uDoFree


uHeap::uHeap() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uHeap()\n", this );
#endif __U_DEBUG_H__
    for ( int i = 0; i < 32; i += 1 ) {			// initialize the free lists
	headers[i].blockSize = 1ul << i;
	headers[i].freeList = (uStorage *)0;
    } // for

    brk( (char *)uCeiling( (long unsigned int)sbrk( 0 ), uAlign() ) ); // move start of heap to multiple of alignment
    uHeapBegin = uHeapEnd = sbrk( 0 );
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uHeap &)0x%p.uHeap() uHeapBegin:0x%p, uHeapEnd:0x%p\n", this, uHeapBegin, uHeapEnd );
#endif __U_DEBUG_H__
    uHeapRemaining = 0;					// don't pre-allocate storage that might not get used

    // TEMPORARY: While debugging the debugger. Problem with GDB and X freeing
    // unallocated memory.

    uStrict = getenv( "__U_DEBUG_MEMORY_NONSTRICT" ) == (char *)0;
} // uHeap::uHeap


uHeap::~uHeap() {
#ifdef __U_DEBUG__
    size_t total = 0;
    for ( int i = 0; i < 32; i += 1 ) {
	size_t size = headers[i].blockSize;
	for ( uStorage *p = headers[i].freeList; p; p = p->uHdr.next ) {
	    total += size;
	} // for
    } // for
    ptrdiff_t diff = (char *)uHeapEnd - (char *)uHeapBegin;

    if ( total != (size_t)diff ) {
        ptrdiff_t underby = diff - total;
	// DO NOT USE STREAMS AS THEY MAY BE CLOSED DOWN BY THIS POINT.
	fprintf( stderr, "uC++ Runtime warning (UNIX pid:%ld) : program terminating with %d(0x%x) bytes of storage allocated but not freed.\n"
		 "Possible cause is unfreed storage allocated by the program or system/library routines called from the program.\n",
		 getpid(), underby, underby );
    } // if
#endif __U_DEBUG__
} // uHeap::~uHeap


inline void *uHeap::uMalloc( size_t size ) {
    return uDoMalloc( size );
} // uHeap::uMalloc

inline void uHeap::uFree( void *addr ) {
    uDoFree( addr );
} // uHeap::uFree

inline void *uHeap::uRealloc( void *addr, size_t size ) {
  if ( addr == NULL ) return uMalloc( size );
  if ( size == 0 ) {
	uFree( addr );
	return NULL;
    } // exit

    size_t s = uUserSize( addr );			// size of user part of storage block
  if ( s >= size ) return addr;				// already sufficient storage
    void *area = uMalloc( size );			// create new area
#ifdef __svr4__
    memcpy( area, addr, s < size ? s : size );
#else
    bcopy( addr, area, s < size ? s : size );
#endif __svr4__
    uFree( addr );
    return area;
} // uHeap::uRealloc

void *uHeap::uCalloc( size_t noOfElems, size_t elemSize ) {
    void *area = uMalloc( noOfElems * elemSize );
#ifdef __svr4__
    memset( area, '\0', noOfElems * elemSize );
#else
    bzero( area, noOfElems * elemSize );
#endif __svr4__
    return area;
} // uHeap::uCalloc


// Unnecessary to redefine __builtin_new, __builtin_vec_new, __builtin_delete
// as they call malloc and free, which are redefined below.


extern "C" {
void *malloc( size_t size ) {
    if ( uKernelModule::uHeapManager == (uHeap *)0 ) {
#ifdef __U_DEBUG__
	if ( uMemoryModule::uMapLock != (uSpinLock *)0 ) { // check for recursion during system boot
	    fprintf( stderr, "malloc( %d ) recursively invoked during system boot.\n", size );
	    _exit( -1 );
	} // if
#endif __U_DEBUG__

	// SKULLDUGGERY: This ensures that calls to uThisCoroutine and
	// scheduling work before the kernel is initialized.

	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uCurrCoroutine = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uInheritTask = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uPriority = 0;
	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uActivePriority = 0;
	uKernelModule::uHeapManager = new((void *)&uHeapStorage) uHeap;
    } // if
    void *area = uKernelModule::uHeapManager->uMalloc( size );
#ifdef __U_DEBUG_H__
    uDebugPrt( "0x%p = malloc( %d )\n", area, size );
#endif __U_DEBUG_H__
    return area;
} // malloc

void *realloc( void *addr, size_t size ) {
    if ( uKernelModule::uHeapManager == (uHeap *)0 ) {
#ifdef __U_DEBUG__
	if ( uMemoryModule::uMapLock != (uSpinLock *)0 ) { // check for recursion during system boot
	    fprintf( stderr, "realloc( 0x%p, %d ) recursively invoked during system boot.\n", addr, size );
	    _exit( -1 );
	} // if
#endif __U_DEBUG__

	// SKULLDUGGERY: This ensures that calls to uThisCoroutine work before
	// the kernel is initialized.

	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uCurrCoroutine = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uInheritTask = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	uKernelModule::uHeapManager = new((void *)&uHeapStorage) uHeap;
    } // if
    void *area = uKernelModule::uHeapManager->uRealloc( addr, size );
#ifdef __U_DEBUG_H__
    uDebugPrt( "0x%p = realloc( 0x%p, %d )\n", area, addr, size );
#endif __U_DEBUG_H__
    return area;
} // realloc

void *calloc( size_t noOfElems, size_t elemSize ) {
    if ( uKernelModule::uHeapManager == (uHeap *)0 ) {
#ifdef __U_DEBUG__
	if ( uMemoryModule::uMapLock != (uSpinLock *)0 ) { // check for recursion during system boot
	    fprintf( stderr, "calloc( %d, %d ) recursively invoked during system boot.\n", noOfElems, elemSize );
	    _exit( -1 );
	} // if
#endif __U_DEBUG__

	// SKULLDUGGERY: This ensures that calls to uThisCoroutine work before
	// the kernel is initialized.

	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uCurrCoroutine = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	((uBaseTask *)&uKernelModule::uTaskBootStorage)->uInheritTask = (uBaseTask *)&uKernelModule::uTaskBootStorage;
	uKernelModule::uHeapManager = new((void *)&uHeapStorage) uHeap;
    } // if
    void *area = uKernelModule::uHeapManager->uCalloc( noOfElems, elemSize );
#ifdef __U_DEBUG_H__
    uDebugPrt( "0x%p = calloc( %d, %d )\n", area, noOfElems, elemSize );
#endif __U_DEBUG_H__
    return area;
} // calloc

void *memalign( size_t alignment, size_t size ) {
    uAbort( "memalign( %d, %d ) not implemented\n", alignment, size );
    return (void *)-1;
} // memalign

void *valloc( size_t size ) {
    uAbort( "valloc( %d ) not implemented", size );
    return (void *)-1;
} // valloc

void free( void *addr ) {
    uKernelModule::uHeapManager->uFree( addr );
#ifdef __U_DEBUG_H__
    uDebugPrt( "free( 0x%p )\n", addr );
#endif __U_DEBUG_H__
} // free
} // extern "C"


// Local Variables: //
// compile-command: "dmake" //
// End: //
