//                              -*- Mode: C++ -*- 
// 
// uC++ Version 4.7, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1994
// 
// uCluster.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Mon Mar 14 17:34:24 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Feb 10 10:30:34 1999
// Update Count     : 306
// 


#define __U_KERNEL__
#ifndef __U_DEBUG__
#define NDEBUG						// turn off assert
#endif
#include <uC++.h>
#include <uAssert.h>
//#include <uDebug.h>

#include <signal.h>					// needed for SIGALRM and kill


#if ! defined( __svr4__ ) && defined( __sun__ )
extern "C" int sigsetmask(int);				// missing from signal.h
extern "C" int sigblock(int);				// missing from signal.h
extern "C" int sigpause(int);				// missing from signal.h
#endif

#if defined( __ultrix__ ) && defined( __dec__ )
extern "C" int sigsetmask(int);				// missing from signal.h
extern "C" int sigblock(int);				// missing from signal.h
extern "C" int sigpause(int);				// missing from signal.h
#endif


extern uCluster &uThisCluster() {
    return *uKernelModule::uActiveCluster;
} // uThisCluster


//######################### uClusterDL #########################


uClusterDL::uClusterDL( uCluster &w ) : uWho( w ) {}
uCluster &uClusterDL::uGet() const { return uWho; }


//######################### uCluster #########################


void uCluster::uWakeProcessor( int uPid ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.uWakeProcessor: waking processor %d\n", this, uPid );
#endif __U_DEBUG_H__
    kill( uPid, SIGALRM );				// wake it up, as it is probably sleeping
} // uCluster::uWakeProcessor


void uCluster::uProcessorPause() {
    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );

    // Check the ready queue to make sure that no task managed to slip onto the
    // queue since the processor last checked.

    uReadyIdleLock.uAcquire();
    if ( uReadyTasksEmpty() && uThisProcessor().uExternal.uEmpty() ) {
	// stop generating SIGALRM signals on this processor until woken up

	uThisProcessor().uSetContextSwitchEvent( uDuration( 0, 0 ) ); // turn off context-switching

	// real-time

	uTime nextAlarm;
	nextAlarm = uThisProcessor().uEvents->uNextAlarm(); // find next event time

	// Block any SIGALRM signals from arriving.
#ifdef __svr4__
	sighold( SIGALRM );
#else
	unsigned int mask = sigblock( sigmask( SIGALRM ) );
#endif __svr4__
	if ( uKernelModule::uInKernelRF ) {		// in kernel roll-forward flag on ?
	        uReadyIdleLock.uRelease();
#ifdef __svr4__
		sigrelse( SIGALRM );
#else
	        sigsetmask( mask );
#endif __svr4__
		uKernelModule::uRollForward( true );	// make sure to do chores
	} else {
	    uIdleProcessors.uAddTail( &(uThisProcessor().uIdleRef) );
	    uReadyIdleLock.uRelease();

	    // Install the old signal mask and wait for a signal to arrive.

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uCluster &)0x%p.uProcessorPause, before sigpause\n", this );
#endif __U_DEBUG_H__
#ifdef __svr4__
	    sigpause( SIGALRM );
	    sigrelse( SIGALRM );
#else
	    sigpause( mask );
	    sigsetmask( mask );
#endif __svr4__

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uCluster &)0x%p.uProcessorPause, after sigpause\n", this );
#endif __U_DEBUG_H__

	    // A UNIX process may be woken by any signal, e.g. SIGCHLD, so it is
	    // necessary to check and remove the processor from the idle queue.
	    // Normally a processor is removed in uMakeTaskReady.

	    if ( uThisProcessor().uIdle() ) {
		uReadyIdleLock.uAcquire();
		if ( uThisProcessor().uIdle() ) {
		    uIdleProcessors.uRemove( &(uThisProcessor().uIdleRef) );
		} // if
		uReadyIdleLock.uRelease();
	    } // if

	    // Just woken up after an alarm but in kernel/library code, so no
	    // actual popping from the event list took place in the sigalarm
	    // handler (i.e., signal alarm handler just ignored the interrupt).
	    // Therefore, do a roll-forward to ensure that any necessary events
	    // are popped from the event list.

	    uKernelModule::uRollForward( true );
	} // if

	// Reset the context-switch

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.uProcessorPause, reset timeslice:%d\n", this, uThisProcessor().uGetPreemption() );
#endif __U_DEBUG_H__
	uThisProcessor().uSetContextSwitchEvent( uThisProcessor().uGetPreemption() );
    } else {
	uReadyIdleLock.uRelease();
    } // if

    uAssert( uKernelModule::uDisableInt && uKernelModule::uDisableIntCnt > 0 );
} // uCluster::uProcessorPause


void uCluster::uMakeProcessorIdle( uProcessor &p ) {
    uReadyIdleLock.uAcquire();
    uIdleProcessors.uAddTail( &(p.uIdleRef) );
    uReadyIdleLock.uRelease();
} // uCluster::uMakeProcessorIdle


void uCluster::uMakeProcessorActive( uProcessor &p ) {
    uReadyIdleLock.uAcquire();
    if ( p.uIdle() ) {					// processor on idle queue ?
	uIdleProcessors.uRemove( &(p.uIdleRef) );
    } // if
    uReadyIdleLock.uRelease();
} // uCluster::uMakeProcessorActive


bool uCluster::uReadyTasksEmpty() {
    return uReadyTasks->uEmpty();
} // uCluster::uReadyTasksEmpty


void uCluster::uMakeTaskReady( uBaseTask &uReadyTask ) {
    uReadyIdleLock.uAcquire();
    if ( &uReadyTask.uBound ) {				// task bound to a specific processor ?
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.uMakeTaskReady(1): task 0x%p (%.256s) makes task 0x%p (%.256s) ready\n",
		  this, &uThisTask(), uThisTask().uGetName(), &uReadyTask, uReadyTask.uGetName() );
#endif __U_DEBUG_H__
	uProcessor *p = &uReadyTask.uBound;		// optimization
	p->uExternal.uAddTail( &(uReadyTask.uReadyRef) ); // add task to end of special ready queue
#ifdef __U_MULTI__
	if ( p->uIdle() ) {				// processor on idle queue ?
	    uIdleProcessors.uRemove( &(p->uIdleRef) );
	    int uPid = p->uPid;
	    uReadyIdleLock.uRelease();			// don't hold lock while sending SIGALRM
	    uWakeProcessor( uPid );
	} else {
	    uReadyIdleLock.uRelease();
	} // if
#else
	uReadyIdleLock.uRelease();
#endif __U_MULTI__
    } else {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.uMakeTaskReady(2): task 0x%p (%.256s) makes task 0x%p (%.256s) ready\n",
		  this, &uThisTask(), uThisTask().uGetName(), &uReadyTask, uReadyTask.uGetName() );
#endif __U_DEBUG_H__
#ifdef __U_MULTI__
	// Wake up an idle processor if the ready task is migrating to another
	// cluster with idle processors or if the ready task is on the same
	// cluster but the ready queue of that cluster is not empty. This check
	// prevents a single task on a cluster, which does a yield, from
	// unnecessarily waking up a processor that has no work to do.

	if ( ! uIdleProcessors.uEmpty() && ( &uThisCluster() != this || ! uReadyTasks->uEmpty() ) ) {
	    uReadyTasks->uAdd( &(uReadyTask.uReadyRef) ); // add task to end of cluster ready queue
	    int uPid = uIdleProcessors.uDropHead()->uGet().uPid;
	    uReadyIdleLock.uRelease();			// don't hold lock while sending SIGALRM
	    uWakeProcessor( uPid );
	} else {
	    uReadyTasks->uAdd( &(uReadyTask.uReadyRef) ); // add task to end of cluster ready queue
	    uReadyIdleLock.uRelease();
	} // if
#else
	uReadyTasks->uAdd( &(uReadyTask.uReadyRef) );	// add task to end of cluster ready queue
	uReadyIdleLock.uRelease();
#endif __U_MULTI__
    } // if
} // uCluster::uMakeTaskReady


uBaseTask &uCluster::uReadyTaskTryRemove() {
    // Select a task from the ready queue of this cluster if there are no ready
    // tasks, return the nil pointer.

    uBaseTask *t;

    uReadyIdleLock.uAcquire();
    if ( ! uReadyTasksEmpty() ) {
	t = &(uReadyTasks->uDrop()->uGet());
    } else {
	t = (uBaseTask *)0;
    } // if
    uReadyIdleLock.uRelease();
    return *t;
} // uCluster::uReadyTaskTryRemove


void uCluster::uTaskAdd( uBaseTask &t ) {
    uReadyIdleLock.uAcquire();
    uTasksOnCluster.uAddTail( &(t.uClusterRef) );
    uReadyTasks->uAddInitialize( uTasksOnCluster );
    uReadyIdleLock.uRelease();
} // uCluster::uTaskAdd


void uCluster::uTaskRemove( uBaseTask &t ) {
    uReadyIdleLock.uAcquire();
    uTasksOnCluster.uRemove( &(t.uClusterRef) );
    uReadyTasks->uRemoveInitialize( uTasksOnCluster );
    uReadyIdleLock.uRelease();
} // uCluster::uTaskRemove


void uCluster::uTaskReschedule( uBaseTask &t ) {
    uReadyIdleLock.uAcquire();
    uReadyTasks->uRescheduleTask( &(t.uClusterRef), uTasksOnCluster );
    uReadyIdleLock.uRelease();
} // uCluster::uTaskReschedule


void uCluster::uTaskResetPriority( uBaseTask &owner, uBaseTask &calling ) {
    uReadyIdleLock.uAcquire();
    if ( &uThisCluster() == owner.uCurrCluster ) {
	if ( uReadyTasks->uCheckPriority( owner.uReadyRef, calling.uReadyRef ) ) {
	    uReadyTasks->uResetPriority( owner.uReadyRef, calling.uReadyRef );
	} // if
    } // if
    uReadyIdleLock.uRelease();
} // uCluster::uTaskResetPriority


void uCluster::uProcessorAdd( uProcessor &p ) {
    uProcessorsOnClusterLock.uAcquire();
    uProcessorsOnCluster.uAddTail( &(p.uProcessorRef) );
    uProcessorsOnClusterLock.uRelease();
} // uCluster::uProcessorAdd


void uCluster::uProcessorRemove( uProcessor &p ) {
    uProcessorsOnClusterLock.uAcquire();
    uProcessorsOnCluster.uRemove( &(p.uProcessorRef) );
    uProcessorsOnClusterLock.uRelease();
} // uCluster::uProcessorRemove

void uCluster::uCreateCluster( unsigned int stacksize, const char *name ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.uCreateCluster\n", this );
#endif __U_DEBUG_H__

#ifdef __U_DEBUG__
#ifdef __U_MULTI__
    uDebugIgnore = false;
#else
    uDebugIgnore = true;
#endif __U_MULTI__
#endif __U_DEBUG__

    uSetName( name );
    uSetStackSize( stacksize );

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif __U_LOCALDEBUGGER_H__

    uKernelModule::uGlobalClusterLock->uAcquire();
    uKernelModule::uGlobalClusters->uAddTail( &uGlobalRef );
    uKernelModule::uGlobalClusterLock->uRelease();

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->createCluster( *this );
#endif __U_LOCALDEBUGGER_H__

    if ( uReadyTasks == NULL ) {
	uReadyTasks = new uDefaultScheduler;
	uDefaultReadyTasks = true;
    } else {
	uDefaultReadyTasks = false;
    } // if
} // uCluster::uCreateCluster


uCluster::uCluster( unsigned int stacksize, const char *name ) : uGlobalRef( *this ), uReadyTasks( NULL ) {
    uCreateCluster( stacksize, name );
} // uCluster::uCluster


uCluster::uCluster( const char *name ) : uGlobalRef( *this ), uReadyTasks( NULL ) {
    uCreateCluster( uDefaultStackSize(), name );
} // uCluster::uCluster


uCluster::uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, unsigned int stacksize, const char *name ) : uGlobalRef( *this ), uReadyTasks( &ReadyQueue ) {
    uCreateCluster( stacksize, name );
} // uCluster::uCluster


uCluster::uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, const char *name ) : uGlobalRef( *this ), uReadyTasks( &ReadyQueue ) {
    uCreateCluster( uDefaultStackSize(), name );
} // uCluster::uCluster


uCluster::~uCluster() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.~uCluster\n", this );
#endif __U_DEBUG_H__

#ifdef __U_DEBUG__
    uBaseTaskDL *tr;
    uReadyIdleLock.uAcquire();
    for ( uSeqGen<uBaseTaskDL> gen1(uTasksOnCluster); gen1 >> tr; ) {
	uAbort( ": attempt to delete cluster 0x%p (%.256s) with task 0x%p (%.256s) still on it.\n"
	       "Possible cause is that the task has not been deleted.",
	       this, this->uGetName(), &(tr->uGet()), tr->uGet().uGetName() );
    } // for
    uReadyIdleLock.uRelease();

    uProcessorDL *pr;
    uProcessorsOnClusterLock.uAcquire();
    for ( uSeqGen<uProcessorDL> gen2(uProcessorsOnCluster); gen2 >> pr; ) {
	uAbort( ": attempt to delete cluster 0x%p (%.256s) with processor 0x%p still on it.\n"
	       "Possible cause is that the processor has not been deleted.",
	       this, this->uGetName(), &(pr->uGet()) );
    } // for
    uProcessorsOnClusterLock.uRelease();
#endif __U_DEBUG__

    if ( uDefaultReadyTasks ) {				// delete if cluster allocated it
	delete uReadyTasks;
    } // if

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->destroyCluster( *this );
#endif __U_LOCALDEBUGGER_H__

    uKernelModule::uGlobalClusterLock->uAcquire();
    uKernelModule::uGlobalClusters->uRemove( &uGlobalRef );
    uKernelModule::uGlobalClusterLock->uRelease();
} // uCluster::~uCluster

const char *uCluster::uSetName( const char *name ) {
    const char *prev = uName;
    uName = name;
    return prev;
} // uCluster::uSetName

const char *uCluster::uGetName() const {
#ifdef __U_DEBUG__
    if ( uName == NULL ) return "*unknown*";		// storage might be scrubbed
#endif __U_DEBUG__
    return uName;
} // uCluster::uGetName

unsigned int uCluster::uSetStackSize( unsigned int stacksize ) {
    unsigned int prev = uStackSize;
    uStackSize = stacksize;
    return prev;
} // uCluster::uSetStackSize

unsigned int uCluster::uGetStackSize() const {
    return uStackSize;
} // uCluster::uGetStackSize


//######################### uIOCluster #########################


void uIOCluster::uProcessorAdd( uProcessor &p ) {
    uProcessorsOnClusterLock.uAcquire();
    if ( uProcessorsOnCluster.uEmpty() ) {
	uProcessorsOnCluster.uAddTail( &(p.uProcessorRef) );
	uProcessorsOnClusterLock.uRelease();
    } else {
	uProcessorsOnClusterLock.uRelease();
	uAbort( ": attempt to add or move processor 0x%p to or from I/O cluster 0x%p (%.256s).\n"
	       "An I/O cluster (uIOCluster) can have only one processor.",
	       &p, this, this->uGetName() );
    } // if
} // uIOCluster::uProcessorAdd


void uIOCluster::uCreateIOCluster() {
#ifdef __U_MULTI__
    NBIO = new uNBIO;
#endif __U_MULTI__
} // uIOCluster::uCreateIOCluster


uIOCluster::uIOCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue ) : uCluster( ReadyQueue, uDefaultStackSize(), "uSystemCluster" ), processor( *this, 1.0 ) {
    uCreateIOCluster();
} // uIOCluster::uIOCluster


uIOCluster::uIOCluster( unsigned int stacksize, const char *name ) : uCluster( stacksize, name ), processor( *this ) {
    uCreateIOCluster();
} // uIOCluster::uIOCluster


uIOCluster::uIOCluster( const char *name ) : uCluster( name ), processor( *this ) {
    uCreateIOCluster();
} // uIOCluster::uIOCluster


uIOCluster::~uIOCluster() {
#ifdef __U_MULTI__
    delete NBIO;
#endif __U_MULTI__
} // uIOCluster::uIOCluster


const int uIOCluster::uReadSelect = 1;
const int uIOCluster::uWriteSelect = 2;
const int uIOCluster::uExceptSelect = 4;


int uIOCluster::uSelect( int fd, int rwe, timeval *timeout ) {
    return NBIO->uSelect( fd, rwe, timeout );
} // uIOCluster::uSelect


int uIOCluster::uSelect( fd_set *rfd, fd_set *wfd, fd_set *efd, timeval *timeout ) {
    return NBIO->uSelect( FD_SETSIZE, rfd, wfd, efd, timeout );
} // uIOCluster::uSelect


int uIOCluster::uSelect( int nfds, fd_set *rfd, fd_set *wfd, fd_set *efd, timeval *timeout ) {
    return NBIO->uSelect( nfds, rfd, wfd, efd, timeout );
} // uIOCluster::uSelect


// Local Variables: //
// compile-command: "dmake" //
// End: //
