/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
file name       :   protocol.cpp
implementation
of class        :   Protocol
author          :   Andreas Rinkel

version         :   1.0
history         :   25.04.96  Manfred Kraess
                    clean-up and modification
--------------------------------------------------------------------------*/

// C++ standards
#include <strstrea.h>

// TOOPS library
#include <toops.h>
#include <str.h>

// Project header
#include "protocol.h"   

//------------------------------------------------------------------------
Protocol:: Protocol(	ToopsProcess *entity, ToopsSocket *to_peer, 
						Synchron_Socket *to_user)
		: _ost ( _str, 200 )						
{ 
	String				_text;

	// connect the protocol to the sockets (for sending messages)
	_entity   = entity;
	_to_peer  = to_peer;
	_to_user  = to_user;

	// create a timer
	_text  = _entity->name();
	_text += ":timer";
	_timer = new ToopsTimer(_text, entity, 0);

	// set initial values for the entity's counters
	_ack_counter     = 0; 
	_send_counter    = 0;
	_receive_counter = 0;
	_timer_counter   = 0;
}

//------------------------------------------------------------------------
void Protocol:: handler (DataReq *data_req)
{  
	_ost << "handler: received DataReq from service user"
		  << ends;
	print_stringstream();
	
	// If the last package has been acknowledged, consumeTime and send the
	// data in a Data package to the peer entity. Increment the 
	// _send_counter (modulo 2) and start a timer.
	// Do not delete the _pdu because it may have to be sent again.
	if ( _ack_counter == _send_counter )
	{
		_pdu = new Data(_send_counter, data_req->getText(), 
						  data_req-> getTimeStamp());
		_ost	<< "handler: forming Data PDU number "
				<< _send_counter << " and incrementing send number" 
				<< ends;
		print_stringstream();
		_send_counter = (_send_counter + 1) % 2;
						  
		_entity-> consumeTime (DELAY_FOR_FORMING_PDU);
		_ost << "handler: sending Data PDU number "
			  << ((Data*)_pdu)->getSendNumber() << ", set timer to "
			  << TIMER_EXPIRATION << " time units" << ends;
		print_stringstream();
		_to_peer-> send (_pdu, SEND_DELAY_FOR_DATA_PDU);
		
		_timer -> start(TIMER_EXPIRATION);
	} 
	else
	{
		_ost << "handler: discarding DataReq, still waiting for receipt"
			  << ends; 
		print_stringstream();
	}
	delete data_req;
}

//------------------------------------------------------------------------
void Protocol:: handler (Data *data)
{
	DataInd *_data_ind;
	ACK     *_ack;

	_ost << "handler: received Data PDU number " 
		  <<  data-> _send_number << ends;
	print_stringstream();

	// If the _send_number is ok (this PDU has not been received before), 
	// then consumeTime and pass the contents on to the user as a DataInd 
	// service primitive. Increment the _receive_counter and the 
	// _ack_counter (modulo 2).
	if ( data-> _send_number == _receive_counter)
	{
		_data_ind = new DataInd( data->getText(), data->getTimeStamp() );
		_entity->  consumeTime (DELAY_FOR_FORMING_DATA_IND);
		_ost << "handler: passing DataInd to service user " 
			 << ends;	
		print_stringstream();			 	
		_to_user-> send(_data_ind);
		delete _data_ind;
		_receive_counter = (_receive_counter + 1) % 2;
	}

	// Return a receipt to the peer entity for the last package received.
	// The number in the receipt is the number of the last packet received 
	// plus 1.
	if ( ! FORGET_TO_SEND_RECEIPT ) 
	{
		_ost << "handler: sending receipt number " 
			  << _receive_counter << ends;		
		print_stringstream();
		_ack = new ACK(_receive_counter);
		_to_peer-> send (_ack, SEND_DELAY_FOR_ACK_PDU);
		delete _ack;
	}
	else
	{
		_ost << "handler:	forgetting to send a receipt ..." << ends;		
	}
	delete  data;
}

//------------------------------------------------------------------------
void Protocol:: handler (ACK *ack)
{
	
	// If the _ack_number is ok (it matches the number of the package
	// sent last plus 1), increment the _ack_counter (modulo 2) and stop 
	// the timer. The processing of the receipt take no time in this model. 
	if ( ack-> _ack_number != _ack_counter)
	{  
		_ost << "handler: received receipt number " 
			  << ack->_ack_number 
			  << ", stopping timer and incrementing _ack_counter" << ends ;
		print_stringstream();
		_ack_counter = ack-> _ack_number;
		if ( _timer->status() == ToopsTimer::active ) 
			_timer-> stop();
		else
		{
			_ost << "error: timer has expired before packet was received"
				  << ends;
			print_stringstream();
		}
		delete _pdu;
	}
	else
	{
		_ost << "handler: received receipt number " 
			  << ack->_ack_number << ", ignoring" << ends;
		print_stringstream();
	}
}

//------------------------------------------------------------------------
void Protocol:: timerHasExpired ()
{ 
	// Timer has expired. 
	// If the timer has not expired more than twice before:
	// * comsumeTime for processing,
	// * send the last PDU again and restart the timer.
	if ( _timer_counter < 2)
	{ 
		_ost << "timer has expired" << ends;
		print_stringstream();
		_entity-> consumeTime(DELAY_FOR_RESENDING_PDU);
		_ost << "sending last PDU again, setting timer to "
			  << TIMER_EXPIRATION << " time units" << ends;
		print_stringstream();
		_to_peer-> send (_pdu, SEND_DELAY_FOR_RESENDING);
		_timer-> start(TIMER_EXPIRATION);
	}
	// else discard the previously sent _pdu
	// (here, in a better protocol, the service user would receive
	// a warning)
	else
	{
		_ost << "timer has expired 3rd time, discarding PDU" << ends;
		print_stringstream();
		delete _pdu;

		// !!! 
		// here we should stop the simulation or reset both entities
		// for a correct implementation of the alternating bit protocol
		
	}
	// Increment the counter, that counts how often the timer
	// has expired
	_timer_counter++;
}

//------------------------------------------------------------------------
void Protocol:: print_stringstream( void )
{
	cout << ToopsSimulation::time() << " " << _entity->name()
		  << ": protocol " << _str << endl;
	_ost.seekp(0);   	
}

