#ifndef PDU_H
#define PDU_H
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
file name       :   pdu.h
class name      :   PDU, Data, ACK
author          :   Andreas Rinkel

documentation   :   Protocol Data Units PDUs: 
                    * PDU (baseclass) derived from ToopsMessage
                    * Data derived from PDU
                    * ACK  derived from PDU

version         :   1.0
history         :   25.04.96  Manfred Kraess
                    clean-up and modification
--------------------------------------------------------------------------*/

// C++ standards

// TOOPS library
#include <message.h>
#include <str.h>


//------------------------------------------------------------------------
// Type definition for type of PDU
typedef enum Prot_Data_Unit { data = 1, ack = 2 };


//------------------------------------------------------------------------
// Base class for PDU (Protocol Data Unit), the information unit that is
// passed to / received from the peer entity by our protocol entity.
class PDU: public ToopsMessage
{
public:
    // return the type of the PDU
    Prot_Data_Unit pdu() 
        {return _type;}

protected:
	// Type of the PDU
    Prot_Data_Unit _type;    
};




//------------------------------------------------------------------------
// Data: a PDU containing data, a time stamp and a serial number
class Data: public PDU
{
friend class Protocol;
public:
	// Constructor
    Data(int num, String str, int time): PDU()
    {  
        _type        = data;
        _send_number  = num;
        _timeStamp    = time;
        _text         = str;
    }

    // duplicte function as required by TOOPS
	virtual ToopsMessage *duplicate()
	{ 
		return new Data(_send_number, _text, _timeStamp); 
	}

	// get content of the message
	// Important: The return value of getText() is only a pointer to
	//            the internal String.              
	const char* getText() { return _text; }
	int		     getSendNumber() { return _send_number; } 
	int		     getTimeStamp() { return _timeStamp; }

private:
    // PDU's contents
    int    _send_number;
    int    _timeStamp;
    String _text;
};



//------------------------------------------------------------------------
// ACK: acknowledgement PDU, contains an acknowledgement number
class ACK: public PDU
{
friend class Protocol;
public:
	// Constructor
    ACK(int num): PDU()
    {  
        _type       = ack; 
        _ack_number = num;
    }

    // duplicte function as required by TOOPS
    virtual ToopsMessage *duplicate()
        { return new ACK(_ack_number); }
	
private:
    // PDU's contents
    int _ack_number;
};


#endif  

