#if !defined(ERROR_H)
#define ERROR_H

//-----------------------------------------------------------------------------
// ToopsError (source code -> error.cpp/.C)
// ====================================
// (class is defined in toops.h)
// An object of this error class is created for each error, warning, ...
// as a static member of class emitting the error.
// For use in your own application from TOOPS' classes create the appropriate
// ToopsError's as static members in your classes.
// To emit an error, a call to ToopsError::err() is required. Because an error
// object holds all basic informations about an error, in many cases no addi-
// tional information is needed for a exact description of the error (if more
// information has to be supplied, use err(char *xwho=0, char *xdesc1=0, ...)).
//
// The look of the error messages is defined as follows (2 examples):
// an error emitted with ToopsError::err()
// TOOPS fatal 1001 ToopsSimulation: start w/o ToopsProcessor defined
//                               |-> parameter desc in ToopsError(...)
//                  |-> paramter who in ToopsError(...)
//             |-> parameter errno in ToopsError(...)
//       |-> built automatically by interpreting severity s
// |-> automatically added
// an error emitted with ToopsError::err("P1", "don't use search()")
// TOOPS warning xxxx ToopsProcessor "P1": name is yet used don't use search()
//                                                      |-> par xdesc1 in err
//                                     |-> parameter desc in ToopsError(...)
//                                |-> parameter xwho in err
//                    |-> parameter who in ToopsError(...)
//               |-> parameter errno in ToopsError(...)
//       |-> built automatically by interpreting severity s
// |-> automatically added
//
// setSeverity allows a userprovided interpretation of all acessable error
// objects. It returns the old value.
// If no error handler exists, all output is sent to cerr.
// Default error handling:
// - fatal: immediate exit (always)
// - error: does not stop the program, but the simulation will be stopped.
// - warning: the program continues normally.
// - ignore is a special case: nothing will be sent cerr or the userdefined
//   error handler. If some errors can be ignored for an application, set
//   their severity to ignore. It is no good idea to do this with a fatal !!
// The userdefined error handler can be installed with setErrorHandler.
// It's type should be ToopsError::ERRFUNC. (Arguments: int errornumber,
// ToopsError::severity, const char*}; the char* will be the whole error message,
// ATTENTION: if this string is to be buffered, copy it, because it will be
// deleted immediatly after the error handler's return).

//-----------------------------------------------------------------------------
// All errors, warnings, ... created by TOOPS:
// - The error numbers are defined here to prevent problems.
// - Naming convention: TE_XXXXX_YYYYY
//                               |-> error type
//                         |-> the class or module where the error occured
//                      |-> short name for TOOPS ERROR
// - Many ToopsErrors (usually fatal errors) are declared as private, so their
//   interpretation cannot be modified (labeled with [p] below).
// - The interpretation of the other ToopsErrors can be modified.
//
//  The definitions of the error numbers, each preceeded by a short
//  description:
//
// fatal [p], an attempt to allocate memory (e.g. for a new ToopsTimeStep, a
// ToopsProcess's context, a copy of a message, ..) failed.
#define TE_OUT_OF_MEMORY               2000
// fatal [p], ToopsSimulation::start() called, when the simulation is not ready
// to be started (i.e.: no TOOPS objects exist (clear), the sim is running
// yet  (running) or the sim is stopped, but at least one TOOPS object of
// the previous simulation exists still (stopped)). A simulation can  only
// be started, if ToopsSimulation::state() returns notStarted.
#define TE_TSIMULATION_CANNOTSTART     1000
// fatal [p], call to ToopsSimulation::start() w/o at least one ToopsProcessor created.
#define TE_TSIMULATION_NOPROCESSOR     1001
// fatal [p], call to ToopsSimulation::start() w/o at least one ToopsProcess created.
#define TE_TSIMULATION_NOPROCESSES     1002

// error, a ToopsNamedObject's name is empty or not unique (if search() is not used
// and the ToopsNamedObject's name is not needed for debugging and displaying,
// turn this error off by ToopsNamedObject::eName.setSeverity(ToopsError::ignore) ).
// Note: ToopsProcessor, ToopsChannel, ... have each their own name space.
// If severity is ignore, no test for the name takes place.
#define TE_TNO_NAME                    1100

// warning, no ToopsProcess exists for this processor
#define TE_TPROCESSOR_NOPROCESS        1200
// fatal[p], destroying a ToopsProcessor with ToopsProcesses while the simulation
// is running
#define TE_TPROCESSOR_DESTROY          1201

// fatal [p], consumeTime() called from another ToopsProcess
#define TE_TPROCESS_CT_NOACCESS        1300
// fatal [p], setWaiting called from another ToopsProcess
#define TE_TPROCESS_SETWAITING_NOACCESS 1301
// warning, ToopsProcess terminated without call to terminate().
#define TE_TPROCESS_NOTERMINATE        1302
// fatal[p], destructor called in running sim for a not terminated ToopsProcess
#define TE_TPROCESS_DESTROY            1303
// fatal[p], a ToopsProcess cannot create another ToopsProcess with a higher priority
// when the simulation is running
#define TE_TPROCESS_PRIORITY_TO_LOW    1304
// fatal[p], a ToopsProcess cannot create a ToopsProcess on another ToopsProcessor
#define TE_TPROCESS_WRONG_OWNER        1305
// fatal[p], while the simulation is running, new ToopsProcesses can only be
// created from behavior().
#define TE_TPROCESS_CANNOT_CREATE      1306
// fatal [p], synchronize() called from another ToopsProcess
#define TE_TPROCESS_SYNC_NOACCESS      1307

// fatal [p], calling a controlling function from a ToopsProcess that is not the
// owner of that ToopsTimer.
#define TE_TTIMER_NOACCESS             1400
// warning, call to ToopsTimer::start() or ToopsTimer::startAbs() ignored, because
// the ToopsTimer's expiry time is less or equal time().
#define TE_TTIMER_IGNOREDSTART         1401
// fatal [p], calling a controlling function before simulation started.
#define TE_TTIMER_SIMNOTSTARTED        1402
// warning, call to ToopsTimer::stop() for a not active ToopsTimer
#define TE_TTIMER_IGNOREDSTOP          1403
// warning, indicates that a setDefault on an active ToopsTimer has no effect
// on the ToopsTimer's actual expiry time.
#define TE_TTIMER_SETDEFAULT           1404
// warning, call to start() or startAbs on active ToopsTimer. The ToopsTimer
// will be set to the new expiry time anyway.
#define TE_TTIMER_STARTACTIVE          1405
// warning, an active ToopsTimer's destructor is called (by ToopsTimer going out
// of scope or delete'ing it). The ToopsTimer will be destroyed, the alert will
// be cancelled.
#define TE_TTIMER_DESTROYACTIVE        1406

// fatal[p], trying to create a ToopsSocket from a ToopsProcess not beeing the owner or
// to create a ToopsSocket from anywhere but not ToopsProcess::behavior() when
// the simulation is running. Occurs also, when calling send, receive or
// synchronize from anywhere but owner()->behavior.
#define TE_TSOCKET_NOACCESS            1500
// fatal [p], calling a controlling function before simulation started.
#define TE_TSOCKET_SIMNOTSTARTED       1501

// fatal [p], deleting a ToopsChannel with connected ToopsSockets
#define TE_TCHANNEL_DESTROY            1600

// fatal [p], instantiating a ToopsMessage before the simulation started.
#define TE_TMESSAGE_SIMNOTSTARTED      1700
// fatal [p], ToopsMessage::duplicate must be overridden.
#define TE_TMESSAGE_NODUPLICATE        1701
#endif // if  !defined(ERROR_H)
