#if !defined(TIMER_H)
#define TIMER_H
#if !defined(TOOPS_H)
#include "toops.h"
#endif

#if !defined(TNO_LIST_H)
#include "tno_list.h"
#endif

class ToopsTimer;
class ToopsProcess;

//-----------------------------------------------------------------------------
// ToopsTimerLink
// ==========
// Set up a link to a ToopsTimer. See also to_list.h, tno_list.h.
class ToopsTimerLink : public ToopsNamedObjectLink
{
public:
    ToopsTimerLink (ToopsTimer *p) : 
         ToopsNamedObjectLink(( ToopsNamedObject*)p) {}
    virtual void write(int depth = 0, int mode = 0) const;
    ToopsTimer* object(void) { return (ToopsTimer*) _object();}
    inline void unLink(void);
    
    DECLARE_CLASSINFO(ToopsTimerLink); // pt 9.94
    
private:
    // prevent compiler from using copy / default constructor and assignment
    ToopsTimerLink();
    //ToopsTimerLink(const ToopsTimerLink&); //mjk 280895
    void operator=(const ToopsTimerLink&);
};

//-----------------------------------------------------------------------------
// ToopsTimer (source code -> timer.cpp/.C)
// ====================================
//
//class ToopsTimer : private ToopsNamedObject / pt 9.94
class ToopsTimer : public ToopsNamedObject
{
public:
    // ToopsTimer's states:
    // - idle: ToopsTimer is stopped or not started
    // - active: ToopsTimer is started or restarted and will alert at alertTime()
    // - expired: ToopsTimer has expired, owner() has been alerted. A ToopsTimer will
    //          remain in this state until started again or isExpired() is
    //          called.
    // - destroy: used internally for removing from lists in destructor
    // - create:  used internally
    enum timerState { idle, active, expired, destroy, create };
    timerState status(void) const { return t_state; }

    int isExpired (void); // Returns 1, if ToopsTimer expired and resets
                          // state to idle.
    // Create ToopsTimer named <name>, belonging to ToopsProcess <where> and having
    // a default time until alert of <defPeriod>
    ToopsTimer(const char *name, ToopsProcess &where, simtime defPeriod = 0);
    ToopsTimer(const char *name, ToopsProcess *where, simtime defPeriod = 0);

    // Make a ToopsTimer alert at time() + <period> (if <period> > 0) or at
    // time() + defPeriod() (if <period> == 0).
    void start(simtime period=0);

    // ToopsTimer alerts at time() == absTime.
    void startAbs(simtime absTime);

    // Stop a ToopsTimer (status = idle), returns left time until alert.
    simtime stop(void);

    simtime setDefault(simtime defPeriod); // set the default period, return
                                    // old value. Has no effect on an active
                                    // ToopsTimer's expiry time !!
    // Get the ToopsTimer's owner.
    ToopsProcess *owner(void) const { return t_owner; }

    // When will the ToopsTimer alert (if active) respectively when did the ToopsTimer
    // alert (if expired). Otherwise 0.
    simtime alertTime(void) const { return t_alertTime; }

    // Returns the actual default time span.
    simtime defPeriod(void) const{ return t_defPeriod; }

    ~ToopsTimer();

    // Get a ToopsTimer* by a given name. Returns 0, if not found.
    inline static ToopsTimer* search(const char *n);
    // Get an array of all ToopsTimers. The array _must_ be deleted by the program.
    // The length of the array can be determined with ToopsSimulation::numTimers().
    inline static ToopsTimer** systemTimers(void);

    static ToopsError wIgnoredStart;
    static ToopsError wIgnoredStop;
    static ToopsError wSetDefault;
    static ToopsError wStartActive;
    static ToopsError wDestroyActive;

/*    // re-export some information lost by the private inheritance:
    ToopsObject::isInheritedFrom;
    ToopsObject::DBGout;
    ToopsObject::isSameClass;
    ToopsNamedObject::name;
    ToopsSimulation::simstate;
    ToopsSimulation::state;
    ToopsSimulation::responsibility;
    ToopsSimulation::responsible;
    ToopsSimulation::stopReason;
    ToopsSimulation::time;
    ToopsSimulation::objects;
    ToopsSimulation::order;
    ToopsSimulation::stopSimulation;
    ToopsSimulation::stopWithError;
    ToopsSimulation::getTimeOut;
    ToopsSimulation::numProcessors;
    ToopsSimulation::numChannels;
    ToopsSimulation::numMessages;
    ToopsSimulation::numProcesses;
    ToopsSimulation::numTimers;
    ToopsSimulation::numSockets;*/	// pt 9.94

    enum dbgInfo {cstr=1, dstr=2, onStart=4, onStartAbs=8, onStop=16,
                  onIsExpired= 32, onSetDefault=32, onActivate=64 };
    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsTimer); // pt 9.94

protected:
    void activate(void);    // don't use this at all

    // explained in toops.h
    virtual void atStart(void) {}
    virtual void atEnd(void) {}
    virtual void atStateChange(void) {}

private:
    simtime t_alertTime;           // absolute simtime for alert
    simtime t_defPeriod;           // default period until alert, set by
                                   // ToopsTimer(..., defPeriod) or set(defPeriod)
    ToopsProcess *t_owner;             // the ToopsProcess to alert
    timerState t_state;            // ToopsTimer's state
    ToopsTimerLink processLink;        // Links to the owner
    ToopsTimerLink sysLink;            //          the system list
    ToopsTimerLink eventLink;          //          the event list

    void checkCaller(const char *n);
    void initTimer( ToopsProcess *where, simtime defPeriod);
    ToopsTimer( ToopsProcess *where, const char *n,
           simtime defPeriod, responsibility r);
    inline void stateChange(void);

    static ToopsError fSimNotStarted;
    static ToopsError fNoAccess;

    // prevent compiler from using copy / default constructor and assignment
    ToopsTimer();
    //ToopsTimer(const ToopsTimer&); //mjk 280895
    void operator=(const ToopsTimer&);
};


//-----------------------------------------------------------------------------
// ToopsTimerList
// ==========
// Class ToopsTimerList forms a doubly linked list with a built in iterator.
// See also to_list.h.
class ToopsTimerList : public ToopsNamedObjectList
{
public:
    // Set up a new (empty) ToopsTimerList. Current is reset.
    ToopsTimerList() : ToopsNamedObjectList() {}
    ~ToopsTimerList() {}
    // Insert a new ToopsTimerLink at the list's top. Current remains unchanged.
    void insert(ToopsTimerLink &r) { insert(&r); }
    void insert(ToopsTimerLink *p) { _insert((ToopsObjectLink*)p); }
    // Append a new ToopsTimerLink at the list's end. Current remains unchanged.
    void append(ToopsTimerLink &r) { append(&r); }
    void append(ToopsTimerLink *p) { _append((ToopsObjectLink*)p); }
    // Insert (append) a new ToopsTimerLink before (after) the current element.
    // Current remains unchanged. If current is reset, insert (append) at the
    // list's top (end).
    void inserthere(ToopsTimerLink &r) { inserthere(&r); }
    void inserthere(ToopsTimerLink *p) { _inserthere((ToopsObjectLink*)p); }
    void appendhere(ToopsTimerLink &r) { appendhere(&r); }
    void appendhere(ToopsTimerLink *p) { _appendhere((ToopsObjectLink*)p); }
    // Read the current (first, last) element. Current remains unchanged.
    ToopsTimer *current(void) { return (ToopsTimer*) _current(); }
    ToopsTimer *first(void) { return (ToopsTimer*) _first(); }
    ToopsTimer *last(void) { return (ToopsTimer*) _last(); }
    // Read the current (first, last) link. Current remains unchanged.
    ToopsTimer *currentLink(void) { return (ToopsTimer*) _currentLink(); }
    ToopsTimer *firstLink(void) { return (ToopsTimer*) _firstLink(); }
    ToopsTimer *lastLink(void) { return (ToopsTimer*) _lastLink(); }
    // Read the next element by passing a (refence to a) ToopsTimer* to next().
    // next() returns 0 (and rp is set to 0 !), if the list is empty or the
    // current element is the last one (in this case current will be reset).
    // Otherwise 1 will be returned , current will advance to the next ToopsTimer
    // in the list and rp will be set to the next ToopsTimer*.
    // If current is 0, it will be set to the first element, rp will point
    // to the first element and 1 will be returned.

    int next(ToopsTimer*& rp) 
    {
        if (_setnext())
            { rp = (ToopsTimer*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    // Read the previous element by passing a (refence to a) ToopsTimer* to prev().
    // prev() returns 0 (and rp is set to 0 !), if the list is empty or the
    // current element is the first one (in this case current will be reset).
    // Otherwise 1 will be returned, current will advance to the previous ToopsTimer
    // in the list and rp will be set to the previous ToopsTimer*.
    // If current is 0, it will be set to the last element, rp will point
    // to the last element and 1 will be returned.
    int prev(ToopsTimer*& rp)
    {
        if (_setprev())
            { rp = (ToopsTimer*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    // Get the next ToopsTimer in the list by passing (a reference to a) ToopsTimer* to
    // getnext(). If there is a next ToopsTimer in the list, 1 will be returned, rp
    // will be set to the next ToopsTimer* and the next ToopsTimer will be _removed_
    // from the list (current remains unchanged).
    // If the list is empty or current points to the last element, 0 will be
    // returned, current will be reset and rp is set to 0.
    // If current is 0, it will remain 0, rp will point to the first element
    // and 1 will be returned.
    int getnext(ToopsTimer*& rp)
    {
        ToopsTimer * pptmp;
        if ((pptmp = (ToopsTimer*) _getnext()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }

    // Get the previous ToopsTimer in the list by passing (a reference to a) ToopsTimer*
    // to getprev(). If there is a previous ToopsTimer in the list, 1 will be returned,
    // rp will be set to the previous ToopsTimer* and the previous ToopsTimer will be
    // _removed_ from the list (current remains unchanged).
    // If the list is empty or current points to the first element, 0  will be
    // returned, current will be reset and rp is set to 0.
    // If current is 0, it will remain 0, rp will point to the last element and
    // 1 will be returned.
    int getprev(ToopsTimer*& rp)
    {
        ToopsTimer* pptmp;
        if ((pptmp = (ToopsTimer*) _getprev()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
    // Read the next link by passing a (refence to a) ToopsTimerLink* to next().
    // See int next(ToopsTimer*&) for behaviour.
    int next(ToopsTimerLink*& rpl)
    {
        if (_setnext())
           { rpl = (ToopsTimerLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    // Read the previous element by passing a (refence to a) ToopsTimerLink* to
    // prev(). See int prev(ToopsTimer*&) for behaviour.
    int prev(ToopsTimerLink*& rpl)
    {
        if (_setprev())
            { rpl = (ToopsTimerLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    // Get the next ToopsTimerLink in the list by passing (a reference to a)
    // ToopsTimerLink* to getnext().  See int getnext(ToopsTimer*&) for behaviour.
    int getnext(ToopsTimerLink*& rpl)
    {
        ToopsTimerLink * pptmp;
        if ((pptmp = (ToopsTimerLink*) _getnextlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    // Get the previous ToopsTimerLink in the list by passing (a reference
    // to a) ToopsTimerLink* to getprev(). See int next(ToopsTimer*&) for behaviour.
    int getprev(ToopsTimerLink*& rpl)
    {
        ToopsTimerLink* pptmp;
        if ((pptmp = (ToopsTimerLink*) _getprevlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    
    // Remove a given link from the list.
    void remove(ToopsTimerLink* t) { _remove((ToopsObjectLink*)t); }

    // Search a ToopsTimer by it's name.
    ToopsTimer *search(const char* n) { return (ToopsTimer*) _search(n); }

    // Return an array of all timers in the list.
    ToopsTimer** touchAll(void) { return (ToopsTimer**) _touchAll(); }

    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsTimerList); // pt 9.94

private:
    // prevent compiler from using copy constructor and assignment
    //ToopsTimerList(const ToopsTimerList&); mjk 280895
    void operator=(const ToopsTimerList&);
};

inline ToopsTimer* ToopsTimer::search(const char *n)
    { return allTimers()->search(n); }

inline ToopsTimer** ToopsTimer::systemTimers(void)
    { return allTimers()->touchAll(); }

// moved here from timer.cpp  mjk 180796
inline void ToopsTimerLink::unLink(void)
    { ((ToopsTimerList*) linkedTo())->remove(this); }

#endif
