#include "toops.h"
#include "tprocess.h"
#include "timer.h"
#include "to_error.h"
#include "timestep.h"
#include "portable.h"
#include "event.h"
#include "strstrea.h"
#include <string.h>

//-----------------------------------------------------------------------------
// ToopsTimerLink (declared in timer.h)
// ================================
IMPLEMENT_CLASSINFO( ToopsTimerLink, ToopsNamedObjectLink)

// moved to timer.h  mjk 180796
//void ToopsTimerLink::unLink(void) 
//   { ((ToopsTimerList*) linkedTo())->remove(this); }

void ToopsTimerLink::write(int depth, int mode) const
{
    ToopsNamedObjectLink::write(depth, mode);
}

//-----------------------------------------------------------------------------
// ToopsTimer (declared in timer.h)
// ============================
IMPLEMENT_CLASSINFO( ToopsTimer, ToopsNamedObject)
void ToopsTimer::stateChange(void) { owner()->remember(&processLink); }
static void sayState(ostrstream &,const ToopsTimer* const); // pt 9.94 'static' hinzugefuegt

// initialize ToopsTimer's statics
ToopsError ToopsTimer::fNoAccess( TE_TTIMER_NOACCESS, ToopsError::fatal,
                          ToopsTimer::ThisClassInfo()->name(),
                          "\n ");
// pt 9.94 'ToopsTimer::' in Zeile 32 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::fSimNotStarted( TE_TTIMER_SIMNOTSTARTED, ToopsError::fatal,
                               ToopsTimer::ThisClassInfo()->name(),
                               "\n " );
// pt 9.94 'ToopsTimer::' in Zeile 36 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::wIgnoredStart( TE_TTIMER_IGNOREDSTART, ToopsError::warning,
                              ToopsTimer::ThisClassInfo()->name(),
                              "\n  alert time <= time(), ignored"   );
// pt 9.94 'ToopsTimer::' in Zeile 40 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::wIgnoredStop( TE_TTIMER_IGNOREDSTOP, ToopsError::warning,
                             ToopsTimer::ThisClassInfo()->name(),
                             "\n  not active, ignored stop()");
// pt 9.94 'ToopsTimer::' in Zeile 44 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::wSetDefault( TE_TTIMER_SETDEFAULT, ToopsError::warning,
                            ToopsTimer::ThisClassInfo()->name(),
                             "\n  setDefault while active has no effect"
                             "on the actual expiry time");
// pt 9.94 'ToopsTimer::' in Zeile 48 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::wStartActive( TE_TTIMER_STARTACTIVE, ToopsError::warning,
                             ToopsTimer::ThisClassInfo()->name(),
                             "\n  start() or startActive() called"
                             "on active ToopsTimer");
// pt 9.94 'ToopsTimer::' in Zeile 53 vor 'ThisClassInfo() ergaenzt
ToopsError ToopsTimer::wDestroyActive( TE_TTIMER_DESTROYACTIVE, ToopsError::warning,
                               ToopsTimer::ThisClassInfo()->name(),
                               "\n destructor call on active ToopsTimer");
// pt 9.94 'ToopsTimer::' in Zeile 58 vor 'ThisClassInfo() ergaenzt

// The constructors
ToopsTimer::ToopsTimer(const char *n, ToopsProcess &where, simtime defPeriod) :
               ToopsNamedObject(n, where.responsible()), sysLink(this),
               processLink(this), eventLink(this)
{
    initTimer(&where, defPeriod);
}


ToopsTimer::ToopsTimer(const char *n, ToopsProcess *where, simtime defPeriod) :
               ToopsNamedObject(n, where->responsible()), sysLink(this),
               processLink(this), eventLink(this)
{
    initTimer(where, defPeriod);
}

// Does the real work of the constructor.
void ToopsTimer::initTimer( ToopsProcess *where, simtime defPeriod)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::"<< ThisClassInfo()->name() << "("
        << name() << ", " << where->name() << ", " << defPeriod <<  ")\n")
    DBG_WRITE(cstr)
    t_owner= where;
    t_defPeriod =defPeriod;
    t_alertTime = 0;

    if ( (ToopsSimulation::state() == ToopsSimulation::running)
         && (where != ToopsProcess::thisProcess()) )
    {
        ostrstream o;
        if (ToopsProcess::thisProcess())
            o << ToopsProcess::thisProcess()->name()
              << " cannot create a ToopsTimer for " << t_owner->name();
        else
            o << "ToopsTimer for " << t_owner->name()
              << " can only be created from " << t_owner->name()
              << "::behavior() in running simulation";
        o << endl << ends;
        char *error = o.str();
        fNoAccess.err(name(),error);
        delete error;
    }

    if (eName.getSeverity() != ToopsError::ignore )
    {
        DBG_OUT(" checking name (ToopsError, if empty or not unique).\n", cstr)
        //if (strlen(name()) == 0 || allTimers()->search(name()))
        if (!allTimers()->isNameValid(name()) )
            eName.err(name());
    }

    t_state = create;

    DBG_OUT(" inserting sysLink in ToopsSimulation::allTimers()\n",cstr)
    allTimers()->insert(&sysLink);

    DBG_OUT(" inserting processLink in owner's ToopsTimerList\n",cstr)
    stateChange();
    t_state = idle;
    _DBG(write(),cstr)
    DBG_OUT(DBGends,cstr)
}

// Checks, if a call to the state modifying functions of ToopsTimer is legal
// (i.e. they can only be called by owner's behavior() and emits
// fatal ToopsError, if call was not allowed.
void ToopsTimer::checkCaller(const char *functionName)
{
    // check, whether caller has access, if not, emit fatal and stop the sim
    if (ToopsProcess::thisProcess() != owner())
    {
        if(ToopsSimulation::state() != ToopsSimulation::running)
            fSimNotStarted.err(name(), ThisClassInfo()->name() ,
                               functionName,
                               "cannot be called before ToopsSimulation::start()");
        else
        {
            if (ToopsProcess::thisProcess())
                fNoAccess.err(name(), functionName, "cannot be called by",
                              ToopsProcess::thisProcess()->name(), ", only by",
                              owner()->name());
            else
                fNoAccess.err(name(), functionName,
                              " callable only by ", owner()->name(),
                              "::behavior()");

        }
    }
}

// Resets state to idle and returns 1,  if the ToopsTimer was expired.
int ToopsTimer::isExpired(void)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::isExpired() checking caller:" )
    DBG_WRITE(onIsExpired)

    checkCaller("isExpired()");

    if (t_state == expired)
    {
        DBG_OUT(" o.k.\n status expired -> idle; returns 1",onIsExpired)
        DBG_OUT(DBGends,onIsExpired)
        t_state = idle;
        t_alertTime = 0;
        atStateChange();
        return 1;
    }
    DBG_OUT(" o.k.\n status was not expired -> not changed; returns 0",
            onIsExpired)
    DBG_OUT(DBGends,onIsExpired)
    return 0;
}

// Start a ToopsTimer
void ToopsTimer::start(simtime period)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::start(" << period << ") checking caller:" )
    DBG_WRITE(onStart)

    const char * cFunc = "start()";
    checkCaller(cFunc);
    DBG_OUT("o.k.\n",onStart)

    if (status() == active)
        wStartActive.err(name());

    if (period || defPeriod())
    {
        DBG_INIT()
        DBG(" ToopsTimer will alert at ")

        t_alertTime = time() + (period ? period : defPeriod() );
        if (timeOverFlow(t_alertTime))
            ToopsSimulation::_stop(timeOverFlowStop);

        t_state = active;
        atStateChange();
        DBG( t_alertTime << ", status: active" << endl)
        DBG( " inserting eventLink in timeSteps:\n")
        DBG_WRITE(onStart)
        timeSteps()->sortIn(&eventLink);
    }
    else
        wIgnoredStart.err(name(), cFunc);
    DBG_OUT(DBGends,onStart)
}

void ToopsTimer::startAbs(simtime absTime)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::startAbs(" << absTime
        << ") checking caller:" )
    DBG_WRITE(onStartAbs)

    const char * cFunc = "startAbs()" ;
    checkCaller(cFunc);

    DBG_OUT("o.k.\n",onStartAbs)

    if (status() == active)
        wStartActive.err(name());

    if (absTime > time() )
    {
        DBG_INIT()
        DBG(" ToopsTimer will alert at ")

        t_alertTime = absTime;
        t_state = active;
        atStateChange();

        DBG( " inserting eventLink in timeSteps:\n")
        DBG_WRITE(onStartAbs)
        timeSteps()->sortIn(&eventLink);
    }
    else
        wIgnoredStart.err(name(), cFunc);
    DBG_OUT(DBGends,onStartAbs)

}

// Stop a ToopsTimer. ToopsTimer's state changes to idle, if the ToopsTimer was active
simtime ToopsTimer::stop(void)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::stop() checking caller:" )
    DBG_WRITE(onStop)

    checkCaller("stop()");

    DBG_OUT("o.k.\n",onStop)

    simtime timeLeft = 0;
    if (status() == active)
    {
        DBG_INIT()
        DBG(" ToopsTimer set to idle " << owner()->name() << endl)
        DBG_WRITE(onStop)
        timeLeft = t_alertTime - time();
        t_alertTime = 0;
        t_state = idle;
        atStateChange();
        eventLink.unLink();
    }
    else
        wIgnoredStop.err(name());

    DBG_OUT(DBGends,onStop)

    return timeLeft;
}


simtime ToopsTimer::setDefault(simtime defP)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::setDefault(" << defP
        << ") checking caller:" )
    DBG_WRITE(onSetDefault)

    checkCaller("setDefault()");

    {
        DBG_INIT()
        DBG("o.k.\n default time set to" << defP << ", returning old: "
            << t_defPeriod << endl << DBGends)
        DBG_WRITE(onSetDefault)
    }

    simtime oldDefPeriod = t_defPeriod;
    t_defPeriod = defP;
    if (status() == active)
        wSetDefault.err(name());
    return oldDefPeriod;
}

ToopsTimer::~ToopsTimer()
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::~"<< ThisClassInfo()->name())
    DBG(" for " << name()  << " on " << owner()->name() << endl)
    DBG(" removing sysLink from ToopsSimulation::allTimers()\n")
    DBG_WRITE(dstr)

    allTimers()->remove(&sysLink);

    if ((status() == active) && (ToopsSimulation::state() == running) )
    {
        wDestroyActive.err(name());
        eventLink.unLink();
    }
    DBG_OUT(" status set to destroy, notifiying owner", dstr)
    t_state = destroy;
    atStateChange();
    stateChange();

    DBG_OUT(DBGends,dstr)
}

// called by ToopsSimulation::start(), when a ToopsTimer shall alert.
void ToopsTimer::activate(void)
{
    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() << "::activate() for " << name()
             << " on " << owner()->name() << "\n status set to expired at "
             << time() << endl)
    DBG_WRITE(onActivate)

    t_state = expired;
    atStateChange();

    stateChange();
    DBG_OUT(" ToopsTimer::activate calling nextEvent()\n\n",onActivate)
    nextEvent();
}


void ToopsTimer::write(int depth, int mode) const
{

    if (depth-1 >= 0)
        ToopsNamedObject::write(depth-1, mode);

    ostrstream o;

    if (depth==0 && !mode)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ":";
    sayState(o, this);
    o << ", owner: " << owner()->name();
    o << " alert time: " << alertTime() << ", default period: " << defPeriod();

    o << endl << ends;

    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;
}

static void sayState(ostrstream &o, const ToopsTimer* const t)
{
    o << "  status: ";
    switch (t->status())
    {
        case ToopsTimer::idle:      o << "idle    "; break;
        case ToopsTimer::active:    o << "active  "; break;
        case ToopsTimer::expired:   o << "expired "; break;
		default:  ; //NOP   mjk 310895
    }
}

//-----------------------------------------------------------------------------
// ToopsTimerList (declared in timer.h)
// ================================

IMPLEMENT_CLASSINFO( ToopsTimerList, ToopsNamedObjectList)

void ToopsTimerList::write(int depth, int mode) const
{
    ToopsObjectList::write(depth, mode);
}


