#include "toops.h"
#include "to_list.h"
#include "tno_list.h"
#include "tprocess.h"
#include "socket.h"
#include "channel.h"
#include "timestep.h"
#include "to_error.h"
#include "event.h"
#include "strstrea.h"
#include <assert.h>

//-----------------------------------------------------------------------------
// ToopsSocketLink (declared in socket.h)
// ==================================
IMPLEMENT_CLASSINFO( ToopsSocketLink, ToopsNamedObjectLink)

// moved to socket.h 180796 mjk
//void ToopsSocketLink::unLink(void)
//        {((ToopsSocketList*)linkedTo())->remove(this); }

void ToopsSocketLink::write(int depth, int mode) const
{
    ToopsNamedObjectLink::write(depth,mode);
}

//-----------------------------------------------------------------------------
// ToopsSocketList (declared in socket.h)
// ==================================
IMPLEMENT_CLASSINFO( ToopsSocketList, ToopsNamedObjectList)

void ToopsSocketList::write(int depth, int mode) const
{
    ToopsObjectList::write(depth,mode);
}

//-----------------------------------------------------------------------------
// ToopsSocket (declared in socket.h)
// ==============================
IMPLEMENT_CLASSINFO( ToopsSocket, ToopsNamedObject)


ToopsError ToopsSocket::fNoAccess(TE_TSOCKET_NOACCESS, ToopsError::fatal,
                          ToopsSocket::ThisClassInfo()->name(),
                          "\n ");
// pt9.94 'ToopsSocket::' in Zeile 43 vor 'ThisClassInfo() eingefuegt
ToopsError ToopsSocket::fSimNotStarted( TE_TSOCKET_SIMNOTSTARTED, ToopsError::fatal,
                              ToopsSocket:: ThisClassInfo()->name(),
                               "\n " );
// pt9.94 'ToopsSocket::' in Zeile 43 vor 'ThisClassInfo() eingefuegt

void ToopsSocket::stateChange(void)
{
    owner()->remember(&processLink);
}



ToopsSocket::ToopsSocket(const char *name, ToopsProcess *where, 
                 ToopsChannel *conn) :
                 ToopsNamedObject(name, where->responsible()),
                 sysLink(this), eventLink(this),
                 processLink(this), channelLink(this)
    { initSocket(where, conn); }

ToopsSocket::ToopsSocket(const char *name, ToopsProcess *where, 
                 ToopsChannel &conn) :
                 ToopsNamedObject(name, where->responsible()),
                 sysLink(this), eventLink(this),
                 processLink(this), channelLink(this)
    { initSocket(where, &conn); }

ToopsSocket::ToopsSocket(const char *name, ToopsProcess &where, 
                 ToopsChannel *conn) :
                 ToopsNamedObject(name, where.responsible()),
                 sysLink(this), eventLink(this),
                 processLink(this), channelLink(this)
    { initSocket(&where, conn); }

ToopsSocket::ToopsSocket(const char *name, ToopsProcess &where, 
                 ToopsChannel &conn):
                 ToopsNamedObject(name, where.responsible()),
                 sysLink(this), eventLink(this),
                 processLink(this), channelLink(this)
    { initSocket(&where, &conn); }


// does most of the constructor's job
void ToopsSocket::initSocket( ToopsProcess* where, ToopsChannel *conn)
{
    assert(where != 0);
    assert(conn != 0);

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::"<< ThisClassInfo()->name() << "("
        << name() << ", " << where->name() << ")\n")
    DBG_WRITE(cstr)

    s_owner = where;
    s_channel = conn;

    if ( (ToopsSimulation::state() == ToopsSimulation::running)
         && (where != ToopsProcess::thisProcess()) )
    {
        ostrstream o;
        if (ToopsProcess::thisProcess())
            o << ToopsProcess::thisProcess()->name()
              << " cannot create a ToopsSocket for " << s_owner->name();
        else
            o << "ToopsSocket for " << s_owner->name()
              << " can only be created from " << s_owner->name()
              << "::behavior() in running simulation";
        o << endl << ends;
        char *error = o.str();
        fNoAccess.err(name(),error);
        delete error;
    }

    if (eName.getSeverity() != ToopsError::ignore )
    {
        DBG_OUT(" checking name (ToopsError, if empty or not unique).\n", cstr)
        if (!allSockets()->isNameValid(name()) )
            eName.err(name());
    }

    s_state = create;

    DBG_OUT(" inserting sysLink in ToopsSimulation::allSockets()\n",cstr)
    allSockets()->insert(&sysLink);

    DBG_OUT(" inserting processLink in owner's ToopsSocketList\n",cstr)
    stateChange();

    DBG_OUT(" inserting channelLink in channels's ToopsSocketList\n",cstr)
    channel()->remember(&channelLink);

    //_DBG(write(),cstr)
    DBG_OUT(DBGends,cstr)

    s_state = ready;
}

ToopsSocket::~ToopsSocket()
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::~" << ThisClassInfo()->name() <<
        " for " << name() << endl)
    DBG_WRITE(dstr)

    s_state = destroy;
    atStateChange();

    DBG_OUT(" Deleting the ToopsMessages in the queues\n", dstr)
    // delete all remaining messages in the ToopsSocket's queues
	ToopsMessage* m; //"register" removed mjk 310895
    
    receivedMessages.reset();
        
    while(receivedMessages.getnext(m))
        delete m;
    messagesToReceive.reset();
    while(messagesToReceive.getnext(m))
        delete m;

    // remove the links
    DBG_OUT(" ToopsSocket::~ToopsSocket:removing the links\n\n", dstr)
    allSockets()->remove(&sysLink);
    stateChange();
    channel()->remember(&channelLink);
    if (eventLink.isLinked())
        eventLink.unLink();
}


void ToopsSocket::checkCaller(const char *functionName)
{
    // check, whether caller has access, if not, emit fatal and stop the sim
    if (ToopsProcess::thisProcess() != owner())
    {
        if(ToopsSimulation::state() != ToopsSimulation::running)
            fSimNotStarted.err(name(), ThisClassInfo()->name() ,
                               functionName,
                               "cannot be called before ToopsSimulation::start()");
        else
        {
            if (ToopsProcess::thisProcess())
                fNoAccess.err(name(), functionName, "cannot be called by",
                              ToopsProcess::thisProcess()->name(), ", only by",
                              owner()->name());
            else
                fNoAccess.err(name(), functionName,
                              " callable only by ", owner()->name(),
                              "::behavior()");

        }
    }
}


void ToopsSocket::send(ToopsMessage* msg, simtime delay)
{
    checkCaller("send()");

    s_state = sending;
    atStateChange();

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::send(ToopsMessage*, " << delay << ")"
        " for " << name() << " at " << time() << endl)
    DBG_WRITE(onSend)

    ToopsSocketList* l = channelLink.linkedTo();
    ToopsSocket* s;      //"register" removed mjk 310895
    ToopsMessage* dup;   //"register" removed mjk 310895

    // send it to all the channel's connected ToopsSockets
    switch (channel()->mode())
    {
        case ToopsChannel::bidirectional:
            DBG_OUT( " One ToopsSocket connected: duplicating the ToopsMessage\n",
                       onSend)

            // copy the message
            dup = msg->duplicate();
            if (dup == 0)
                fOutOfMem.err("","ToopsSocket: internal");

            // check, if the channel will let the message arrrive
            DBG_OUT( " Calling ToopsChannel::distort(): ",onSend)
            if (!channel()->distort(dup, this, s))
            {
                DBG_OUT("returned 0, sending ToopsMessage\n", onSend)
                // Ok, the message shall arrive
                // 'stamp' the ToopsMessage with the time to be received
                dup->m_timeToReceive =
                   time() + delay + channel()->delay(dup, this, s);
                // remember, only one other ToopsSocket on the channel
                if ((s = l->first()) == this)
                    s = l->last();

                DBG_INIT()
                DBG( " ToopsMessage will arrive at " << s->name() <<
                      " time: " << dup-> m_timeToReceive <<
                      ", calling takeMessage()" )
                DBG_WRITE(onSend)
                s->takeMessage(dup);
            }
            else
            {
                DBG_OUT("returned 1, ToopsMessage will be deleted\n", onSend)
                delete dup;
            }
            break;

        case ToopsChannel::broadcast:
            {
                DBG_INIT()
                DBG( " " << l->len() <<
                    "ToopsSockets connected, sending ToopsMessage\n")
                DBG_WRITE(onSend)
            }
            l->reset();
            while (l->next(s))
            {
                if(s != this)
                {
                    // copy the message
                    dup = msg->duplicate();
                    if (dup == 0)
                        fOutOfMem.err("","ToopsSocket: internal");

                    // check, if the channel will let the message arrrive
                    if (!channel()->distort(dup, this, s))
                    {
                        // Ok, the message shall arrive
                        // 'stamp' the ToopsMessage with the time to be received
                        dup->m_timeToReceive =
                         time() + delay + channel()->delay(dup, this, s);
                        DBG_INIT()
                        DBG(" copy sent to " << s->name() <<
                            ", will arrive at " << dup->m_timeToReceive <<
                            endl)
                        DBG_WRITE(onSend)
                        s->takeMessage(dup);
                    }
                    else
                        delete dup;
                }
            }
        case ToopsChannel::noMode:
        DBG_OUT(" no ToopsSockets connected, call ignored\n", onSend)
        ;
    }
    s_state = ready;
}

ToopsMessage* ToopsSocket::receive(void)
{
    ToopsMessage* msg;
    checkCaller("receive()");

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::receive()" <<
        " for " << name() << " at " << time() << endl)
    DBG_WRITE(onReceive)

    if (!receivedMessages.len())
    {
        DBG_OUT(" no messages, calling owner()->remember()\n", onReceive)
        s_state = inSync;
        atStateChange();
        stateChange();

    }

    s_state = ready;

    receivedMessages.reset();
    receivedMessages.getnext(msg);
    return msg;
}

void ToopsSocket::synchronize(void)
{
    checkCaller("synchronize()");

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::synchronize()" <<
        " for " << name() << " at " << time() << endl)
    DBG_WRITE(onSynchronize)

    if (!receivedMessages.len())
    {
        DBG_OUT(" no messages, calling owner()->remember()\n", onSynchronize)
        s_state = inSync;
        atStateChange();
        stateChange();

    }

    s_state = ready;
}

void ToopsSocket::takeMessage(ToopsMessage* msg)
{

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::takeMessage()" <<
        " for " << name() << " at " << time() << endl )

    // put the ToopsMessage in the messagesToReceive queue
    ToopsMessage* m;  //"register" removed mjk 310895
    messagesToReceive.reset();
    while(messagesToReceive.next(m) &&
           (m->receivedAt() <= msg->receivedAt()) );

    if(m)
        messagesToReceive.inserthere(msg);
    else
        messagesToReceive.append(msg);

    // insert the ToopsSocket as event at the time of the first message to
    // be received. (this could be done faster !!)
    s_nextTime = messagesToReceive.first()->receivedAt();
    if (eventLink.isLinked())
        eventLink.unLink();

    DBG(" inserting ToopsSocket as event at " << s_nextTime << endl)
    DBG_WRITE(onTakeMessage)

    timeSteps()->sortIn(&eventLink);
}


void ToopsSocket::activate(void)
{
    // one or more of the ToopsMessages in messagesToReceive must be received now
    // put the ToopsMessage(s) in the receivedMessages queue

    assert(messagesToReceive.len() > 0);
    assert(messagesToReceive.first()->receivedAt() == time());

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::activate()" <<
        " for " << name() << " at " << time() << endl )

    ToopsMessage* m; //"register" removed mjk 310895
    while(messagesToReceive.len() &&
          messagesToReceive.first()->receivedAt() == time())
    {
        messagesToReceive.reset();
        messagesToReceive.getnext(m);
        receivedMessages.append(m);
    }
     
    // if there are still messagesToReceive, make the ToopsSocket an event again
    // note: eventLink is always unlinked at the time activate() is called.
    if (messagesToReceive.len())
    {
        s_nextTime = messagesToReceive.first()->receivedAt();
        timeSteps()->sortIn(&eventLink);
        DBG(" ToopsSocket is new event at: " <<
              messagesToReceive.first()->receivedAt() << endl)
    }
    else
        s_nextTime = 0;

    DBG_WRITE(onActivate)

    // check, whether the ToopsSocket is inSync
    if (status() == inSync)
    {
        DBG_OUT(" ToopsSocket is synchronizing, setting all other sync"
                 "\n ToopsSockets ready via cancelSync()\n", onActivate)
        // processorLink.linkedTo() returns the list !!!
        ToopsSocketList *list = processLink.linkedTo();
        ToopsSocket* socket;
        list->reset();
        while (list->next(socket))
            socket->cancelSync();

        DBG_OUT("\n calling owner()->remember()\n", onActivate)
        // notify the owner
        s_state = gotMessage;
        stateChange();

    }

    s_state = gotMessage;
    atStateChange();

    s_state = ready;

    nextEvent();

}



void ToopsSocket::cancelSync(void)
{

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::cancelSync()" <<
        " for " << name() << " at " << time() << ", status: ready" << endl )
    DBG_WRITE(onCancelSync)
    s_state = ready;
    atStateChange();
}


void ToopsSocket::startSync(void)
{
    checkCaller("");

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::startSync()" <<
        " for " << name() << " at " << time() << ", status: inSync" << endl )
    DBG_WRITE(onCancelSync)

    s_state = inSync;
    atStateChange();
}


void ToopsSocket::write(int depth, int mode) const
{
    if (depth-1 >= 0)
        ToopsNamedObject::write(depth-1, mode);

    ostrstream o;

    if (depth==0 && !mode)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ": name " << name() << " on "
      << owner()->name() << ", status: ";
    switch (status())
    {
        case ready:
            o << "ready"; break;
        case inSync:
            o << "inSync"; break;
        case gotMessage:
            o << "gotMessage"; break;
        case sending:
            o << "sending"; break;
        case create:
            o << "create"; break;
        case destroy:
            o << "destroy"; break;
    }

    o << endl << " next event at: " << nextTime() << ", "
      << receivedMessages.len() << " received ToopsMessages\n";
    o << messagesToReceive.len() << " ToopsMessages to receive";


    o << endl << ends;

    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;

    if (!mode)
    {
        DBGout(" receivedMessages:\n");
        receivedMessages.write(3,0);
        DBGout(" messagesToReceive:\n");
        messagesToReceive.write(3,0);
    }

}

