#ifndef protocol_H
#define protocol_H
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
file name       :   protocol.h
class name      :   Protocol
author          :   Andreas Rinkel

documentation   :   Implementation of the Alternating Bit Protocol.
					  
					  The protocol can roughly be described as follows:
					  * If the entity receives a data request from the
						service user, it generates a protocol data unit
						PDU that will be sent to the peer entity. The Data
						PDU contains data, a time stamp and a send number.
						The PDU and its derived classes Data and ACK are 
						described in file pdu.h.
						Upon sending the PDU, a timer is set and the
						acknowledgement counter is increasesed (to match
						the number of the next acknowledgement expected).
						If the timer expires before the PDU has been
						acknowledged by the peer entity, it is sent again.
					  * If the entity receives a Data PDU from the peer,
						it sends back a numbered ACK PDU to acknowledge
						the PDU. It passes on the received data as a
						DataInd service primitive to the service user,
						if the send number of the Data PDU matches the
						packet counter of the protocol entity (i.e. this
						PDU has not been received before).
					  * If the entity receives an ACK PDU, it checks if
						it matches the packet couter (i.e. the receipt
						really acknowledges the PDU sent last). In this
						case, the timer is stopped and the previously
						sent PDU is destroyed.
					  * If a timer expires, the protocol checks, if it has
						expired less than twice before, in which case the
						last PDU is sent again. Otherwise it is just 
						discarded.
					  
					  The allowed values for the packet and acknowledgement
					  counters are 0 and 1 ("altenating bit protocol").						

version         :   1.0
history         :   25.04.96  Manfred Kraess
                    clean-up and modification
--------------------------------------------------------------------------*/

// C++ standards

// TOOPS library
#include <socket.h>
#include <tprocess.h>
#include <timer.h>
#include <strstrea.h>

// Project header
#include "sdu.h"
#include "pdu.h"
#include "synsoc.h"

// Manipulate these constants to change the behavior.
// You may also generate an interactive simulation by adding 
// routines to manipulate these or other values at runtime.
const int SEND_DELAY_FOR_DATA_PDU    = 1;
const int SEND_DELAY_FOR_ACK_PDU     = 2;
const int SEND_DELAY_FOR_RESENDING   = 1;
const int DELAY_FOR_FORMING_PDU      = 2;
const int DELAY_FOR_FORMING_DATA_IND = 2;
const int DELAY_FOR_RESENDING_PDU	 = 0;
const int TIMER_EXPIRATION           = 4;
const int FORGET_TO_SEND_RECEIPT     = 0;


//------------------------------------------------------------------------
class Protocol 
{
public:
	// Constructor:
	// * connect the protocol to the sockets (for sending messages)
	// * create a timer
	// * set initial values for the entity's counters
	Protocol( ToopsProcess    *entity,
			   ToopsSocket     *to_peer,
			   Synchron_Socket *to_user);

	// protocol handlers for various messages
	void handler		(DataReq *data_req);
	void handler		(Data    *data);
	void handler		(ACK     *data);

	// routine to handle expired timers	
	void timerHasExpired ();

private:
	// print_stringstream(): internal method to print info
	// To use printstringstream(), the info has to be written into the output
	// string stream _ost before printstringstream() is called. 
	char _str[200];
	ostrstream _ost;
	void print_stringstream( void );

	// reference to the process using this protocol and to the sockets
	ToopsProcess		*_entity;
	ToopsSocket		*_to_peer;
	Synchron_Socket	*_to_user;

	// Reference for the PDU to be send to the peer entity.
	// The PDU is stored until it has been acknowledged, because
	// it may have to be sent again.
	PDU					*_pdu;

	// timer for resending packages that have not been acknowledged
	// and a counter for the number of times the timer has expired
	ToopsTimer			*_timer;
	int					_timer_counter;   // 0, 1, 2

	// Important: All the following counters count modulo 2, i.e.
	//            the allowed values are only 0 and 1. 
	// number of the next package to send
	int					_send_counter;    // 0, 1
	// number of the next packet expected
	int					_receive_counter; // 0, 1
	// number of the next receipt expected
	int					_ack_counter;     // 0, 1

};
#endif  

