/****************************************************************************/
/*                                                                          */
/* wammtracelib.c - libreria instrumentata di WAMM                          */
/*                                                                          */
/****************************************************************************/

/*
 * 
 *
 *               WAMM: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, M. Cosso, G. Faieta, M. Formica, 
 *                      D. Laforenza, M. Nicosia 
 *                   (C) 1997 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * We want thanks Brad Topol of the Georgia Institute of Technology, Atlanta.
 */



#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <varargs.h>
#include <errno.h>
#include <sys/resource.h>


#ifdef IMA_SUN4SOL2 
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/signal.h>
#include <sys/fault.h>
#include <sys/syscall.h>
#include <sys/procfs.h>
#endif



#include "pvm3.h"

#ifdef hpux 
#include <sys/syscall.h>
#define getrusage(a, b) syscall(SYS_GETRUSAGE, a, b)
#endif

/********************************************************/

#define SIZE_OF_BYTE (sizeof(char))
#define SIZE_OF_COMPLEX (2*sizeof(float))
#define SIZE_OF_DCOMPLEX (2*sizeof(double))
#define SIZE_OF_DOUBLE (sizeof(double))
#define SIZE_OF_FLOAT (sizeof(float))
#define SIZE_OF_INT (sizeof(int))
#define SIZE_OF_LONG (sizeof(long))
#define SIZE_OF_SHORT (sizeof(short))

#define TRACEBUFSIZ 1400
#define TRACETAG 0x22cb173b
#define TRACEBARRTAG 0x22cb173c
#define INTERACTTAG 0x22cb173d
#define NAMELEN 256
#define ITER 100000 /* iterazioni di attesa era 1000 (21/5/97) */

#define SBLOCKING 13
#define MAXUSEREVENT 256
#define OL_CHECKIN 1
#define OL_STATISTIC 2
#define OL_CHECKOUT 3
#define OL_SCAN     0x22cb173e
#define OL_PRINT    5
#define REQ_SBLOCKING 14
#define RESTART 16
#define OL_MODIFYMINSECS 11
#define OL_EXIT_MON 15

/*******************************/
/* Variabili globali (private) */
/*******************************/

static int mypos=0;		/* posizione di questo task all'interno dell'array dei tid */
static int mytid=0;		/*il mio tid	*/
static int mon = 0;  	/* esprime il tipo di monitoraggio richiesto */
static int *tids=NULL;		/* array dei tid */
static int *punt=NULL;		/*array di appoggio per la realloc*/
static int lenght=0;		/* lunghezza dell'array tids */
static FILE *logfile=stderr;	/* nome del file di log */
static int lam =0;            /* timestamp di Lamport */
static struct timeval ref_time;	/*  timestamp di riferimento	*/
static struct timeval tt;	
static struct timeval tt1;	
static struct timeval ttsample_bg;	
static struct timeval ttsample_end;
static struct timeval tt_sample;
static struct timeval tt_idle;	/* Tempo di  idle  */
static struct timeval tt_send;	/* Tempo di  send  */
static struct timezone *tz=NULL;/* indica la timezone	*/
static int barriertotal=0;	/* serve per la barrier */
static char logname[256];	/* nome del file di traccia */
static char *outfilenm=NULL;     /*nome opzionale per il file di traccia*/
static char *prefixfilenm=NULL;  /* prefisso opzionale per il file di traccia*/
static int called_exit=0;
static char hostname[NAMELEN];
static int cont_close = 0;	/* conteggia il numero di fasi di monitoraggio */

static struct timeval previous_time={0,0};
static int wammtid=0;         /*tid del monitor*/
static int function_cnt=0;      /*contatore di funzioni */
static int total_sends=0;
static int total_receives=0;
static int interval_sends=0;
static int interval_receives=0;
static int interval_volume_rcvd=0;
static int interval_volume_snt=0;
static int total_volume_snt=0;
static int total_volume_rcvd=0;
static struct timeval int_idle_time={0,0};
static struct timeval tot_idle_time={0,0};
static struct timeval int_send_time={0,0};
static struct timeval tot_send_time={0,0};
static struct timeval int_program_time={0,0};
static struct timeval tot_program_time={0,0};

static int *send_vector;
static int *int_send_vector;
static int *send_byte_vector;
static int *receive_vector;
static int *receive_byte_vector;


static int stats_frequency=10;   /*frequenza per la raccolta delle informazioni
									sulla comunicazione */
static int probe_frequency=1;    /*frequency to probe*/
static int minimum_delay = 5;   /*frequenza di default per il campionamento*/

/*******************************/
/* Funzioni globali (private)  */
/*******************************/

static void diffTime();
static void addTime();
static void checkSampling();
static int wamm_scaninternal();
int wammOL_print();

/********************/
/* Funzioni globali */
/********************/

#ifdef IMA_SUN4SOL2 
static int getpagesize();
#endif


#ifdef hpux
static int getpagesize() 
{
  return (int) sysconf(3001);
}
#endif


/******************************************/
/* rec_sends - memorizza statistiche send */
/******************************************/

static void rec_sends(volume,send_time)
int volume;
struct timeval send_time;
{
	interval_sends++;
    total_sends++;
    interval_volume_snt+=volume;
    total_volume_snt+=volume;
    addTime(&int_send_time,&send_time,&int_send_time);
    addTime(&tot_send_time,&send_time,&tot_send_time);
}

/*************************************************/
/* rec_receives - memorizza statistiche receives */
/*************************************************/

static void rec_receives(volume,idle_time)
int volume;
struct timeval idle_time;
{
    interval_receives++;
    total_receives++;
    interval_volume_rcvd+=volume;
    total_volume_rcvd+=volume;
    addTime(&int_idle_time,&idle_time,&int_idle_time);
    addTime(&tot_idle_time,&idle_time,&tot_idle_time);
}


/**********************************/
/* maketimestamp - crea timestamp */
/**********************************/

static void maketimestamp(t)
struct timeval* t;
{
    struct timeval tx;
    double m1, m2;
    gettimeofday(&tx,tz);
    if (tx.tv_usec >= ref_time.tv_usec) {
	t->tv_usec = tx.tv_usec-ref_time.tv_usec;
	t->tv_sec = tx.tv_sec-ref_time.tv_sec;
    }
    else {
	/* borrow a second */
	t->tv_usec = (tx.tv_usec+1000000)-ref_time.tv_usec;
	t->tv_sec = (tx.tv_sec-1)-ref_time.tv_sec;
    }    
}

/********************************************/
/* send_stats - spedisce statistiche a wamm */
/********************************************/

static void send_stats()
{
    register int i;
    int info,oldbuf, sbuf;
	struct rusage usageinfo;

    maketimestamp(&ttsample_end);
    diffTime(&ttsample_end,&ttsample_bg,&tt_sample);
    addTime(&int_program_time,&tt_sample,&int_program_time);
    addTime(&tot_program_time,&tt_sample,&tot_program_time);
	getrusage(RUSAGE_SELF,&usageinfo);
    if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0) 
	pvm_perror("wamm mkbuf");
    if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
	pvm_perror("wamm setsbuf");
	pvm_pklong(&usageinfo.ru_utime.tv_sec, 1, 1);
    pvm_pklong(&usageinfo.ru_utime.tv_usec, 1, 1);
    pvm_pklong(&usageinfo.ru_stime.tv_sec, 1, 1);
    pvm_pklong(&usageinfo.ru_stime.tv_usec, 1, 1);
    pvm_pklong(&usageinfo.ru_maxrss, 1, 1);
    pvm_pkint(&interval_sends, 1, 1);
    pvm_pkint(&total_sends, 1, 1);
    pvm_pkint(&interval_receives, 1, 1);
    pvm_pkint(&total_receives, 1, 1);
    pvm_pkint(&interval_volume_snt, 1, 1);
    pvm_pkint(&total_volume_snt, 1, 1);
    pvm_pkint(&interval_volume_rcvd, 1, 1);
    pvm_pkint(&total_volume_rcvd, 1, 1);
    pvm_pklong(&int_idle_time.tv_sec, 1, 1);
    pvm_pklong(&int_idle_time.tv_usec, 1, 1);
    pvm_pklong(&tot_idle_time.tv_sec, 1, 1);
    pvm_pklong(&tot_idle_time.tv_usec, 1, 1);
    pvm_pklong(&int_send_time.tv_sec, 1, 1);
    pvm_pklong(&int_send_time.tv_usec, 1, 1);
    pvm_pklong(&tot_send_time.tv_sec, 1, 1);
    pvm_pklong(&tot_send_time.tv_usec, 1, 1);   
    pvm_pklong(&int_program_time.tv_sec, 1, 1);
    pvm_pklong(&int_program_time.tv_usec, 1, 1);
    pvm_pklong(&tot_program_time.tv_sec, 1, 1);
    pvm_pklong(&tot_program_time.tv_usec, 1, 1);
	pvm_pkint(&lenght, 1, 1);
    pvm_pkint(send_vector,lenght,1);
    pvm_pkint(int_send_vector,lenght,1);
    pvm_pkint(send_byte_vector,lenght, 1);
    pvm_pkint(receive_vector,lenght, 1);
    pvm_pkint(receive_byte_vector,lenght, 1); 
    pvm_send(wammtid, OL_STATISTIC);
    pvm_setsbuf(oldbuf);
    pvm_freebuf(sbuf);
    
    for (i=0; i<lenght; i++) {
	int_send_vector[i] = 0;
    }
    interval_sends = interval_receives = 0;
    interval_volume_snt = interval_volume_rcvd = 0;
    int_idle_time.tv_sec= int_send_time.tv_sec =int_program_time.tv_sec= 0;
    int_idle_time.tv_usec= int_send_time.tv_sec=int_program_time.tv_sec = 0;
    maketimestamp(&ttsample_bg);
}

/*********************************************************/
/* probe_interaction - gestisce messaggio di interazione */
/*********************************************************/

static void probe_interaction()
{

    int info,oldbuf, sbuf;
    int type;
    int minsecs;
    oldbuf = pvm_setrbuf(0);
    info = pvm_probe(wammtid, INTERACTTAG);
    if (info == 0)  {
        pvm_freebuf(pvm_setrbuf(oldbuf));
		return;
    }
    if (info < 0) {
        pvm_perror("probe_interaction");
        pvm_freebuf(pvm_setrbuf(oldbuf));
    	return;
    }
    else {
        pvm_recv(wammtid, INTERACTTAG);
    	pvm_upkint(&type, 1, 1);
    }
	if (type == OL_MODIFYMINSECS) {
	    pvm_upkint(&minsecs, 1, 1);
        wamm_setsampling(stats_frequency, minsecs);
	}
    pvm_freebuf(pvm_setrbuf(oldbuf));
  	return;
}

/*********************************************************/
/* check_interaction - gestisce messaggio di interazione */
/*                     ricevuto in una receive           */
/*********************************************************/

static void check_interaction()
{
    int type;
    int minsecs;
    
 
    pvm_upkint(&type, 1, 1);
    if (type == OL_MODIFYMINSECS) {
		pvm_upkint(&minsecs, 1, 1);
		wamm_setsampling(stats_frequency, minsecs);
    }
    return;

}

/**************************************************/
/* checkSampling - controllo scadenza del periodo */
/*                 di campionamento               */
/**************************************************/

static void checkSampling(cur_time)
struct timeval cur_time;
{
    if (cur_time.tv_sec - previous_time.tv_sec >= minimum_delay) {
		if ((mon > 1) && (function_cnt%stats_frequency==0)) {
	    	send_stats();
	    	previous_time.tv_sec = cur_time.tv_sec;
		}
	}
    if ((mon > 1) && (function_cnt%probe_frequency==0)) probe_interaction ();
    function_cnt++;
}


/***************************************/
/* db_index - ricerca tid nel database */
/***************************************/

static int db_index(int tid)
{
    int i,who,cc;
    char str[10];

    sprintf(str, "%d", tid);
	while ((cc=pvm_lookup(str, 0, &who))==PvmNoEntry) {
		for (i=0; i<ITER; i++){
		;
		}
	}
	return who;
}


/**********************************************/
/* tid_index - ricerca tid nell'array dei tid */ 
/**********************************************/

static int tid_index(int tid)
{
    int i;
    int m = -1;           
    for (i=0; i<lenght; i++) {
		if(tids[i]==tid) {
	    	m=i;
	   	 	break;
		}
    }
    return m;
}

/***************/
/* getdatasize */
/***************/

static int getdatasize(dtype)
     int dtype;
{

switch(dtype) {
case PVM_STR:
     return(SIZE_OF_BYTE);
case PVM_BYTE:
     return(SIZE_OF_BYTE);
case PVM_USHORT:
case PVM_SHORT:
     return(SIZE_OF_SHORT);
case PVM_UINT:
case PVM_INT:
     return(SIZE_OF_INT);
case PVM_ULONG:
case PVM_LONG:
     return(SIZE_OF_LONG);
case PVM_FLOAT:
     return(SIZE_OF_FLOAT);
case PVM_DOUBLE:
     return(SIZE_OF_DOUBLE);
case PVM_CPLX:
     return(SIZE_OF_COMPLEX);
case PVM_DCPLX:
     return(SIZE_OF_DCOMPLEX);

}
}

/************************************************/
/*	wammOL_print - invia output all'interfaccia	*/
/************************************************/

int wammOL_print(va_alist)
va_dcl
{
	static char tbuf[2049];
    int info,oldbuf, sbuf;
    char *fmt2;
    va_list ap;
    va_start(ap);

	if (mon>1) {
		fmt2 = va_arg (ap, char *);
		vsprintf(tbuf, fmt2, ap);
   	 	if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0) 
			pvm_perror("errore nella mkbuf");
    	if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
       		pvm_perror("errore nella setsbuf");
    	pvm_pkstr(tbuf);
    	pvm_send(wammtid, OL_PRINT);
    	pvm_setsbuf(oldbuf);
    	pvm_freebuf(sbuf);
		va_end(ap);
	}
	else {
		fmt2 = va_arg (ap, char *);
    	vprintf(fmt2, ap);
	}
	return 0;
}

/**********************************/
/* wamm_scan* - funzioni di input */
/**********************************/

int wamm_scanstring(prompt, str)
char *prompt, *str;
{
    int type=PVM_STR;
    return(wamm_scaninternal(prompt, str, type));
}

int wamm_scanint(prompt, value)
char *prompt;
int  *value;
{
    char str[256];
    int type=PVM_INT;
    wamm_scaninternal(prompt, str, type);
    sscanf(str,"%d", value);
    return(0);
}

int wamm_scanfloat(prompt, value)
char *prompt;
float *value;
{
    char str[256];
    int type=PVM_FLOAT;
    wamm_scaninternal(prompt, str, type);
    sscanf(str,"%f", value);
    return(0);
}

int wamm_scanlong(prompt, value)
char *prompt;
long  *value;
{
    char str[256];
    int type=PVM_LONG;
    wamm_scaninternal(prompt, str, type);
    sscanf(str,"%ld", value);
    return(0);
}

int wamm_scandouble(prompt, value)
char *prompt;
double *value;
{
    char str[256];
    int type=PVM_DOUBLE;
    wamm_scaninternal(prompt, str, type);
    sscanf(str,"%lf", value);
    return(0);
}

/********************************************************/
/* wamm_scaninternal - effettua la ricezione dell'input */
/********************************************************/

static int wamm_scaninternal(prompt, str, type)
char *prompt, *str;
int type;
{
    int info,oldbuf, sbuf;
    if (!wammtid){
    	pvm_lookup("monitor", 1, &wammtid);
    }
    if ((prompt != NULL) && (str != NULL)) {
	if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0) 
	    pvm_perror("wammtids mkbuf");
	if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
	    pvm_perror("wammtids setsbuf");
        pvm_pkint(&type, 1, 1);
	pvm_pkstr(prompt);
	pvm_send(wammtid, OL_SCAN);
	pvm_setsbuf(oldbuf);
	pvm_freebuf(sbuf);

	oldbuf = pvm_setrbuf(0);

	if (pvm_recv(wammtid, OL_SCAN)< 0) 
	    pvm_perror("wammtids pvm_recv");
	pvm_upkstr(str);
	pvm_freebuf(pvm_setrbuf(oldbuf));	
    }
   
    return 0;
}

/***********************************************************/
/* wamm_setsampling - cambia la frequenza di campionamento */
/***********************************************************/

int wamm_setsampling(newfreq,newdelay)
int newfreq;
int newdelay;
{
    
	/*if (mon < 2) return 0;*/
    function_cnt=0;
    if (newfreq < 2)  newfreq = 2;
    if (newdelay < 2) newdelay=2;
    minimum_delay = newdelay;
    stats_frequency = newfreq;
    return 0;
}

    
/**********************************************************/			
/*  wamm_mytid - effettua la pvm_mytid ed inizializza le  */
/*               strutture per il monitoraggio            */
/**********************************************************/

int wamm_mytid()
{
	char str[10];
	int pagesize;
	int oldbuf, sbuf;	
	FILE *fd;
	
	if (mytid) return(mytid);			/* caso in cui e' gia' stata 
										invocata una pvm_mytid()*/
	mytid = pvm_mytid();
	pvm_lookup("monitor",0,&mon);	     /* scopro che monitoraggio devo fare */
	if (!mon) {
		return(mytid);		     /* non devo monitorare */
	}
	pvm_lookup("monitor", 1, &wammtid);  /* devo monitorare */
	mypos = pvm_insert("tid", -1, mytid);  /* scopre la sua posizione in tids */
	sprintf(str, "%d", mytid);
	pvm_insert(str, 0, mypos); 	     /* rende nota agli altri task la sua posizione */
	gettimeofday(&ref_time,tz);
	tids = (int *) calloc(mypos+1, SIZE_OF_INT);
	lenght = mypos+1;
	tids[mypos] = mytid;
	if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
		pvm_perror("errore nella mkbuf");
	if ((oldbuf = pvm_setsbuf(sbuf)) <0)
		pvm_perror("errore nella setsbuf");
	pvm_pkint(&mypos, 1, 1);
	
	switch (mon) {
		case 1:
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			if (prefixfilenm) {
				sprintf(logname, "/tmp/%s.pvanimfile.%d.%s.%d", prefixfilenm,
				getuid(), hostname, getpid());
			}
			else {
				sprintf(logname, "/tmp/pvanimfile.%d.%s.%d", getuid(),
		 		hostname, getpid());
			}
			fd = fopen(logname, "w+");
			if (fd != 0) {
				logfile = fd;
				setvbuf(logfile, NULL, _IOFBF, 10240);
			}
			else {
				printf("la wamm_mytid non riesce ad aprire il
				logfile\n");
			}
			maketimestamp(&tt);
			fprintf (logfile, "1 %d %d %d %d 0 0 0\n", lam++, tt.tv_sec,
			tt.tv_usec, mypos);
		break;
		case 2:
			gethostname(hostname, NAMELEN);
			pagesize = getpagesize();
	
	/* alloco le strutture necessarie */
	
			send_vector = (int *) calloc(mypos+1, SIZE_OF_INT);
			int_send_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			send_byte_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			receive_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			receive_byte_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		   		
		
			pvm_pkstr(hostname);
			pvm_pkint(&pagesize, 1, 1);
			pvm_pkint(&minimum_delay, 1, 1);
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			maketimestamp(&ttsample_bg);
			send_stats();
		break;
		case 3:
			gethostname(hostname, NAMELEN);
			pagesize = getpagesize();
			send_vector = (int *) calloc(mypos+1, SIZE_OF_INT);
			int_send_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			send_byte_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			receive_vector = (int *) calloc(mypos+1, SIZE_OF_INT);		
			receive_byte_vector = (int *) calloc(mypos+1, SIZE_OF_INT);	
			
			pvm_pkstr(hostname);
			pvm_pkint(&pagesize, 1, 1);
			pvm_pkint(&minimum_delay, 1, 1);
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			if (prefixfilenm) {
				sprintf(logname, "/tmp/%s.pvanimfile.%d.%s.%d", prefixfilenm,
				getuid(), hostname, getpid());
			}
			else {
				sprintf(logname, "/tmp/pvanimfile.%d.%s.%d", getuid(),
		 		hostname, getpid());
			}
			fd = fopen(logname, "w+");
			if (fd != 0) {
				logfile = fd;
				setvbuf(logfile, NULL, _IOFBF, 10240);
			}
			else {
				printf("la wamm_mytid non riesce ad aprire il
				logfile\n");
			}
			maketimestamp(&tt);
			fprintf (logfile, "1 %d %d %d %d 0 0 0\n", lam++, tt.tv_sec,
			tt.tv_usec, mypos);
		break;
		default: 
			break;
	}
return (mytid);
}		

/****************************************/
/* wamm_parent - effettua la pvm_parent */
/****************************************/

int wamm_parent()
{
	int val;
	if ((val = pvm_parent()) == wammtid) {
		return PvmNoParent;
	}
	else 
		return (val);
}

/**************************************************************/

int wamm_chprefix(nme)
char *nme;
{
    if (mon == 0 || mon == 2) return 0;		/* se non traccio */
	prefixfilenm = malloc(strlen(nme) + 1);
	strcpy (prefixfilenm,nme);
    return(0);
}

/********************************************************/
/* wamm_print - registra evento specificato dall'utente */
/********************************************************/

int wamm_print(evt)
char *evt;
{
    char buf[MAXUSEREVENT+1];
    if (mon == 0 || mon == 2) return(0);	/* se non traccio */
	maketimestamp(&tt);
	strncpy(buf, evt, MAXUSEREVENT);
	buf[MAXUSEREVENT] = '\0';
	fprintf (logfile, "20 %d %d %d %d %s \n", 
		 lam++,tt.tv_sec, tt.tv_usec, mytid, buf);
}

/***************************************************/
/* wamm_open - apre una nuova fase di monitoraggio */
/***************************************************/

int wamm_open()
{
	char str[10];
	int pagesize;
	int oldbuf, sbuf;	
	FILE *fd;
	int i;
	
	
	pvm_lookup("monitor",0,&mon);	  /* scopro che monitoraggio devo fare */
	if (!mon) {
		return(0);		     /* non devo monitorare */
	}
	
	called_exit = 0;
 	
	gettimeofday(&ref_time,tz);	
	

	/* riazzero le strutture necessarie */
	
	if (mon > 1) {
		previous_time.tv_usec=0;
		previous_time.tv_sec=0;
		total_sends=0;
		total_receives=0;
		total_volume_snt=0;
		total_volume_rcvd=0;
		tot_idle_time.tv_usec=0;
		tot_idle_time.tv_sec=0;
		tot_send_time.tv_usec=0;
		tot_send_time.tv_sec=0;
		tot_program_time.tv_usec=0;
		tot_program_time.tv_sec=0;
	
		for (i=0; i<lenght; i++) {
			send_vector[i] = 0;		
			send_byte_vector[i] = 0;		
			receive_vector[i] = 0;		
			receive_byte_vector[i] = 0;
		}		
	}        		
	if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
		pvm_perror("errore nella mkbuf");
	if ((oldbuf = pvm_setsbuf(sbuf)) <0)
		pvm_perror("errore nella setsbuf");
	pvm_pkint(&mypos, 1, 1);
	
	switch (mon) {
		case 1:
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			lam = 0;
			if (prefixfilenm) {
				sprintf(logname, "/tmp/%s.pvanimfile.%d.%s.%d.%d", prefixfilenm,
				getuid(), hostname, getpid(),cont_close);
			}
			else {
				sprintf(logname, "/tmp/pvanimfile.%d.%s.%d.%d", getuid(),
		 		hostname, getpid(),cont_close);
			}
			fd = fopen(logname, "w+");
			if (fd != 0) {
				logfile = fd;
				setvbuf(logfile, NULL, _IOFBF, 10240);
			}
			else {
				printf("la wamm_open non riesce ad aprire il
				logfile\n");
			}
			maketimestamp(&tt);
			fprintf (logfile, "1 %d %d %d %d 0 0 0\n", lam++, tt.tv_sec,
			tt.tv_usec, mypos);
		break;
		case 2:
			pvm_pkstr(hostname);
			pvm_pkint(&pagesize, 1, 1);
			pvm_pkint(&minimum_delay, 1, 1);
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			maketimestamp(&ttsample_bg);
			send_stats();
		break;
		case 3:
			pvm_pkstr(hostname);
			pvm_pkint(&pagesize, 1, 1);
			pvm_pkint(&minimum_delay, 1, 1);
			pvm_send(wammtid, OL_CHECKIN);
			pvm_setsbuf(oldbuf);
			pvm_freebuf(sbuf);
			maketimestamp(&ttsample_bg);
			send_stats();
			lam = 0;
			if (prefixfilenm) {
				sprintf(logname, "/tmp/%s.pvanimfile.%d.%s.%d.%d", prefixfilenm,
				getuid(), hostname, getpid(),cont_close);
			}
			else {
				sprintf(logname, "/tmp/pvanimfile.%d.%s.%d.%d", getuid(),
		 		hostname, getpid(),cont_close);
			}
			fd = fopen(logname, "w+");
			if (fd != 0) {
				logfile = fd;
				setvbuf(logfile, NULL, _IOFBF, 10240);
			}
			else {
				printf("la wamm_open non riesce ad aprire il
				logfile\n");
			}
			maketimestamp(&tt);
			fprintf (logfile, "1 %d %d %d %d 0 0 0\n", lam++, tt.tv_sec,
			tt.tv_usec, mypos);
		break;
		default: break;
	}
return (0);
}		


/************************************************/
/* wamm_close - chiude una fase di monitoraggio */
/************************************************/

int wamm_close()			/* esco dal monitoraggio ma non dal pvm */
{
    int inum,retval,sbuf, oldbuf;
    int f1,n,i;
    int bufid, nb, tag, rid;
    char buf[TRACEBUFSIZ];
    if (mon == 0 || called_exit)  return 0; /* se non traccio */
    called_exit=1;
	switch(mon) {

		case 1:
			maketimestamp(&tt);
			fprintf (logfile, "19 %d %d %d %d 0\n", lam++, tt.tv_sec, tt.tv_usec, mypos);
			if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
				pvm_perror("oltids mkbuf");
			if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
				pvm_perror("oltids setsbuf");
			pvm_send(wammtid, REQ_SBLOCKING);
			pvm_recv(wammtid, SBLOCKING);
			fclose(logfile);
			if ((f1 = open(logname,O_RDONLY,0)) == -1) {
		   		printf("wamm_close: cannot reopen tracefile");
			}
			else {
		   		unlink(logname);
		   		while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					pvm_initsend(PvmDataDefault);
					pvm_pkint(&n, 1, 1);
					pvm_pkbyte(buf, n, 1);
					pvm_send(wammtid, TRACETAG);
		    	} 
		    	n= -1;
		    	pvm_initsend(PvmDataDefault);
		    	pvm_pkint(&n, 1, 1);
		    	pvm_send(wammtid, TRACETAG);
		    	close(f1);
		    	pvm_recv(wammtid, RESTART);
			}
			break;

		case 2:
			send_stats();
			break;

		case 3:
			send_stats();
			maketimestamp(&tt);
			fprintf (logfile, "19 %d %d %d %d 0\n", lam++, tt.tv_sec, tt.tv_usec, mypos);
			if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
				pvm_perror("oltids mkbuf");
			if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
				pvm_perror("oltids setsbuf");
			pvm_send(wammtid, REQ_SBLOCKING);
			pvm_setsbuf(oldbuf);
    		pvm_freebuf(sbuf);
			fclose(logfile);
			pvm_recv(wammtid, SBLOCKING);
			fclose(logfile);
			if ((f1 = open(logname,O_RDONLY,0)) == -1) {
		   		printf("wamm_close: cannot reopen tracefile");
			}
			else {
		   		unlink(logname);
		   		while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					pvm_initsend(PvmDataDefault);
					pvm_pkint(&n, 1, 1);
					pvm_pkbyte(buf, n, 1);
					pvm_send(wammtid, TRACETAG);
		    	} 
		    	n= -1;
		    	pvm_initsend(PvmDataDefault);
		    	pvm_pkint(&n, 1, 1);
		    	pvm_send(wammtid, TRACETAG);
		    	close(f1);
		    	pvm_recv(wammtid, RESTART);
			}
			break;

		default: 
			break;
	}
	    
    if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0) 
		pvm_perror("wammtids mkbuf");
    if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
		pvm_perror("wammtids setsbuf");
	wammOL_print("wamm_close: closing\n");
  	pvm_send(wammtid, OL_EXIT_MON);
    pvm_setsbuf(oldbuf);
    pvm_freebuf(sbuf);
	cont_close++;
	mon = 0;
	return 0;
}

/******************************************************/
/* wamm_exit - chiude il monitoraggio ed esce dal PVM */
/******************************************************/

int wamm_exit()
{
    int inum,retval,sbuf, oldbuf;
    int f1,n,i;
    int bufid, nb, tag, rid;
    char buf[TRACEBUFSIZ];
    if (mon == 0 || called_exit)  return pvm_exit(); /* se non traccio */
    called_exit=1;
	switch(mon) {

		case 1:
			maketimestamp(&tt);
			fprintf (logfile, "19 %d %d %d %d 0\n", lam++, tt.tv_sec, tt.tv_usec, mypos);
			if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
				pvm_perror("oltids mkbuf");
			if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
				pvm_perror("oltids setsbuf");
			pvm_send(wammtid, REQ_SBLOCKING);
			pvm_recv(wammtid, SBLOCKING);
			fclose(logfile);
			if ((f1 = open(logname,O_RDONLY,0)) == -1) {
		   		printf("wamm_exit: cannot reopen tracefile");
			}
			else {
		   		unlink(logname);
		   		while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					pvm_initsend(PvmDataDefault);
					pvm_pkint(&n, 1, 1);
					pvm_pkbyte(buf, n, 1);
					pvm_send(wammtid, TRACETAG);
		    	} 
		    	n= -1;
		    	pvm_initsend(PvmDataDefault);
		    	pvm_pkint(&n, 1, 1);
		    	pvm_send(wammtid, TRACETAG);
		    	close(f1);
			}
			break;

		case 2:
			send_stats();
			break;

		case 3:
			send_stats();
			maketimestamp(&tt);
			fprintf (logfile, "19 %d %d %d %d 0\n", lam++, tt.tv_sec, tt.tv_usec, mypos);
			if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0)
				pvm_perror("oltids mkbuf");
			if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
				pvm_perror("oltids setsbuf");
			pvm_send(wammtid, REQ_SBLOCKING);
			pvm_setsbuf(oldbuf);
    		pvm_freebuf(sbuf);
			fclose(logfile);
			pvm_recv(wammtid, SBLOCKING);
			fclose(logfile);
			if ((f1 = open(logname,O_RDONLY,0)) == -1) {
		   		printf("wamm_exit: cannot reopen tracefile");
			}
			else {
		   		unlink(logname);
		   		while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					pvm_initsend(PvmDataDefault);
					pvm_pkint(&n, 1, 1);
					pvm_pkbyte(buf, n, 1);
					pvm_send(wammtid, TRACETAG);
		    	} 
		    	n= -1;
		    	pvm_initsend(PvmDataDefault);
		    	pvm_pkint(&n, 1, 1);
		    	pvm_send(wammtid, TRACETAG);
		    	close(f1);
			}
			break;

		default: 
			break;
	}
	    
	printf("wamm_exit: calling pvm_exit\n");
    if ((sbuf = pvm_mkbuf(PvmDataDefault)) < 0) 
		pvm_perror("wammtids mkbuf");
    if ((oldbuf = pvm_setsbuf(sbuf)) < 0)
		pvm_perror("wammtids setsbuf");
  	pvm_send(wammtid, OL_CHECKOUT);
    pvm_setsbuf(oldbuf);
    pvm_freebuf(sbuf);
    return pvm_exit();

/****************************************************/
/* wamm_recv - effettua la pvm_recv, monitoraggio e */
/*             gestisce messaggi di steering        */
/****************************************************/

int wamm_recv(tid, msgtag)
	int tid;
	int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who, i;
	char str[10];
	if (mon == 0) return(pvm_recv(tid, msgtag));
	maketimestamp(&tt);
	info = bufid = pvm_recv(tid, msgtag);
	if (bufid>0) {
		(void)pvm_bufinfo(bufid, &nb, &tag, &rid); 
		switch (mon) {
			case 1: 
				while(tag == TRACEBARRTAG) {	/* deve solo tracciare */
					barriertotal++;
					info = bufid = pvm_recv(tid,msgtag);
					if (info <= 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
					}
				who=tid_index(rid);
				if (who == -1) {
					sprintf(str, "%d", rid);
					pvm_lookup(str, 0, &who);
					if (who>= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<= who; i++)
							tids[i] = 0;
						lenght = who + 1;
					}
					tids[who] = rid;
				}					
				fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
					lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
				maketimestamp(&tt1);		/* decidere */
				fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
					lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
				break;							
		
		
			case 2: 
				while(rid == wammtid && tag == INTERACTTAG) {	/* deve solo campionare */				
					check_interaction();
					info = bufid = pvm_recv(tid,msgtag);
					if (info <= 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
					}
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = rid;
					}
					maketimestamp(&tt1);
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
					break;
					
			case 3: 
				while((tag == TRACEBARRTAG) ||
				     (rid == wammtid && tag == INTERACTTAG)) {
					if (tag == TRACEBARRTAG) {
						barriertotal++;
					}	
					else 
					if (rid == wammtid && tag == INTERACTTAG) {	
						check_interaction();
					}
					info = bufid = pvm_recv(tid,msgtag);
						if (info <= 0) return info;
							(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
						}
						who=tid_index(rid);
						if (who == -1) {
							sprintf(str, "%d", rid);
							pvm_lookup(str, 0, &who);
							if (who>= lenght) {
								punt = tids;
								tids = realloc(punt, (who+1)*SIZE_OF_INT);
								punt = receive_vector;
								receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
								punt = receive_byte_vector;
								receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
								punt = send_vector;
								send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
								punt = send_byte_vector;
								send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
								punt = int_send_vector;
								int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
								for (i = lenght; i<= who; i++) {
									tids[i] = receive_vector[i] = 0;
									receive_byte_vector[i] = 0;
									send_vector[i] = send_byte_vector[i] = 0;
									int_send_vector[i] = 0;
								}
								lenght = who + 1;
							}
							tids[who] = rid;
						}
						receive_vector[who]+= 1;
						receive_byte_vector[who]+= nb;
						fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
							lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
						maketimestamp(&tt1);		/* decidere */
						fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
							lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
						break;
					
			default: break;
		}
	if (mon > 1 ) {
		diffTime(&tt1, &tt, &tt_idle);
		rec_receives(nb, tt_idle);
		checkSampling(tt1);
	}
	} /*chiude bufid > 0 */
	return info;
}

/******************************************************/
/* wamm_nrecv - effettua la pvm_nrecv, monitoraggio e */
/*             gestisce messaggi di steering          */
/******************************************************/

int wamm_nrecv(tid, msgtag)
	int tid;
	int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who, i;
	char str[10];				
	if (mon == 0) return(pvm_nrecv(tid, msgtag));
	maketimestamp(&tt);
	info = bufid = pvm_nrecv(tid, msgtag);
	if (bufid>0) {
		(void)pvm_bufinfo(bufid, &nb, &tag, &rid); 
		switch (mon) {
			case 1:	
				while(tag == TRACEBARRTAG) {	/* deve solo tracciare */
					barriertotal++;
					info = bufid = pvm_nrecv(tid,msgtag);
					if (info < 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) { 
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++)
								tids[i] = 0;
							lenght = who + 1;
						}
						tids[who] = rid;
					}					
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
				}
				break;							
		
		
			case 2: 
				while(rid == wammtid && tag == INTERACTTAG) {	/* deve solo campionare */				
					check_interaction();
					info = bufid = pvm_nrecv(tid,msgtag);
						if (info < 0) return info;
						(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = rid;
					}
					maketimestamp(&tt1);
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
				}
				break;
					
			case 3: 
				while((tag == TRACEBARRTAG) ||
				     (rid == wammtid && tag == INTERACTTAG)) {
					if (tag == TRACEBARRTAG) {
						barriertotal++;
					}	
					else 
					if (rid == wammtid && tag == INTERACTTAG) {	
						check_interaction();
					}
					info = bufid = pvm_nrecv(tid,msgtag);
					if (info < 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = rid;
					}
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
				}
				break;
					
			default: break;
		}
	if ((mon > 1) && (bufid > 0) ) {
		diffTime(&tt1, &tt, &tt_idle);
		rec_receives(nb, tt_idle);
		checkSampling(tt1);
	}
	if ((mon > 1) && (bufid == 0) ) {
		diffTime(&tt1, &tt, &tt_idle);
		rec_receives(0, tt_idle);
		checkSampling(tt1);
	}
	} /*chiude bufid > 0 */
	return info;
}

/******************************************************/
/* wamm_trecv - effettua la pvm_trecv, monitoraggio e */
/*             gestisce messaggi di steering          */
/******************************************************/

int wamm_trecv(tid, msgtag, tmout)
	int tid;
	int msgtag;
	struct timeval *tmout;
{
	int info;
	int nb, rid, tag, bufid, who, i;
	char str[10];
	if (mon == 0) return(pvm_trecv(tid, msgtag, tmout));
	maketimestamp(&tt);
	info = bufid = pvm_trecv(tid, msgtag, tmout);
	if (bufid>0) {
		(void)pvm_bufinfo(bufid, &nb, &tag, &rid); 
		switch (mon) {
			case 1:
				while(tag == TRACEBARRTAG) {	/* deve solo tracciare */
					barriertotal++;
					info = bufid = pvm_trecv(tid,msgtag, tmout);
					if (info < 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++)
								tids[i] = 0;
							lenght = who + 1;
						}
						tids[who] = rid;
					}					
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
				}
				break;							
		
		
			case 2: 
				while(rid == wammtid && tag == INTERACTTAG) {	/* deve solo campionare */				
					check_interaction();
					info = bufid = pvm_trecv(tid,msgtag, tmout);
					if (info < 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = rid;
					}
					maketimestamp(&tt1);
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
				}
				break;
					
			case 3: 
				while((tag == TRACEBARRTAG) ||
				     (rid == wammtid && tag == INTERACTTAG)) {
					if (tag == TRACEBARRTAG) {
						barriertotal++;
					}	
					else 
					if (rid == wammtid && tag == INTERACTTAG) {	
						check_interaction();
					}
					info = bufid = pvm_trecv(tid,msgtag, tmout);
					if (info < 0) return info;
					(void)pvm_bufinfo(bufid, &nb, &tag, &rid);
				}
				if (bufid > 0) {
					who=tid_index(rid);
					if (who == -1) {
						sprintf(str, "%d", rid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = rid;
					}
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, tag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, tag, nb);
				}
				break;
					
			default: break;
		}
	if ((mon > 1) && (bufid > 0) ) {
		diffTime(&tt1, &tt, &tt_idle);
		rec_receives(nb, tt_idle);
		checkSampling(tt1);
	}
	if ((mon > 1) && (bufid == 0) ) {
		diffTime(&tt1, &tt, &tt_idle);
		rec_receives(0, tt_idle);
		checkSampling(tt1);
	}
	} /*chiude bufid > 0 */
	return info;
}

/******************************************************/
/* wamm_precv - effettua la pvm_precv, monitoraggio e */
/*             gestisce messaggi di steering          */
/******************************************************/

int wamm_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt)
    int tid, msgtag,nitems,dtype,*rtid,*rtag,*rcnt;
    void* buf;
{
    int info, nb, who, i;
    int len, rbf, bufid;
    char str[10];
    if (mon == 0) return (pvm_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt));
	maketimestamp(&tt);
	
    /* determina la lunghezza del messaggio */
    
    switch (dtype) {
    case PVM_BYTE:
	len = nitems * sizeof(char);
	break;
	
    case PVM_SHORT:
    case PVM_USHORT:
	len = nitems * sizeof(short);
	break;
	
    case PVM_INT:
    case PVM_UINT:
	len = nitems * sizeof(int);
	break;
	
    case PVM_LONG:
    case PVM_ULONG:
	len = nitems * sizeof(long);
	break;
	
    case PVM_FLOAT:
	len = nitems * sizeof(float);
	break;
	
    case PVM_CPLX:
	len = nitems * sizeof(float) * 2;
	break;
	
    case PVM_DOUBLE:
	len = nitems * sizeof(double);
	break;
	
    case PVM_DCPLX:
	len = nitems * sizeof(double) * 2;
	break;
	
    default:
	wammOL_print("Error--pvm_precv datatype not recognized\n");
        return (PvmBadParam);
	break;
    }

    rbf = pvm_setrbuf(0);
    info = bufid = pvm_recv(tid, msgtag);
    if (bufid > 0) {
		(void) pvm_bufinfo(bufid, rcnt, rtag, rtid);
		switch (mon) {
			case 1:	
				while(*rtag == TRACEBARRTAG) {	/* deve solo tracciare */
					barriertotal++;
					info = bufid = pvm_recv(tid,msgtag);
					(void)pvm_bufinfo(bufid, rcnt, rtag, rtid);
					if (info < 0) {
						pvm_setrbuf(rbf);
						return info;
					}
				}
				if (*rcnt < len)
					len = *rcnt;
	    			pvm_upkbyte((char *) buf, len, 1);
	    			pvm_freebuf(info);
		    		pvm_setrbuf(rbf);
					who=tid_index(*rtid);
	    			nb = (getdatasize(dtype))*(*rcnt);
					if (who == -1) {
						sprintf(str, "%d", *rtid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++)
								tids[i] = 0;
							lenght = who + 1;
						}
						tids[who] = *rtid;
					}					
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, *rtag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, *rtag, nb);
				break;
					
			case 2: 
				while(*rtid == wammtid && *rtag == INTERACTTAG) {	/* deve solo campionare */				
					check_interaction();
					info = bufid = pvm_recv(tid,msgtag);
					(void)pvm_bufinfo(bufid, rcnt, rtag, rtid);
					if (info < 0) {
						pvm_setrbuf(rbf);
						return info;
					}
				}
				if (*rcnt < len)
					len = *rcnt;
	    			pvm_upkbyte((char *) buf, len, 1);
	    			pvm_freebuf(info);
		    		pvm_setrbuf(rbf);
					who=tid_index(*rtid);
					nb = (getdatasize(dtype))*(*rcnt);
					if (who == -1) {
						sprintf(str, "%d", *rtid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = *rtid;
					}
					maketimestamp(&tt1);
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
				break;
				
			case 3: 
				while((*rtag == TRACEBARRTAG) ||
				     (*rtid == wammtid && *rtag == INTERACTTAG)) {
					if (*rtag == TRACEBARRTAG) {
						barriertotal++;
					}	
					else 
					if (*rtid == wammtid && *rtag == INTERACTTAG) {	
						check_interaction();
					}
					info = bufid = pvm_recv(tid,msgtag);
					(void)pvm_bufinfo(bufid, rcnt, rtag, rtid);
					if (info < 0) {
						pvm_setrbuf(rbf);
						return info;
					}
				}
				if (*rcnt < len)
					len = *rcnt;
	    			pvm_upkbyte((char *) buf, len, 1);
	    			pvm_freebuf(info);
		    		pvm_setrbuf(rbf);
					who=tid_index(*rtid);
					nb = (getdatasize(dtype))*(*rcnt);
					if (who == -1) {
						sprintf(str, "%d", *rtid);
						pvm_lookup(str, 0, &who);
						if (who>= lenght) {
							punt = tids;
							tids = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_vector;
							receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = receive_byte_vector;
							receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_vector;
							send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = send_byte_vector;
							send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							punt = int_send_vector;
							int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
							for (i = lenght; i<= who; i++) {
								tids[i] = receive_vector[i] = 0;
								receive_byte_vector[i] = 0;
								send_vector[i] = send_byte_vector[i] = 0;
								int_send_vector[i] = 0;
							}
							lenght = who + 1;
						}
						tids[who] = *rtid;
					}
					receive_vector[who]+= 1;
					receive_byte_vector[who]+= nb;
					fprintf (logfile, "7 %d %d %d %d %d %d %d\n",
						lam++, tt.tv_sec, tt.tv_usec, mypos, who, *rtag, nb);
					maketimestamp(&tt1);		/* decidere */
					fprintf (logfile, "8 %d %d %d %d %d %d %d\n",
						lam++, tt1.tv_sec, tt1.tv_usec, mypos, who, *rtag, nb);
				break;
					
			default: break;
		}
		if (mon > 1) {
		diffTime(&tt1,&tt,&tt_idle);
	    rec_receives(nb,tt_idle);
	    checkSampling(tt1);
	    }
    } 					/* chiude bufid > 0 */
	return info;
}

/*********************************************************/		
/* wamm_mcast - effettua la pvm_mcast ed il monitoraggio */
/*********************************************************/

int wamm_mcast(ttids, ntid, msgtag)
	int *ttids, ntid, msgtag;
{ 
	int *tmp;
	char str[10];
	int info,i,bufid, nb, a, b, who;
	int groupsize = 0, found_one = 0, max = 0;
	
	if (!mon) {
		return(pvm_mcast(ttids, ntid, msgtag));
	}
	tmp = (int *) calloc(ntid, SIZE_OF_INT);
	maketimestamp(&tt);
	info = pvm_mcast(ttids, ntid, msgtag);
	if (info == 0) {
		bufid = pvm_getsbuf();
		(void)pvm_bufinfo(bufid, &nb, &a, &b);
		switch (mon) {
			
			case 1:
				for (i = 0; i < ntid; i++) {
					who = tid_index(ttids[i]);
					if (who == -1) {
						who = db_index(ttids[i]);
						tmp[i] = who;
						if (who >= max) {
							max = who;
						}
					}
					else {
						tmp[i] = who;
					}
					fprintf(logfile, "4 %d %d %d %d %d %d %d\n", lam++,
					tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				}
				if (max >= lenght) {
					punt = tids;
					tids = realloc(punt, (max+1)*SIZE_OF_INT);
					for (i = lenght; i <= max; i++) {
						tids[i] = 0;
					}
					lenght = max+1;
				}
				for (i = 0; i < ntid; i++) {
					tids[tmp[i]] = ttids[i];
				}
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				free(tmp);
				break;
						  
			case 2:
				for (i = 0; i < ntid; i++) {
					who = tid_index(ttids[i]);
					if (who == -1) {
						who = db_index(ttids[i]);
						tmp[i] = who;
						if (who >= max) {
							max = who;
						}
					}
					else {
						tmp[i] = who;
					}
				} 
				if (max >= lenght) {
					punt = tids;
					tids = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = receive_vector;
					receive_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = receive_byte_vector;
					receive_byte_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = send_vector;
					send_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = send_byte_vector;
					send_byte_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = int_send_vector;
					int_send_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					for (i = lenght; i<= max; i++) {
						tids[i] = receive_vector[i] = 0;
						receive_byte_vector[i] = 0;
						send_vector[i] = send_byte_vector[i] = 0;
						int_send_vector[i] = 0;
					}
					lenght = max + 1;
				}
				maketimestamp(&tt1);	
				for (i = 0; i < ntid; i++) {
					tids[tmp[i]] = ttids[i];
					send_vector[tmp[i]]+= 1;
					int_send_vector[tmp[i]]+= 1;
					send_byte_vector[tmp[i]]+=nb;
				}
				free(tmp);
				break;

			case 3:
				for (i = 0; i < ntid; i++) {
					who = tid_index(ttids[i]);
					if (who == -1) {
						who = db_index(ttids[i]);
						tmp[i] = who;
						if (who >= max) {
							max = who;
						}
					}
					else {
						tmp[i] = who;
					}
					fprintf(logfile, "4 %d %d %d %d %d %d %d\n", lam++,
					tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				}
				if (max >= lenght) {
					punt = tids;
					tids = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = receive_vector;
					receive_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = receive_byte_vector;
					receive_byte_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = send_vector;
					send_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = send_byte_vector;
					send_byte_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					punt = int_send_vector;
					int_send_vector = realloc(punt, (max+1)*SIZE_OF_INT);
					for (i = lenght; i<= max; i++) {
						tids[i] = receive_vector[i] = 0;
						receive_byte_vector[i] = 0;
						send_vector[i] = send_byte_vector[i] = 0;
						int_send_vector[i] = 0;
					}
					lenght = max + 1;
				}	
				for (i = 0; i < ntid; i++) {
					tids[tmp[i]] = ttids[i];
					send_vector[tmp[i]]+= 1;
					int_send_vector[tmp[i]]+= 1;
					send_byte_vector[tmp[i]]+=nb;
				}
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				free(tmp);
				break;

			default:
				break;
		}
		if (mon>1) {
			diffTime(&tt1, &tt, &tt_send);
			rec_sends(nb*ntid, tt_send);
			checkSampling(tt1);
		}
	}
	return (info);
}



/*******************************************************/		
/* wamm_send - effettua la pvm_send ed il monitoraggio */
/*******************************************************/	
					
int wamm_send(tid, msgtag)
	int tid;
	int msgtag;
{
	int info, who, bufid, nb, a, b, i;
	char str[10];
	
	if (!mon) {
		return(pvm_send(tid, msgtag));
	}
	maketimestamp(&tt);
	info = pvm_send(tid, msgtag);
	if (info == 0) {
		bufid = pvm_getsbuf();
		(void)pvm_bufinfo(bufid, &nb, &a, &b);
		switch (mon) {
			case 1:
				who = tid_index(tid);      /* solo tracing */
				if (who == -1) {
					who = db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<=who; i++) {
								tids[i] = 0;
						}
						lenght = who+1;
					}
					tids[who] = tid;
				}
				fprintf(logfile, "4 %d %d %d %d %d %d %d\n",  lam++,
				tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				break;
					
			case 2:
				who = tid_index(tid);      /* solo sampling */
				if (who == -1) {
					who=db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_vector;
						receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_byte_vector;
						receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_vector;
						send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_byte_vector;
						send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = int_send_vector;
						int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<= who; i++) {
							tids[i] = receive_vector[i] = 0;
						   	receive_byte_vector[i] = 0;
							send_vector[i] = send_byte_vector[i] = 0;
							int_send_vector[i] = 0;
						}
						lenght = who + 1;	
					}
					tids[who] = tid;
				}
				maketimestamp(&tt1);
				send_vector[who]+= 1;
				int_send_vector[who]+= 1;
				send_byte_vector[who]+=nb;
				break;
					
			case 3:
				who = tid_index(tid);      /* tracing e sampling */
				if (who == -1) {
					who = db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_vector;
						receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_byte_vector;
						receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_vector;
						send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_byte_vector;
						send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = int_send_vector;
						int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<= who; i++) {
							tids[i] = receive_vector[i] = 0;
							receive_byte_vector[i] = 0;
							send_vector[i] = send_byte_vector[i] = 0;
							int_send_vector[i] = 0;
						}
						lenght = who + 1;	
					}
					tids[who] = tid;
				}
				send_vector[who]+= 1;
				int_send_vector[who]+= 1;
				send_byte_vector[who]+=nb;
				fprintf(logfile, "4 %d %d %d %d %d %d %d\n",  lam++,
				tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				break;
					
			default:
				break;
		}
		if (mon>1) {
			diffTime(&tt1, &tt, &tt_send);
			rec_sends(nb, tt_send);
			checkSampling(tt1);
		}
	}
	return (info);
}
		
/*********************************************************/		
/* wamm_psend - effettua la pvm_psend ed il monitoraggio */
/*********************************************************/

int wamm_psend(tid, msgtag, buf, nitems, dtype)
int tid, msgtag, nitems, dtype;
void *buf;
{
	int info, who, i;
	char str[10];
	int nb = getdatasize(dtype)*nitems;
	if (!mon) {
		return(pvm_psend(tid, msgtag, buf, nitems, dtype));
	}
	maketimestamp(&tt);
	info = pvm_psend(tid, msgtag, buf, nitems, dtype);
	if (info == 0) {
		switch (mon) {
			case 1:
				who = tid_index(tid);				/* solo tracing */
				if (who == -1) {
					who = db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<=who; i++) {
								tids[i] = 0;
						}
						lenght = who+1;
					}
					tids[who] = tid;
				}
				fprintf(logfile, "4 %d %d %d %d %d %d %d\n",  lam++,
				tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				break;

			case 2:
				who = tid_index(tid);				/* solo sampling */
				if (who == -1) {
					who = db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_vector;
						receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_byte_vector;
						receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_vector;
						send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_byte_vector;
						send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = int_send_vector;
						int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<= who; i++) {
							tids[i] = receive_vector[i] = 0;
						   	receive_byte_vector[i] = 0;
							send_vector[i] = send_byte_vector[i] = 0;
							int_send_vector[i] = 0;
						}
						lenght = who + 1;	
					}
					tids[who] = tid;
				}
				maketimestamp(&tt1);
				send_vector[who]+= 1;
				int_send_vector[who]+= 1;
				send_byte_vector[who]+=nb;
				break;

			case 3:
				who = tid_index(tid);			/* tracing e sampling */
				if (who == -1) {
					who = db_index(tid);
					if (who >= lenght) {
						punt = tids;
						tids = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_vector;
						receive_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = receive_byte_vector;
						receive_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_vector;
						send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = send_byte_vector;
						send_byte_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						punt = int_send_vector;
						int_send_vector = realloc(punt, (who+1)*SIZE_OF_INT);
						for (i = lenght; i<= who; i++) {
							tids[i] = receive_vector[i] = 0;
							receive_byte_vector[i] = 0;
							send_vector[i] = send_byte_vector[i] = 0;
							int_send_vector[i] = 0;
						}
						lenght = who + 1;	
					}
					tids[who] = tid;
				}
				send_vector[who]+= 1;
				int_send_vector[who]+= 1;
				send_byte_vector[who]+=nb;
				fprintf(logfile, "4 %d %d %d %d %d %d %d\n",  lam++,
				tt.tv_sec, tt.tv_usec, mypos, who, msgtag, nb);
				maketimestamp(&tt1);
				fprintf(logfile, "11 %d %d %d %d %d %d\n",  lam++,
				tt1.tv_sec, tt1.tv_usec, mypos, 0, 0);
				break;
			
			default:
				break;
		}
		if (mon>1) {
			diffTime(&tt1, &tt, &tt_send);
			rec_sends(nb, tt_send);
			checkSampling(tt1);
		}
	}
	return (info);
}

/***********************************************/
/* diffTime - effettua la differenza tra tempi */
/***********************************************/

static void
diffTime(t1, t2, t)
     struct timeval *t1, *t2, *t;
{    
  struct timeval tx;
  if ((t1->tv_sec*1000000 + t1->tv_usec) < (t2->tv_sec*1000000+ t2->tv_usec)) {
	  printf("Error: diffTimes: t1 < t2");
	  exit(1);
  }
  if (t1->tv_usec >= t2->tv_usec) {
	  tx.tv_usec = t1->tv_usec -t2->tv_usec;
	  tx.tv_sec = t1->tv_sec -t2->tv_sec;
  }
  else {
      tx.tv_usec = (t1->tv_usec+1000000) - t2->tv_usec;
      tx.tv_sec = (t1->tv_sec-1)- t2->tv_sec;
  }
  *t = tx;

}

/*****************************************/
/* addTime - effettua la somma tra tempi */
/*****************************************/

static void
addTime(t1, t2, t)
     struct timeval *t1, *t2, *t;
{    
struct timeval tx;
tx.tv_sec = 0;  
tx.tv_usec = 0;

tx.tv_usec = t1->tv_usec + t2->tv_usec;
if (tx.tv_usec > 999999) {
	tx.tv_usec -= 1000000;
	tx.tv_sec += 1;
}
tx.tv_sec += (t1->tv_sec + t2->tv_sec);

*t = tx;

}


/***************/
/* per Solaris */
/***************/

#ifdef IMA_SUN4SOL2 
static int getpagesize()
{
  int           fd1;
  char          file_name[100];
  long pid=0;

  int psize;
  prmap_t       *prmap;
  int temp;

  pid = (long) getpid();
  /* print the string "/proc/<process id>" into file_name  */
  (void)sprintf(file_name, "/proc/%05ld", (long) pid);
  
  /* open the /proc/<process id> file to examine values */
  if((fd1 = open (file_name, O_RDONLY, 0)) == -1) {
      return (-1);
  }

  if (ioctl(fd1, PIOCNMAP, &temp) == -1) {
      (void)close(fd1);
      pid = 0;
      pvanimOL_print("error with ioctl PIOCNMAP\n");
      return (-1);
  }
  
  prmap = (prmap_t *) malloc (temp*sizeof(prmap_t));

  if (ioctl(fd1, PIOCMAP, prmap) == -1) {
      (void)close(fd1);
      pid = 0;
      pvanimOL_print("error with ioctl PIOCMAP\n");
      return (-1);
  }
 
  psize = (int) prmap[0].pr_pagesize;
  
  (void)close(fd1);


  return (psize);

}
#endif

