/*****************************************************************************\
*                                                                             *
*  modTasks.c - Gestione tasks			                                      *
*                                                                             *
\*****************************************************************************/

/*
 * 
 *
 *               WAMM: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, M. Cosso, G. Faieta, M. Formica, 
 *                      D. Laforenza, M. Nicosia 
 *                   (C) 1997 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * We want thanks Brad Topol of the Georgia Institute of Technology, Atlanta.
 */


#include "modTasks.h"

/*********************/
/* strutture private */
/*********************/

struct TaskList {
	struct Task * first;	/* primo task della lista */
	struct Task * last;		/* ultimo task della lista */
	int ntask;				/* numero di tasks */
};

struct Task {
	struct Task * prev;		/* task precedente nella lista */
	struct Task * next;		/* task successivo nella lista */
	int tid;				/* tid del task */
	int zombie;				/* 1=task terminato */
	int system;				/* 1=task di sistema (PVMHoster, PVMTasker, PVMMaker) */
	char * name;			/* nome del task */
	char * host;			/* indirizzo del'host su cui gira il task */
	char * out;				/* output prodotto dal task */
	Widget Woutshell;		/* finestra output (0 = non aperta) */
	Widget Wouttext;		/* text output (0 = non aperta) */
};

/*******************************/
/* variabili globali (private) */
/*******************************/

static struct TaskList * TL;	/* la lista dei tasks */

static Widget Wtoplevel;		/* toplevel shell */
static Widget Wtaskshell;		/* finestra con la lista (0 = finestra non aperta) */
static Widget Wtasklist;		/* widget lista task */

int showsystem;					/* 1 = mostra anche i task di sistema */

/***********/
/* globali */
/***********/

extern char * sys_errlist[];	/* messaggi di errore */
extern int done_count;
extern int lung_array;
extern int mon_end;
extern int montype; /* YYY */

/********************/
/* Funzioni private */
/********************/

static struct Task *	FindTaskByIndex (int);
static void 			WriteTaskEntry (struct Task *, char *);

/**********************/
/* Funzioni pubbliche */
/**********************/

struct Task * 	FindTaskByTid (int);

/**********************/
/* Funzioni importate */
/**********************/

extern int mevalue(int);


/*********************/
/* Callbacks private */
/*********************/

static void CloseCB (Widget, Widget *, int);
static void SignalCB (Widget, int, int);
static void OutputCB (Widget, int, int);
static void CleanupCB (Widget, int, int);
static void ToggleSystemCB (Widget, int, int);
static void SaveCB (Widget, struct Task *, int);

/***************************************/
/* TasksInit - inizializzazione modulo */
/***************************************/

void TasksInit (Widget toplevel)
{
	Wtoplevel = toplevel;
	
	/****************************/
	/* Inizializza la task list */
	/****************************/
	
	TL = calloc (1, sizeof (struct TaskList));
	return;
}

/*********************************/
/* TasksEnd - distruzione modulo */
/*********************************/

void TasksEnd (void)
{
	/* XXX per ora nulla */
}

/*******************************/
/* TasksAdd - aggiunge un task */
/*******************************/

struct Task * TasksAdd (int tid, char * name, char * host)
{
	struct Task * task;
	char buffer [500];
	
	/************************/
	/* alloca il nuovo nodo */
	/************************/
	
	task = calloc (1, sizeof (struct Task));
	task -> tid = tid;
	if (name) task-> name = strdup (name);
	else task -> name = strdup ("");
	if (host) task -> host = strdup (host);
	else task -> host = strdup("");
	if ((!strcmp(task->name, "PVMHoster")) || (!strcmp(task->name, "PVMTasker")) || (!strcmp(task->name, "PVMMaker"))) task->system = 1;
	
	/********************/
	/* inserimento nodo */
	/********************/
	
	/**** lista vuota ****/
	
	if (!TL->first) {
		TL->first = TL->last = task;
		TL->ntask = 1;
	}

	/**** lista non vuota, inserimento in FONDO ****/
	
	else {	
		task -> next = NULL;
		task -> prev = TL->last;
		
		TL -> last -> next = task;
		TL -> last = task;

		TL -> ntask++;
	}	
	
	/*********************************/
	/* Inserisce il nodo nella lista */
	/*********************************/
		
	if (Wtaskshell && (showsystem || (!task->system))) {
		WriteTaskEntry (task, buffer);
		XmListAddItem (Wtasklist, XmStringCreate(buffer, DEFSET), 0);
		XmListSetBottomPos (Wtasklist, 0);
	}
	
	return (task);
}

/******************************/
/* TasksDel - rimuove un task */
/******************************/

void TasksDel (int tid)
{
	int n;
	struct Task * task, * task2;
	char buffer[1000];
	XmString newitem;

	/*****************/
	/* Cerca il task */
	/*****************/
	
	task = FindTaskByTid (tid);
	if (!task) return;				/* task non esistente! */

	/***********************/
	/* Segnala "terminato" */
	/***********************/

	task -> zombie = 1;
	
	/*************************************/
	/* Aggiornamento lista sullo schermo */
	/*************************************/
	
	if (!Wtaskshell || ((!showsystem) && task->system)) return;
	
	/**** calcola la posizione del task nella lista ****/
	
	n = 0;
	task2 = TL->first;
	while (task != task2) {
		if (showsystem || (!task2->system)) n++; /* conta solo i task mostrati */
		task2 = task2->next;
	}
	
	WriteTaskEntry (task, buffer);
	newitem = XmStringCreate(buffer, DEFSET);
	XmListReplaceItemsPos (Wtasklist, &newitem, 1, n+1);
}

/*************************************************/
/* TasksDelAll - rimuove tutti i task di un nodo */
/*************************************************/

void TasksDelAll (char * addr)
{
	int n;
	struct Task * task;
	char buffer[1000];
	XmString newitem;

	/******************************/
	/* Segnala tutti i task morti */
	/******************************/
	
	task = TL->first;
	n = 0;
	
	while (task) {
	
		if (!strcmp(addr, task->host)) task->zombie = 1;
		
		/***********************************/
		/* Aggiorna la lista sullo schermo */
		/***********************************/
			
		if (Wtaskshell && (showsystem || (!task->system))) {		
			n++;
			if (!strcmp(addr, task->host)) {		
				WriteTaskEntry (task, buffer);
				newitem = XmStringCreate (buffer, DEFSET);
				XmListReplaceItemsPos (Wtasklist, &newitem, 1, n);
			}
		}
	
		task = task->next;
	}
}

/***************/
/* TasksOutput */
/***************/

void TasksOutput (int tid, char * out)
{
	struct Task * task;
	
	task = FindTaskByTid (tid);
	if (!task) return;

	/**** inserisce la nuova stringa nel buffer ****/
	
	if (!task->out) task->out = strdup (out);
	else {
		task->out = realloc (task->out, strlen(task->out)+strlen(out)+1);
		strcat (task->out, out);
	}
	
	/**** se la finestra di output e` aperta, aggiunge out ****/
	
	if (task->Woutshell) {
		XmTextInsert (task->Wouttext, XmTextGetLastPosition(task->Wouttext), out);
		XmTextShowPosition (task->Wouttext, XmTextGetLastPosition(task->Wouttext));
	}
}

/*******************************/
/* TasksWindow - finestra task */
/*******************************/

void TasksWindow (void)
{
	Arg args[20];
	int n;
	Widget Wform, Wmenubar, Wmenupane;

	struct Task * task;
	char buffer[1000];

	/********************/
	/* finestra aperta? */
	/********************/
	
	if (Wtaskshell) {
		XMapRaised (XtDisplay(Wtaskshell), XtWindow(Wtaskshell));
		return;
	}

	/**********************/
	/* creazione finestra */
	/**********************/
	
	/**** shell ****/
	
	n = 0;
	XtSetArg (args[n], XmNtitle, "Tasks"); n++;
	XtSetArg (args[n], XmNdeleteResponse, XmDO_NOTHING); n++;
	XtSetArg (args[n], XmNwidth, 600); n++;
	XtSetArg (args[n], XmNheight, 250); n++;
	Wtaskshell = XtAppCreateShell ("Tasks", "Wamm", applicationShellWidgetClass, XtDisplay(Wtoplevel), args, n);
	MotifProtectShell (Wtaskshell, (XtCallbackProc)CloseCB, &Wtaskshell);
	XtRealizeWidget (Wtaskshell);

    /**** Editres ****/
    
    XtAddEventHandler (Wtaskshell, (EventMask)0, True, _XEditResCheckMessages, NULL);

	/**** form ****/

	n = 0;
	Wform = XmCreateForm (Wtaskshell, "taskform", args, n);
	XtManageChild (Wform);
	
	/**** menubar ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	Wmenubar = XmCreateMenuBar (Wform, "taskmenubar", args, n);
	XtManageChild (Wmenubar);

	/**** menu ****/
	
	Wmenupane = MotifCreateMenu (Wmenubar, "File", 'F', 0, 0);
	MotifAddMenuToggle (Wmenupane, "Show system tasks", 'S', (XtCallbackProc)ToggleSystemCB, NULL, showsystem);
	MotifAddMenuItem (Wmenupane, "Remove old tasks", 'R', (XtCallbackProc)CleanupCB, NULL); 
	MotifAddMenuItem (Wmenupane, "Close", 'C', (XtCallbackProc)CloseCB, &Wtaskshell);
	
	Wmenupane = MotifCreateMenu (Wmenubar, "Task", 'T', 0, 0);
	MotifAddMenuItem (Wmenupane, "Output", 'O', (XtCallbackProc)OutputCB, &Wtaskshell);
	MotifAddMenuItem (Wmenupane, "Kill", 'K', (XtCallbackProc)SignalCB, (XtPointer)SIGTERM);
	Wmenupane = MotifCreateMenu (Wmenupane, "Signal", 'S', 0, 0);
	MotifAddMenuItem (Wmenupane, "01 SIGHUP", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGHUP);
	MotifAddMenuItem (Wmenupane, "02 SIGINT", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGINT);
	MotifAddMenuItem (Wmenupane, "03 SIGQUIT", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGQUIT);
	MotifAddMenuItem (Wmenupane, "04 SIGILL", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGILL);
	MotifAddMenuItem (Wmenupane, "05 SIGTRAP", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGTRAP);
	MotifAddMenuItem (Wmenupane, "06 SIGABRT", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGABRT);
	MotifAddMenuItem (Wmenupane, "07 SIGEMT", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGEMT);
	MotifAddMenuItem (Wmenupane, "08 SIGFPE", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGFPE);
	MotifAddMenuItem (Wmenupane, "09 SIGKILL", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGKILL);	
	MotifAddMenuItem (Wmenupane, "15 SIGTERM", 0, (XtCallbackProc)SignalCB, (XtPointer)SIGTERM);	

	/**** lista ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wmenubar); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNselectionPolicy, XmEXTENDED_SELECT); n++;
	XtSetArg (args[n], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE); n++;
	XtSetArg (args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n++;
	Wtasklist = XmCreateScrolledList (Wform, "tasklist", args, n);
	XtManageChild (Wtasklist);
	XtAddCallback (Wtasklist, XmNdefaultActionCallback, (XtCallbackProc)OutputCB, NULL);
	
	/**** inserimento tasks ****/

	task = TL->first;
	while (task) {

		if (showsystem || (!task->system)) {
			WriteTaskEntry (task, buffer);
			XmListAddItem (Wtasklist, XmStringCreate(buffer, DEFSET), 0);
		}
		
		task = task->next;
	}
}

/**************************************************/
/* OutputWindow (task) - mostra l'output del task */
/**************************************************/

void OutputWindow (struct Task * task)
{
	int n;
	Arg args[20];
	Widget Wshell, Wform, Wmenubar, Wmenupane, Wtext, Wbutton;
	char buffer [1000];
	
	if (task->Woutshell) {
		XMapRaised (XtDisplay(task->Woutshell), XtWindow(task->Woutshell));
		return;
	}

	/**** shell ****/

	sprintf (buffer, "Output of task %x (%s) on %s", task->tid, (task->name ? task->name : (char *)"no name"), task->host);

	n = 0;
	XtSetArg (args[n], XmNtitle, buffer); n++;
	XtSetArg (args[n], XmNdeleteResponse, XmDO_NOTHING); n++;
	XtSetArg (args[n], XmNwidth, 700); n++;
	XtSetArg (args[n], XmNheight, 350); n++;
	Wshell = XtAppCreateShell ("output", "Wamm", applicationShellWidgetClass,	XtDisplay(Wtoplevel), args, n);
	task->Woutshell = Wshell;
	MotifProtectShell (Wshell, (XtCallbackProc)CloseCB, &task->Woutshell);
	XtRealizeWidget (Wshell);

    /**** Editres ****/
    
    XtAddEventHandler (Wshell, (EventMask)0, True, _XEditResCheckMessages, NULL);

	/**** form ****/

	n = 0;
	Wform = XmCreateForm (Wshell, "outform", args, n);
	XtManageChild (Wform);
	
	/**** menubar ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	Wmenubar = XmCreateMenuBar (Wform, "outputmenubar", args, n);
	XtManageChild (Wmenubar);

	/**** menu ****/
	
	Wmenupane = MotifCreateMenu (Wmenubar, "File", 'F', 0, 0);
	MotifAddMenuItem (Wmenupane, "Save As...", 'S', (XtCallbackProc)SaveCB, task); 
	MotifAddMenuItem (Wmenupane, "Close", 'C', (XtCallbackProc)CloseCB, &task->Woutshell);
	
	/**** scrolled text ****/
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wmenubar); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomOffset, 45); n++;
	XtSetArg (args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
	XtSetArg (args[n], XmNeditable, False); n++;
	XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
	XtSetArg (args[n], XmNvalue, task->out); n++;
	Wtext = XmCreateScrolledText (Wform, "outtext", args, n);
	XtManageChild (Wtext);
	task -> Wouttext = Wtext;
	XmTextShowPosition (task->Wouttext, XmTextGetLastPosition(task->Wouttext));
	
	/**** OK ****/
	
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("OK", DEFSET)); n++;
	XtSetArg (args[n], XmNwidth, 80); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomOffset, 5); n++;
	Wbutton = XmCreatePushButton (Wform, "outok", args, n);
	XtManageChild (Wbutton);
	XtAddCallback (Wbutton, XmNactivateCallback, (XtCallbackProc)CloseCB, &task->Woutshell);
}

/******************************************************************************/
/******************************************************************************/

/*******************************/
/* CloseCB - callback chiusura */
/*******************************/

void CloseCB (Widget w, Widget * wid, int dummy2)
{
	XtDestroyWidget (*wid);
	*wid = 0;
}
	
/****************************/
/* SignalCB - invia segnali */
/****************************/

void SignalCB (Widget w, int sig, int dummy2)
{
	int * tasklist;
	int num, n, cc, data; /* YYY */
	char temp_str[256]; /* YYY */
	Boolean mem;
	struct Task * task;
	int found = 0;
		
	/******************************/
	/* Preleva i task dalla lista */
	/******************************/
	
	num = 0;
	mem = XmListGetSelectedPos (Wtasklist, &tasklist, &num);
	
	/***********/
	/* Segnale */
	/***********/
	
	for (n=0; n<num; n++) {
		task = FindTaskByIndex (tasklist[n]-1);
		pvm_sendsig (task->tid, sig);
		if ((!mon_end) && (sig == SIGTERM && (mevalue(task->tid)!=-1))){
			found++;
		}
	}		
	if ((!mon_end) && (found)){
		MotifWarning(Wtoplevel, "Warning", "You have killed a monitored task!\nThe monitoring isn't reliable.");
		done_count = done_count + found;
		if (done_count == lung_array) { 
			mon_end = 1;
			montype = 0; 
			while((cc=pvm_lookup("tid", -1, &data))!= PvmNoEntry) { /* YYY */
 				sprintf(temp_str,"%d",data);
 				pvm_delete("tid",cc);
 				pvm_delete(temp_str,0);
 			}/* YYY */
		}
	}
	
	/***********************/
	/* Rilascia il vettore */
	/***********************/
	
	if (mem) free (tasklist);			
}

/******************************/
/* OutputCB - finestre output */
/******************************/

void OutputCB (Widget w, int dummy1, int dummy2)
{
	int * tasklist;
	int num, n;
	Boolean mem;
	struct Task * task;
		
	/******************************/
	/* Preleva i task dalla lista */
	/******************************/
	
	num = 0;
	mem = XmListGetSelectedPos (Wtasklist, &tasklist, &num);
	
	/************/
	/* Finestre */
	/************/
	
	for (n=0; n<num; n++) {
		task = FindTaskByIndex (tasklist[n]-1);
		OutputWindow (task);
	}		

	/***********************/
	/* Rilascia il vettore */
	/***********************/
	
	if (mem) free (tasklist);			
}

/************************************/
/* CleanupCB - elimina i task morti */
/************************************/

void CleanupCB (Widget w, int dummy1, int dummy2)
{
	struct Task * task, * tmp;
	int n = 1;
	
	task = TL->first;
	while (task) {
		
		tmp = task->next;
		
		if (task->zombie) {
		
			/***********************************/
			/* Elimina il nodo dalla Wtasklist */
			/***********************************/

			if (showsystem || (!task->system)) XmListDeletePos (Wtasklist, n);
		
			/***********************/
			/* svuota la struttura */
			/***********************/

			free (task->name);
			free (task->host);
			if (task->out) free (task->out);
			if (task->Woutshell) XtDestroyWidget (task->Woutshell);
		
			/*******************/
			/* Sgancia il nodo */
			/*******************/
	
			if (task->prev) task->prev->next = task->next;
			if (task->next) task->next->prev = task->prev;
	
			if (task == TL->first) TL->first =	task->next;
			if (task == TL->last) TL->last = task->prev;
		
			free (task);
			TL -> ntask--;
		}
	
		else {
		
			/**** Se il task era mostrato e non e` stato tolto, bisogna
			      incrementare la posizione ****/
	
			if (showsystem || (!task->system)) n++;
		}
		
		task = tmp;
	}
}

/*****************************************************/
/* SystemToggleCB - mostra/rimuove i task di sistema */
/*****************************************************/

void ToggleSystemCB (Widget w, int dummy1, int dummy2)
{
	struct Task * task;
	char buffer[1000];

	/**** cambia lo stato ****/

	showsystem ^= 1;
	
	/**** refresh lista ****/

	XmListDeleteAllItems (Wtasklist);

	task = TL->first;
	while (task) {

		if (showsystem || (!task->system)) {
			WriteTaskEntry (task, buffer);
			XmListAddItem (Wtasklist, XmStringCreate(buffer, DEFSET), 0);
		}
		
		task = task->next;
	}
}

/**********/
/* SaveCB */
/**********/

void SaveCB (Widget w, struct Task * task, int dummy)
{
	char * file;
	FILE * fh;
	char buffer[1000];
	int res;
	
	file = MotifAskFile (task->Woutshell, "Please insert file name", NULL);
	if (!file) return;
	
	fh = fopen (file, "w");
	if (!fh) {
		sprintf (buffer, "Can't save output on file %s:\n%s.", file, sys_errlist[errno]);
		MotifError (task->Woutshell, "Save failed", buffer);
		free (file);
		return;
	}
	
	res = fwrite (task->out, 1, strlen(task->out), fh);
	if (res != strlen(task->out)) {
		sprintf (buffer, "Can't write output file %s:\n%s", file, sys_errlist[errno]);
		MotifError (task->Woutshell, "Save failed", buffer);
		fclose (fh);
		free (file);
		return;
	}
	
	fclose (fh);
	free (file);
}

/******************************************************************************/
/******************************************************************************/

/*******************/
/* FindTaskByIndex */
/*******************/

/* NOTA: qui l'indice parte da ZERO; le liste Motif partono da UNO. */

struct Task * FindTaskByIndex (int i)
{
	struct Task * task;
	int n;
	
	task = TL->first;
	
	n = 0;
	while (n<i) {
		if (showsystem || (!task->system)) n++;	/* conta solo quelli mostrati */
		task = task->next;
	}
	
	/**** salta tutti i task di sistema finali, se non sono mostrati ****/
	
	if (!showsystem) while (task->system) task = task->next;

	return task;
}

/*****************/
/* FindTaskByTid */
/*****************/

struct Task * FindTaskByTid (int tid)
{
	struct Task * task;
	
	task = TL->first;
	while (task && (task->tid != tid)) task = task->next;
	return task;
}

/******************/
/* WriteTaskEntry */
/******************/

void WriteTaskEntry (struct Task * task, char * buffer)
{
	sprintf (buffer, "Tid: %-10x", task->tid);
	if (!task->zombie) sprintf (buffer+strlen(buffer), "%-20s", "(running)");
	else sprintf (buffer+strlen(buffer), "%-20s", "(finished)");
	if (task->name[0]) sprintf (buffer+strlen(buffer), " Name: %-20s", task->name);
	else sprintf (buffer+strlen(buffer), " Name: %-20s", "--------");
	if (task->host[0]) sprintf (buffer+strlen(buffer), " Host: %-20s", task->host);
	else sprintf (buffer+strlen(buffer), " Host: %-20s", "--------");
}
