/*****************************************************************************\
*                                                                             *
*  modPVM.c - gestione PVM							                          *
*                                                                             *
\*****************************************************************************/

/*
 * 
 *
 *               WAMM: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, M. Cosso, G. Faieta, M. Formica, 
 *                      D. Laforenza, M. Nicosia 
 *                   (C) 1997 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * We want thanks Brad Topol of the Georgia Institute of Technology, Atlanta.
 */


#include "modPVM.h"
#include "modRedraw.h"
#include "modMON.h"

/***************/
/* Definizioni */
/***************/

#define PVM_DELAY	50		/* ms di pausa tra un controllo e il successivo */

#define MAXRANGECHOICES 7
#define SBLOCKING 13
#define TRACETAG 0x22cb173b
#define INTERACTTAG 0x22cb173d

#define OL_CHECKIN 1
#define OL_STATISTIC 2
#define OL_CHECKOUT 3
#define OL_SCAN      0x22cb173e
#define OL_PRINT    5
#define OL_PROCINFO 6
#define OL_MODIFYSTATUSF 7
#define OL_MODIFYSTATSF  8
#define OL_MODIFYPROBEF  9
#define OL_MODIFYSTATUST 10
#define OL_MODIFYMINSECS 11
#define REQ_SBLOCKING 14
#define RESTART 16
#define OL_EXIT_MON 15
#define TRACEBUFSIZ 1400

#define STARTSLAVE 500
#define KILLSLAVE 600
#define KILLSLAVE_ACK 650
#define SLAVEKILLED 800
#define SAMPLEDELAY 700

#define ACK 1  /* indica che gli slave devono essere uccisi con un ack */
#define NORM 0 /* indica che gli slave devono essere uccisi normalmente */


/*********************/
/* strutture private */
/*********************/

struct TaskList_mon {
	struct Task_mon * first;	/* primo task della lista */
	struct Task_mon * last;		/* ultimo task della lista */
	int ntask_mon;				/* numero di tasks */
};

struct Task_mon {
	struct Task_mon * next;		/* task successivo nella lista */
	int tid;				/* tid del task */
};


/*********************/
/* variabili globali */
/*********************/

int commlegendrangelevel;
int maxprocs;
int numhosts;
int lung_array; /* specifica la dimensione di tids_array */
XmTextPosition wpr_position;
int done_count;
struct TaskList_mon * TL_mon;	/* la lista dei tasks */


/*******************************/
/* variabili globali (private) */
/*******************************/


static Widget Wtoplevel;
static Display *display;
static int procs_checked_in;
static struct NetworkObj * WAN;
static int MyTid;
static int HosterTid;
static int cont_mon = 0;	/* conta le fasi di monitoraggio */


/**************************/
/* variabili importate    */
/**************************/

extern int montype;
extern loadentry *loads;
extern hostentry *hosts;
extern rangeentry rangechoices[MAXRANGECHOICES];
extern int *tids_array;
extern int pvm_useruid;
extern char path_str[64]; 

/************** Widget ****************/

extern Widget  loadinfo, text_output;
extern Widget  totalutil;
extern Widget  messagesent;
extern Widget  matrixcomm; 
extern Widget  hostlist, hostutil;
extern Widget  memuse; 
extern Widget  scale;
extern Widget  tasklist;


/**********************/
/* funzioni pubbliche */
/**********************/

void	Timeout (void);
int	cerca_file (void);
void NFY_Sample_Delay(int);
int mevalue(int);

/********************/
/* funzioni private */
/********************/

static void INT2_AddEnd (void);
static void NFY_AddHost (void);
static void NFY_DelHost (void);
static void NFY_Kill_Slave(int);
static void TSKR_New (void);
static void TSKR_End (void);
static void TSKR_Cmd (void);
static void HSTR_Psw (void);
static void MKR_Msg (int);
static void MKR_MsgEnd (int);
static void MKR_Out (void);
static void TASK_Out (void);
static void TASK_Trace (void);

static void addhost(char *,int);
static int hostindex(char *);
static void OL_Checkin(int);
static void OL_Statistic(int);
static void OL_Print(int);
static void OL_Scan(int);
static void OL_Checkout(int);
static void OL_Exit_Mon(int);
static void OL_Procinfo();
static void REQ_Sblocking();
static void RefreshStatus (struct NetworkObj *);
static void free_db();
static void TasksAdd_mon (int, int);

/**********************/
/* funzioni importate */
/**********************/

extern char * pvmdsockfile (void);


/****************/
/* mevalue()    */
/****************/

int mevalue(int tid)
{
    int i;
    int m = -1;           /* Assume host process if not found */
    for (i=0; i<lung_array; i++) {
    	if(tids_array[i]==tid) {
        	m=i;
        	break;
         }
    }
    return m;
}

/******************/
/* post_process() */
/******************/

static void post_process(char *nome_file)
{
	int child;
	char *p1=NULL, *p2=NULL;
	char comm[256];

	child=fork();
	if (!child){
		p1 = getenv("PVM_ROOT");
		p2 = getenv("PVM_ARCH");
		sprintf(comm,"%s/bin/%s/PVANIMSORT",p1,p2);
		execl(comm, comm, nome_file, NULL);
	}
}	 

/***********/
/* PVMInit */
/***********/

void PVMInit (struct NetworkObj * wan, Widget toplevel)
{
	int n;
	char buffer[1000];
	
	struct NetworkObj * no;			/* oggetto corrente */
	int res;						/* risultato */
	char * errmsg;					/* messaggio di errore */
	char * verargv[2];				/* argomenti per tasker/hoster */

	char * hostfile;				/* nome hostfile */
	FILE * fh;						/* hostfile */
	char * opt;						/* opzioni */
	struct stat * st;				/* controllo /tmp/pvmd.xx.yy */
	char sockfile[1000];			/* nome sockfile */
			
	int hostnum;					/* numero host */
	int archnum;					/* numero architetture */
	struct pvmhostinfo * hosts;		/* informazioni sugli host */
	
	int * tids;						/* tids PVMTaskers */
	int tid;						/* tid PVMTasker */
	int ack;						/* 1=PVMtasker ok */
		
	/*******************************/
	/* Inizializzazione modulo PVM */
	/*******************************/
	
	/*  1. Viene estratto un host file dal file di configurazione
	*   2. Se necessario si fa partire il master daemon.
	*   3. Viene attivato il notify PvmHostAdd
	*	4. Viene lanciato l'hoster
	*   5. Viene determinata la configurazione iniziale (pvm_conf)
	*	6. Per ogni host in PVM e nel file di configurazione si prende il
	       dtid e si registra un notify PvmHostDelete
	    7. Si lanciano tutti i PVMTasker
	    8. Si registrano i loro tid, eliminando i doppioni
	    9. Si fanno partire manualmente i PVMTasker sugli host "mancati";
	       si controllano gli ack di tutti.
	*/

	/**** Finestra per i messaggi ****/
	
	WAN = wan;
	
	/********************************/
	
	Wtoplevel = toplevel;
	display = XtDisplay(Wtoplevel);
			
	/*********************/
	/* Crea un host file */
	/*********************/

	hostfile = tmpnam(NULL);
	fh = fopen (hostfile, "w");
	
	no = NetworkObjFirst ();
	while (no) {
	
		if (no->type == NET_HOST) {
			opt = no->options;
			if (opt && opt[0] == '&') {
				fprintf (fh, "&");
				opt++;
				while (*opt && (*opt == ' ' || *opt == '\t')) opt++;
			}

			fprintf (fh, "%s", no->addr);
			if (opt) fprintf (fh, "\t\t\t%s\n", opt);
			else fprintf (fh, "\n");
		}
		
		no = no->next;
	}
	
	fclose (fh);
	
	/**** prepara il nome del sockfile ****/
	
	pvm_useruid = getuid();
	sprintf (sockfile, "%s", pvmdsockfile());

	/***********************/
	/* Opzioni globali PVM */
	/***********************/
	
	pvm_setopt (PvmAutoErr, 0);		/* disabilita la stampa degli errori */

	/*******************/
	/* Attivazione PVM */
	/*******************/
	
	st = calloc (1, sizeof (struct stat));
	if (stat(sockfile, st)) {
	
		/**** PVM non attivo. Lo lancia. ****/
		
		MotifStatusMsg (wan->Wstatus, "PVM is not running, starting it for you... ");
	  
		sprintf (buffer, "pvmd %s 1>/dev/null 2>/dev/null &", hostfile);
		system (buffer);
	
		/**** aspetta finche` non compare... ****/

		do {
			sleep(1);
			MotifStatusMsg (wan->Wstatus, ".");
		} while (stat(sockfile, st));
		
		MotifStatusMsg (wan->Wstatus, " OK; now starting setup:\n"); 
	}
	else MotifStatusMsg (wan->Wstatus, "PVM already running; now starting setup:\n");	
		
	unlink (hostfile);
	free (st);

	/*********/
	/* MyTid */
	/*********/

	MotifStatusMsg (wan->Wstatus, "- joining PVM... ");
			
	MyTid = pvm_mytid();
	if (MyTid < 0) {
		errmsg = PVMError (MyTid);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		return;
	}
	pvm_insert("monitor", 1, MyTid);	/*inserisce il tid di wamm nel data base*/
	MotifStatusMsg (wan->Wstatus, "OK.\n");

	/*********************/
	/* Notify PvmHostAdd */
	/*********************/
	
	pvm_notify (PvmHostAdd, T_NFY_ADDHOST, -1, NULL);
	
	/**********/
	/* Hoster */
	/**********/
	
	verargv[0] = VER_HSTR;
	verargv[1] = NULL;

	MotifStatusMsg (wan->Wstatus, "- spawning PVMHoster on local host... ");
	
	res = pvm_spawn ("PVMHoster", verargv, PvmTaskHost, ".", 1, &HosterTid);
	if (res != 1) {
		errmsg = PVMError (HosterTid);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		HosterTid = 0;
	}
	
	/**** attende l'ack ****/
		
	if (HosterTid > 0) {
		
		pvm_recv (HosterTid, T_HSTR_ACK);
		pvm_unpackf ("%d", &ack);
		if (!ack) {
			MotifStatusMsg (wan->Wstatus, "FAILED: PVMHoster has a wrong version.\n");
			HosterTid = 0;
		}
	}

	if (HosterTid) MotifStatusMsg (wan->Wstatus, "OK.\n");
	else MotifStatusMsg (wan->Wstatus, "  You can use Wamm anyway, but won't get password dialogs.\n  Use shell to insert passwords when required.\n");
	
	/***************************/
	/* Configurazione corrente */
	/***************************/
	
	MotifStatusMsg (wan->Wstatus, "- getting current configuration... ");
	
	res = pvm_config (&hostnum, &archnum, &hosts);
	if (res < 0) {
		errmsg = PVMError (res);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		return;
	}
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*********************************************/
	/* Registrazione dtid e notify PvmHostDelete */
	/*********************************************/
	
	MotifStatusMsg (wan->Wstatus, "- getting pvmd tids for active hosts... ");
		
	no = NetworkObjFirst();
	while (no) {
	
		if (no->type == NET_HOST) {
				
			/**** controlla se e` attivo ****/
	
			for (n=0; n<hostnum; n++) if (!strcmp(no->addr, hosts[n].hi_name)) break;
			if (n<hostnum) {
			
				/**** registra il dtid ****/
		
				no->dtid = hosts[n].hi_tid;
		
				/**** notifica rimozione ****/
		
				pvm_notify (PvmHostDelete, T_NFY_DELHOST, 1, &no->dtid);
			}
		}	
		
		no = no->next;
	}
	
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*************************/
	/* Attivazione PVMTasker */
	/*************************/
	
	verargv[0] = VER_TSKR;
	verargv[1] = NULL;
	
	MotifStatusMsg (wan->Wstatus, "- spawning PVMTasker on all active hosts... ");
	
	tids = calloc (hostnum, sizeof (int));
	res = pvm_spawn ("PVMTasker", verargv, PvmTaskDefault, NULL, hostnum, tids);

	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/********************************/
	/* Registrazione tids PVMTasker */
	/********************************/
	
	MotifStatusMsg (wan->Wstatus, "- saving PVMTasker tids... ");
	
	for (n=0; n<hostnum; n++) {
	
		if (tids[n] <= 0) continue;		/* salta quelli falliti */
		
		/**** cerca l'host su cui e` stato lanciato questo task. Se non lo
		      trova, l'host non e` nel file di configurazione: uccide il
		      tasker (non ci serve) ****/
		
		no = NetworkObjFindByDtid (pvm_tidtohost(tids[n]));
		if (!no) pvm_kill (tids[n]);
		
		/**** se gia` e` stato registrato un tasker, questo e` un doppione
		      che va eliminato... ****/
		
		else {		
			if (no->taskertid) pvm_kill (tids[n]);
			else no->taskertid = tids[n];
		}
	}
	
	free (tids);
	
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*********************************/
	/* Controllo nodi mancanti e ack */
	/*********************************/
	
	MotifStatusMsg (wan->Wstatus, "- checking all PVMTaskers (activation and version)...\n");
	
	no = NetworkObjFirst ();
	n = 1;
	while (no) {
	
		if (no->type == NET_HOST) {
		
			XmListSetBottomPos (wan->Wlist, n);
			XmListSelectPos (wan->Wlist, n, False);
			XmUpdateDisplay (wan->Wlist);
			n++;
		
			if (!no->taskertid && no->dtid) {
			
				/**** bisogna provare a lanciarlo "a mano"... ****/
				
				res = pvm_spawn ("PVMTasker", verargv, PvmTaskHost, no->addr, 1, &tid);
				if (res != 1) {
					errmsg = PVMError (tid);
					MotifStatusMsg (wan->Wstatus, "  PVMTasker on %s can't be started: %s.\n", no->addr, errmsg);
					free (errmsg);
				}
				else no->taskertid = tid;
			}
		
			if (no->taskertid && no->dtid) {
			
				/**** attende l'ack ****/
				
				pvm_recv (no->taskertid, T_TSKR_ACK);
				pvm_unpackf ("%d", &ack);
				if (!ack) {
					MotifStatusMsg (wan->Wstatus, "  PVMTasker on %s has a wrong version and can't be used.\n", no->addr);
					no->taskertid = 0;
				}
			}
		}
		
		no = no->next;
	}
	
	XmListDeselectAllItems (wan->Wlist);
	XmListSetPos (wan->Wlist, 1);
	
	MotifStatusMsg (wan->Wstatus, "Done; setup completed.\n");
	
	/**** attiva il primo timeout ****/
	
	XtAppAddTimeOut (XtWidgetToApplicationContext (wan->Wshell), 1, (XtTimerCallbackProc)Timeout,
	NULL);
	
	return;	
}

/**********/
/* PVMEnd */
/**********/

void PVMEnd (int mode)
{	
	struct NetworkObj * no;

	if (mode != PVM_ENDFAIL) {
		
		/*** Uccide gli slave attendendo un ack ***/
		
		NFY_Kill_Slave(ACK);
		

		/**** Uccide i tasker ****/
	
		no = NetworkObjFirst();
		while (no) {
			if (no->taskertid) {
				pvm_recv(no->taskertid, SLAVEKILLED);
				pvm_sendsig (no->taskertid, SIGKILL);
			}
			no = no->next;
		}
	
		/**** Uccide l'hoster ****/
	
		if (HosterTid) pvm_sendsig (HosterTid, SIGKILL);
	}
	
	if (mode == PVM_ENDSOFT) {
		free_db();
		pvm_exit();
	}
	else if (mode == PVM_ENDHARD) pvm_halt();

	/**** per PVM_ENDFAIL non si puo` fare piu` nulla... ****/
}

/**********/
/* PVMTid */
/**********/

int PVMTid (void)
{
	return MyTid;
}

/****************************************************/
/* PVMError (code) - prepara un messaggio di errore */
/****************************************************/

char * PVMError (int code)
{
	char * buffer;
	
	buffer = calloc (1000,1);
	switch (code) {
	
		case PvmOk:			sprintf (buffer, "OK!");
							break;
		case PvmBadParam:	sprintf (buffer, "some parameter is not valid");
							break;
		case PvmNoHost:		sprintf (buffer, "host not in PVM or not existing");
							break;
		case PvmNoFile:		sprintf (buffer, "can't find executable file");
							break;
		case PvmNoMem:		sprintf (buffer, "malloc failed");
							break;
		case PvmSysErr:		sprintf (buffer, "local pvmd doesn't respond (PVM halted?)");
							break;
		case PvmHostFail:	sprintf (buffer, "host failed");
							break;
		
		case PvmBadVersion: sprintf (buffer, "remote pvmd has a wrong version");
							break;
		case PvmOutOfRes:	sprintf (buffer, "no resources");
							break;
		case PvmDupHost:	sprintf (buffer, "host is already in PVM");
							break;
		case PvmCantStart:	sprintf (buffer, "can't start pvmd");
							break;
		case PvmAlready:	sprintf (buffer, "operation already in progress");
							break;
		default:			sprintf (buffer, "error %d", code);
	}
	
	return (buffer);
}

/**********/
/* PVMAdd */
/**********/

void PVMAdd (struct NetworkObj * no, int num, char ** hosts, Boolean * mem)
{
	int * infos = NULL;
	int n, res;

	if (!fork()) {
	
		/**********/
		/* figlio */
		/**********/
		
		pvmendtask();
		pvm_mytid();
		
		infos = calloc (num, sizeof(int));

		res = pvm_addhosts ((char **)hosts, num, infos);
		
		/**** PvmAlready e PvmSysErr sono restituiti in res, NON in infos */
		
		if ((res == PvmAlready) || (res == PvmSysErr))
			for (n=0; n<num; n++) infos[n] = res; 
			
		pvm_packf ("%+ %d %d", PvmDataDefault, no, num);
		for (n=0; n<num; n++) pvm_packf ("%s %d", hosts[n], infos[n]);
		pvm_send (PVMTid(), T_INT2_ADDEND);
		
		for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
		free (mem);
		free (hosts);	
		free (infos);
		
		pvm_exit();
		exit(0);
	}
	
	/* XXX ci vorrebbe un controllo per fork == -1 */
	
	else return;
}

/**********/
/* PVMDel */
/**********/

void PVMDel (struct NetworkObj * no, int num, char ** hosts, Boolean * mem)
{
	int * infos;
	char * tmp;
	int n;
	
	/* Il delete e` veloce, non occorre farlo con una fork. */
	
	infos = calloc (num, sizeof(int));
	pvm_delhosts ((char **)hosts, num, infos);
	
	for (n=0; n<num; n++) if (infos[n]<0) {
	
		if (infos[n] != PvmNoHost) {		
			tmp = PVMError (infos[n]);
			MotifStatusMsg (no->Wstatus, "%s: %s\n", hosts[n], tmp);
			free (tmp);
		}
		
		else MotifStatusMsg (no->Wstatus, "%s: not in PVM\n", hosts[n]);
	}
	
	free (infos);	
}

/************/
/* PVMCheck */
/************/

void PVMCheck (struct NetworkObj * no, int num, char ** hosts, Boolean * mem)
{
	char * tmp;
	int n, res;
	
	/* Il check e` veloce, non occorre farlo con una fork. */
	
	for (n=0; n<num; n++) {
	
		MotifStatusMsg (no->Wstatus, "%s: ", hosts[n]);
		res = pvm_mstat (hosts[n]);
		
		/**** controlla l'esito ****/
		
		tmp = PVMError (res);
		MotifStatusMsg (no->Wstatus, "%s\n", tmp);
		free (tmp);
	}
}

/*************/
/* PVMRepair */
/*************/

void PVMRepair (struct NetworkObj * no, int num, char ** hosts, Boolean * mem)
{

	/* XXX bugs noti:
	
		- i messaggi di errore di rexec vengono mostrati su shell (orrendo...)
	*/

	struct servent * servent;		/* servent per rexec */
	char * localname;				/* nome utente locale */
	char remotename[1000];			/* nome utente remoto */
	
	char buffer[1000];
	int n;
	struct NetworkObj * no2;
	char * psw;
	int out;
	char * tmp;
	
	/**** porta per rexec e nome utente locale ****/
	
	servent = getservbyname ("exec", "tcp");
	localname = getpwuid (getuid()) -> pw_name; 
	
	for (n=0; n<num; n++) {
	
		MotifStatusMsg (no->Wstatus, "%s... ", hosts[n]);
		no2 = NetworkObjFindByAddr (hosts[n]);
	
		/**** Nome utente remoto ****/
		
		tmp = strstr(no2->options, "lo=");
		if (tmp) {
			strcpy (remotename, tmp+3);
			tmp = strpbrk (remotename, " \t");
			if (tmp) *tmp = 0;
		}
		else strcpy (remotename, localname);
		
		/**** Controlla se deve usare rsh o rexec ****/
		
		no2 = NetworkObjFindByAddr (hosts[n]);
		if (no2->options && strstr(no2->options, "so=pw")) {
		
			/**** Usa rexec ****/
			
			psw = no2->password;
			if (! psw) {
				sprintf (buffer, "Please insert password for %s, user %s:", hosts[n], remotename);
				psw = MotifAskPsw (WAN->Wshell, "Password", buffer);
				if (!psw) {
					MotifStatusMsg (no->Wstatus, "failed (must use password!)\n");
					continue;
				}
			}
	
			out = rexec ((char **)&hosts[n], servent->s_port, remotename, psw, "PVMKill", 0);
			if (out < 0) MotifStatusMsg (no->Wstatus, "failed.\n");
			else {
				MotifStatusMsg (no->Wstatus, "repaired.\n");
				close (out);
				if (!no2->password) no2->password = strdup (psw);
			}
		}
		
		else {
		
			/**** Usa rsh ****/
		
			sprintf (buffer, "rsh < /dev/null %s -l %s PVMKill", hosts[n], remotename);
			system (buffer);
			MotifStatusMsg (no->Wstatus, "repaired.\n");
		}		
	}
}

/******************************************************************************/
/******************************************************************************/

/***********/
/* Timeout */
/***********/

void Timeout (void)
{	
	int msgid, msglen, msgtag, msgtid;	/* dati del messaggio ricevuto */
	int done_flag = 0;
	XEvent evt;
	XtAppContext context;
	
	context = XtWidgetToApplicationContext(WAN->Wshell);	
	/*****************************************/
	/* Controllo messaggi e ricarica timeout */
	/*****************************************/
/*while(!done_flag) {
		while(XtAppPending(context)) {
			XtAppNextEvent(context,&evt);
			XtDispatchEvent(&evt);
		}*/
	msgid = pvm_nrecv (-1, -1);
	if (msgid<0) {
		
		/* gravi problemi con il PVM... */
		
		MotifError (WAN->Wshell, "FATAL ERROR!",
			"Local pvmd doesn't respond. Maybe PVM was halted.\n"
			"Please select OK to quit Wamm.");
		
		/*************************/
		/* Chiude tutti i moduli */
		/*************************/
	
		PVMEnd (PVM_ENDFAIL);
		InternetEnd ();
		WANEnd ();
		MANEnd ();
		LANEnd ();
		TasksEnd ();
		SpawnEnd ();
		MakeEnd ();
		NetworkObjEnd ();
		
		/*XXX aggiungere chiusura del monitor */
	
		exit (1);
	}
	
	/* Tutto ok, puo` ricaricare il timeout */
	
	XtAppAddTimeOut (XtWidgetToApplicationContext (WAN->Wshell), PVM_DELAY, (XtTimerCallbackProc)Timeout,
	NULL);

	if (msgid) {
	pvm_bufinfo (msgid, &msglen, &msgtag, &msgtid);
	switch (msgtag) {
	
		/*********************/
		/* AddHost terminato */
		/*********************/
		
		case T_INT2_ADDEND:		INT2_AddEnd();
								break;

	
		/*********************/
		/* Notify nuovo host */
		/*********************/
		
		case T_NFY_ADDHOST:		NFY_AddHost ();
								break;
													
		/***********************/
		/* Notify host rimosso */
		/***********************/
		
		case T_NFY_DELHOST:		NFY_DelHost ();
								break;
												
		/**********************/
		/* Tasker: nuovo task */
		/**********************/
		
		case T_TSKR_NEW:		TSKR_New ();
								break;
															
		/******************/
		/* Task terminato */
		/******************/
		
		case T_TSKR_END:		TSKR_End ();
								break;
				
		/*********************/
		/* Comando terminato */
		/*********************/
		
		case T_TSKR_CMD:		TSKR_Cmd ();
								break;
												
		/**********************/
		/* Richiesta password */
		/**********************/
		
		case T_HSTR_PSW:		HSTR_Psw ();
								break;
								
		/***********************/
		/* Messaggio dal maker */
		/***********************/
		
		case T_MKR_MSG:			MKR_Msg (msgtid);
								break;	
								
		/******************************/
		/* Messaggio finale dal maker */
		/******************************/
		
		case T_MKR_MSGEND:		MKR_MsgEnd (msgtid);
								break;
								
		/**********************/
		/* Output da un maker */
		/**********************/
		
		case T_MKR_OUT:			MKR_Out ();
								break;
								
		/*********************/
		/* Output da un task */
		/*********************/
		
		case T_TASK_OUT:		TASK_Out ();
								break;
								
		/**************/
		/* Trace task  		DA ELIMINARE */ 
		/**************/
		
		case T_TASK_TRACE:		TASK_Trace ();
								break;
		
		/**************************************************/
		/* Ingresso di un task nella fase di monitoraggio */
		/**************************************************/
		
		case OL_CHECKIN:		OL_Checkin(msgtid);
								break;
								
		/************************************************/
		/* Statistiche sulla comunicazione (da un task) */
		/************************************************/						
		
		case OL_STATISTIC:		OL_Statistic(msgtid);
								break;
								
		/*********************************************/
		/* Statistiche sulle macchine (da un tasker) */
		/*********************************************/						
		
		case OL_PROCINFO:		OL_Procinfo();
								break;						
								
		/*************************************************/
		/* Output da un task per la finestra del monitor */
		/*************************************************/	
		
		case OL_PRINT:		OL_Print(msgtid);
							break;
								
		/*********************/
		/* Input da un task  */
		/*********************/	
		
		case OL_SCAN:		OL_Scan(msgtid);
							break;
								
		/*****************************/
		/* Uscita di un task dal PVM */
		/*****************************/	
		
		case OL_CHECKOUT:		OL_Checkout(msgtid);
								break;
		
		/*************************************************/
		/* Uscita di un task da una fase di monitoraggio */
		/*************************************************/	
		
		case OL_EXIT_MON:		OL_Exit_Mon(msgtid);
								break;
								
		/*****************************************************/
		/* Richiesta per la spedizione di un file di traccia */
		/*****************************************************/	
		
		case REQ_SBLOCKING:		REQ_Sblocking();
								break;
																				
	/*}*/																								
	}
}
}

/************************************************/
/* INT2_AddEnd - segnalazione AddHost terminato */
/************************************************/

void INT2_AddEnd (void)
{
	int n;
	char * tmp;
	struct NetworkObj * no;			/* nodo da usare per i messaggi */
	int num;						/* numero nodi aggiunti */
	struct NetworkObj * hostno;		/* nodo aggiunto */
	char host[1000];				/* indirizzo host */
	int dtid;						/* dtid nuovo host (o errore se < 0) */

	pvm_unpackf ("%d %d", &no, &num);
												
	for (n=0; n<num; n++) {
							
		pvm_unpackf ("%s %d", host, &dtid);
		hostno = NetworkObjFindByAddr (host);						
								
		if (dtid > 0) continue;
		
		if (dtid == PvmDupHost) {
			if (no->Wshell) MotifStatusMsg (no->Wstatus, "%s: already in PVM\n", host);
			continue;
		}

		if (hostno->password) {
			free (hostno->password);
			hostno->password = NULL;
		}
		
		tmp = PVMError (dtid);
		if (no->Wshell) MotifStatusMsg (no->Wstatus, "%s: %s\n", host, tmp);
		free (tmp);
	}
}

/***********************************/
/* NFY_AddHost - notify nuovo host */
/***********************************/
	
void NFY_AddHost (void)
{
	int n, res;
	char * tmp;
	
	int confnum, confarchs;				/* risultati di pvm_config */
	struct pvmhostinfo * confinfo;
	struct NetworkObj * no;				/* nuovo host */
	char * argv[2];						/* argv tasker */
	int tid;							/* tid tasker */
	int ack;							/* ack versione (1=ok) */
	
	argv[0] = VER_TSKR;
	argv[1] = NULL;

	res = pvm_config (&confnum, &confarchs, &confinfo);
	if (res<0) return;
				
	for (n=0; n<confnum; n++) {
		
		no = NetworkObjFindByAddr (confinfo[n].hi_name);
									
		/**** Solo per gli host "nuovi" e riconosciuti ****/
									
		if (no && (!no->dtid)) {
					
			no->dtid = confinfo[n].hi_tid;
			
			/**** notifica rimozione ****/
		
			pvm_notify (PvmHostDelete, T_NFY_DELHOST, 1, &confinfo[n].hi_tid);
			
			/**** Segnala nuovo host ****/
			
			RefreshStatus (no);
			MotifStatusMsg (WAN->Wstatus, "NEW HOST: %s. Spawning PVMtasker... ", no->addr);
										
			/**** tasker ****/
									
			res = pvm_spawn ("PVMTasker", argv, PvmTaskHost, no->addr, 1, &tid);
			if (res != 1) {
				tmp = PVMError (tid);
				MotifStatusMsg (WAN->Wstatus, "FAILED! (%s).\n", tmp);
				tid = 0;
				free (tmp);
			}
		
			/**** attende l'ack ****/
		
			else {
				pvm_recv (tid, T_TSKR_ACK);
				pvm_unpackf ("%d", &ack);
				if (!ack) {
					MotifStatusMsg (WAN->Wstatus, "FAILED! Remote PVMTasker has a wrong version.\n");
					tid = 0;
				}
				else if(montype > 1) {
					pvm_initsend(PvmDataDefault);
					pvm_send(tid, STARTSLAVE);
					}
			}
			
			no->taskertid = tid;
			if (tid) MotifStatusMsg (WAN->Wstatus, "OK.\n");
		}
	}
}

/*************************************/
/* NFY_DelHost - notify host rimosso */
/*************************************/
	
void NFY_DelHost (void)
{
	int dtid;
	struct NetworkObj * no;

	pvm_upkint (&dtid, 1, 1);
			
	/**** Cerca l'host e cambia lo status PVM ****/
		
	no = NetworkObjFindByDtid (dtid);
	if (!no) return;
	
	no->dtid = 0;
	RefreshStatus (no);
	no->taskertid = 0;
	
	MotifStatusMsg (WAN->Wstatus, "HOST DELETED: %s\n", no->addr);
	
	/**** Segnala che tutti i task in esecuzione sull'host sono morti */
	
	TasksDelAll (no->addr);
}
	
/*************************/
/* TSKR_New - nuovo task */
/*************************/
	
void TSKR_New (void)
{
	int tid, ntask, position;
	char temp_str[256];
	XmString xmstr;
	struct pvmtaskinfo * ti;
	struct NetworkObj * no;	
	
	pvm_upkint (&tid, 1, 1);
	if (tid<=0) return;
	
	/* Se ntask != 1 ti NON contiene dati validi! */
	
	if (pvm_tasks(tid, &ntask, &ti) >=0) {
		if (ntask != 1) return;
		no = NetworkObjFindByDtid (ti->ti_host);
		TasksAdd (tid, ti->ti_a_out, no->addr);
	}
		
}	
	
/*****************************/
/* TSKR_End - task terminato */
/*****************************/
	
void TSKR_End (void)
{
	int tid;
	
	pvm_upkint (&tid, 1, 1);
	if (tid) TasksDel (tid);
}

/********************************/
/* TSKR_Cmd - comando terminato */
/********************************/
	
void TSKR_Cmd (void)
{
	int res, len;
	struct NetworkObj * no;
	char * out;
	char * err;
	
	pvm_unpackf ("%d %d %d", (int *)&no, &res, &len);
	
	/**** finestra aperta ? ****/
	
	if (!no->Wshell) return;

	out = calloc (len+1, 1);
	pvm_unpackf ("%s %d", out, &len);
	err = calloc (len+1, 1);
	pvm_unpackf ("%s", err);
											
	if (!res) {
		if (strlen(out)) MotifStatusMsg (no->Wstatus, "%s:\n%s\n", no->addr, out);
		else MotifStatusMsg (no->Wstatus, "%s: OK (no output).\n\n", no->addr);
	}
	
	else {
		if (strlen(err)) MotifStatusMsg (no->Wstatus, "%s:\n%s\n", no->addr, err);
		else MotifStatusMsg (no->Wstatus, "%s: error (%d)\n\n", no->addr, res);
	}
	
	free (out);
	free (err);
}

/********************************/
/* HSTR_Psw - richiede password */
/********************************/
	
void HSTR_Psw (void)
{
	struct NetworkObj * no;
	char host [1000];
	char buffer [1000];
	char * tmp;
	
	pvm_upkstr (host);
	no = NetworkObjFindByAddr (host);
							
	/**** Se esiste, usa la password gia` inserita ****/
							
	if (no && no->password) {
		pvm_packf ("%+ %s", PvmDataDefault, no->password);
		pvm_send (HosterTid, T_HSTR_PSW);
		return;
	}
							
	/**** Nessuna password: la richiede ****/
							
	sprintf (buffer, "Please insert password for %s:", host);
	tmp = MotifAskPsw (WAN->Wshell, "Password", buffer);
	if (tmp) {
		if (no) no->password = strdup (tmp);
		pvm_packf ("%+ %s", PvmDataDefault, tmp);
	}
	else pvm_packf ("%+ %s", PvmDataDefault, "");
	
	pvm_send (HosterTid, T_HSTR_PSW);
}

/*********************************/
/* MKR_Msg - messaggio dal maker */
/*********************************/

void MKR_Msg (int tid)
{
	char msg[1000];
	
	pvm_unpackf ("%s", msg);
	MakeMsg (tid, msg);
}

/*******************************************/
/* MKR_MsgEnd - messaggio finale dal maker */
/*******************************************/

void MKR_MsgEnd (int tid)
{
	char msg[1000];
	int res;
	
	pvm_unpackf ("%s %d", msg, &res);
	MakeMsgEnd (tid, msg, res);
}

/********************************/
/* MKR_Out - output da un maker */
/********************************/
	
void MKR_Out (void)
{
	int tid, len;
	char * out = NULL;
	
	pvm_unpackf ("%d%d", &tid, &len);
	if (len>0) {
		out = calloc (len+1, 1);
		pvm_upkbyte (out, len, 1);
		MakeOutput (tid, out);
		free (out);
	}
}

/********************************/
/* TASK_Out - output da un task */
/********************************/
	
void TASK_Out (void)
{
	int tid, len;
	char * out = NULL;
	
	pvm_unpackf ("%d%d", &tid, &len);
	if (len>0) {
		out = calloc (len+1, 1);
		pvm_upkbyte (out, len, 1);
		TasksOutput (tid, out);
		free (out);
	}
}

/***************************/
/* TASK_Trace - trace task */
/***************************/
	
void TASK_Trace (void)
{
	int sec, usec, tid, eid;
	
	pvm_unpackf ("%d %d %d %d", &sec, &usec, &tid, &eid);
	printf ("Trace: TID=%x, EID=%d\n", tid, eid);
}



/**************/
/* OL_Checkin */
/**************/

void OL_Checkin(int msgtid)
{
	int pos,lung, temp_pagesize,sample_delay;
	int n;
	struct pvmtaskinfo * ti;
	char *temp_str; 
	char *temp_str2;
	XEvent evt2;
	XmString xmstr;
	
	pvm_upkint(&pos, 1, 1); 
	lung=pos+1;
	if (lung >= lung_array) {
		lung_array = lung;
		maxprocs =lung_array;
	}
	tids_array[pos] = msgtid;
	procs_checked_in++;
	if (montype > 1) {
		temp_str2 = calloc(256,sizeof(char));
		temp_str = calloc(256,sizeof(char));
		pvm_tasks(msgtid, &n, &ti); 		
		pvm_upkstr(temp_str);
		pvm_upkint(&temp_pagesize, 1, 1);
		pvm_upkint(&sample_delay, 1, 1);
        loads[pos].hostname = (char *) realloc(NULL,strlen(temp_str)+1);
		strcpy(loads[pos].hostname,temp_str);
		addhost(temp_str,temp_pagesize);
		evt2.xexpose.display = display;
        evt2.xexpose.window = XtWindow(hostlist);
        redrawHostListWindow(&evt2);
		loads[pos].taskid = msgtid;
		sprintf(temp_str2,"proc %d, tid=%x on %s has checked in!\n",
				pos,msgtid,temp_str);
		XmTextInsert(text_output, wpr_position, temp_str2);
		wpr_position += strlen(temp_str2);
		XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
		XmTextShowPosition(text_output, wpr_position);
		
		TasksAdd_mon (msgtid,pos); /*aggiunge un task alla tasklist */
		
		free(temp_str2);
		
		if (procs_checked_in == 1) {
            /* clearwindow one more time*/
		    evt2.xexpose.window = XtWindow(hostutil);
		    redrawHostUtilWindow(&evt2, -2);
		    XmScaleSetValue(scale,sample_delay);
		    if (sample_delay != 5) {
		    	NFY_Sample_Delay(sample_delay);
		    }  	
		}
	}
}

/****************/
/* OL_Statistic */
/****************/

void OL_Statistic(int msgtid)
{

	struct rusage usageinfo;
	int total_sends=0;
    int total_receives=0;
    int interval_sends=0;
    int interval_receives=0;
    int interval_volume_rcvd=0;
    int interval_volume_snt=0;
    int total_volume_snt=0;
    int total_volume_rcvd=0;
    struct timeval int_idle_time;
    struct timeval tot_idle_time;
    struct timeval int_send_time;
    struct timeval tot_send_time;
    struct timeval int_program_time;
    struct timeval tot_program_time;
    int spot, lung, i, index;
    XEvent evt2;
    static int numberofmessages;
    
	pvm_upklong(&usageinfo.ru_utime.tv_sec, 1, 1);
	pvm_upklong(&usageinfo.ru_utime.tv_usec, 1, 1);
	pvm_upklong(&usageinfo.ru_stime.tv_sec, 1, 1);
	pvm_upklong(&usageinfo.ru_stime.tv_usec, 1, 1);
	pvm_upklong(&usageinfo.ru_maxrss, 1, 1);
	pvm_upkint(&interval_sends, 1, 1);
	pvm_upkint(&total_sends, 1, 1);		
	pvm_upkint(&interval_receives, 1, 1);
	pvm_upkint(&total_receives, 1, 1);		
	pvm_upkint(&interval_volume_snt, 1, 1);
	pvm_upkint(&total_volume_snt, 1, 1);		
	pvm_upkint(&interval_volume_rcvd, 1, 1);
	pvm_upkint(&total_volume_rcvd, 1, 1);		
	pvm_upklong(&int_idle_time.tv_sec, 1, 1);
	pvm_upklong(&int_idle_time.tv_usec, 1, 1);
	pvm_upklong(&tot_idle_time.tv_sec, 1, 1);
	pvm_upklong(&tot_idle_time.tv_usec, 1, 1);
	pvm_upklong(&int_send_time.tv_sec, 1, 1);
	pvm_upklong(&int_send_time.tv_usec, 1, 1);
	pvm_upklong(&tot_send_time.tv_sec, 1, 1);
	pvm_upklong(&tot_send_time.tv_usec, 1, 1);		
	pvm_upklong(&int_program_time.tv_sec, 1, 1);
	pvm_upklong(&int_program_time.tv_usec, 1, 1);
	pvm_upklong(&tot_program_time.tv_sec, 1, 1);
	pvm_upklong(&tot_program_time.tv_usec, 1, 1);	
	spot = mevalue(msgtid);		
	pvm_upkint(&lung, 1, 1);
	if (lung > lung_array) maxprocs = lung_array = lung; /* caso in cui il monitor non
													ha ancora ricevuto OL_CHECKIN 
													da qualche task che sta gia 
													comunicando */
	pvm_upkint(loads[spot].send_vector, lung, 1);
	pvm_upkint(loads[spot].int_send_vector, lung, 1);
	pvm_upkint(loads[spot].send_byte_vector, lung, 1);
	pvm_upkint(loads[spot].receive_vector, lung, 1);
	pvm_upkint(loads[spot].receive_byte_vector, lung,1);		
	loads[spot].interval_sent = interval_sends;
	loads[spot].interval_bytes_sent = interval_volume_snt;
	for (i=0; i < lung_array; i++) {
		if (loads[spot].send_vector[i] > numberofmessages)
		numberofmessages = loads[spot].send_vector[i];
	}

	/*update comm legend if necessary*/
	if ((commlegendrangelevel < MAXRANGECHOICES-1) &&
		(numberofmessages >=  
		rangechoices[commlegendrangelevel].rangemax[7])) {
		for (i=commlegendrangelevel+1; i < MAXRANGECHOICES-1;i++) {
			commlegendrangelevel++;
			if (rangechoices[commlegendrangelevel].rangemax[7] >
			    numberofmessages || commlegendrangelevel == 6)
			    break;
		    }
		}
		loads[spot].maxrss = usageinfo.ru_maxrss;
		
		loads[spot].intsend_time = int_send_time.tv_sec * 1.0e3
		    + int_send_time.tv_usec * 1.0e-3;
		
		loads[spot].totsend_time = tot_send_time.tv_sec * 1.0e3
		    + tot_send_time.tv_usec * 1.0e-3;

		loads[spot].intidle_time = int_idle_time.tv_sec * 1.0e3
		    + int_idle_time.tv_usec * 1.0e-3;
		
		loads[spot].totidle_time = tot_idle_time.tv_sec * 1.0e3
		    + tot_idle_time.tv_usec * 1.0e-3;

		loads[spot].intprog_time = int_program_time.tv_sec * 1.0e3
		    + int_program_time.tv_usec * 1.0e-3;
		
		loads[spot].totprog_time = tot_program_time.tv_sec * 1.0e3
		    + tot_program_time.tv_usec * 1.0e-3;

		loads[spot].intcompute_time = loads[spot].intprog_time -
		    (loads[spot].intsend_time + loads[spot].intidle_time);

		loads[spot].totcompute_time = loads[spot].totprog_time -
		    (loads[spot].totsend_time + loads[spot].totidle_time);
		
		index = hostindex(loads[spot].hostname);

		hosts[index].totsend_time = tot_send_time.tv_sec * 1.0e3
		    + tot_send_time.tv_usec * 1.0e-3;

		hosts[index].totidle_time = tot_idle_time.tv_sec * 1.0e3
		    + tot_idle_time.tv_usec * 1.0e-3;

		hosts[index].totprog_time = tot_program_time.tv_sec * 1.0e3
		    + tot_program_time.tv_usec * 1.0e-3;

		hosts[index].totcompute_time = hosts[index].totprog_time -
		    (hosts[index].totsend_time + hosts[index].totidle_time);

		
		
		evt2.xexpose.display = display;
        evt2.xexpose.window = XtWindow(totalutil); 
		redrawTotalUtilWindow(&evt2);

		evt2.xexpose.display = display; 
		evt2.xexpose.window = XtWindow(messagesent);
		redrawMessageSentWindow(&evt2);	
		
		evt2.xexpose.window = XtWindow(matrixcomm);
		redrawMatrixCommWindow(&evt2);
		
		evt2.xexpose.window = XtWindow(hostutil);
		redrawHostUtilWindow(&evt2, index);
		
		evt2.xexpose.window = XtWindow(memuse);
		redrawMemUseWindow(&evt2);		   		
}

/***************/
/* OL_Procinfo */
/***************/

void OL_Procinfo()
{
	int bcount,index;
	char infobuf[10240];
	char temp_str[256];
	char name_host[64];
	XEvent evt2;
	
	/* OL_PROCINFO now for only uptime info*/
	pvm_upkint(&bcount, 1, 1);
	pvm_upkbyte(infobuf, bcount, 1);
	pvm_upkstr(name_host); /* modifica */
	infobuf[bcount]='\0';
	if((index = hostindex(name_host)) != -1) {
		/*  XXX may need to be arch specific*/
		sscanf(infobuf,"%*s %*s %*s %*s %*s %s",temp_str);
		if (!strncmp(temp_str,"min",3))
			sscanf(infobuf,"%s %*s %*s %*s %*s %*s %d %*s %*s %*s %lf",
			    hosts[index].time, &hosts[index].users, 
			    &hosts[index].loadvalue);
		else if (!strncmp(temp_str,"loa",3))
				sscanf(infobuf,"%s %*s %*s %d %*s %*s %*s %lf",
			    	hosts[index].time, &hosts[index].users, 
			    	&hosts[index].loadvalue);
		else if (!strncmp(temp_str,"use",3)) 
				sscanf(infobuf,"%s %*s %*s %*s %d %*s %*s %*s %lf",
			    	hosts[index].time, &hosts[index].users, 
			    	&hosts[index].loadvalue);
		else
		    sscanf(infobuf,"%s %*s %*s %*s %*s %d %*s %*s %*s %lf",
			    hosts[index].time, &hosts[index].users, 
			    &hosts[index].loadvalue);
		evt2.xexpose.display = display;
        evt2.xexpose.window = XtWindow(loadinfo);
		redrawLoadWindow(&evt2); 
	}
}

/************/
/* OL_Print */
/************/

void OL_Print(msgtid)
{	
	char print_str[2048];
	char temp_str[256];
	
	
	pvm_upkstr(print_str);
	sprintf(temp_str,"task %d:%s", mevalue(msgtid),print_str);
	XmTextInsert(text_output, wpr_position, temp_str);
	wpr_position += strlen(temp_str);
	XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
	XmTextShowPosition(text_output, wpr_position);
}

/***********/
/* OL_Scan */
/***********/

void OL_Scan(msgtid)
{
	int scan_type;
	char print_str[2048];
	char temp_str[256];
	char input_string[200];
	XmTextPosition pos1=0;
	XEvent evt;
	char *string;
	
	pvm_upkint(&scan_type, 1, 1);
	pvm_upkstr(print_str);
	switch (scan_type) {
		case PVM_STR:
		    sprintf(temp_str,"SCANSTRING  tasktid=%d: %s:",
					msgtid,print_str);
		    break;
		
		case PVM_INT:
		    sprintf(temp_str,"SCANINT  tasktid=%d: %s:",
			    	msgtid,print_str);
		    break;
		
		case PVM_LONG:
		    sprintf(temp_str,"SCANLONG  tasktid=%d: %s:",
					 msgtid,print_str);
		    break;
		
		case PVM_FLOAT:
		    sprintf(temp_str, 
			    "SCANFLOAT  tasktid=%d: %s:",
			    msgtid,print_str);
		    break;
		
		case PVM_DOUBLE:
		    sprintf(temp_str, 
			    "SCANDOUBLE  tasktid=%d: %s:",
			    msgtid,print_str);
			 break;
	}
	string = MotifAskInput (Wtoplevel,temp_str);
	strcpy(input_string, string);
	pvm_initsend(PvmDataDefault);
	pvm_pkstr(input_string);
	pvm_send(msgtid,OL_SCAN);
}

/***************/
/* OL_Checkout */
/***************/

void OL_Checkout(msgtid)
{
	int spot,cc,data;
	char temp_str[256];
	XmTextPosition pos1;
	
	
	if (montype >1) {
		spot = mevalue(msgtid);
		sprintf(temp_str,"task %d, tid=%d has exited pvm !\n",spot,msgtid);
		XmTextInsert(text_output, wpr_position, temp_str);
		wpr_position += strlen(temp_str);
		XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
		XmTextShowPosition(text_output, wpr_position);
	}
	done_count++;
	if (done_count == lung_array) {
		while((cc=pvm_lookup("tid", -1, &data))!= PvmNoEntry) {
			sprintf(temp_str,"%d",data);
			pvm_delete("tid",cc);
			pvm_delete(temp_str,0);
		}
		if (montype >1) {
			sprintf(temp_str,"All monitored processes have exited pvm!\n");
			pos1 = wpr_position;
			XmTextInsert(text_output, wpr_position, temp_str);
			wpr_position += strlen(temp_str);
			XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
			XmTextShowPosition(text_output, wpr_position);
			XmTextSetHighlight(text_output, pos1, wpr_position, XmHIGHLIGHT_SELECTED);
			NFY_Kill_Slave(NORM);
		}
		if(montype == 3) {
			pos1 = wpr_position;
			sprintf(temp_str,"Ordering tracefile ...\n");
			XmTextInsert(text_output, wpr_position, temp_str);
			wpr_position += strlen(temp_str);
			XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
			XmTextShowPosition(text_output, wpr_position);
			XmTextSetHighlight(text_output,pos1,wpr_position,XmHIGHLIGHT_SELECTED);
		}
		end_monitoring();
		procs_checked_in = 0;
		cont_mon = 0;
	}
}

/***************/
/* OL_Exit_Mon */
/***************/

void OL_Exit_Mon(msgtid)
{
	int spot=0;
	char temp_str[256];
	
	if (montype >1) {
		spot = mevalue(msgtid);
		sprintf(temp_str,"task %d, tid=%d has finished monitoring !\n",spot,msgtid);
		XmTextInsert(text_output, wpr_position, temp_str);
		wpr_position += strlen(temp_str);
		XtVaSetValues(text_output, XmNcursorPosition, wpr_position, NULL);
		XmTextShowPosition(text_output, wpr_position);
	}
}

/*****************/
/* REQ_Sblocking */
/*****************/

void REQ_Sblocking()
{
	int i,n;
	FILE *fp;
	char logname[256];
	char buf[TRACEBUFSIZ];
	
	if (montype == 1 || montype == 3) {
		procs_checked_in--;
		if (procs_checked_in == 0)	{
			pvm_initsend(PvmDataDefault);
			pvm_mcast(tids_array, lung_array, SBLOCKING);
			if(path_str[0] == '\0'){
				sprintf(logname, "/tmp/wammfile.%d.%d",getuid(),cont_mon);
			}
			else {
				sprintf(logname, "%s.%d.%d",path_str,getuid(),cont_mon);
			}
			if ((fp = fopen(logname, "w+"))== NULL){
				printf("wamm: cannot open MAIN tracefile");
			} 
			i=0;
			while (i < lung_array) {
				pvm_recv(tids_array[i], TRACETAG);
				pvm_upkint(&n, 1, 1);
				if (n != -1) {
					pvm_upkbyte(buf, n, 1);
					if (fwrite(buf,1, n,fp) != n) {
						printf("wamm: write error \n while writing %d\n",i);
					}
				}
				else i++;			
			}
			fclose(fp);
			pvm_mcast(tids_array, lung_array, RESTART);	/*XXX solo wamm_close */
			post_process(logname);
			
			cont_mon++;	
		}
	}
}

/************************/
/* RefreshStatus (node) */
/************************/

void RefreshStatus (struct NetworkObj * no)
{
	int n;
	Arg args[5];

	if (!no->Wshell) return;
	
	n = 0;
	if (no->dtid) {XtSetArg (args[n], XmNlabelString, XmStringCreate ("PVM", DEFSET)); n++;}
	else {XtSetArg (args[n], XmNlabelString, XmStringCreate (" ", DEFSET)); n++;}
	XtSetValues (no->Wpvm, args, n);
}



/***********/
/* addhost */
/***********/

void addhost (char *temp_str,int pagesize)
{
    int i;
	for (i=0; i < numhosts; i++) {
		if (strcmp(hosts[i].hostname,temp_str)== 0) break;
    }

    if (i==numhosts) {
		hosts[numhosts].hostname = (char *) realloc(NULL,strlen(temp_str)+1);
		strcpy(hosts[numhosts].hostname, temp_str);
		hosts[numhosts].pagesize = pagesize;
		hosts[numhosts].totsend_time = 0;
        hosts[numhosts].totcompute_time = 0;
        hosts[numhosts].totidle_time = 0;
		hosts[numhosts].totprog_time = 0;
        hosts[numhosts].maxrss = 1;
		numhosts++;
    }

}

/*************/
/* hostindex */
/*************/

int hostindex(char *temp_str)
{
    int i;
    for (i=0; i < numhosts; i++) {
		if (strcmp(hosts[i].hostname, temp_str) == 0)
			return i;
    }
    if (i==numhosts) {
		return -1;
    }

}

/******************/
/* NFY_Kill_Slave */
/******************/


void NFY_Kill_Slave(int cond)
{
	struct NetworkObj * no;
	
	no = NetworkObjFirst ();
	while (no) {
		if (no->type == NET_HOST && no->taskertid) {
			pvm_initsend(PvmDataDefault);
			if (cond == ACK){
				pvm_send(no->taskertid, KILLSLAVE_ACK);
			}
			else pvm_send(no->taskertid, KILLSLAVE);
		}
		no = no->next;
	}
}

/********************/
/* NFY_Sample_Delay */
/********************/


void NFY_Sample_Delay(int s_d)
{
	struct NetworkObj * no;
	
	no = NetworkObjFirst ();
	while (no) {
		if (no->type == NET_HOST && no->taskertid) {
			pvm_initsend(PvmDataDefault);
			pvm_pkint(&s_d,1,1);			
			pvm_send(no->taskertid, SAMPLEDELAY);
		}
		no = no->next;
	}
}
/****************************************/
/*	free_db	  -- ripulisce il data base	*/
/****************************************/

void free_db(void)
{
	int i=0,cc,mon;
	char str[10];
	
	while (pvm_lookup("tid",i,&cc)!=PvmNoEntry){
		pvm_delete("tid",i);
		sprintf(str,"%d",cc);
		pvm_delete(str,0); 
		i+=1;
	}
	i=0;
	while(pvm_lookup("monitor",i,&mon)!=PvmNoEntry) {
		pvm_delete("monitor",i);
		i+=1;
	}
}


/*********************************************/
/*	cerca_file - controlla se il file esiste */
/*********************************************/

int cerca_file()
{
	DIR * dp;
	struct dirent * ep;
	FILE * fd; 
	int trovato=0;
	int pos, pos1;
	char *str_dir;
	char *str_file;
	char *str2;
	int i,j;
	
	str2 = calloc(64,1);
	str_dir = calloc(64,1);
	str_file = calloc(64,1);
	pos1= pos = strlen(path_str);
	while(path_str[pos] != '/') {
		pos--;
	}
	strncpy(str_dir,path_str,pos+1);
	str_dir[pos + 2] = '\0'; 
	for (i = pos+1, j=0; i<= pos1; i++, j++) {
		str_file[j] = path_str[i];
	}
	sprintf(str2,".%d.%d.%s",getuid(),cont_mon,"anim");
	strcat(str_file,str2);
	dp = opendir(str_dir);
	if (dp != NULL) {
		while (((ep = readdir(dp))!= NULL) && (!trovato)) {
			if (!strcmp(str_file,ep->d_name)) {
				trovato =1;
				closedir(dp);
			}
		}			
		if (!trovato) {
			closedir(dp);
		}
	}
	free(str_dir);
	free(str_file);
	free(str2);
	return (trovato);
}

/***********************************/
/* TasksAdd_mon - aggiunge un task */
/***********************************/

void TasksAdd_mon (int tid, int pos)
{
	struct Task_mon * task;
	char *temp_str;
	struct pvmtaskinfo * ti;
	int n;
	XmString xmstr;
	
	temp_str = calloc(256,sizeof(char));
	
	/************************/
	/* alloca il nuovo nodo */
	/************************/
	
	task = calloc (1, sizeof (struct Task_mon));
	task -> tid = tid;

	/********************/
	/* inserimento nodo */
	/********************/
	
	
	if (procs_checked_in == 1 ){	
		
		/*inizializza la task list */
		TL_mon = calloc (1, sizeof (struct TaskList_mon));
	}
	

	
	/**** lista vuota ****/
	
	if (!TL_mon->first) {
		TL_mon->first = TL_mon->last = task;
		TL_mon->ntask_mon = 1;
	}

	/**** lista non vuota, inserimento in FONDO ****/
	
	else {	
		task -> next = NULL;
		
		TL_mon -> last -> next = task;
		TL_mon -> last = task;

		TL_mon -> ntask_mon++;
	}
	
	/*********************************/
	/* Inserisce il nodo nella lista */
	/*********************************/
	
	
	pvm_tasks(tid, &n, &ti); 
	
	sprintf(temp_str,"%3d  %6x  %s", pos, tid, ti->ti_a_out);
	xmstr = XmStringCreateLtoR(temp_str, DEFSET);
	XmListAddItem(tasklist, xmstr, 0);
	
	XmStringFree(xmstr);
	free(temp_str);
}
	
	
