/*****************************************************************************\
*                                                                             *
*  modMake.c - Make "parallelo"			                                      *
*                                                                             *
\*****************************************************************************/

/*
 * 
 *
 *               WAMM: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, M. Cosso, G. Faieta, M. Formica, 
 *                      D. Laforenza, M. Nicosia 
 *                   (C) 1997 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * We want thanks Brad Topol of the Georgia Institute of Technology, Atlanta.
 */


#include "modMake.h"

/***************/
/* Definizioni */
/***************/

/*********************/
/* strutture private */
/*********************/

struct MakeList {
	struct Make * first;	/* primo make della lista */
	struct Make * last;		/* ultimo make della lista */
	int nmake;				/* numero di make */
};

struct Make {
	struct Make * prev;		/* make precedente nella lista */
	struct Make * next;		/* make successivo nella lista */
	char * host;			/* indirizzo dell'host */
	int tid;				/* tid PVMMaker */
	int zombie;				/* 1 se make terminato */
	char * status;			/* ultimo messaggio inviato dal PVMMaker */
	int res;				/* risultato dell'ultimo comando */
	char * out;				/* output maker */
	Widget Woutshell;		/* finestra output */
	Widget Wouttext;
};

/*******************************/
/* variabili globali (private) */
/*******************************/

static struct MakeList * ML;	/* la lista dei make */

static Widget Wtoplevel;		/* toplevel shell */
static Widget Wmakeshell;		/* finestra make (0 = finestra non aperta) */
static Widget Wmakelist;		/* lista make */

static int stopmake;			/* se 1, interrompe l'avvio dei PVMMaker */

/***********/
/* globali */
/***********/

extern char * sys_errlist[];	/* messaggi di errore */

/********************/
/* Funzioni private */
/********************/

static Boolean			MakeDialog (Widget shell, char **, char **);
static struct Make *	FindMakeByTid (int, int *);
static struct Make *	FindMakeByIndex (int);
static void 			WriteMakeEntry (struct Make *, char *);
static void				ChangeMakeStatus (struct Make *, char *, int);
static void				ChangeMakeTitle (struct Make *);
static void				OutputWindow (struct Make *);
static Boolean			CheckStop (Widget);

/*********************/
/* Callbacks private */
/*********************/

static void CloseCB (Widget, Widget *, int);
static void DirCB (Widget, int *, int dummy);
static void CleanupCB (Widget, int, int);
static void OutputCB (Widget, int, int);
static void KillCB (Widget, int, int);

/**************************************/
/* MakeInit - inizializzazione modulo */
/**************************************/

void MakeInit (Widget toplevel)
{
	
	Wtoplevel = toplevel;
	
	/****************************/
	/* Inizializza la make list */
	/****************************/
	
	ML = calloc (1, sizeof (struct MakeList));
		
	return;
}

/********************************/
/* MakeEnd - distruzione modulo */
/********************************/

void MakeEnd (void)
{
	/* XXX per ora nulla */
}

/******************************/
/* MakeWindow - finestra make */
/******************************/

void MakeWindow (void)
{
	Arg args[20];
	int n;
	Widget Wform, Wmenubar, Wmenupane;

	struct Make * make;
	char buffer[1000];

	/********************/
	/* finestra aperta? */
	/********************/
	
	if (Wmakeshell) {
		XMapRaised (XtDisplay(Wmakeshell), XtWindow(Wmakeshell));
		return;
	}

	/**********************/
	/* creazione finestra */
	/**********************/
	
	/**** shell ****/
	
	n = 0;
	XtSetArg (args[n], XmNtitle, "Make"); n++;
	XtSetArg (args[n], XmNdeleteResponse, XmDO_NOTHING); n++;
	XtSetArg (args[n], XmNwidth, 700); n++;
	XtSetArg (args[n], XmNheight, 300); n++;
	Wmakeshell = XtAppCreateShell ("Make", "Wamm", applicationShellWidgetClass,	XtDisplay(Wtoplevel), args, n);
	MotifProtectShell (Wmakeshell, (XtCallbackProc)CloseCB, &Wmakeshell);
	XtRealizeWidget (Wmakeshell);

    /**** Editres ****/
    
    XtAddEventHandler (Wmakeshell, (EventMask)0, True, _XEditResCheckMessages, NULL);

	/**** form ****/

	n = 0;
	Wform = XmCreateForm (Wmakeshell, "makeform", args, n);
	XtManageChild (Wform);
	
	/**** menubar ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	Wmenubar = XmCreateMenuBar (Wform, "makemenubar", args, n);
	XtManageChild (Wmenubar);

	/**** menu ****/
	
	Wmenupane = MotifCreateMenu (Wmenubar, "File", 'F', 0, 0);
	MotifAddMenuItem (Wmenupane, "Stop", 'S', (XtCallbackProc)KillCB, NULL);
	MotifAddMenuItem (Wmenupane, "Cleanup", 0, (XtCallbackProc)CleanupCB, NULL); 
	MotifAddMenuItem (Wmenupane, "Close", 'C', (XtCallbackProc)CloseCB, &Wmakeshell);

	Wmenupane = MotifCreateMenu (Wmenubar, "Hosts", 'H', 0, 0);
	MotifAddMenuItem (Wmenupane, "Stdout/Stderr", 'S', (XtCallbackProc)OutputCB, NULL);

	/**** lista ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wmenubar); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNselectionPolicy, XmEXTENDED_SELECT); n++;
	XtSetArg (args[n], XmNlistSizePolicy, XmRESIZE_IF_POSSIBLE); n++;
	XtSetArg (args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n++;
	Wmakelist = XmCreateScrolledList (Wform, "makelist", args, n);
	XtManageChild (Wmakelist);
	XtAddCallback (Wmakelist, XmNdefaultActionCallback, (XtCallbackProc)OutputCB, NULL);
	
	/**** inserimento make ****/

	make = ML->first;
	while (make) {

		WriteMakeEntry (make, buffer);
		XmListAddItem (Wmakelist, XmStringCreate(buffer, DEFSET), 0);
		
		make = make->next;
	}
}

/***********/
/* MakeAdd */
/***********/

void MakeAdd (Widget Wshell, int num, char ** hosts)
{
	int n;
	char buffer[1000];

	char * makedir;		/* argomenti make */
	char * makeargs;
	
	char olddir[100];		/* directory iniziale */
	char * crunchedfile;	/* nome directory compressa */
	char * crunchedbuf;		/* buffer directory compressa */
	int crunchedlen;		/* lunghezza di crunchedbuf */
	
	Widget Wwait;			/* wait per crunch directory */
	
	char * argv[3];			/* argv per PVMMaker */
	struct Make * make;		/* make */	
	int makertid;			/* tid maker */	
	char * errmsg;			/* messaggio di errore */
	int ack;				/* ack versione */
	
	/**** Resetta "stopmake" (puo` essere a 1 per via di compilazioni
	      precedenti ****/
	      
	stopmake = 0;
	
	/**** preleva gli argomenti ****/
	
	if (! MakeDialog(Wshell, &makedir, &makeargs)) return;

	/**************************/
	/* Preparazione directory */
	/**************************/

	/**** cambia directory ****/

	getcwd (olddir, 100);
	if (chdir(makedir)) {
		sprintf (buffer, "CD to directory %s failed:\n%s.", makedir, sys_errlist[errno]);
		MotifError (Wshell, "Error", buffer);
		free (makedir);
		free (makeargs);
		return;
	}
	
	/**** comprime la directory ****/
	
	/* NOTA: prima di eseguire la CompressDir occorre porre SIGCHLD al valore di
	   default, pena "disturbi" (No Child Processes) nell'esecuzione delle system
	   necessarie. */
	
	Wwait = MotifWait (Wshell, "Please wait...", "Running Tar+Compress.");
	signal (SIGCHLD, SIG_DFL);
	crunchedfile = CompressDir ("*", 0);
	signal (SIGCHLD, SIG_IGN);
	XtUnmanageChild (Wwait);
	XtDestroyWidget (Wwait);
	
	if (!crunchedfile) {
		sprintf (buffer, "Can't compress directory %s:\n%s.", makedir, sys_errlist[errno]);
		MotifError (Wshell, "Error", buffer);
		free (makedir);
		free (makeargs);
		return;
	}
	
	free (makedir);
	chdir (olddir);
	
	/**** carica il file prodotto ****/
	
	crunchedlen = ReadFile (crunchedfile, &crunchedbuf);
	unlink (crunchedfile);
	
	/**********************/
	/* Attivazione makers */
	/**********************/
	
	argv[0] = VER_MKR;
	argv[1] = makeargs;
	argv[2] = NULL;
	
	/**** apre la finestra ****/
	
	MakeWindow();
	
	for (n=0; n<num; n++) {
	
		/**** crea un entry nella lista (in FONDO) ****/
	
		make = calloc (1, sizeof (struct Make));
		make -> host = strdup (hosts[n]);
	
		if (!ML->first) {
			ML->first = ML->last = make;
			ML->nmake = 1;
		}
		
		else {
			make -> next = NULL;
			make -> prev = ML->last;
			
			ML -> last -> next = make;
			ML -> last = make;
			
			ML -> nmake ++;
		}
		
		/**** inserisce il make nella Wmakelist ****/
		
		if (Wmakeshell) {
			WriteMakeEntry (make, buffer);
			XmListAddItem (Wmakelist, XmStringCreate(buffer, DEFSET), 0);
			XmListSetBottomPos (Wmakelist, 0);
		}
				
		/**** lancia il maker (con cattura output) ****/
		
		ChangeMakeStatus (make, "spawning PVMMaker...", ML->nmake);
		
		pvm_setopt (PvmOutputTid, PVMTid());
		pvm_setopt (PvmOutputCode, T_MKR_OUT);
		pvm_spawn ("PVMMaker", (char **)argv, PvmTaskHost, hosts[n], 1, &makertid);
		pvm_setopt (PvmOutputTid, 0);
		
		if (makertid < 0) {
			errmsg = PVMError (makertid);
			sprintf (buffer, "can't spawn PVMMaker (%s).", errmsg);
			free (errmsg);
			make -> zombie = 1;
			ChangeMakeStatus (make, buffer, ML -> nmake);
			makertid = 0;
		}
				
		/**** attende l'ack ****/
		
		if (makertid > 0) {
			ChangeMakeStatus (make, "waiting ack...", ML->nmake);
			pvm_recv (makertid, T_MKR_ACK);
			pvm_unpackf ("%d", &ack);
			if (!ack) {
				make -> zombie = 1;
				ChangeMakeStatus (make, "PVMMaker has a wrong version", ML -> nmake);
				makertid = 0;
			}
		}
				
		if (!makertid) continue;
		make -> tid = makertid;
		
		/**** controllo eventi X ****/
		
		if (CheckStop(Wshell)) break;
		
		/**** invia la directory ****/
		
		ChangeMakeStatus (make, "sending directory...", ML -> nmake);
		
		pvm_packf ("%+ %d", PvmDataDefault, crunchedlen);
		pvm_pkbyte (crunchedbuf, crunchedlen, 1);
		pvm_send (makertid, T_MKR_DIR);
		
		/**** fine; arriveranno i messaggi dal PVMMaker ****/
		
		ChangeMakeStatus (make, "waiting for messages...", ML -> nmake);
		
		/**** controllo eventi X ****/
		
		if (CheckStop(Wshell)) break;
	}
	
	free (crunchedbuf);
	free (makeargs);
}

/**** funzione di controllo eventi X + stop ****/

Boolean CheckStop (Widget W)
{
	while (XtAppPending(XtWidgetToApplicationContext(W)))
		XtAppProcessEvent (XtWidgetToApplicationContext(W), XtIMAll);

	if (stopmake) return True;
	else return False;
}

/*********************************/
/* MakeMsg - messaggio dal maker */
/*********************************/

void MakeMsg (int tid, char * msg)
{
	struct Make * make;
	int pos;
	
	/* scarta i messaggi provenienti da processi "ignoti" oppure ufficialmente
	   morti (in questo caso possono essere messaggi ancora in viaggio sulla
	   rete quando il processo e` gia` terminato) */
	
	make = FindMakeByTid (tid, &pos);
	if (!make || make->zombie) return;
	
	ChangeMakeStatus (make, msg, pos+1);
}

/*******************************************/
/* MakeMsgEnd - messaggio finale dal maker */
/*******************************************/

void MakeMsgEnd (int tid, char * msg, int res)
{
	struct Make * make;
	int pos;
	
	/* scarta i messaggi provenienti da processi "ignoti" oppure ufficialmente
	   morti (in questo caso possono essere messaggi ancora in viaggio sulla
	   rete quando il processo e` gia` terminato) */
	
	make = FindMakeByTid (tid, &pos);
	if (!make || make->zombie) return;
	
	make->zombie = 1;
	make->res = res;
	ChangeMakeStatus (make, msg, pos+1);
	ChangeMakeTitle (make);
}

/*********************************/
/* MakeOutput - output dal maker */
/*********************************/

void MakeOutput (int tid, char * out)
{
	struct Make * make;
	int pos;
	
	/* scarta i messaggi provenienti da processi "ignoti" oppure ufficialmente
	   morti (in questo caso possono essere messaggi ancora in viaggio sulla
	   rete quando il processo e` gia` terminato) */
	
	make = FindMakeByTid (tid, &pos);
	if (!make || make->zombie) return;

	/**** inserisce la nuova stringa nel buffer ****/	

	if (!make->out) make->out = strdup (out);
	else {
		make->out = realloc (make->out, strlen(make->out)+strlen(out)+1);
		strcat (make->out, out);
	}
	
	/**** se la finestra di output e` aperta, aggiunge out ****/
	
	if (make->Woutshell) {
		XmTextInsert(make->Wouttext, XmTextGetLastPosition(make->Wouttext), out);
		XmTextShowPosition (make->Wouttext, XmTextGetLastPosition(make->Wouttext));
	}
}

/******************************************************************************/
/******************************************************************************/

/*******************************/
/* CloseCB - callback chiusura */
/*******************************/

void CloseCB (Widget w, Widget * wid, int dummy2)
{
	XtDestroyWidget (*wid);
	*wid = 0;
}

/************************************/
/* CleanupCB - elimina i make morti */
/************************************/

void CleanupCB (Widget w, int dummy1, int dummy2)
{
	struct Make * make, * tmp;
	int n = 0;
	
	make = ML->first;
	while (make) {
	
		tmp = make->next;
		
		if (make->zombie) {
		
			/***********************/
			/* svuota la struttura */
			/***********************/

			if (make->host) free (make->host);
			if (make->status) free (make->status);
			if (make->out) free (make->out);
			if (make->Woutshell) XtDestroyWidget (make->Woutshell);
		
			/*******************/
			/* Sgancia il nodo */
			/*******************/
	
			if (make->prev) make->prev->next = make->next;
			if (make->next) make->next->prev = make->prev;
	
			if (make == ML->first) ML->first =	make->next;
			if (make == ML->last) ML->last = make->prev;
		
			free (make);
			ML -> nmake--;
	
			/***********************************/
			/* Elimina il nodo dalla Wmakelist */
			/***********************************/
	
			XmListDeletePos (Wmakelist, n+1);
			n--;
		}
	
		make = tmp;	
		n++;
	}
}

/******************************/
/* OutputCB - finestre output */
/******************************/

void OutputCB (Widget w, int dummy1, int dummy2)
{
	int * makelist;
	int num, n;
	Boolean mem;
	struct Make * make;
		
	/******************************/
	/* Preleva i make dalla lista */
	/******************************/
	
	num = 0;
	mem = XmListGetSelectedPos (Wmakelist, &makelist, &num);

	/************/
	/* Finestre */
	/************/
	
	for (n=0; n<num; n++) {
		make = FindMakeByIndex (makelist[n]-1);
		OutputWindow (make);	
	}
	
	/***********************/
	/* Rilascia il vettore */
	/***********************/
	
	if (mem) free (makelist);			
}

/***************************************/
/* KillCB - interrompe la compilazione */
/***************************************/

void KillCB (Widget w, int dummy1, int dummy2)
{
	struct Make * make;
	int n;

	make = ML->first;
	n = 0;
	
	while (make) {	
		if (!make->zombie) {
			pvm_kill (make->tid);
			make->zombie = 1;
			ChangeMakeStatus (make, "killed!", n+1);
			ChangeMakeTitle (make);
		}
		n++;
		make = make->next;
	}
	
	/* impedisce l'attivazione di eventuali altri maker */
	
	stopmake = 1;
}

/******************************************************************************/
/******************************************************************************/

/**************/
/* MakeDialog */
/**************/

Boolean MakeDialog (Widget Wshell, char ** makedir, char ** makeargs)
{
	Arg args[20];
	int n;
	char buffer [80];
	
	Widget Wdialog;
	Widget Wlocalbutton, Wlocaltext;
	Widget Wremotelabel, Wremotetext;
	Widget Wargslabel, Wargstext;
	Widget Wsep, Wok, Wcancel;
	
	int btn;
	char * ptr;
	
	/**** Dialog ****/
	
	n = 0;
	XtSetArg (args[n], XmNdialogTitle, XmStringCreate("Make", DEFSET)); n++;
	XtSetArg (args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); n++;
	XtSetArg (args[n], XmNwidth, 600); n++;
	XtSetArg (args[n], XmNheight, 200); n++;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_GROW); n++;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	Wdialog = XmCreateFormDialog (Wshell, "makedialog", args, n);
	XtManageChild (Wdialog);

	/**** Directory locale ****/
	
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("Makefile...", DEFSET)); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNtopOffset, 10); n++;
	XtSetArg (args[n], XmNwidth, 200); n++;
	Wlocalbutton = XmCreatePushButton (Wdialog, "localbutton", args, n);
	XtManageChild (Wlocalbutton);
	XtAddCallback (Wlocalbutton, XmNactivateCallback, (XtCallbackProc)DirCB, &btn);
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, Wlocalbutton); n++;
	XtSetArg (args[n], XmNleftOffset, 20); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNtopOffset, 10); n++;
	Wlocaltext = XmCreateText (Wdialog, "localtext", args, n);
	XtManageChild (Wlocaltext);
	
	/**** Directory remota ****/
	
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("Remote Directory:", DEFSET)); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wlocalbutton); n++;
	XtSetArg (args[n], XmNtopOffset, 20); n++;
	XtSetArg (args[n], XmNwidth, 200); n++;
	Wremotelabel = XmCreateLabel (Wdialog, "remotelabel", args, n);
	XtManageChild (Wremotelabel);
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, Wlocaltext); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wlocalbutton); n++;
	XtSetArg (args[n], XmNtopOffset, 20); n++;
	XtSetArg (args[n], XmNvalue, "$PVM_ROOT/tmp"); n++;
	XtSetArg (args[n], XmNeditable, False); n++;
	XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
	Wremotetext = XmCreateText (Wdialog, "remotetext", args, n);
	XtManageChild (Wremotetext);
	
	/**** argomenti ****/
		
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("Args for make:", DEFSET)); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wremotelabel); n++;
	XtSetArg (args[n], XmNtopOffset, 20); n++;
	XtSetArg (args[n], XmNwidth, 200); n++;
	Wargslabel = XmCreateLabel (Wdialog, "argslabel", args, n);
	XtManageChild (Wargslabel);
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET); n++;
	XtSetArg (args[n], XmNleftWidget, Wlocaltext); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wremotelabel); n++;
	XtSetArg (args[n], XmNtopOffset, 20); n++;
	Wargstext = XmCreateText (Wdialog, "argstext", args, n);
	XtManageChild (Wargstext);
	
	/**** separatore ****/
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightOffset, 10); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wargslabel); n++;
	XtSetArg (args[n], XmNtopOffset, 30); n++;
	Wsep = XmCreateSeparator (Wdialog, "sep", args, n);
	XtManageChild (Wsep);
	
	/**** ok ****/
	
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("OK", DEFSET)); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 30); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wsep); n++;
	XtSetArg (args[n], XmNtopOffset, 10); n++;
	XtSetArg (args[n], XmNwidth, 80); n++;
	Wok = XmCreatePushButton (Wdialog, "ok", args, n);
	XtManageChild (Wok);
	XtAddCallback (Wok, XmNactivateCallback, (XtCallbackProc)OkCB, &btn);

	/**** cancel ****/

	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("Cancel", DEFSET)); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 50); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wsep); n++;
	XtSetArg (args[n], XmNtopOffset, 10); n++;
	XtSetArg (args[n], XmNwidth, 80); n++;
	Wcancel = XmCreatePushButton (Wdialog, "cancel", args, n);
	XtManageChild (Wcancel);
	XtAddCallback (Wcancel, XmNactivateCallback, (XtCallbackProc)CancelCB, &btn);
	
	/**** Chiusura = Cancel ****/
	
	MotifProtectShell (XtParent(Wdialog), (XtCallbackProc)CancelCB, &btn);

	/**** Main Loop ****/
	
	btn = 0;
	
	for (;;) {
		XtAppProcessEvent (XtWidgetToApplicationContext(Wdialog), XtIMAll);
		switch (btn) {
		
			case 0:		break;
			
			case -1:	/**** Cancella ****/
			
						XtUnmapWidget (XtParent(Wdialog));
						XtDestroyWidget (XtParent(Wdialog));
						return False;
					 	break;
			
			case 1:		/**** OK ****/
						
						btn = 0;
						
						*makedir = XmTextGetString (Wlocaltext);
						ptr = strstr (*makedir, "Makefile");
						
						/**** Cerca "Makefile" ****/
						
						if (!ptr || (strcmp(ptr, "Makefile"))) {
							XBell (XtDisplay(Wdialog), 0);
							free (*makedir);
							break;
						}
						
						/**** Risale alla directory (path completo) ****/
						
						if (*makedir[0] != '/') {
						
							/* e` un path relativo: recupera quello assoluto */
						
							getcwd (buffer, 80);
							sprintf (buffer+strlen(buffer), "/%s", *makedir);
							free (*makedir);
							*makedir = strdup (buffer);
							ptr = strstr (*makedir, "Makefile");
						}
						
						/**** Taglia "Makefile" ****/
						
						if (strcmp(*makedir, "/Makefile")) ptr--;
						*ptr = 0;
						
						/* Preleva gli argomenti ed esce */
						
						*makeargs = XmTextGetString (Wargstext);
						XtUnmapWidget (XtParent(Wdialog));
						XtDestroyWidget (XtParent(Wdialog));
						return True;
						break;
			
			case 2:		/**** Bottone Makefile ****/		
			
						btn = 0;
						
						*makedir = MotifAskFile (XtParent(Wdialog), "Select Makefile to use:", "Makefile");
						if (!*makedir) break;
						XmTextSetString (Wlocaltext, *makedir);
						XmTextShowPosition (Wlocaltext, XmTextGetLastPosition(Wlocaltext));
						free (*makedir);
						break;
		}
	}
}

/**** callback per la directory locale ****/

void DirCB (Widget W, int * btn, int dummy)
{
	*btn = 2;
}

/**************************************************/
/* OutputWindow (make) - mostra l'output del make */
/**************************************************/

void OutputWindow (struct Make * make)
{
	int n;
	Arg args[20];
	Widget Wshell, Wform, Wmenubar, Wmenupane, Wtext, Wbutton;
	
	if (make->Woutshell) {
		XMapRaised (XtDisplay(make->Woutshell), XtWindow(make->Woutshell));
		return;
	}

	/**** shell ****/

	n = 0;
	XtSetArg (args[n], XmNdeleteResponse, XmDO_NOTHING); n++;
	XtSetArg (args[n], XmNwidth, 700); n++;
	XtSetArg (args[n], XmNheight, 350); n++;
	Wshell = XtAppCreateShell ("outshell", "Wamm", applicationShellWidgetClass,	XtDisplay(Wtoplevel), args, n);
	make->Woutshell = Wshell;
	MotifProtectShell (Wshell, (XtCallbackProc)CloseCB, &make->Woutshell);
	XtRealizeWidget (Wshell);
	ChangeMakeTitle (make);

    /**** Editres ****/
    
    XtAddEventHandler (Wshell, (EventMask)0, True, _XEditResCheckMessages, NULL);

	/**** form ****/

	n = 0;
	Wform = XmCreateForm (Wshell, "outform", args, n);
	XtManageChild (Wform);
	
	/**** menubar ****/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	Wmenubar = XmCreateMenuBar (Wform, "outmenubar", args, n);
	XtManageChild (Wmenubar);

	/**** menu ****/
	
	Wmenupane = MotifCreateMenu (Wmenubar, "File", 'F', 0, 0);
	MotifAddMenuItem (Wmenupane, "Close", 'C', (XtCallbackProc)CloseCB, &make->Woutshell);
		
	/**** scrolled text ****/
	
	n = 0;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wmenubar); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomOffset, 45); n++;
	XtSetArg (args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
	XtSetArg (args[n], XmNeditable, False); n++;
	XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
	XtSetArg (args[n], XmNvalue, make->out); n++;
	Wtext = XmCreateScrolledText (Wform, "outtext", args, n);
	XtManageChild (Wtext);
	make -> Wouttext = Wtext;
	XmTextShowPosition (make->Wouttext, XmTextGetLastPosition(make->Wouttext));
	
	/**** OK ****/
	
	n = 0;
	XtSetArg (args[n], XmNlabelString, XmStringCreate ("OK", DEFSET)); n++;
	XtSetArg (args[n], XmNwidth, 80); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftOffset, 10); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomOffset, 5); n++;
	Wbutton = XmCreatePushButton (Wform, "outok", args, n);
	XtManageChild (Wbutton);
	XtAddCallback (Wbutton, XmNactivateCallback, (XtCallbackProc)CloseCB, &make->Woutshell);
}

/*****************/
/* FindMakeByTid */
/*****************/

struct Make * FindMakeByTid (int tid, int * pos)
{
	struct Make * make;
	
	make = ML->first;
	*pos = 0;
	
	while (make && (make->tid != tid)) {
		make = make->next;
		(*pos)++;
	}
	return make;
}

/*******************/
/* FindMakeByIndex */
/*******************/

struct Make * FindMakeByIndex (int i)
{
	struct Make * make;
	int n;
	
	if (i >= ML->nmake) return NULL;
	
	make = ML->first;
	for (n=0; n<i; n++) make = make->next;
	return make;
}

/******************/
/* WriteMakeEntry */
/******************/

void WriteMakeEntry (struct Make * make, char * buffer)
{
	sprintf (buffer, "Host: %-30s", make->host);
	if (!make->zombie) sprintf (buffer+strlen(buffer), "%-20s", "(running)");
	else sprintf (buffer+strlen(buffer), "%-20s", "(completed)");
	if (make->status) sprintf (buffer+strlen(buffer), " Status: %s", make->status);
	else sprintf (buffer+strlen(buffer), " Status: %s", "setup");	
}

/********************/
/* ChangeMakeStatus */
/********************/

void ChangeMakeStatus (struct Make * make, char * status, int pos)
{
	char buffer[1000];
	XmString newitem;
	
	if (make->status) free (make->status);
	make->status = strdup (status);
	
	if (!Wmakeshell) return;
	
	WriteMakeEntry (make, buffer);
	newitem = XmStringCreate (buffer, DEFSET);
	XmListReplaceItemsPos (Wmakelist, &newitem, 1, pos);
	XmUpdateDisplay (Wmakelist);
}

/********************/
/* ChangeMakeTitle */
/********************/

void ChangeMakeTitle (struct Make * make)
{
	char buffer[1000];
	Arg args[10];
	int n;
	
	if (!make->Woutshell) return;

	sprintf (buffer, "Output of make on %s", make->host);
	if (!make->zombie) sprintf (buffer+strlen(buffer), " - running");
	else sprintf (buffer+strlen(buffer), " - completed with exit code %d", make->res);

	n = 0;
	XtSetArg (args[n], XmNtitle, buffer); n++;
	XtSetValues (make->Woutshell, args, n);
}
