/*****************************************************************************\
*                                                                             *
*  PVMHoster.c - hoster for Wamm					                          *
*                                                                             *
\*****************************************************************************/

/*
 * 
 *
 *               WAMM: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, M. Cosso, G. Faieta, M. Formica, 
 *                      D. Laforenza, M. Nicosia 
 *                   (C) 1997 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *
 * We want thanks Brad Topol of the Georgia Institute of Technology, Atlanta.
 */

 
#include <stdio.h>			/* I/O */
#include <sys/types.h>		/* typedefs */
#include <stdlib.h>			/* lots of functions */
#include <unistd.h>			/* lots of functions */
#include <errno.h>			/* error codes */
#include <signal.h>			/* signals */
#include <string.h>			/* string functions */
#include <ctype.h>			/* isprintf */
#include <pwd.h>			/* getpwuid */
#include <netdb.h>			/* getservbyname */
#include <sys/socket.h>		/* socket functions */
#include <pvm3.h>
#include <pvmsdpro.h>

#ifdef _AIX
	#include <sys/select.h>		/* FD_* macros */
#endif


#include "tags.h"
#include "version.h"

/***************/
/* Definizioni */
/***************/

#define DELAY		100000			/* ritardo (us) tra un controllo e l'altro */
#define CHECK		10000000		/* us tra un controllo sul master e l'altro */
#define	RSHTIMEOUT	60				/* secondi timeout per rsh */
#define	RSHNPLL		5				/* numero max attivazioni parallele */

/***************************/
/* Operazioni con le liste */
/***************************/

#define	LISTPUTAFTER(o,n,f,r) \
	{ (n)->f=(o)->f; (n)->r=o; (o)->f->r=n; (o)->f=n; }
#define	LISTPUTBEFORE(o,n,f,r) \
	{ (n)->r=(o)->r; (n)->f=o; (o)->r->f=n; (o)->r=n; }
#define	LISTDELETE(e,f,r) \
	{ (e)->f->r=(e)->r; (e)->r->f=(e)->f; (e)->r=(e)->f=0; }

/********************************/
/* Aritmetica con i timeval (!) */
/********************************/

/* tv = 0 */

#define	TVCLEAR(tvp)	((tvp)->tv_sec = (tvp)->tv_usec = 0)

/* tv != 0 */

#define	TVISSET(tvp)	((tvp)->tv_sec || (tvp)->tv_usec)

/* xtv < ytv */

#define	TVXLTY(xtv, ytv) \
	((xtv)->tv_sec < (ytv)->tv_sec || \
		((xtv)->tv_sec == (ytv)->tv_sec && (xtv)->tv_usec < (ytv)->tv_usec))

/* ztv = xtv+ytv */

#define	TVXADDY(ztv, xtv, ytv)	\
	if (((ztv)->tv_usec = (xtv)->tv_usec + (ytv)->tv_usec) < 1000000) {	\
		(ztv)->tv_sec = (xtv)->tv_sec + (ytv)->tv_sec;	\
	} else {	\
		(ztv)->tv_usec -= 1000000;	\
		(ztv)->tv_sec = (xtv)->tv_sec + (ytv)->tv_sec + 1;	\
	}

/* ztv = xtv-ytv */

#define	TVXSUBY(ztv, xtv, ytv)	\
	if ((xtv)->tv_usec >= (ytv)->tv_usec) {	\
		(ztv)->tv_sec = (xtv)->tv_sec - (ytv)->tv_sec;	\
		(ztv)->tv_usec = (xtv)->tv_usec - (ytv)->tv_usec;	\
	} else {	\
		(ztv)->tv_sec = (xtv)->tv_sec - (ytv)->tv_sec - 1;	\
		(ztv)->tv_usec = (xtv)->tv_usec + 1000000 - (ytv)->tv_usec;	\
	}

/*************/
/* Strutture */
/*************/

struct Host {
	int h_tid;			/* tid demone */
	char * h_name;		/* indirizzo */
	char * h_login;		/* nome utente */
	char * h_sopts;		/* opzioni startup (so=xx): pw oppure ms */
	int h_flag;			/* opzioni startup (HST_PASSWORD, HST_MANUAL) */
	char * h_cmd;		/* ? */
	char * h_result;	/* risultato add */
};

#define	HST_PASSWORD	1		/* usa rexec (so=pw) */
#define	HST_MANUAL		2		/* startup manuale (so=ms) */

struct Slot {
	struct Slot *s_link, *s_rlink;		/* successivo, precedente */
	struct Host *s_Host;				/* host */
	struct timeval s_bail;				/* timeout */
	int s_rfd, s_wfd, s_efd;			/* stdout/in/err figlio */
	int s_pid;							/* pid figlio */
	char s_buf[256];					/* buffer startup manuale */
	int s_len;							/* lunghezza di s_buf */
};

/***********/
/* Globali */
/***********/

char *username = 0;

struct Slot slots[RSHNPLL+2];	/* slots */
struct Slot *slfree = 0;		/* lista di slot liberi */

/*************/
/* Prototipi */
/*************/

void	Hoster (void);
int		PlStartup (int, struct Host **);
void 	CloseSlot (struct Slot *);
int		Phase1 (struct Slot *);

extern void	pvmlogperror(char *);
extern void	pvmbailout(int);
extern int	getdtablesize (void);

/********/
/* Main */
/********/

void main (int argc, char ** argv)
{
	int parenttid;			/* tid interfaccia */
	int ver;				/* 1=versione ok */
	struct timeval tv;		/* timeval per pvm_trecv */
	int bufid, tag, tid;	/* informazioni sul messaggio ricevuto */
	int check;				/* contatore controllo master */
	struct passwd * psw;	/* struttura password utente locale */
	int uid;				/* userid utente locale */

	/****************************/
	/* Informazioni sull'utente */
	/****************************/

	uid = getuid ();
	if (uid == -1) {
		fprintf (stderr, "PVMHoster: getuid failed!\n");
		exit(1);
	}

	psw = getpwuid (uid);
	if (!psw) {
		fprintf (stderr, "PVMHoster: getpwuid failed!\n");
	}
	
	username = strdup (psw->pw_name);
	endpwent();

	/****************************/
	/* Controllo versione e ack */
	/****************************/
	
	pvm_mytid();
	parenttid = pvm_parent();
	
	ver = (!strcmp(VER_HSTR,argv[1]));
	
	pvm_packf ("%+ %d", PvmDataDefault, ver);
	pvm_send (parenttid, T_HSTR_ACK);
	
	if (!ver) {
		pvm_exit();
		exit(1);
	}

	/************************/
	/* Registrazione hoster */
	/************************/
	
	pvm_setopt(PvmResvTids, 1);
	pvm_reg_hoster();
	
	signal (SIGTERM, SIG_IGN);
	signal (SIGINT, SIG_IGN);
	signal (SIGCHLD, SIG_IGN);		/* niente zombies! */

	/*************/
	/* Main Loop */
	/*************/

	tv.tv_sec = 0;
	tv.tv_usec = DELAY;
	
	check = 0;
	
	for (;;) {
	
		/**** attende un messaggio o il timeout ****/
	
		bufid = pvm_trecv (-1, -1, &tv);
		
		/**** controllo master ****/
		
		check += DELAY;
		if (check >= CHECK) {

			if (pvm_pstat (parenttid) < 0 ) {
				fprintf (stderr, "PVMHoster: exit (can't find interface!)\n");
				fflush (stderr);
				pvm_exit ();
				exit (1);
			}
			
			check = 0;
		}

		/**** messaggi ****/

		if (bufid<=0) continue;
		pvm_bufinfo (bufid, NULL, &tag, &tid);
		
		switch (tag) {
				
			/*************/
			/* SM_STHOST */
			/*************/
			
			case SM_STHOST:		Hoster();
								break;
								
			default:			break;
		}
	}	

	pvm_exit();
	exit(0);
}

/**********/
/* Hoster */
/**********/

void Hoster (void)
{
	int num;
	int i;
	struct Host **hostlist = NULL;
	struct Host *hp;
	char *p;
	char sopts[64];
	char lognam[256];
	char cmd[512];
	int fromtid;
	int wid;

	/***********************************/
	/* preleva il messaggio di startup */
	/***********************************/

	/* Formato del messaggio:
	 *
	 *	- numero host da inserire
	 *	- id richiesta (wid)
	 *
	 *	Seguono, per ogni host:
	 *
	 *	- tid da usare per il pvmd
	 *	- sopts (HST_PASSWORD o HST_MANUAL)
	 *	- logname ("utente" oppure "utente@indirizzo")
	 *	- cmd (?)
	 */
	 
	pvm_bufinfo(pvm_getrbuf(), (int *)0, (int *)0, &fromtid);
	pvm_unpackf("%d", &num);
	wid = pvm_getmwid(pvm_getrbuf());

	if (num > 0) {

		hostlist = calloc(num, sizeof (struct Host *));
		
		for (i = 0; i < num; i++) {
			
			hp = calloc (1, sizeof (struct Host));
			hostlist[i] = hp;
			
			pvm_unpackf("%d %s %s %s", &hp->h_tid, sopts, lognam, cmd);
			
			hp->h_flag = 0;
			hp->h_result = 0;
			hp->h_sopts = strdup(sopts);
			hp->h_login = strdup(lognam);
			hp->h_cmd = strdup(cmd);
			if ((p = strchr(hp->h_login, '@'))) {
				
				/**** nome@indirizzo: spezza le due parti ****/
				
				hp->h_name = strdup(p + 1);
				*p = 0;
				p = strdup(hp->h_login);
				free (hp->h_login);
				hp->h_login = p;

			} else {
				hp->h_name = hp->h_login;
				hp->h_login = 0;
			}
			
			if (!strcmp(hp->h_sopts, "pw"))
				hp->h_flag |= HST_PASSWORD;
			if (!strcmp(hp->h_sopts, "ms"))
				hp->h_flag |= HST_MANUAL;
		}
	}

	/**** Lancia i demoni ****/
	
	PlStartup(num, hostlist);

	/**** Spedisce i risultati ****/

	pvm_packf("%+ %d", PvmDataFoo, num);
	for (i = 0; i < num; i++) {
		pvm_packf("%d", hostlist[i]->h_tid);
		pvm_packf("%s", hostlist[i]->h_result ? hostlist[i]->h_result : "PvmDSysErr");
	}

	pvm_setmwid(pvm_getsbuf(), wid);
	pvm_send(fromtid, SM_STHOSTACK);

	return;
}

/*************/
/* PlStartup */
/*************/

int PlStartup (int num, struct Host ** hostlist)
{
	int nxth = 0;						/* prossimo host da inserire */
	struct Slot *slact = 0;				/* lista slot attivi */
	struct Host *hp;
	struct Slot *sp, *sp2;
	struct timeval tnow;
	struct timeval tout;
	struct fd_set rfds;
	int nfds;
	int i;
	int n;
	char *p;
	char ebuf[256];						/* per leggere stderr */

	/************************************/
	/* Inizializza le liste degli slot  */
	/************************************/

	slfree = &slots[RSHNPLL+1];
	slfree->s_link = slfree->s_rlink = slfree;
	slact = &slots[RSHNPLL];
	slact->s_link = slact->s_rlink = slact;
	for (i = RSHNPLL; i-- > 0; )
		LISTPUTAFTER(slfree, &slots[i], s_link, s_rlink);

	/*********************************************************/
	/* Ripete finche' tutti gli host non sono stati inseriti */
	/*********************************************************/

	for (;;) {

		for (;;) {

			/* se ci sono ancora host da aggiungere attende finche' non
			   trova uno slot libero */

			if (slfree->s_link != slfree && nxth < num) hp = hostlist[nxth++];
			else break;

			/* trovato: lo toglie dalla lista di quelli liberi */

			sp = slfree->s_link;
			LISTDELETE(sp, s_link, s_rlink);
			sp->s_Host = hp;
			sp->s_len = 0;

			/* lancia rsh/rexec/manuale */

			Phase1(sp);

			/* Nel caso di rexec / partenza manuale / errore lo slot e` subito
			   disponibile. Nel caso di rsh viene lanciato un processo figlio
			   e lo slot deve essere inserito nella lista degli slot attivi
			   in attesa di conoscere l'esito. */

			if (hp->h_result) LISTPUTBEFORE(slfree, sp, s_link, s_rlink)
			else {

				LISTPUTBEFORE(slact, sp, s_link, s_rlink);
				
				/* Per l'rsh viene preparato il timeout. Viene fissato per
				   <ora corrente> + RSHTIMEOUT e inserito in sp->s_bail */
				
				gettimeofday(&sp->s_bail, (struct timezone*)0);
				tout.tv_sec = RSHTIMEOUT;
				tout.tv_usec = 0;
				TVXADDY(&sp->s_bail, &sp->s_bail, &tout)
			}
		}

		/* Si arriva qui quando gli slot sono finiti o quando non ci sono piu`
		   nodi da attivare. Se non risultano slot attivi siamo nel secondo
		   caso e si puo` terminare. */

		if (slact->s_link == slact) break;

		/****************************************/
		/* Controllo host attivi (fermi su rsh) */
		/****************************************/

		FD_ZERO(&rfds);
		nfds = 0;
		
		TVCLEAR(&tout);
		gettimeofday(&tnow, (struct timezone*)0);
		
		for (sp = slact->s_link; sp != slact; sp = sp->s_link) {
			
			/**** se uno slot e` andato in timeout, segnala PvmCantStart ****/
			
			if (TVXLTY(&sp->s_bail, &tnow)) {
				fprintf(stderr, "PlStartup() %s timed out after %d secs\n",
						sp->s_Host->h_name, RSHTIMEOUT);
				sp->s_Host->h_result = strdup("PvmCantStart");
				sp2 = sp->s_rlink;
				kill (sp->s_pid, SIGKILL);
				CloseSlot(sp);
				sp = sp2;
				continue;
			}

			if (!TVISSET(&tout) || TVXLTY(&sp->s_bail, &tout))
				tout = sp->s_bail;
			
			if (sp->s_rfd >= 0)
				FD_SET(sp->s_rfd, &rfds);
			if (sp->s_rfd > nfds)
				nfds = sp->s_rfd;
			if (sp->s_efd >= 0)
				FD_SET(sp->s_efd, &rfds);
			if (sp->s_efd > nfds)
				nfds = sp->s_efd;
		}

		/**** Se tutti gli rsh sono morti, si puo` terminare ****/

		if (slact->s_link == slact) break;

		nfds++;

		if (TVXLTY(&tnow, &tout)) TVXSUBY(&tout, &tout, &tnow)
		else TVCLEAR(&tout);

		/**** Attende il timeout o stdout/stderr dai figli ****/

		if ((n = select(nfds, (int *)&rfds, 0, 0, &tout)) == -1) {
			if (errno != EINTR) {
				pvmlogperror("PlStartup() select");
				pvmbailout(0);
			}
		}

		if (n < 1) {
			if (n == -1 && errno != EINTR) {
				pvmlogperror("PlStartup() select");
				pvmbailout(0);
			}
			continue;
		}

		/**** Controlla stdout e stderr ****/

		for (sp = slact->s_link; sp != slact; sp = sp->s_link) {

			/* stdout: e` il messaggio prodotto dal pvmd. Lo legge nel buffer */
			
			if (sp->s_rfd >= 0 && FD_ISSET(sp->s_rfd, &rfds)) {
				
				n = read(sp->s_rfd, sp->s_buf + sp->s_len, sizeof(sp->s_buf) - sp->s_len);
				if (n > 0) {
					
					sp->s_len += n;
					if (sp->s_len >= sizeof(sp->s_buf)) {
						fprintf(stderr, "PlStartup() pvmd@%s: big read\n",
								sp->s_Host->h_name);
						sp->s_Host->h_result = strdup("PvmCantStart");
					}
					sp->s_buf[sp->s_len] = 0;
					if ((p = strchr(sp->s_buf + sp->s_len - n, '\n'))) {
						*p = 0;
						sp->s_Host->h_result = strdup(sp->s_buf);
					}

				} else {
					if (n) {
						fprintf(stderr, "PlStartup() pvmd@%s\n",
								sp->s_Host->h_name);
						perror("");
					} else {
						fprintf(stderr, "PlStartup() pvmd@%s: EOF\n",
								sp->s_Host->h_name);
					}
					sp->s_Host->h_result = strdup("PvmCantStart");
				}

				/* Se e` stato ottenuto un risultato, questo rsh e` concluso */

				if (sp->s_Host->h_result) {
					sp2 = sp->s_rlink;
					CloseSlot(sp);
					sp = sp2;
					continue;
				}
			}

			/**** Risposta su stderr: registra sul file di log ****/
			
			if (sp->s_efd >= 0 && FD_ISSET(sp->s_efd, &rfds)) {
				
				if ((n = read(sp->s_efd, ebuf, sizeof(ebuf)-1)) > 0) {
					
					char *p = ebuf, c;

					ebuf[n] = 0;
					fprintf(stderr, "pvmd@%s: ", sp->s_Host->h_name);
					
					while ((c = *p++ & 0x7f)) {
						if (isprint(c))
							fputc(c, stderr);

						else {
							fputc('^', stderr);
							fputc((c + '@') & 0x7f, stderr);
						}
					}
					fputc('\n', stderr);

				} else {
					(void)close(sp->s_efd);
					sp->s_efd = -1;
				}
			}
		}
	}
	
	return 0;
}

/**********/
/* Phase1 */
/**********/

int Phase1 (struct Slot * sp)
{
	struct Host *hp;
	char *hn;
	char *av[16];						/* argomenti rsh */
	int ac;
	char buf[512];
	int pid = -1;						/* pid di rsh */
	char *p;

	char buffer [1024];					/* buffer password */

	struct servent *se;
	static u_short execport = 0;

	/**** Se e` la prima volta, preleva la porta per rexec ****/

	if (!execport) {
		if (!(se = getservbyname("exec", "tcp"))) {
			fprintf(stderr, "PVMHoster: getservbyname() exec failed!\n");
			pvmbailout(0);
		}
		execport = se->s_port;
		endservent();
	}

	hp = sp->s_Host;
	hn = hp->h_name;
	sp->s_rfd = sp->s_wfd = sp->s_efd = -1;

	/*******************/
	/* Startup manuale */
	/*******************/
	
	if (hp->h_flag & HST_MANUAL) {
		fprintf(stderr, "*** Manual startup ***\n");
		fprintf(stderr, "Login to \"%s\" and type:\n", hn);
		fprintf(stderr, "%s\n", hp->h_cmd);

		fprintf(stderr, "Type response: ");
		fflush(stderr);
		if (!(fgets(buf, sizeof(buf), stdin))) {
			fprintf(stderr, "host %s read error\n", hn);
			goto oops;
		}
		p = buf + strlen(buf) - 1;
		if (*p == '\n')
			*p = 0;
		hp->h_result = strdup(buf);
		fprintf(stderr, "Thanks\n");
		fflush(stderr);
		return 0;
	}

	/*******************/
	/* Startup con RSH */
	/*******************/

	if (!(hp->h_flag & HST_PASSWORD)) {
	
		int wpfd[2], rpfd[2], epfd[2];
		int i;

#ifdef	IMA_TITN
		if (socketpair(AF_UNIX, SOCK_STREAM, 0, wpfd) == -1
		|| socketpair(AF_UNIX, SOCK_STREAM, 0, rpfd) == -1
		|| socketpair(AF_UNIX, SOCK_STREAM, 0, epfd) == -1) {
			pvmlogperror("phase1() socketpair");
			goto oops;
		}
#else
		if (pipe(wpfd) == -1 || pipe(rpfd) == -1 || pipe(epfd) == -1) {
			pvmlogperror("phase1() pipe");
			goto oops;
		}
#endif

		if ((pid = fork()) == -1) {
			pvmlogperror("Phase1() fork");
			pvmbailout(0);
		}
		
		if (!pid) {
		
			/**** Figlio ****/
		
			(void)dup2(wpfd[0], 0);		/* manda in,out ed err al padre */
			(void)dup2(rpfd[1], 1);
			(void)dup2(epfd[1], 2);
			
			for (i = getdtablesize(); --i > 2; ) close(i);
			
			/**** prepara gli argomenti ****/
			
			ac = 0;
			av[ac++] = RSHPATH;
			av[ac++] = hn;
			
			if (hp->h_login) {
				av[ac++] = "-l";
				av[ac++] = hp->h_login;
			}
			
			av[ac++] = hp->h_cmd;
			av[ac++] = 0;
			
			execvp(av[0], av);
			fputs("Phase1() execvp failed\n", stderr);
			fflush(stderr);
			_exit(1);
		}
		
		/**** Padre ****/
		
		close(wpfd[0]);
		close(rpfd[1]);
		close(epfd[1]);
		
		sp->s_pid = pid;
		sp->s_wfd = wpfd[1];
		sp->s_rfd = rpfd[0];
		sp->s_efd = epfd[0];

	} else {
	
		/*********************/
		/* Startup con REXEC */
		/*********************/
		
		/**** Chiede la password all'interfaccia ****/
		
		pvm_packf ("%+ %s", PvmDataDefault, hn);
		pvm_send (pvm_parent(), T_HSTR_PSW);
		
		/**** Attende risposta ****/
		
		pvm_recv (pvm_parent(), T_HSTR_PSW);
		pvm_upkstr (buffer);
				
		if ((sp->s_wfd = sp->s_rfd = rexec(&hn, execport,
				(hp->h_login ? hp->h_login : username),
				buffer, hp->h_cmd, &sp->s_efd))
		== -1) {
			fprintf(stderr, "Phase1() rexec failed for host %s\n", hn);
			goto oops;
		}
	}
	
	return 0;

oops:
	hp->h_result = strdup("PvmCantStart");

	if (sp->s_wfd != -1)
		close(sp->s_wfd);
	if (sp->s_rfd != -1)
		close(sp->s_rfd);
	if (sp->s_efd != -1)
		close(sp->s_efd);
	sp->s_wfd = sp->s_rfd = sp->s_efd = -1;
	return 1;
}

/*************/
/* CloseSlot */
/*************/

void CloseSlot (struct Slot * sp)
{

	/**** chiude stdout, stdin, stderr ****/

	if (sp->s_wfd != -1)
		(void)close(sp->s_wfd);
	
	if (sp->s_rfd != -1)
		(void)close(sp->s_rfd);
	
	if (sp->s_efd != -1)
		(void)close(sp->s_efd);
	
	LISTDELETE(sp, s_link, s_rlink);
	LISTPUTBEFORE(slfree, sp, s_link, s_rlink);
}
