/*****************************************************************************\
*                                                                             *
*  modParser.c - parser file di configurazione                                *
*  1.0 - 17/6/95                                                              *
*                                                                             *
\*****************************************************************************/

/*
 *
 *          WAMM version 1.0: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, G. Faieta, M. Formica, D. Laforenza
 *                   (C) 1995 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

#include "modParser.h"

/***********************/
/* Definizioni private */
/***********************/

#define PARSER_NOSTAT		1		/* stat fallita */
#define PARSER_NOOPEN		2		/* fopen fallita */
#define PARSER_NOREAD		3		/* fread fallita */
#define PARSER_DUPNODE		4		/* nodo gia` dichiarato */
#define PARSER_WRONGCLASS	5		/* non e` WAN, MAN, LAN o HOST */
#define PARSER_SYNTAXERR	6		/* errore di sintassi */
#define PARSER_BADEND		7		/* } mancante */
#define PARSER_BADSTART		8		/* { mancante */
#define PARSER_NOCHILDREN	9		/* rete senza figli */
#define PARSER_NONODE		10		/* struttura non dichiarata */
#define PARSER_NOARG		11		/* argomento non specificato */
#define PARSER_NOWAN		12		/* WAN non dichiarata */
#define PARSER_TOOMANYWAN	13		/* WAN dichiarata piu` volte */
#define PARSER_NOADDR		14		/* indirizzo non dichiarato */
#define PARSER_NOARCH		15		/* architettura non dichiarata */
#define PARSER_NOPARENT		16		/* struttura senza padre */

/**************************/
/* Strutture dati private */
/**************************/

struct ParserData {
	unsigned char * filename;      /* Nome del network file */
    unsigned char * buffer;        /* Buffer con il file di configurazione */
    unsigned char * ptr;           /* Puntatore corrente nel buffer */
    unsigned char * word;          /* Copia temporanea della parola corrente */
    int line;                      /* Numero di linea corrente */
};

/*******************************/
/* variabili globali (private) */
/*******************************/

struct ParserData * PD;

/********************/
/* Funzioni private */
/********************/

static unsigned char * ParserHeader (unsigned char *, unsigned char **, int *);
static unsigned char * ParserTryChild (unsigned char *, unsigned char *, unsigned char **, int *, int *);
static unsigned char * ParserTryArg1 (unsigned char *, unsigned char *, unsigned char **);
static unsigned char * ParserTryArg2 (unsigned char *, unsigned char *, unsigned char **, unsigned char **);
static void ParserAddCmd (struct NetworkObj *, unsigned char *, unsigned char *, int);

static unsigned char * ParserNextWord (void);
static unsigned char * ParserNextChar (void);

static unsigned char * ParserError (int, unsigned char *);

/**************/
/* ParserInit */
/**************/

unsigned char * ParserInit (unsigned char * filename)
{
	FILE * fh;
	struct stat * st;
	int size;

	/*********************************/
	/* semplice reinizializzazione ? */
	/*********************************/
	
	if (!filename) {	
		PD->ptr = PD->buffer;
		PD->line = 1;
		return NULL;
	}

	/***************************************************/
	/* prima inizializzazione: tenta di aprire il file */
	/***************************************************/

	if (!PD) PD = calloc (1, sizeof (struct ParserData));
	else if (PD->filename) free (PD->filename);
	PD->filename = strdup (filename);
	
	/* alloca il buffer */
	
	st = calloc (1, sizeof(struct stat));
	if (stat(filename, st)) return ParserError (PARSER_NOSTAT, NULL);
	size = st->st_size;
	free (st);
	PD -> buffer = calloc (size+1, 1);
	
	/* carica il file */
	
	fh = fopen (filename, "r");
	if (!fh) return ParserError (PARSER_NOOPEN, NULL);
	if (fread (PD->buffer, 1, size, fh) != size) return ParserError (PARSER_NOREAD, NULL);
	fclose (fh);
	
	/* inizializzazione strutture */
	
	PD->ptr = PD->buffer;
	PD->line = 1;

	return NULL;
}

/*************/
/* ParserEnd */
/*************/

void ParserEnd (void)
{
	if (PD->buffer) free (PD->buffer);
	if (PD->filename) free (PD->filename);
	free (PD);
}

/**********/
/* Parser */
/**********/

unsigned char * Parser (void)
{
	unsigned char * res;		/* risultato o messaggio di errore */
	unsigned char * word;		/* parola corrente */
	unsigned char * tmp;		/* parola temporanea */
	unsigned char * name;		/* nome della struttura */
	int type;					/* tipo della struttura */
	unsigned char * cmdname;	/* nome comando (per NET_HOST) */
	unsigned char * cmd;		/* comando (per NET_HOST) */
	int x, y;					/* posizione bottone (NET_WAN, NET_LAN) */
	struct NetworkObj * no;		/* struttura corrente */
	struct NetworkObj * WAN;	/* WAN (0 = non trovata) */
	struct NetworkObj *childno; /* NetworkObject figlio */
	int numchildren;			/* numero di figli */
	unsigned char * pixpath;	/* path per le pixmaps (opzionale) */
	unsigned char buffer[80];	/* buffer all-purpose */

	/*****************/
    /* Prima passata */
    /*****************/
    
	/*  Non appena si incontra la definizione di un nodo si alloca la
	 *	struttura relativa (se non esiste gia`). Scandendo il nodo vengono
	 *	inseriti nell'oggetto i valori incontrati. Si contano i figli. Al
	 *	termine si alloca il vettore dei figli.
	 */

	WAN = NULL;			/* la WAN non esiste ancora */
	pixpath = NULL;		/* nessun path predefinito per le pixmaps */

    while ((word = ParserNextWord ())) {
		
		/***************************/
		/* direttive "single-line" */
		/***************************/
		
		/**** path pixmaps ****/
		
		res = ParserTryArg1 (word, "PIXPATH", &tmp);
		if (res>0) return (res);
        if (!res && tmp) {
        	if (pixpath) {
        		free (pixpath);
        		pixpath = NULL;
        	}
        	if (strcmp(tmp, "---")) pixpath = tmp;
        	word = ParserNextWord();
        }
		
		/*************/
		/* strutture */
		/*************/
		
		/**** preleva il nome della struttura, controlla che non esista gia` ****/

		res = ParserHeader (word, &name, &type);
		if (res > 0) return res;
		if (NetworkObjFind (name, type)) return ParserError (PARSER_DUPNODE, NULL);
        
        /**** alloca l'oggetto ****/
        
        no = NetworkObjAdd (name, type);      

		/**** Scansione interna della struttura ****/
		
		numchildren = 0; 
		
 	    while ((word = ParserNextWord()) && (strcmp(word, "}"))) {
         
		switch (type) {
            
            	case NET_WAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						no->title = tmp;
            						break;
            					}
            					 					
            					res = ParserTryArg1 (word, "PICT", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						if (pixpath) {
            							sprintf (buffer, "%s/%s", pixpath, tmp);
            							no->pict = strdup (buffer);
            							free (tmp);
            						}
            						else no->pict = tmp;
            						break;
            					}
            					
            					res = ParserTryChild (word, "MAN", &tmp, &x, &y);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					res = ParserTryChild (word, "LAN", &tmp, &x, &y);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					return ParserError (PARSER_SYNTAXERR, NULL);
            					
            	case NET_MAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						no->title = tmp;
            						break;
             					}
            
            					res = ParserTryArg1 (word, "PICT", &tmp);
            					if (res>0) return (res);
								if (!res && tmp) {
            						if (pixpath) {
            							sprintf (buffer, "%s/%s", pixpath, tmp);
            							no->pict = strdup (buffer);
            							free (tmp);
            						}
            						else no->pict = tmp;
            						break;
            					}
            					
            					res = ParserTryChild (word, "LAN", &tmp, &x, &y);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					return ParserError (PARSER_SYNTAXERR, NULL);
            	
            	case NET_LAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						no->title = tmp;
            						break;
            					}
            	
            					res = ParserTryArg1 (word, "HOST", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						numchildren++;
            						free (tmp);
            						break;
            					}
            	
            					return ParserError (PARSER_SYNTAXERR, NULL);
            	
            	case NET_HOST:  res = ParserTryArg1 (word, "PICT", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						if (pixpath) {
            							sprintf (buffer, "%s/%s", pixpath, tmp);
            							no->pict = strdup (buffer);
            							free (tmp);
            						}
            						else no->pict = tmp;
            						break;
            					}
            	
            					res = ParserTryArg1 (word, "ADDRESS", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						no->addr = tmp;
            						break;
            					}
            
            					res = ParserTryArg1 (word, "ARCH", &tmp);
            					if (res>0) return (res);
            					if (!res && tmp) {
            						no->arch = tmp;
            						break;
            					}
             
             					res = ParserTryArg1 (word, "INFO", &tmp);
             					if (res>0) return (res);
             					if (!res && tmp) {
             						no->info = tmp;
             						break;
             					}
 
 								res = ParserTryArg1 (word, "OPTIONS", &tmp);
             					if (res>0) return (res);
             					if (!res && tmp) {
             						no->options = tmp;
             						break;
             					}
 
	    						res = ParserTryArg2 (word, "CMD", &cmdname, &cmd);
	    						if (res>0) return (res);
	    						if (!res && cmdname) {
	    							ParserAddCmd (no, cmdname, cmd, 0);
	    							free (cmdname);
	    							free (cmd);
	    							break;
	    						}
	
	    						res = ParserTryArg2 (word, "XCMD", &cmdname, &cmd);
	    						if (res>0) return (res);
	    						if (!res && cmdname) {
	    							ParserAddCmd (no, cmdname, cmd, 1);
	    							free (cmdname);
	    							free (cmd);
	    							break;
	    						}
     
     							return ParserError (PARSER_SYNTAXERR, NULL);
     		}
    	}

		/**** fine struttura. Controlla se e` chiusa correttamente ****/
		
		if (!word) return ParserError (PARSER_BADEND, NULL);
        
        /************************************/
        /* controllo direttive obbligatorie */
        /************************************/
        
        if (type == NET_HOST) {
        	if (!no->addr) return ParserError (PARSER_NOADDR, NULL);
        	if (!no->arch) return ParserError (PARSER_NOARCH, NULL);
        }
        
        else if (!numchildren) return ParserError (PARSER_NOCHILDREN, NULL);

		/*********************/
        /* valori di default */
        /*********************/
        
        if (type == NET_HOST) {
        	if (!no->pict) {
        		if (pixpath) sprintf (buffer, "%s/%s.xpm", pixpath, no->arch);
        		else sprintf (buffer, "%s.xpm", no->arch);
        		no->pict = strdup (buffer);
        	}
        }
        
        /*********************/
        /* registrazione WAN */
        /*********************/
        
        if (type == NET_WAN && WAN) return ParserError (PARSER_TOOMANYWAN, NULL);
        WAN = no;
        
        /*********/
        /* figli */
        /*********/

        if (type != NET_HOST) {
        	no->nchildren = numchildren;
        	no->children = calloc (numchildren, sizeof (struct NetworkObj *));
        }
        
        /**** Rilascia il nome e ricomincia ****/
        
        free (name);
    }
    
	/*******************/
    /* Seconda passata */
    /*******************/

	/*  Vengono controllati solo i figli. Per ogni figlio utilizzato in una
	 *  struttura si cerca il suo nome, si estrae l'indirizzo e si inserisce
	 *  nell'oggetto. Il figlio puo` settare il puntatore al padre.
	 */

    ParserInit (NULL);

    while ((word = ParserNextWord ())) {
    
    	/***************************/
		/* direttive "single-line" */
		/***************************/
		
		/**** path pixmaps ****/
		
		res = ParserTryArg1 (word, "PIXPATH", &tmp);
  		if (!res && tmp) {
       		free (tmp);
        	word = ParserNextWord();
    	}

		/*************/
    	/* strutture */
    	/*************/
    
   		ParserHeader (word, &name, &type);
		numchildren = 0;
		no = NetworkObjFind (name, type);

	    while (strcmp(word = ParserNextWord(), "}")) {
            
            switch (type) {
            
            	case NET_WAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (!res && tmp) {
            						free (tmp); 
            						break;
            					}         
            					
            					res = ParserTryArg1 (word, "PICT", &tmp);
            					if (!res && tmp) {
            						free (tmp);
            						break;
            					}          					

            					res = ParserTryChild (word, "MAN", &tmp, &x, &y);
            					if (!res && tmp) {
            						childno = NetworkObjFind (tmp, NET_MAN);
            						if (!childno) return ParserError (PARSER_NONODE, NULL);
            						no->children[numchildren] = childno;
            						childno->parent = no;
									childno->x = x;
									childno->y = y;
            						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					res = ParserTryChild (word, "LAN", &tmp, &x, &y);
            					if (!res && tmp) {
            						childno = NetworkObjFind (tmp, NET_LAN);
            						if (!childno) return ParserError (PARSER_NONODE, NULL);
            						no->children[numchildren] = childno;
            						childno->parent = no;
            						childno->x = x;
            						childno->y = y;
             						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					break;
            					
            	case NET_MAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (!res && tmp) {
            						free (tmp);
            						break;
            					}           
            					
            					res = ParserTryArg1 (word, "PICT", &tmp);
            					if (!res && tmp) {
            						free (tmp);
            						break;
            					}
            					
            					res = ParserTryChild (word, "LAN", &tmp, &x, &y);
            					if (!res && tmp) {
									childno = NetworkObjFind (tmp, NET_LAN);
            						if (!childno) return ParserError (PARSER_NONODE, NULL);
            						no->children[numchildren] = childno;
            						childno->parent = no;
            						childno->x = x;
            						childno->y = y;
             						numchildren++;
            						free (tmp);
            						break;
            					}
            					
            					break;
            	
            	case NET_LAN:	res = ParserTryArg1 (word, "TITLE", &tmp);
            					if (!res && tmp) {
            						free (tmp);
            						break;
            					}
           	
            					res = ParserTryArg1 (word, "HOST", &tmp);
            					if (!res && tmp) {
            						childno = NetworkObjFind (tmp, NET_HOST);
            						if (!childno) return ParserError (PARSER_NONODE, NULL);
            						no->children[numchildren] = childno;
            						childno->parent = no;
            						childno->x = x;
            						childno->y = y;
             						numchildren++;
            						free (tmp);
            						break;
            					}
            	
            					break;
            	
            	case NET_HOST:	res = ParserTryArg1 (word, "PICT", &tmp); 
            					if (!res && tmp) {
            						free (tmp); 
            						break;
            					}         
	           					
	           					res = ParserTryArg1 (word, "ADDRESS", &tmp); 
            					if (!res && tmp) {
            						free (tmp); 
            						break;
            					}
	           					
	           					res = ParserTryArg1 (word, "ARCH", &tmp); 
	           					if (!res && tmp) {
	           						free (tmp); 
	           						break;
	           					}         
             					
             					res = ParserTryArg1 (word, "INFO", &tmp);
             					if (!res && tmp) {
             						free (tmp);
             						break;
             					}
             					
             					res = ParserTryArg1 (word, "OPTIONS", &tmp);
             					if (!res && tmp) {
             						free (tmp);
             						break;
             					}
	    						
	    						res = ParserTryArg2 (word, "CMD", &cmdname, &cmd);
	    						if (!res && cmdname) {
	    							free (cmdname);
	    							free (cmd);
	    							break;
	    						}	
	    						
	    						res = ParserTryArg2 (word, "XCMD", &cmdname, &cmd);
	    						if (!res && cmdname) {
	    							free (cmdname);
	    							free (cmd);
	    							break;
	    						}
     
     							break;
     		}
 		}
    
    	free (name);		/* fine struttura, ricomincia il ciclo */
    }
	
	/*************************************************/
	/* controlli finali sulla consistenza della rete */
	/*************************************************/

	/* - deve esistere la radice
	 * - non devono esistere strutture senza padre (eccetto la WAN)
	 *
	 */
	
	if (! WAN) return ParserError (PARSER_NOWAN, NULL);
	
	no = NetworkObjFirst ();
	while (no) {
		if ((no->type != NET_WAN) && (!no->parent)) return ParserError (PARSER_NOPARENT, no->name);
		no = no->next;
	}
	
    return NULL;
}

/*******************************************************/
/* ParserHeader (class, &name, &type) - parsing header */
/*******************************************************/

/*  Esegue il parsing di una riga tipo: "WAN|MAN|LAN|HOST <nome> {"
 *  Inserisce nome e tipo della struttura trovata in name e type.
 *  NOTA: "name" va rilasciato!
 *  Restituisce 0 se tutto e` ok, >0 = stringa errore
 */

unsigned char * ParserHeader (unsigned char * class, unsigned char ** name, int * type)
{   
    unsigned char * word;

    if (!strcmp(class, "WAN")) * type = NET_WAN;
    else if (!strcmp(class, "MAN")) * type = NET_MAN;
    else if (!strcmp(class, "LAN")) * type = NET_LAN;
    else if (!strcmp(class, "HOST")) * type = NET_HOST;
    else return ParserError (PARSER_WRONGCLASS, NULL);

    word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *name = strdup (word);

    word = ParserNextWord ();
    if (strcmp(word, "{")) return ParserError (PARSER_BADSTART, NULL);

    return NULL;
}
                                                                
/*****************************************************************************/
/* ParserTryChild (curword, childtype, &name, &x, &y) - prova parsing figlio */
/*****************************************************************************/

/* Prova ad eseguire il parsing di una riga "<childtype> <name> <x> <y>".
 * NOTA: "name" va rilasciato!
 * Restituisce:	 0 e *name == NULL: non e` una riga "childtype"
 * 				 0 e *name != NULL: ok
 * 				 >0 : messaggio di errore
*/

unsigned char * ParserTryChild (unsigned char * curword, unsigned char * childtype, unsigned char ** name, int * x, int * y)
{
    unsigned char * word;

	*name = NULL;
    if (strcmp(curword, childtype)) return NULL;
    
    word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *name = strdup (word);

    word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *x = atoi(word);

    word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *y = atoi(word);

    return NULL;
}    

/**********************************************************************/
/* ParserTryArg1 (curword, argtype, &arg) - prova parsing 1 argomento */
/**********************************************************************/

/* Prova ad eseguire il parsing di una riga "<argtype> <arg>".
 * NOTA: "arg" va rilasciato!
 * Restituisce: 0 e *arg = NULL: non e` una riga "argtype"
 *				0 e *arg != NULL: ok
 *				>0 : messaggio di errore
 */

unsigned char * ParserTryArg1 (unsigned char * curword, unsigned char * argtype, unsigned char ** arg)
{  
	unsigned char * word;
	
	*arg = NULL;
    if (strcmp(curword, argtype)) return NULL;
    
    word = ParserNextWord ();
    if (!word) return ParserError(PARSER_NOARG, NULL);
    *arg = strdup (word);
    return NULL;
}

/******************************************************************************/
/* ParserTryArg2 (curword, argtype, &arg1, &arg2) - prova parsing 2 argomenti */
/******************************************************************************/

/* Prova ad eseguire il parsing di una riga "<argtype> <arg1> <arg2>".
 * NOTA: "arg1" e "arg2" vanno rilasciati!
 * Restituisce: 0 e *arg1 == NULL: non e` una riga "argtype"
 *				0 e *arg1 != NULL: ok
 *				>0 : messaggio di errore
 */

unsigned char * ParserTryArg2 (unsigned char * curword, unsigned char * argtype, unsigned char ** arg1, unsigned char ** arg2)
{  
	unsigned char * word;
	
	*arg1 = NULL;
    if (strcmp(curword, argtype)) return NULL;
    
	word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *arg1 = strdup (word);

	word = ParserNextWord ();
    if (!word) return ParserError (PARSER_NOARG, NULL);
    *arg2 = strdup (word);

    return NULL;
}
         
/************************************************************/
/* ParserAddCmd (no, cmdname, cmd, x) - aggiunge un comando */
/************************************************************/

void ParserAddCmd (struct NetworkObj * no, unsigned char * cmdname, unsigned char * cmd, int x)
{
	struct HostCmd * hc;
	
	hc = calloc (1, sizeof(struct HostCmd));
	hc -> no = no;
	hc -> name = strdup (cmdname);
	hc -> cmd = strdup (cmd);
	hc -> x = x;
	
	if (!no->numcmd) no->firstcmd = no->lastcmd = hc;
	else {
		no->lastcmd->next = hc;
		no->lastcmd = hc;
	}
	no->numcmd++;
}
                                                
/********************************************************/
/* ParserNextWord () - restituisce la parola successiva */
/********************************************************/

/* Restitusce la prossima parola "valida" (i commenti vengono saltati); aggiorna
 * eventualmente il numero della riga corrente. NULL = file file.
 * NOTA: la parola sara` distrutta alla prossima chiamata della funzione: se
 * serve, occorre crearne una copia privata usando strdup!
*/

unsigned char * ParserNextWord (void)
{
    unsigned char * ptr, * tmp;
    
    /* Cerca il primo carattere valido che non sia l'inizio di un commento  */

    ptr = ParserNextChar ();
    while (*ptr == '#') {
		while (*ptr && (*ptr != '\n')) ptr++;
		PD->ptr = ptr;
        ptr = ParserNextChar ();
    }
    if (!*ptr) return NULL;
    
    /* Trovata una parola valida. Se inizia per " puo` contenere spazi e
       deve terminare con ". */

    if (*ptr == '"') {
		ptr++;
		tmp = ptr;
		while (*tmp && *tmp != '"') tmp++;
		if (!*tmp) return NULL;
    }
    else {
		tmp = ptr;
		while (*tmp > 32 && *tmp < 127) tmp++;
    }
    
    /* Butta la parola precedente */
    
    if (PD->word) free (PD->word);
    
    PD -> word = calloc (tmp-ptr+1, 1);
    strncpy (PD->word, ptr, tmp-ptr);
    
    if (*tmp == '"') tmp++;
    PD->ptr = tmp;
    return (PD->word);
}
                                                                   
/****************************************************************/
/* ParserNextChar () - restituisce il prossimo carattere valido */
/****************************************************************/

/*  Restituisce il primo carattere diverso da \n, \t; se occorre, incrementa
 *  il numero di riga
 */

unsigned char * ParserNextChar (void)
{
	unsigned char * ptr;

    ptr = PD->ptr;
    while (*ptr && (*ptr < 33 || *ptr > 126)) {
		if (*ptr == '\n') PD ->line = PD->line + 1;
		ptr++;
    }

    PD -> ptr = ptr;
    return (ptr);
}

/*************************************************************/
/* ParserError (code, name) - prepara un messaggio di errore */
/*************************************************************/

unsigned char * ParserError (int code, unsigned char * name)
{
	unsigned char * buffer;

	buffer = calloc (1000,1);
	
	switch (code) {
	
		case PARSER_NOSTAT:		sprintf (buffer, "Can't stat file %s", PD->filename);
								break;
		case PARSER_NOOPEN: 	sprintf (buffer, "Can't open file %s", PD->filename);
								break;
		case PARSER_NOREAD: 	sprintf (buffer, "Can't read file %s", PD->filename);
								break;
		case PARSER_DUPNODE:	sprintf (buffer, "File %s, line %d:\nnode already defined", PD->filename, PD->line);
								break;
		case PARSER_WRONGCLASS:	sprintf (buffer, "File %s, line %d:\nnode is not WAN, MAN, LAN or HOST", PD->filename, PD->line);
								break;
		case PARSER_SYNTAXERR:	sprintf (buffer, "File %s, line %d:\nsyntax error", PD->filename, PD->line);
								break;
		case PARSER_BADEND:		sprintf (buffer, "File %s, line %d:\nmissing '}'", PD->filename, PD->line);
								break;
		case PARSER_BADSTART:	sprintf (buffer, "File %s, line %d:\nmissing '{'", PD->filename, PD->line);
								break;
		case PARSER_NOCHILDREN: sprintf (buffer, "File %s, line %d:\nnode must have at least one child", PD->filename, PD->line);
								break;
		case PARSER_NONODE: 	sprintf (buffer, "File %s, line %d:\nnode not defined", PD->filename, PD->line);
								break;
		case PARSER_NOARG: 		sprintf (buffer, "File %s, line %d:\nmissing argument", PD->filename, PD->line);
								break;
		case PARSER_NOWAN: 		sprintf (buffer, "File %s:\nWAN not defined", PD->filename);
								break;
		case PARSER_TOOMANYWAN: sprintf (buffer, "File %s:\nWAN must be defined exactly once", PD->filename);
								break;
		case PARSER_NOADDR:		sprintf (buffer, "File %s, line %d:\nmissing address", PD->filename, PD->line);
								break;
		case PARSER_NOARCH:		sprintf (buffer, "File %s, line %d:\nmissing architecture", PD->filename, PD->line);
								break;
		case PARSER_NOPARENT:	sprintf (buffer, "File %s:\nnode %s has no parent", PD->filename, name);
								break;
		default:				sprintf (buffer, "File %s, line %d:\nerror %d", PD->filename, PD->line, code);
	}
	
	return (buffer);
}
