/*****************************************************************************\
*                                                                             *
*  modPVM.c - gestione PVM							                          *
*  1.0 - 18/5/95                                                              *
*                                                                             *
\*****************************************************************************/

/*
 *
 *          WAMM version 1.0: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, G. Faieta, M. Formica, D. Laforenza
 *                   (C) 1995 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

#include "modPVM.h"

/***************/
/* Definizioni */
/***************/

#define PVM_DELAY	50		/* ms di pausa tra un controllo e il successivo */
#define VERSION		"1.0"	/* versione del PVMtasker da usare */

/*******************************/
/* variabili globali (private) */
/*******************************/

static struct NetworkObj * WAN;
static int MyTid;
static int HosterTid;

extern int pvm_useruid;

/********************/
/* funzioni private */
/********************/

extern unsigned char * pvmdsockfile (void);

static void	Timeout (void);

static void INT2_AddEnd (void);
static void NFY_AddHost (void);
static void NFY_DelHost (void);
static void TSKR_New (void);
static void TSKR_End (void);
static void TSKR_Cmd (void);
static void HSTR_Psw (void);
static void MKR_Msg (int);
static void MKR_MsgEnd (int);
static void MKR_Out (void);
static void TASK_Out (void);
static void TASK_Trace (void);

static void RefreshStatus (struct NetworkObj *);

/***********/
/* PVMInit */
/***********/

void PVMInit (struct NetworkObj * wan)
{
	int n;
	unsigned char buffer[1000];
	
	struct NetworkObj * no;			/* oggetto corrente */
	int res;						/* risultato */
	unsigned char * errmsg;			/* messaggio di errore */
	char * verargv[2];				/* argomenti per tasker/hoster */

	char * hostfile;				/* nome hostfile */
	FILE * fh;						/* hostfile */
	unsigned char * opt;			/* opzioni */
	struct stat * st;				/* controllo /tmp/pvmd.xx.yy */
	unsigned char sockfile[1000];	/* nome sockfile */
			
	int hostnum;					/* numero host */
	int archnum;					/* numero architetture */
	struct pvmhostinfo * hosts;		/* informazioni sugli host */
	
	int * tids;						/* tids PVMTaskers */
	int tid;						/* tid PVMTasker */
	int ack;						/* 1=PVMtasker ok */
		
	/*******************************/
	/* Inizializzazione modulo PVM */
	/*******************************/
	
	/*  1. Viene estratto un host file dal file di configurazione
	*   2. Se necessario si fa partire il master daemon.
	*   3. Viene attivato il notify PvmHostAdd
	*	4. Viene lanciato l'hoster
	*   5. Viene determinata la configurazione iniziale (pvm_conf)
	*	6. Per ogni host in PVM e nel file di configurazione si prende il
	       dtid e si registra un notify PvmHostDelete
	    7. Si lanciano tutti i PVMTasker
	    8. Si registrano i loro tid, eliminando i doppioni
	    9. Si fanno partire manualmente i PVMTasker sugli host "mancati";
	       si controllano gli ack di tutti.
	*/

	/**** Finestra per i messaggi ****/
	
	WAN = wan;
			
	verargv[0] = VERSION;	/* versione di PVMTasker/PVMHoster da usare */
	verargv[1] = NULL;

	/*********************/
	/* Crea un host file */
	/*********************/

	hostfile = tmpnam(NULL);
	fh = fopen (hostfile, "w");
	
	no = NetworkObjFirst ();
	while (no) {
	
		if (no->type == NET_HOST) {
			opt = no->options;
			if (opt && opt[0] == '&') {
				fprintf (fh, "&");
				opt++;
				while (*opt && (*opt == ' ' || *opt == '\t')) opt++;
			}

			fprintf (fh, "%s", no->addr);
			if (opt) fprintf (fh, "\t\t\t%s\n", opt);
			else fprintf (fh, "\n");
		}
		
		no = no->next;
	}
	
	fclose (fh);
	
	/**** prepara il nome del sockfile ****/
	
	pvm_useruid = getuid();
	sprintf (sockfile, "%s", pvmdsockfile());

	/***********************/
	/* Opzioni globali PVM */
	/***********************/
	
	pvm_setopt (PvmAutoErr, 0);		/* disabilita la stampa degli errori */

	/*******************/
	/* Attivazione PVM */
	/*******************/
	
	st = calloc (1, sizeof (struct stat));
	if (stat(sockfile, st)) {
	
		/**** PVM non attivo. Lo lancia. ****/
		
		MotifStatusMsg (wan->Wstatus, "PVM is not running, starting it for you... ");
	  
		sprintf (buffer, "pvmd %s 1>/dev/null 2>/dev/null &", hostfile);
		system (buffer);
	
		/**** aspetta finche` non compare... ****/

		do {
			sleep(1);
			MotifStatusMsg (wan->Wstatus, ".");
		} while (stat(sockfile, st));
		
		MotifStatusMsg (wan->Wstatus, " OK; now starting setup:\n"); 
	}
	
	else MotifStatusMsg (wan->Wstatus, "PVM already running; now starting setup:\n");	
		
	unlink (hostfile);
	free (st);

	/*********/
	/* MyTid */
	/*********/

	MotifStatusMsg (wan->Wstatus, "- joining PVM... ");
			
	MyTid = pvm_mytid();
	if (MyTid < 0) {
		errmsg = PVMError (MyTid);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		return;
	}
	
	MotifStatusMsg (wan->Wstatus, "OK.\n");

	/*********************/
	/* Notify PvmHostAdd */
	/*********************/
	
	pvm_notify (PvmHostAdd, T_NFY_ADDHOST, -1, NULL);
	
	/**********/
	/* Hoster */
	/**********/
	
	MotifStatusMsg (wan->Wstatus, "- spawning PVMHoster on local host... ");
	
	res = pvm_spawn ("PVMHoster", verargv, PvmTaskHost, ".", 1, &HosterTid);
	if (res != 1) {
		errmsg = PVMError (HosterTid);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		HosterTid = 0;
	}
	
	/**** attende l'ack ****/
		
	if (HosterTid > 0) {
		
		pvm_recv (HosterTid, T_HSTR_ACK);
		pvm_unpackf ("%d", &ack);
		if (!ack) {
			MotifStatusMsg (wan->Wstatus, "FAILED: PVMHoster has a wrong version.\n");
			HosterTid = 0;
		}
	}

	if (HosterTid) MotifStatusMsg (wan->Wstatus, "OK.\n");
	else MotifStatusMsg (wan->Wstatus, "  You can use Wamm anyway, but won't get password dialogs.\n  Use shell to insert passwords when required.\n");
	
	/***************************/
	/* Configurazione corrente */
	/***************************/
	
	MotifStatusMsg (wan->Wstatus, "- getting current configuration... ");
	
	res = pvm_config (&hostnum, &archnum, &hosts);
	if (res < 0) {
		errmsg = PVMError (res);
		MotifStatusMsg (wan->Wstatus, "FAILED: %s!\n", errmsg);
		free (errmsg);
		return;
	}
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*********************************************/
	/* Registrazione dtid e notify PvmHostDelete */
	/*********************************************/
	
	MotifStatusMsg (wan->Wstatus, "- getting pvmd tids for active hosts... ");
		
	no = NetworkObjFirst();
	while (no) {
	
		if (no->type == NET_HOST) {
				
			/**** controlla se e` attivo ****/
	
			for (n=0; n<hostnum; n++) if (!strcmp(no->addr, hosts[n].hi_name)) break;
			if (n<hostnum) {
			
				/**** registra il dtid ****/
		
				no->dtid = hosts[n].hi_tid;
		
				/**** notifica rimozione ****/
		
				pvm_notify (PvmHostDelete, T_NFY_DELHOST, 1, &no->dtid);
			}
		}	
		
		no = no->next;
	}
	
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*************************/
	/* Attivazione PVMTasker */
	/*************************/
	
	MotifStatusMsg (wan->Wstatus, "- spawning PVMTasker on all active hosts... ");
	
	tids = calloc (hostnum, sizeof (int));
	res = pvm_spawn ("PVMTasker", verargv, PvmTaskDefault, NULL, hostnum, tids);

	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/********************************/
	/* Registrazione tids PVMTasker */
	/********************************/
	
	MotifStatusMsg (wan->Wstatus, "- saving PVMTasker tids... ");
	
	for (n=0; n<hostnum; n++) {
	
		if (tids[n] <= 0) continue;		/* salta quelli falliti */
		
		/**** cerca l'host su cui e` stato lanciato questo task. Se non lo
		      trova, l'host non e` nel file di configurazione: uccide il
		      tasker (non ci serve) ****/
		
		no = NetworkObjFindByDtid (pvm_tidtohost(tids[n]));
		if (!no) pvm_kill (tids[n]);
		
		/**** se gia` e` stato registrato un tasker, questo e` un doppione
		      che va eliminato... ****/
		
		else {		
			if (no->taskertid) pvm_kill (tids[n]);
			else no->taskertid = tids[n];
		}
	}
	
	free (tids);
	
	MotifStatusMsg (wan->Wstatus, "OK.\n");
	
	/*********************************/
	/* Controllo nodi mancanti e ack */
	/*********************************/
	
	MotifStatusMsg (wan->Wstatus, "- checking all PVMTaskers (activation and version)...\n");
	
	no = NetworkObjFirst ();
	n = 1;
	while (no) {
	
		if (no->type == NET_HOST) {
		
			XmListSetBottomPos (wan->Wlist, n);
			XmListSelectPos (wan->Wlist, n, False);
			XmUpdateDisplay (wan->Wlist);
			n++;
		
			if (!no->taskertid && no->dtid) {
			
				/**** bisogna provare a lanciarlo "a mano"... ****/
				
				res = pvm_spawn ("PVMTasker", verargv, PvmTaskHost, no->addr, 1, &tid);
				if (res != 1) {
					errmsg = PVMError (tid);
					MotifStatusMsg (wan->Wstatus, "  PVMTasker on %s can't be started: %s.\n", no->addr, errmsg);
					free (errmsg);
				}
				else no->taskertid = tid;
			}
		
			if (no->taskertid && no->dtid) {
			
				/**** attende l'ack ****/
				
				pvm_recv (no->taskertid, T_TSKR_ACK);
				pvm_unpackf ("%d", &ack);
				if (!ack) {
					MotifStatusMsg (wan->Wstatus, "  PVMTasker on %s has a wrong version and can't be used.\n", no->addr);
					no->taskertid = 0;
				}
			}
		}
		
		no = no->next;
	}
	
	XmListDeselectAllItems (wan->Wlist);
	XmListSetPos (wan->Wlist, 1);
	
	MotifStatusMsg (wan->Wstatus, "Done; setup completed.\n");
	
	/**** attiva il primo timeout ****/
	
	XtAppAddTimeOut (XtWidgetToApplicationContext (wan->Wshell), 1, (XtTimerCallbackProc)Timeout, NULL);
	
	return;	
}

/**********/
/* PVMEnd */
/**********/

void PVMEnd (int mode)
{	
	struct NetworkObj * no;

	if (mode != PVM_ENDFAIL) {

		/**** Uccide i tasker ****/
	
		no = NetworkObjFirst();
		while (no) {
			if (no->taskertid) pvm_sendsig (no->taskertid, SIGKILL);
			no = no->next;
		}
	
		/**** Uccide l'hoster ****/
	
		if (HosterTid) pvm_sendsig (HosterTid, SIGKILL);
	}
	
	if (mode == PVM_ENDSOFT) pvm_exit();
	else if (mode == PVM_ENDHARD) pvm_halt();

	/**** per PVM_ENDFAIL non si puo` fare piu` nulla... ****/
}

/**********/
/* PVMTid */
/**********/

int PVMTid (void)
{
	return MyTid;
}

/****************************************************/
/* PVMError (code) - prepara un messaggio di errore */
/****************************************************/

unsigned char * PVMError (int code)
{
	unsigned char * buffer;
	
	buffer = calloc (1000,1);
	switch (code) {
	
		case PvmOk:			sprintf (buffer, "OK!");
							break;
		case PvmBadParam:	sprintf (buffer, "some parameter is not valid");
							break;
		case PvmNoHost:		sprintf (buffer, "host not in PVM or not existing");
							break;
		case PvmNoFile:		sprintf (buffer, "can't find executable file");
							break;
		case PvmNoMem:		sprintf (buffer, "malloc failed");
							break;
		case PvmSysErr:		sprintf (buffer, "local pvmd doesn't respond (PVM halted?)");
							break;
		case PvmHostFail:	sprintf (buffer, "host failed");
							break;
		case PvmBadVersion: sprintf (buffer, "remote pvmd has a wrong version");
							break;
		case PvmOutOfRes:	sprintf (buffer, "no resources");
							break;
		case PvmDupHost:	sprintf (buffer, "host is already in PVM");
							break;
		case PvmCantStart:	sprintf (buffer, "can't start pvmd");
							break;
		case PvmAlready:	sprintf (buffer, "operation already in progress");
							break;
		default:			sprintf (buffer, "error %d", code);
	}
	
	return (buffer);
}

/**********/
/* PVMAdd */
/**********/

void PVMAdd (struct NetworkObj * no, int num, unsigned char ** hosts, Boolean * mem)
{
	int * infos = NULL;
	int n, res;

	if (!fork()) {
	
		/**********/
		/* figlio */
		/**********/
		
		pvmendtask();
		pvm_mytid();
		
		infos = calloc (num, sizeof(int));

		res = pvm_addhosts ((char **)hosts, num, infos);
		
		/**** PvmAlready e PvmSysErr sono restituiti in res, NON in infos */
		
		if ((res == PvmAlready) || (res == PvmSysErr))
			for (n=0; n<num; n++) infos[n] = res; 
			
		pvm_packf ("%+ %d %d", PvmDataDefault, no, num);
		for (n=0; n<num; n++) pvm_packf ("%s %d", hosts[n], infos[n]);
		pvm_send (PVMTid(), T_INT2_ADDEND);
		
		for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
		free (mem);
		free (hosts);	
		free (infos);
		
		pvm_exit();
		exit(0);
	}
	
	/* XXX ci vorrebbe un controllo per fork == -1 */
	
	else return;
}

/**********/
/* PVMDel */
/**********/

void PVMDel (struct NetworkObj * no, int num, unsigned char ** hosts, Boolean * mem)
{
	int * infos;
	unsigned char * tmp;
	int n;
	
	/* Il delete e` veloce, non occorre farlo con una fork. */
	
	infos = calloc (num, sizeof(int));
	pvm_delhosts ((char **)hosts, num, infos);
	
	for (n=0; n<num; n++) if (infos[n]<0) {
	
		if (infos[n] != PvmNoHost) {		
			tmp = PVMError (infos[n]);
			MotifStatusMsg (no->Wstatus, "%s: %s\n", hosts[n], tmp);
			free (tmp);
		}
		
		else MotifStatusMsg (no->Wstatus, "%s: not in PVM\n", hosts[n]);
	}
	
	free (infos);	
}

/************/
/* PVMCheck */
/************/

void PVMCheck (struct NetworkObj * no, int num, unsigned char ** hosts, Boolean * mem)
{
	unsigned char * tmp;
	int n, res;
	
	/* Il check e` veloce, non occorre farlo con una fork. */
	
	for (n=0; n<num; n++) {
	
		MotifStatusMsg (no->Wstatus, "%s: ", hosts[n]);
		res = pvm_mstat (hosts[n]);
		
		/**** controlla l'esito ****/
		
		tmp = PVMError (res);
		MotifStatusMsg (no->Wstatus, "%s\n", tmp);
		free (tmp);
	}
}

/*************/
/* PVMRepair */
/*************/

void PVMRepair (struct NetworkObj * no, int num, unsigned char ** hosts, Boolean * mem)
{

	/* XXX bugs noti:
	
		- i messaggi di errore di rexec vengono mostrati su shell (orrendo...)
	*/

	struct servent * servent;		/* servent per rexec */
	unsigned char * localname;		/* nome utente locale */
	unsigned char remotename[1000];	/* nome utente remoto */
	
	unsigned char buffer[1000];
	int n;
	struct NetworkObj * no2;
	unsigned char * psw;
	int out;
	unsigned char * tmp;
	
	/**** porta per rexec e nome utente locale ****/
	
	servent = getservbyname ("exec", "tcp");
	localname = getpwuid (getuid()) -> pw_name; 
	
	for (n=0; n<num; n++) {
	
		MotifStatusMsg (no->Wstatus, "%s... ", hosts[n]);
		no2 = NetworkObjFindByAddr (hosts[n]);
	
		/**** Nome utente remoto ****/
		
		tmp = strstr(no2->options, "lo=");
		if (tmp) {
			strcpy (remotename, tmp+3);
			tmp = strpbrk (remotename, " \t");
			if (tmp) *tmp = 0;
		}
		else strcpy (remotename, localname);
		
		/**** Controlla se deve usare rsh o rexec ****/
		
		no2 = NetworkObjFindByAddr (hosts[n]);
		if (no2->options && strstr(no2->options, "so=pw")) {
		
			/**** Usa rexec ****/
			
			psw = no2->password;
			if (! psw) {
				sprintf (buffer, "Please insert password for %s, user %s:", hosts[n], remotename);
				psw = MotifAskPsw (WAN->Wshell, "Password", buffer);
				if (!psw) {
					MotifStatusMsg (no->Wstatus, "failed (must use password!)\n");
					continue;
				}
			}
	
			out = rexec ((char **)&hosts[n], servent->s_port, remotename, psw, "PVMKill", 0);
			if (out < 0) MotifStatusMsg (no->Wstatus, "failed.\n");
			else {
				MotifStatusMsg (no->Wstatus, "repaired.\n");
				close (out);
				if (!no2->password) no2->password = strdup (psw);
			}
		}
		
		else {
		
			/**** Usa rsh ****/
		
			sprintf (buffer, "rsh < /dev/null %s -l %s PVMKill", hosts[n], remotename);
			system (buffer);
			MotifStatusMsg (no->Wstatus, "repaired.\n");
		}		
	}
}

/******************************************************************************/
/******************************************************************************/

/***********/
/* Timeout */
/***********/

void Timeout (void)
{	
	int msgid, msglen, msgtag, msgtid;	/* dati del messaggio ricevuto */
		
	/*****************************************/
	/* Controllo messaggi e ricarica timeout */
	/*****************************************/
	
	msgid = pvm_nrecv (-1, -1);
	
	if (msgid<0) {
		
		/* gravi problemi con il PVM... */
		
		MotifError (WAN->Wshell, "FATAL ERROR!",
			"Local pvmd doesn't respond. Maybe PVM was halted.\n"
			"Please select OK to quit Wamm.");
		
		/*************************/
		/* Chiude tutti i moduli */
		/*************************/
	
		PVMEnd (PVM_ENDFAIL);
		InternetEnd ();
		WANEnd ();
		MANEnd ();
		LANEnd ();
		TasksEnd ();
		SpawnEnd ();
		MakeEnd ();
		NetworkObjEnd ();

		exit (1);
	}
	
	/* Tutto ok, puo` ricaricare il timeout */
	
	XtAppAddTimeOut (XtWidgetToApplicationContext (WAN->Wshell), PVM_DELAY, (XtTimerCallbackProc)Timeout, NULL);

	if (!msgid) return;
	pvm_bufinfo (msgid, &msglen, &msgtag, &msgtid);
	switch (msgtag) {
	
		/*********************/
		/* AddHost terminato */
		/*********************/
		
		case T_INT2_ADDEND:		INT2_AddEnd();
								break;

	
		/*********************/
		/* Notify nuovo host */
		/*********************/
		
		case T_NFY_ADDHOST:		NFY_AddHost ();
								break;
													
		/***********************/
		/* Notify host rimosso */
		/***********************/
		
		case T_NFY_DELHOST:		NFY_DelHost ();
								break;
												
		/**********************/
		/* Tasker: nuovo task */
		/**********************/
		
		case T_TSKR_NEW:		TSKR_New ();
								break;
															
		/******************/
		/* Task terminato */
		/******************/
		
		case T_TSKR_END:		TSKR_End ();
								break;
				
		/*********************/
		/* Comando terminato */
		/*********************/
		
		case T_TSKR_CMD:		TSKR_Cmd ();
								break;
												
		/**********************/
		/* Richiesta password */
		/**********************/
		
		case T_HSTR_PSW:		HSTR_Psw ();
								break;
								
		/***********************/
		/* Messaggio dal maker */
		/***********************/
		
		case T_MKR_MSG:			MKR_Msg (msgtid);
								break;	
								
		/******************************/
		/* Messaggio finale dal maker */
		/******************************/
		
		case T_MKR_MSGEND:		MKR_MsgEnd (msgtid);
								break;
								
		/**********************/
		/* Output da un maker */
		/**********************/
		
		case T_MKR_OUT:			MKR_Out ();
								break;
								
		/*********************/
		/* Output da un task */
		/*********************/
		
		case T_TASK_OUT:		TASK_Out ();
								break;
								
		/**************/
		/* Trace task */
		/**************/
		
		case T_TASK_TRACE:		TASK_Trace ();
								break;		
	}
}

/************************************************/
/* INT2_AddEnd - segnalazione AddHost terminato */
/************************************************/

void INT2_AddEnd (void)
{
	int n;
	unsigned char * tmp;
	struct NetworkObj * no;			/* nodo da usare per i messaggi */
	int num;						/* numero nodi aggiunti */
	struct NetworkObj * hostno;		/* nodo aggiunto */
	unsigned char host[1000];		/* indirizzo host */
	int dtid;						/* dtid nuovo host (o errore se < 0) */

	pvm_unpackf ("%d %d", &no, &num);
												
	for (n=0; n<num; n++) {
							
		pvm_unpackf ("%s %d", host, &dtid);
		hostno = NetworkObjFindByAddr (host);						
								
		if (dtid > 0) continue;
		
		if (dtid == PvmDupHost) {
			if (no->Wshell) MotifStatusMsg (no->Wstatus, "%s: already in PVM\n", host);
			continue;
		}

		if (hostno->password) {
			free (hostno->password);
			hostno->password = NULL;
		}
		
		tmp = PVMError (dtid);
		if (no->Wshell) MotifStatusMsg (no->Wstatus, "%s: %s\n", host, tmp);
		free (tmp);
	}
}

/***********************************/
/* NFY_AddHost - notify nuovo host */
/***********************************/
	
void NFY_AddHost (void)
{
	int n, res;
	unsigned char * tmp;
	
	int confnum, confarchs;				/* risultati di pvm_config */
	struct pvmhostinfo * confinfo;
	struct NetworkObj * no;				/* nuovo host */
	char * argv[2];						/* argv tasker */
	int tid;							/* tid tasker */
	int ack;							/* ack versione (1=ok) */
	
	argv[0] = VERSION;
	argv[1] = NULL;

	res = pvm_config (&confnum, &confarchs, &confinfo);
	if (res<0) return;
				
	for (n=0; n<confnum; n++) {
		
		no = NetworkObjFindByAddr (confinfo[n].hi_name);
									
		/**** Solo per gli host "nuovi" e riconosciuti ****/
									
		if (no && (!no->dtid)) {
					
			no->dtid = confinfo[n].hi_tid;
			
			/**** notifica rimozione ****/
		
			pvm_notify (PvmHostDelete, T_NFY_DELHOST, 1, &confinfo[n].hi_tid);
			
			/**** Segnala nuovo host ****/
			
			RefreshStatus (no);
			MotifStatusMsg (WAN->Wstatus, "NEW HOST: %s. Spawning PVMtasker... ", no->addr);
										
			/**** tasker ****/
									
			res = pvm_spawn ("PVMTasker", argv, PvmTaskHost, no->addr, 1, &tid);
			if (res != 1) {
				tmp = PVMError (tid);
				MotifStatusMsg (WAN->Wstatus, "FAILED! (%s).\n", tmp);
				tid = 0;
				free (tmp);
			}
		
			/**** attende l'ack ****/
		
			else {
				pvm_recv (tid, T_TSKR_ACK);
				pvm_unpackf ("%d", &ack);
				if (!ack) {
					MotifStatusMsg (WAN->Wstatus, "FAILED! Remote PVMTasker has a wrong version.\n");
					tid = 0;
				}
			}
			
			no->taskertid = tid;
			if (tid) MotifStatusMsg (WAN->Wstatus, "OK.\n");
		}
	}
}

/*************************************/
/* NFY_DelHost - notify host rimosso */
/*************************************/
	
void NFY_DelHost (void)
{
	int dtid;
	struct NetworkObj * no;

	pvm_upkint (&dtid, 1, 1);
			
	/**** Cerca l'host e cambia lo status PVM ****/
		
	no = NetworkObjFindByDtid (dtid);
	if (!no) return;
	
	no->dtid = 0;
	RefreshStatus (no);
	no->taskertid = 0;
	
	MotifStatusMsg (WAN->Wstatus, "HOST DELETED: %s\n", no->addr);
	
	/**** Segnala che tutti i task in esecuzione sull'host sono morti */
	
	TasksDelAll (no->addr);
}
	
/*************************/
/* TSKR_New - nuovo task */
/*************************/
	
void TSKR_New (void)
{
	int tid, ntask;
	struct pvmtaskinfo * ti;
	struct NetworkObj * no;	
	
	pvm_upkint (&tid, 1, 1);
	if (tid<=0) return;
	
	/* Se ntask != 1 ti NON contiene dati validi! */
	
	if (pvm_tasks(tid, &ntask, &ti) >=0) {
		if (ntask != 1) return;
		no = NetworkObjFindByDtid (ti->ti_host);
		TasksAdd (tid, ti->ti_a_out, no->addr);
	}
}	
	
/*****************************/
/* TSKR_End - task terminato */
/*****************************/
	
void TSKR_End (void)
{
	int tid;
	
	pvm_upkint (&tid, 1, 1);
	if (tid) TasksDel (tid);
}

/********************************/
/* TSKR_Cmd - comando terminato */
/********************************/
	
void TSKR_Cmd (void)
{
	int res, len;
	struct NetworkObj * no;
	unsigned char * out;
	unsigned char * err;
	
	pvm_unpackf ("%d %d %d", (int *)&no, &res, &len);
	
	/**** finestra aperta ? ****/
	
	if (!no->Wshell) return;

	out = calloc (len+1, 1);
	pvm_unpackf ("%s %d", out, &len);
	err = calloc (len+1, 1);
	pvm_unpackf ("%s", err);
												
	if (!res) {
		if (strlen(out)) MotifStatusMsg (no->Wstatus, "%s:\n%s\n", no->addr, out);
		else MotifStatusMsg (no->Wstatus, "%s: OK (no output).\n\n", no->addr);
	}
	
	else {
		if (strlen(err)) MotifStatusMsg (no->Wstatus, "%s:\n%s\n", no->addr, err);
		else MotifStatusMsg (no->Wstatus, "%s: error (%d)\n\n", no->addr, res);
	}
	
	free (out);
	free (err);
}

/********************************/
/* HSTR_Psw - richiede password */
/********************************/
	
void HSTR_Psw (void)
{
	struct NetworkObj * no;
	unsigned char host [1000];
	unsigned char buffer [1000];
	unsigned char * tmp;
	
	pvm_upkstr (host);
	no = NetworkObjFindByAddr (host);
							
	/**** Se esiste, usa la password gia` inserita ****/
							
	if (no && no->password) {
		pvm_packf ("%+ %s", PvmDataDefault, no->password);
		pvm_send (HosterTid, T_HSTR_PSW);
		return;
	}
							
	/**** Nessuna password: la richiede ****/
							
	sprintf (buffer, "Please insert password for %s:", host);
	tmp = MotifAskPsw (WAN->Wshell, "Password", buffer);
	if (tmp) {
		if (no) no->password = strdup (tmp);
		pvm_packf ("%+ %s", PvmDataDefault, tmp);
	}
	else pvm_packf ("%+ %s", PvmDataDefault, "");
	
	pvm_send (HosterTid, T_HSTR_PSW);
}

/*********************************/
/* MKR_Msg - messaggio dal maker */
/*********************************/

void MKR_Msg (int tid)
{
	unsigned char msg[1000];
	
	pvm_unpackf ("%s", msg);
	MakeMsg (tid, msg);
}

/*******************************************/
/* MKR_MsgEnd - messaggio finale dal maker */
/*******************************************/

void MKR_MsgEnd (int tid)
{
	unsigned char msg[1000];
	int res;
	
	pvm_unpackf ("%s %d", msg, &res);
	MakeMsgEnd (tid, msg, res);
}

/********************************/
/* MKR_Out - output da un maker */
/********************************/
	
void MKR_Out (void)
{
	int tid, len;
	unsigned char * out = NULL;
	
	pvm_unpackf ("%d%d", &tid, &len);
	if (len>0) {
		out = calloc (len+1, 1);
		pvm_upkbyte (out, len, 1);
		MakeOutput (tid, out);
		free (out);
	}
}

/********************************/
/* TASK_Out - output da un task */
/********************************/
	
void TASK_Out (void)
{
	int tid, len;
	unsigned char * out = NULL;
	
	pvm_unpackf ("%d%d", &tid, &len);
	if (len>0) {
		out = calloc (len+1, 1);
		pvm_upkbyte (out, len, 1);
		TasksOutput (tid, out);
		free (out);
	}
}

/***************************/
/* TASK_Trace - trace task */
/***************************/
	
void TASK_Trace (void)
{
	int sec, usec, tid, eid;
	
	pvm_unpackf ("%d %d %d %d", &sec, &usec, &tid, &eid);
	printf ("Trace: TID=%x, EID=%d\n", tid, eid);
}

/************************/
/* RefreshStatus (node) */
/************************/

void RefreshStatus (struct NetworkObj * no)
{
	int n;
	Arg args[5];

	if (!no->Wshell) return;
	
	n = 0;
	if (no->dtid) {XtSetArg (args[n], XmNlabelString, XmStringCreate ("PVM", DEFSET)); n++;}
	else {XtSetArg (args[n], XmNlabelString, XmStringCreate (" ", DEFSET)); n++;}
	XtSetValues (no->Wpvm, args, n);
}
