/*****************************************************************************\
*                                                                             *
*  modLAN.c - Gestione LAN				                                      *
*  1.0 - 17/6/95                                                              *
*                                                                             *
\*****************************************************************************/

/*
 *
 *          WAMM version 1.0: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, G. Faieta, M. Formica, D. Laforenza
 *                   (C) 1995 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

#include "modLAN.h"

/***************/
/* definizioni */
/***************/

#define ICONHEIGHT  80		/* Altezza delle icone degli host */
#define AREAWIDTH   300		/* Largezza drawingarea LAN */
#define AREAOFFSET  20		/* Offset tra le icone e i bordi della drawingarea */
#define ICONX		40		/* Posizione x icone */
#define ETHERX		20		/* Posizione x cavo ethernet */
#define ETHERWIDTH	6		/* Spessore cavo ethernet */
#define MENUX		150		/* Posizione x menu */
#define INFOWIDTH	350		/* Larghezza label informazioni */
#define STATUSWIDTH	100		/* Ampiezza delle label PVM e Selezione */

/*******************************/
/* variabili globali (private) */
/*******************************/

static Widget Wtoplevel;
static unsigned char * displayaddr;
static int xhosterr;

/********************/
/* funzioni private */
/********************/

static int ErrorHandler (Display *, XErrorEvent *);			/* handler errori X */
static void PopupHandler (Widget, struct NetworkObj *, XEvent *);	/* gestione popup */

/*********************/
/* callbacks private */
/*********************/

static void ExposeCB (Widget, struct NetworkObj *, int);		/* expose LAN */
static void CloseCB (Widget, struct NetworkObj *, int);			/* close */
static void SelectionCB (Widget, struct NetworkObj *, int);		/* selezione icona */
static void SelectAllCB (Widget, struct NetworkObj *, int);		/* select all lista */
static void DeselectAllCB (Widget, struct NetworkObj *, int);	/* deselect all lista */
static void ClearStatusCB (Widget, struct NetworkObj *, int);	/* reset text status */

static void AddCB (Widget, struct NetworkObj *, int);		/* add pvm */
static void DelCB (Widget, struct NetworkObj *, int);		/* del pvm */
static void CheckCB (Widget, struct NetworkObj *, int);		/* check pvm */
static void MakeCB (Widget, struct NetworkObj *, int);			/* make */
static void SpawnCB (Widget, struct NetworkObj *, int);		/* finestra spawn */
static void CmdCB (Widget, struct HostCmd *, int);			/* comandi utente */
static void TasksWinCB (Widget, struct NetworkObj *, int);		/* finestra tasks */
static void MakeWinCB (Widget, struct NetworkObj *, int);		/* finestra make */
static void RepairCB (Widget, struct NetworkObj *, int);		/* clear /tmp/pvmd */
static void PingCB (Widget, struct NetworkObj *, int);			/* ping */
static void TracerouteCB (Widget, struct NetworkObj *, int);	/* traceroute */

/************/
/* LANInit */
/************/

void LANInit (Widget toplevel)
{
	Wtoplevel = toplevel;
	displayaddr = MotifGetDisplayAddr (XtDisplay(toplevel));
	
	/**** Handler per gli errori di MotifXHost ****/
	
	XSetErrorHandler ((XErrorHandler)ErrorHandler);
	
}

/*******************************/
/* LANEnd - distruzione modulo */
/*******************************/

void LANEnd (void)
{
	free (displayaddr);
}

/*********************************/
/* LANWindow (no) - finestra LAN */
/*********************************/

void LANWindow (struct NetworkObj * no)
{
	Arg args[20];
	int n, i;
	Widget Wshell, Wmainform, Wmenubar, Wmenupane, Wpanedwin, Wtopform;
	Widget Wscrollwin, Wdrawingarea, Wstatus;
	Widget Wicon, Wframe, Wselection, Wpvm, Winfo, Wpopup;
	
	XGCValues xgcvals;
	
	unsigned char buffer[1000];
	
	struct NetworkObj * no2;
	struct HostCmd * hc;
	
	/**************************************************/
	/* Se la finestra esiste, la porta in primo piano */
	/**************************************************/
	
	if (no->Wshell) {
		XMapRaised (XtDisplay(no->Wshell), XtWindow(no->Wshell));
		return;
	}
	
	/*********/
	/* Shell */
	/*********/
	
	n = 0;
	XtSetArg (args[n], XmNwidth, 600); n++;
    XtSetArg (args[n], XmNheight, 550); n++;
    XtSetArg (args[n], XmNtitle, no->title); n++;
    XtSetArg (args[n], XmNdeleteResponse, XmDO_NOTHING); n++;
 	Wshell = XtAppCreateShell (no->name, "Wamm", applicationShellWidgetClass, XtDisplay(Wtoplevel), args, n);
    MotifProtectShell (Wshell, (XtCallbackProc)CloseCB, no);
 	no->Wshell = Wshell;
 
 	/**** Editres ****/
    
    XtAddEventHandler (Wshell, (EventMask)0, True, _XEditResCheckMessages, NULL);

	/******************/
	/* Form "globale" */
	/******************/
	
	n = 0;
	Wmainform = XmCreateForm (Wshell, "mainform", args, n);
	XtManageChild (Wmainform);
	
	/********/	
	/* Menu */
	/********/

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	Wmenubar = XmCreateMenuBar (Wmainform, "menubar", args, n);
	XtManageChild (Wmenubar);

	Wmenupane = MotifCreateMenu (Wmenubar, "File", 'F', 0, 0);
	MotifAddMenuItem (Wmenupane, "Close", 'C', (XtCallbackProc)CloseCB, no);
	
	Wmenupane = MotifCreateMenu (Wmenubar, "Host", 'H', 0, 0);
	MotifAddMenuItem (Wmenupane, "Select all", 'S', (XtCallbackProc)SelectAllCB, no);
	MotifAddMenuItem (Wmenupane, "Deselect all", 'D', (XtCallbackProc)DeselectAllCB, no);
	MotifAddMenuSep (Wmenupane);
	MotifAddMenuItem (Wmenupane, "PVM Add", 'A', (XtCallbackProc)AddCB, no);
	MotifAddMenuItem (Wmenupane, "PVM Remove", 'R', (XtCallbackProc)DelCB, no);
	MotifAddMenuItem (Wmenupane, "PVM Check", 'C', (XtCallbackProc)CheckCB, no);
	MotifAddMenuItem (Wmenupane, "PVM Repair...", 'e', (XtCallbackProc)RepairCB, no);
	MotifAddMenuSep (Wmenupane);
	
	/**** Ping e traceroute sono opzionali... ****/

#ifdef PINGPATH
	MotifAddMenuItem (Wmenupane, "Ping...", 'P', (XtCallbackProc)PingCB, no);
#endif
#ifdef TROUTEPATH
	MotifAddMenuItem (Wmenupane, "Traceroute...", 'T',(XtCallbackProc)TracerouteCB, no);
#endif

	Wmenupane = MotifCreateMenu (Wmenubar, "Apps", 'A', 0, 0);
	MotifAddMenuItem (Wmenupane, "Make...", 'M', (XtCallbackProc)MakeCB, no);
	MotifAddMenuItem (Wmenupane, "Spawn...", 'S', (XtCallbackProc)SpawnCB, no);
	
	Wmenupane = MotifCreateMenu (Wmenubar, "Windows", 'W', 0, 0);
	MotifAddMenuItem (Wmenupane, "Make", 'M', (XtCallbackProc)MakeWinCB, no);
	MotifAddMenuItem (Wmenupane, "Tasks", 'T', (XtCallbackProc)TasksWinCB, no);
	
	Wmenupane = MotifCreateMenu (Wmenubar, "Misc", 'M', 0, 0);
	MotifAddMenuItem (Wmenupane, "Clear messages", 'C', (XtCallbackProc)ClearStatusCB, no);

	/****************/
	/* Paned Window */
	/****************/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, Wmenubar); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	Wpanedwin = XmCreatePanedWindow (Wmainform, "panedwin", args, n);
	XtManageChild (Wpanedwin);
	
	/*******************************/
	/* TopForm (drawingarea+lista) */
	/*******************************/
	
	n = 0;
	XtSetArg (args[n], XmNheight, 350); n++;
	Wtopform = XmCreateForm (Wpanedwin, "topform", args, n);
	XtManageChild (Wtopform);
		
	/****************/
	/* Drawing Area */
	/****************/
	
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNscrollingPolicy, XmAUTOMATIC); n++;
	XtSetArg (args[n], XmNscrollBarDisplayPolicy, XmSTATIC); n++;
	Wscrollwin = XmCreateScrolledWindow (Wtopform, "mapSW", args, n);
	XtManageChild (Wscrollwin);

	n = 0;
	XtSetArg (args[n], XmNresizePolicy, XmRESIZE_GROW); n++;
	XtSetArg (args[n], XmNwidth, AREAWIDTH); n++;
	XtSetArg (args[n], XmNheight, no->nchildren*ICONHEIGHT+ 2 * AREAOFFSET); n++;
	Wdrawingarea = XmCreateDrawingArea (Wscrollwin, "map", args, n);
	XtManageChild (Wdrawingarea);
	XtAddCallback (Wdrawingarea, XmNexposeCallback, (XtCallbackProc)ExposeCB, no);

	/*******************/
	/* Graphic Context */
	/*******************/
	
	xgcvals.foreground = 0;
	xgcvals.line_width = ETHERWIDTH;
	no->gc = XtGetGC (Wdrawingarea, GCForeground | GCLineWidth, &xgcvals);	
	
	/**********/
	/* Status */
	/**********/

	n = 0;
	XtSetArg (args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
	XtSetArg (args[n], XmNeditable, False); n++;
	XtSetArg (args[n], XmNcursorPositionVisible, False); n++;
	Wstatus = XmCreateScrolledText (Wpanedwin, "status", args, n);
	XtManageChild (Wstatus);
	no->Wstatus = Wstatus;
	
	/***************/
	/* Disegno LAN */
	/***************/
	
	XtRealizeWidget (Wshell);
	XmUpdateDisplay (Wshell);
	
	for (i=0; i < no->nchildren; i++) {
	
		no2 = no->children[i];
		no2 -> Wshell = Wshell;		/* i nodi hanno la stessa shell della LAN */
		no2 -> Wstatus = Wstatus;	/* idem per lo status */

		/*********/
		/* Icona */
		/*********/

		n = 0;
		XtSetArg (args[n], XmNx, ICONX); n++;
		XtSetArg (args[n], XmNy, AREAOFFSET+ICONHEIGHT*i); n++;

		if (no2->pict && (no2->pmres == XpmSuccess)) {

			XtSetArg (args[n], XmNlabelPixmap, no2->pm); n++;
			XtSetArg (args[n], XmNlabelType, XmPIXMAP); n++;
		}
			
		else {

			if (no2->pict) switch (no2->pmres) {
				
				case XpmColorFailed:
				case XpmColorError:		MotifStatusMsg (Wstatus, "colors not available for %s\n", no2->pict);
										break;
				case XpmOpenFailed:		MotifStatusMsg (Wstatus, "file %s doesn't exist or is not readable\n", no2->pict);
										break;
				case XpmFileInvalid:	MotifStatusMsg (Wstatus, "%s is not a valid pixmap\n", no2->pict);
										break;
				case XpmNoMemory:		MotifStatusMsg (Wstatus, "not enough memory to load %s\n", no2->pict);
										break;
				default:				MotifStatusMsg (Wstatus, "can't use pixmap %s (error %d)", no2->pict, no2->pmres);
			}
			
			XtSetArg (args[n], XmNlabelString, XmStringCreate (no2->name, DEFSET)); n++;
		}

		Wicon = XmCreatePushButton (Wdrawingarea, "icon", args, n);
		XtManageChild (Wicon);
		XtAddCallback (Wicon, XmNactivateCallback, (XtCallbackProc)SelectionCB, no2);
		XmUpdateDisplay (Wshell);
		
		/*************/
		/* Selection */
		/*************/
		
		n = 0;
		XtSetArg (args[n], XmNx, ICONX+120); n++;
		XtSetArg (args[n], XmNy, AREAOFFSET+ICONHEIGHT*i); n++;
		Wframe = XmCreateFrame (Wdrawingarea, "frame", args, n);
		XtManageChild (Wframe);	
	
		n = 0;
		XtSetArg (args[n], XmNwidth, STATUSWIDTH); n++;
		XtSetArg (args[n], XmNrecomputeSize, False); n++;
		XtSetArg (args[n], XmNlabelString, XmStringCreate (" ", DEFSET)); n++;
		Wselection = XmCreateLabel (Wframe, "selection", args, n);
		XtManageChild (Wselection);
		no2->Wselection = Wselection;

		/*******/
		/* Pvm */
		/*******/

		n = 0;
		XtSetArg (args[n], XmNx, ICONX+120); n++;
		XtSetArg (args[n], XmNy, AREAOFFSET+ICONHEIGHT*i+30); n++;
		Wframe = XmCreateFrame (Wdrawingarea, "frame", args, n);
		XtManageChild (Wframe);
	
		n = 0;
		XtSetArg (args[n], XmNwidth, STATUSWIDTH); n++;
		XtSetArg (args[n], XmNrecomputeSize, False); n++;
		if (no2->dtid) { XtSetArg (args[n], XmNlabelString, XmStringCreate ("PVM", DEFSET)); n++;}
		else {XtSetArg (args[n], XmNlabelString, XmStringCreate (" ", DEFSET)); n++; } 
		Wpvm = XmCreateLabel (Wframe, "pvm", args, n);
		XtManageChild (Wpvm);
		no2 -> Wpvm = Wpvm;

		/****************/
		/* Informazioni */
		/****************/
		
		n = 0;
		XtSetArg (args[n], XmNx, ICONX+250); n++;
		XtSetArg (args[n], XmNy, AREAOFFSET+ICONHEIGHT*i); n++;
		Wframe = XmCreateFrame (Wdrawingarea, "frame", args, n);
		XtManageChild (Wframe);
	
		sprintf (buffer, "%s (%s)\n%s", no2->addr, no2->arch, no2->info);
	
		n = 0;
		XtSetArg (args[n], XmNwidth, INFOWIDTH); n++;
		XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
		XtSetArg (args[n], XmNmarginWidth, 5); n++;
		XtSetArg (args[n], XmNmarginHeight, 5); n++;
		XtSetArg (args[n], XmNlabelString, XmStringCreateLtoR (buffer, DEFSET)); n++;
		Winfo = XmCreateLabel (Wframe, "info", args, n);
		XtManageChild (Winfo);
		
		/**************/
		/* Menu Popup */
		/**************/
		
		n = 0;
		Wpopup = XmCreatePopupMenu (Wicon, "popupmenu", args, n);
		XtAddEventHandler (Wicon, ButtonPressMask, FALSE, (XtEventHandler)PopupHandler, no2);
		no2->Wpopup = Wpopup;
		
		MotifAddMenuLabel (Wpopup, no2->name);
		MotifAddMenuSep (Wpopup);
		MotifAddMenuItem (Wpopup, "PVM Add", 'A', (XtCallbackProc)AddCB, no2);
		MotifAddMenuItem (Wpopup, "PVM Remove", 'R', (XtCallbackProc)DelCB, no2);
		MotifAddMenuItem (Wpopup, "PVM Check", 'C', (XtCallbackProc)CheckCB, no2);
		MotifAddMenuItem (Wpopup, "PVM Repair...", 'e', (XtCallbackProc)RepairCB, no2);
		MotifAddMenuSep (Wpopup);
		
		/**** Ping e traceroute sono opzionali... ****/

#ifdef PINGPATH	
		MotifAddMenuItem (Wpopup, "Ping...", 'P', (XtCallbackProc)PingCB, no2);
#endif
#ifdef TROUTEPATH
		MotifAddMenuItem (Wpopup, "Traceroute...", 'T',(XtCallbackProc)TracerouteCB, no2);
#endif

		MotifAddMenuSep (Wpopup);
		MotifAddMenuItem (Wpopup, "Make...", 'M', (XtCallbackProc)MakeCB, no2);
		MotifAddMenuItem (Wpopup, "Spawn...", 'S', (XtCallbackProc)SpawnCB, no2);
		MotifAddMenuSep (Wpopup);
	
		/*************************/
		/* comandi configurabili */
		/*************************/
	
		hc = no2->firstcmd;
		while (hc) {
			if (!strcmp(hc->name, "---")) MotifAddMenuSep (Wpopup);
			else MotifAddMenuItem (Wpopup, hc->name, 0, (XtCallbackProc)CmdCB, hc);
			hc = hc->next;
		}	
	}
}

/******************************************************************************/
/******************************************************************************/

/****************/
/* ErrorHandler */
/****************/

int ErrorHandler (Display * dis, XErrorEvent *xev)
{
	if ((xev->error_code == BadAccess) && (xev->request_code == X_ChangeHosts)) {
		xhosterr = 1;
		return 1;
	}

	XmuPrintDefaultErrorMessage (dis, xev, stderr);
    return 0;
}

/*******************************/
/* LANSelection (node, status) */
/*******************************/

void LANSelection (struct NetworkObj * no, int status)
{
	int n;
	Arg args[5];
	
	no->selection = status;
	if (!no->Wshell) return;
	
	n = 0;
	if (status) {XtSetArg (args[n], XmNlabelString, XmStringCreate ("Selected", DEFSET)); n++;}
	else {XtSetArg (args[n], XmNlabelString, XmStringCreate (" ", DEFSET)); n++;}
	XtSetValues (no->Wselection, args, n);
}

/*************************************/
/* SelectedItems (node, &addr, &mem) */
/*************************************/

int SelectedItems (struct NetworkObj * no, unsigned char *** addr, Boolean ** mem)
{
	int n, i;
	int num;					/* numero di elementi selezionati */
	
	unsigned char ** addrlist;	/* elementi selezionati (formato leggibile) */
	Boolean * memlist;			/* True = rilasciare la memoria */
	
	/****************/
	/* singolo nodo */
	/****************/
	
	if (no->type == NET_HOST) {
		addrlist = calloc (1, sizeof (char *));
		memlist = calloc (1, sizeof (Boolean));
		addrlist[0] = strdup (no->addr);
		memlist[0] = True;
		*addr = addrlist;
		*mem = memlist;
		return (1);
	}
	
	/********************/
	/* nodi selezionati */
	/********************/
	
	num = 0;
	for (n=0; n<no->nchildren; n++) if (no->children[n]->selection) num++;
	if (!num) return (0);
	
	addrlist = calloc (num, sizeof (char *));
	memlist = calloc (num, sizeof(Boolean));
	
	i = 0;
	for (n=0; n<no->nchildren; n++)
		if (no->children[n]->selection) {
			addrlist[i] = strdup (no->children[n]->addr);
			memlist[i] = True;
			i++;
		}
	
	*addr = addrlist;
	*mem = memlist;
	return num;
}

/******************************************************************************/
/******************************************************************************/

/************/
/* ExposeCB */
/************/

void ExposeCB (Widget w, struct NetworkObj * no, int dummy)
{
	int i, y;

	/**** Disegna il cavo e i segmenti ethernet (!) ****/
	
	XDrawLine (XtDisplay(w), XtWindow(w), no->gc, ETHERX, AREAOFFSET/2, ETHERX, no->nchildren * ICONHEIGHT + AREAOFFSET * 3/2);
	
	for (i=0; i < no->nchildren; i++) {
		y = AREAOFFSET+ICONHEIGHT*i+10;
		XDrawLine (XtDisplay(w), XtWindow(w), no->gc, ETHERX, y, ICONX, y);
	}
}

/***************/
/* SelectionCB */
/***************/

void SelectionCB (Widget w, struct NetworkObj * no, int dummy)
{
	LANSelection (no, 1-no->selection);
}

/***********/
/* CloseCB */
/***********/

void CloseCB (Widget w, struct NetworkObj * no, int dummy)
{	
	XtUnmapWidget (no->Wshell);
}

/***************/
/* SelectAllCB */
/***************/

void SelectAllCB (Widget w, struct NetworkObj * no, int dummy)
{	
	int n;
	
	for (n=0; n<no->nchildren; n++) LANSelection (no->children[n], 1);	
}

/*****************/
/* DeselectAllCB */
/*****************/

void DeselectAllCB (Widget w, struct NetworkObj * no, int dummy)
{
	int n;
	
	for (n=0; n<no->nchildren; n++) LANSelection (no->children[n], 0);	
}

/*****************/
/* ClearStatusCB */
/*****************/

void ClearStatusCB (Widget w, struct NetworkObj * no, int dummy)
{	
	MotifClearText (no->Wstatus);
}

/****************/
/* PopupHandler */
/****************/

void PopupHandler (Widget w, struct NetworkObj * no, XEvent * event)
{
	if (event->xbutton.button==Button3) {
		XmMenuPosition (no->Wpopup, (XButtonPressedEvent *)event);
		XtManageChild (no->Wpopup);
	}
}

/******************************************************************************/
/******************************************************************************/

/***********/
/* CheckCB */
/***********/

void CheckCB (Widget w, struct NetworkObj * no, int dummy)
{	
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
	
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "Nessun host selezionato!\n");
		return;
	}
	
	/*********/
	/* Check */
	/*********/
		
	PVMCheck (no, num, hosts, mem);

	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);						
}

/***********/
/* RepairCB */
/***********/

void RepairCB (Widget w, struct NetworkObj * no, int dummy)
{	
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
	
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "Nessun host selezionato!\n");
		return;
	}

	/**********/
	/* Repair */
	/**********/
		
	if (MotifQuestion(no->Wshell, "Repair", "Are you sure you want to repair selected hosts?\n" "If PVM daemons are still working on those hosts,\n" "this operation will destroy them."))
		PVMRepair (no, num, hosts, mem);

	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);						
}

/**********/
/* PingCB */
/**********/

void PingCB (Widget w, struct NetworkObj * no, int dummy1)
{
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
		
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "No host selected!\n");
		return;
	}

	/********/
	/* Ping */
	/********/
	
	for (n=0; n<num; n++) InternetPing (no, hosts[n]);
	
	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);					
}

/****************/
/* TracerouteCB */
/****************/

void TracerouteCB (Widget w, struct NetworkObj * no, int dummy1)
{
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
		
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "No host selected!\n");
		return;
	}

	/**************/
	/* Traceroute */
	/**************/
	
	for (n=0; n<num; n++) InternetTraceroute (no, hosts[n]);
	
	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);					
}

/*********/
/* AddCB */
/*********/

void AddCB (Widget w, struct NetworkObj * no, int dummy)
{	
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
	
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "No host selected!\n");
		return;
	}
	
	/*******/
	/* Add */
	/*******/
	
	MotifStatusMsg (no->Wstatus, "Running AddHosts. Hosts will be added in a few seconds.\n");
	PVMAdd (no, num, hosts, mem);
	
	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);					
}

/*********/
/* DelCB */
/*********/

void DelCB (Widget w, struct NetworkObj * no, int dummy)
{	
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
		
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "No host selected!\n");
		return;
	}
	
	/*******/
	/* Del */
	/*******/
		
	MotifStatusMsg (no->Wstatus, "Running DelHosts. Hosts will be deleted in a few seconds.\n");
	PVMDel (no, num, hosts, mem);
		
	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);

	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);					
}

/**********/
/* MakeCB */
/**********/

void MakeCB (Widget w, struct NetworkObj * no, int dummy1)
{
	int num, n;
	unsigned char ** hosts;
	Boolean * mem;
	
	/********************************/
	/* Preleva gli host dalla lista */
	/********************************/
	
	num = SelectedItems (no, &hosts, &mem);
	if (!num) {
		MotifStatusMsg (no->Wstatus, "No host selected!\n");
		return;
	}

	/********/
	/* Make */
	/********/

	MakeAdd (no->Wshell, num, hosts);
	
	/**********************************************/
	/* deseleziona gli host e rilascia la memoria */
	/**********************************************/
	
	if (no->type == NET_LAN) for (n=0; n<no->nchildren; n++)
		LANSelection (no->children[n], 0);
	
	for (n=0; n<num; n++) if (mem[n]) free (hosts[n]);
	free (mem);
	free (hosts);					
}

/*********/
/* CmdCB */
/*********/

void CmdCB (Widget w, struct HostCmd * hc, int dummy)
{
	unsigned char buffer[1024];
	int res;
	unsigned char * errmsg;

	/********************/
	/* Controllo tasker */
	/********************/

	if (! hc->no->dtid) {
		MotifStatusMsg (hc->no->Wstatus, "Can't execute remote command: host not in PVM.\n");
		return;
	}
	
	/**********************/
	/* Prepara il comando */
	/**********************/

	if (hc->x) {
		if (!hc->no->xhost) {
	
			/* Tenta un xhost */
	
			xhosterr = 0;
			MotifXHost (XtDisplay(w), hc->no->addr, True);
			if (xhosterr) MotifStatusMsg (hc->no->Wstatus, "Can't execute 'xhost' for this display: command could fail.\n");
			else hc->no->xhost = 1;
		}
		
		sprintf (buffer, "export DISPLAY; DISPLAY=%s ; %s", displayaddr, hc->cmd);
	}
	else sprintf (buffer, "%s", hc->cmd);

	/*******************/
	/* Invio richiesta */
	/*******************/

	pvm_packf ("%+ %d %s", PvmDataDefault, (int)hc->no, buffer);
	res = pvm_send (hc->no->taskertid, T_TSKR_CMD);
	if (res<0) {
		errmsg = PVMError (res);
		MotifStatusMsg (hc->no->Wstatus, "command failed: %s\n", errmsg);
		free (errmsg);
	}
}

/***********/
/* SpawnCB */
/***********/

void SpawnCB (Widget w, struct NetworkObj * no, int dummy)
{
	SpawnWindow (no);
}

/**************/
/* TasksWinCB */
/**************/

void TasksWinCB (Widget w, struct NetworkObj * no, int dummy)
{
	TasksWindow();
}

/*************/
/* MakeWinCB */
/*************/

void MakeWinCB (Widget w, struct NetworkObj * no, int dummy)
{
	MakeWindow();
}

