/*****************************************************************************\
*                                                                             *
*  PVMTasker.c - PSEUDO tasker ed "esecutore remoto" per interfaccia          *
*  1.0 - 29/3/95                                                              *
*                                                                             *
\*****************************************************************************/

/*
 *
 *          WAMM version 1.0: Wide Area Metacomputer Manager
 *     CNUCE - Institute of the Italian National Research Council
 *      Authors:  R. Baraglia, G. Faieta, M. Formica, D. Laforenza
 *                   (C) 1995 All Rights Reserved
 *
 *                              NOTICE
 *
 *
 * Permission is hereby granted, without written agreement and without license
 * or royalty fees, to use, copy, modify, and distribute this software and
 * its documentation for educational and research purpose only, provided that
 * the above copyright notice and the following two paragraphs appear in all
 * copies of this software and in the supporting documentation. No charge,
 * other than an "at-cost" distribution fee, may be charged for copies,
 * derivations, or distributions of this material without the express written
 * consent of the copyright holder.
 * 
 * IN NO EVENT SHALL THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS BE LIABLE TO
 * ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 * DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THEINSTITUTION OR THE AUTHORS HAS BEEN ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * THE INSTITUTION (CNUCE-CNR) AND THE AUTHORS SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUTHORS HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <pvm3.h>
#include <sys/wait.h>
#include <signal.h>
#include "tags.h"
#include "util.h"

/*************/
/* Strutture */
/*************/

struct Task {
	struct Task * prev, * next;		/* precedente, successivo */
	int pid;						/* pid processo */
	int tid;						/* tid (solo per task PVM) */
	int checking;					/* 1 = controllo (solo task PVM) */
	int no;							/* NetworkObj (solo task normali) */
	unsigned char * out;			/* stdout (solo task normali) */
	unsigned char * err;			/* stderr (solo task normali) */
};

/*********************/
/* variabili globali */
/*********************/

struct Task * TaskList;
struct Task * PVMTaskList;

int MyTid;		/* tid tasker */
int ParentTid;	/* tid interfaccia */
int DTid;		/* tid demone locale */

/***************/
/* definizioni */
/***************/

#define VERSION		"1.0"		/* versione tasker */
#define DELAY		500000		/* ritardo (us) tra un controllo e l'altro */
#define CHECK		10000000	/* us tra un controllo sul master e l'altro */

/************/
/* funzioni */
/************/

extern void pvmendtask (void);

struct Task *		AddTask (struct Task **);
void				DelTask (struct Task **, struct Task *);
void				CheckTasks (int);
struct Task *		FindPVMTask (int);

void 				StartCmd (void);

/********/
/* main */
/********/

void main (int argc, char ** argv)
{	
	int ver;					/* 1=versione ok */
	struct timeval tv;			/* timeval per pvm_trecv */
	int bufid, tag, tid;		/* informazioni sul messaggio ricevuto */
	int check;					/* contatore controllo master */
	
	/*************************/
	/* Registra il programma */
	/*************************/
	
	MyTid = pvm_mytid();
	ParentTid = pvm_parent();
	DTid = pvm_tidtohost (MyTid);
	
	pvm_setopt (PvmRoute, PvmDontRoute);
	
	signal (SIGTERM, SIG_IGN);
	signal (SIGINT, SIG_IGN);
	
	/****************************/
	/* Controllo versione e ack */
	/****************************/
		
	ver = (!strcmp(VERSION,argv[1]));
	
	pvm_packf ("%+ %d", PvmDataDefault, ver);
	pvm_send (ParentTid, T_TSKR_ACK);
	
	if (!ver) {
		pvm_exit();
		exit(1);
	}
		
	/*************/
	/* Main Loop */
	/*************/
	
	tv.tv_sec = 0;
	tv.tv_usec = DELAY;
	
	check = 0;
	
	for (;;) {
	
		/**** attende un messaggio o il timeout ****/
	
		if ((bufid = pvm_trecv (-1, -1, &tv)) < 0) break;
		
		/**** controllo master ****/
		
		check += DELAY;
		if (check >= CHECK) {

			if (pvm_pstat (ParentTid) < 0 ) {
			
				/**** l'interfaccia non esiste piu`: usciamo. ****/
			
				pvm_exit ();
				exit (1);
			}
			
			check = 0;
		}
		
		/**** controlla i task ****/
				
		CheckTasks (DTid);		
		
		/**** messaggi ****/
		
		if (bufid<=0) continue;
		pvm_bufinfo (bufid, NULL, &tag, &tid);
	
		switch (tag) {
			
			/**************/					
			/* T_TSKR_CMD */
			/**************/
	
			case T_TSKR_CMD:	StartCmd ();
								break;
		}
	}
	
	/**** il pvmd e` morto: e` bene uscire... ****/
	
	/* pvm_exit(); XXX e` inutile... */
	exit (0);	
}

/*******************************************************/
/* StartCmd () - esegue un comando (T_TSKR_CMD no cmd) */
/*******************************************************/

void StartCmd ()
{
	int no;					/* NetworkObj (ignorato dal tasker) */
	char buffer[1024];		/* comando da eseguire */

	struct Task * task;		/* processo figlio */
	int pid;

	unsigned char * out;	/* stdout */
	unsigned char * err;	/* stderr */

	/**** preleva gli argomenti ****/

	pvm_unpackf ("%d %s", &no, buffer);
	
	/**** prepara il comando ****/
	
	out = strdup (tmpnam(NULL));
	err = strdup (tmpnam(NULL));
	sprintf (buffer+strlen(buffer), " 1> %s 2> %s", out, err);
	
	/**** Fork ****/
	
	pid = fork();
	if (!pid) {
	
		/**** Figlio ****/
		
		/* chiude il socket con il pvmd e resetta mytid interno */
		
		pvmendtask();
		
		/* il figlio deve avere segnali standard! */
		
		signal (SIGTERM, SIG_DFL);
		signal (SIGINT, SIG_DFL);
		
		execlp ("sh", "sh", "-c", buffer, NULL);
		
		/**** shell non trovata (!) ****/

		_exit(1);
	}
	
	/**** Padre ****/
	
	if (pid == -1) {
		/* XXX inviare messaggio di errore all'interfaccia. */
		free (out);
		free (err);
		return;
	}
	
	task = AddTask(&TaskList);
	task->pid = pid;
	task->no = no;
	task->out = out;
	task->err = err;
}

/**************/
/* CheckTasks */
/**************/

void CheckTasks (int dtid)
{
	struct Task * task, * tmp;
	int n;
	
	int ntask;					/* risultati pvm_tasks */
	struct pvmtaskinfo * ti;

	int res;					/* risultato processo */
	unsigned char * outbuf;		/* buffer per stdout */
	unsigned char * errbuf;		/* buffer per stderr */
	
	/****************************/
	/* Controlla i task non PVM */
	/****************************/
	
	task = TaskList;
	while (task) {
	
		tmp = task->next;
	
		if (waitpid (task->pid, &res, WNOHANG)) {
	
			/*************************/
			/* Invia stdout e stderr */
			/*************************/
				
			pvm_packf ("%+ %d %d", PvmDataDefault, task->no, res);
			pvm_packf ("%d %s", ReadFile (task->out, &outbuf), outbuf);
			pvm_packf ("%d %s", ReadFile (task->err, &errbuf), errbuf);
			pvm_send (ParentTid, T_TSKR_CMD);
			unlink (task->out);
			unlink (task->err);
			
			DelTask (&TaskList, task);
		}
		
		task = tmp;
	}
	
	/************************/
	/* Controlla i task PVM */
	/************************/
	
	if ((pvm_tasks (DTid, &ntask, &ti)) < 0) return;
	
	/**** Marca tutti i task come "checking" ****/
	
	task = PVMTaskList;
	while (task) {
		task->checking = 1;
		task = task->next;
	}
	
	/**** Esame dei task ****/
	
	for (n=0; n<ntask; n++) {
		
		task = FindPVMTask (ti[n].ti_tid);
		if (task) task->checking = 0;			/* era gia` presente */
		
		else {
			
			/**************/
			/* Nuovo task */
			/**************/
			
			task = AddTask (&PVMTaskList);
			task -> tid = ti[n].ti_tid;
			
			/**** informa l'interfaccia ****/
								
			pvm_packf ("%+ %d", PvmDataDefault, ti[n].ti_tid);
			pvm_send (ParentTid, T_TSKR_NEW);
		}	
	}
	
	/**** Tutti i task rimasti con "checking" sono terminati ****/
	
	task = PVMTaskList;
	while (task) {
	
		tmp = task->next;
		
		if (task->checking) {
		
			/**** informa l'interfaccia ****/
			
			pvm_packf ("%+ %d", PvmDataDefault, task->tid);
			pvm_send (ParentTid, T_TSKR_END);
			
			DelTask (&PVMTaskList, task);
		}
		
		task = tmp;
	}
}

/*************************************************/
/* AddTask (&list) - aggiunge un task alla lista */
/*************************************************/

struct Task * AddTask (struct Task ** list)
{
	struct Task * task;
	
	task = calloc (1, sizeof (struct Task));
	
	/**** aggiunge il task all'inizio ****/
	
	task->prev = NULL;
	task->next = *list;
	
	if (*list) (*list)->prev = task;
	*list = task;
	
	return (task);
}

/*******************************************************/
/* DelTask (&list, task) - rimuove un task dalla lista */
/*******************************************************/

void DelTask (struct Task ** list, struct Task * task)
{
	/**** sgancia l'elemento ****/
	
	if (task->prev) task->prev->next = task->next;
	if (task->next) task->next->prev = task->prev;
	
	if (task == *list) *list = task->next;
	
	/**** svuota la struttura ****/
	
	if (task->out) free (task->out);
	if (task->err) free (task->err);
	free (task);
}

/*****************************************/
/* FindPVMTask (tid) - cerca un task PVM */
/*****************************************/

struct Task * FindPVMTask (int tid)
{
	struct Task * task;
	
	task = PVMTaskList;	
	while (task && (task->tid != tid)) task = task->next;
	return (task);
}
