#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#ifndef NO_STRING_H
#   include <string.h>
#else
#   include <strings.h>
#endif

#include "tkpvmInt.h"
#include "patchlevel.h"

#if defined(__WIN32__)
#   define WIN32_LEAN_AND_MEAN
#   include <windows.h>
#   undef WIN32_LEAN_AND_MEAN

/*
 * VC++ has an alternate entry point called DllMain, so we need to rename
 * our entry point.
 */

#   if defined(_MSC_VER)
#	define EXPORT(a,b) __declspec(dllexport) a b
#	define DllEntryPoint DllMain
#   else
#	if defined(__BORLANDC__)
#	    define EXPORT(a,b) a _export b
#	else
#	    define EXPORT(a,b) a b
#	endif
#   endif
#else
#   define EXPORT(a,b) a b
#endif

/*
 * Declarations for functions defined in this file.
 */

EXTERN EXPORT(int,Pvm_Init) _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN EXPORT(int,Pvm_SafeInit) _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * Default directory in which to look for libraries:
 */

static char defaultLibraryDir[200] = PVM_LIBRARY;

/*
 * The following string is the startup script executed in new
 * interpreters.  It looks on disk in several different directories
 * for a script "pvm.tcl" that is compatible with this version
 * of Tkpvm.  The pvm.tcl script does all of the real work of
 * initialization.
 */

static char initScript[] =
"proc tkpvmInit {} {\n\
    global pvm_library tkpvm_version tkpvm_patchLevel env\n\
    rename tkpvmInit {}\n\
    set dirs {}\n\
    if [info exists env(PVM_LIBRARY)] {\n\
	lappend dirs $env(PVM_LIBRARY)\n\
    }\n\
    if [info exists env(EXT_FOLDER)] {\n\
	lappend dirs [file join $env(EXT_FOLDER) \"Tool Command Language\" lib tkpvm$tkpvm_version]\n\
    }\n\
    lappend dirs $pvm_library\n\
    set pvm_library {}\n\
    if ![catch {uplevel #0 source -rsrc pvm}] {\n\
	return\n\
    }\n\
    lappend dirs [file join [file dirname [info library]] tkpvm$tkpvm_version]\n\
    set parentDir [file dirname [file dirname [info nameofexecutable]]]\n\
    lappend dirs [file join $parentDir lib tkpvm$tkpvm_version]\n\
    if {![regexp {.*[ab][12345]} $tkpvm_patchLevel lib]} {\n\
	set lib tkpvm$tkpvm_version\n\
    }\n\
    lappend dirs [file join [file dirname $parentDir] $lib library]\n\
    lappend dirs [file join $parentDir library]\n\
    foreach i $dirs {\n\
	set pvm_library $i\n\
	if ![catch {uplevel #0 source [list [file join $i pvm.tcl]]}] {\n\
	    return\n\
	}\n\
    }\n\
    set pvm_library {}\n\
    set msg \"Can't find a usable pvm.tcl in the following directories: \n\"\n\
    append msg \"    $dirs\n\"\n\
    append msg \"This probably means that Tkpvm wasn't installed properly.\n\"\n\
    error $msg\n\
}\n\
tkpvmInit";

static char safeInitScript[] =
"proc tkpvmInit {} {\n\
    global pvm_library tkpvm_version tkpvm_patchLevel env\n\
    rename tkpvmInit {}\n\
    set dirs {}\n\
    if [info exists env(PVM_LIBRARY)] {\n\
	lappend dirs $env(PVM_LIBRARY)\n\
    }\n\
    if [info exists env(EXT_FOLDER)] {\n\
	lappend dirs [file join $env(EXT_FOLDER) \"Tool Command Language\" lib tkpvm$tkpvm_version]\n\
    }\n\
    lappend dirs $pvm_library\n\
    set pvm_library {}\n\
    if ![catch {uplevel #0 source -rsrc pvm}] {\n\
	return\n\
    }\n\
    lappend dirs [file join [file dirname [info library]] tkpvm$tkpvm_version]\n\
    set parentDir [file dirname [file dirname [info nameofexecutable]]]\n\
    lappend dirs [file join $parentDir lib tkpvm$tkpvm_version]\n\
    if {![regexp {.*[ab][12345]} $tkpvm_patchLevel lib]} {\n\
	set lib tkpvm$tkpvm_version\n\
    }\n\
    lappend dirs [file join [file dirname [file dirname [info library]]] $lib library]\n\
    foreach i $dirs {\n\
	set pvm_library $i\n\
	if ![catch {uplevel #0 source [list [file join $i pvm.tcl]]}] {\n\
	    return\n\
	}\n\
    }\n\
    set pvm_library {}\n\
    set msg \"Can't find a usable pvm.tcl in the following directories: \n\"\n\
    append msg \"    $dirs\n\"\n\
    append msg \"This probably means that Tkpvm wasn't installed properly.\n\"\n\
    error $msg\n\
}\n\
tkpvmInit";

static char **initScripts = NULL;

typedef struct {
    char *name;			/* Name of command. */
    int (*cmdProc) _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
	    int argc, char **argv));
				/* Command procedure. */
    int flags;			/* If flag 0 is set, this command will be
                                 * exposed in a safe interpreter. Otherwise
                                 * it will be hidden in a safe interpreter.
				 * If flag 1 is set, this command is availabe
				 * as is, otherwise it is only availabe as
				 * option for the "pvm" command. */
} TkpvmCmd;

static TkpvmCmd commands[] = {
    {"addhost",		Pvm_AddhostCmd,		2},
    {"bind",		Pvm_BindCmd,		1},
    {"conf",		Pvm_ConfCmd,		3},
    {"delhost",		Pvm_DelhostCmd,		2},
    {"halt",		Pvm_HaltCmd,		2},
    {"joingroup",	Pvm_JoingroupCmd,	3},
    {"kill",		Pvm_KillCmd,		0},
    {"leavegroup",	Pvm_LeavegroupCmd,	3},
    {"parent",		Pvm_ParentCmd,		3},
    {"pvm",		Pvm_PvmCmd,		3},
    {"recv",		Pvm_RecvCmd,		3},
    {"send",		Pvm_SendCmd,		1},
    {"spawn",		Pvm_SpawnCmd,		2},
    {"tasks",		Pvm_TasksCmd,		3},
    {"tid",		Pvm_TidCmd,		3},
    {(char *) NULL,	(int (*) _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **))) NULL}
};

/*
 *----------------------------------------------------------------------
 *
 * Pvm_PvmCmd --
 *
 *	This procedure is invoked to process the "pvm" Pvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_PvmCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    size_t length;
    char *temp, *argv1;
    TkpvmCmd *cmdPtr;

    int c, isSafe, result;
    int (*optfn) _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **)) =
	    (int (*) _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **))) NULL;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    isSafe = Tcl_IsSafe(interp);
    for (cmdPtr = commands; cmdPtr->name != NULL; cmdPtr++) {
	if (((!isSafe) || (cmdPtr->flags & 1)) &&
		!strncmp(cmdPtr->name, argv[1], length)) {
	    if (optfn != NULL) {
		optfn = NULL;
		break;
	    }
	    optfn = cmdPtr->cmdProc;
	}
    }
    if (optfn != NULL) {
	temp = (char *) ckalloc (strlen(argv[0]) + strlen(argv[1]) + 2);
	strcpy(temp,argv[0]);
	strcat(temp," ");
	strcat(temp,argv[1]);
	argv1 = argv[1];
	argv[1] = temp;
	result = (* optfn) (dummy, interp, argc-1, argv+1);
	argv[1] = argv1;
	ckfree(temp);
	return result;
    }
    Tcl_AppendResult(interp, "bad or ambigious option \"", argv[1],
	    "\": should be addhost, bind, delhost, halt, ",
	    "joingroup, kill, leavegroup, parent, recv, send, ",
	    "spawn, tasks or tid,", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_TidCmd --
 *
 *	This procedure is invoked to process the "tid" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_TidCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int tid;
    char storage[16];

    if (argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    tid = pvm_mytid();
    if (tid==PvmSysErr) {
	return Pvm_ReturnError(interp,PvmSysErr);
    }
    sprintf(storage,"0x%x",tid);
    Tcl_AppendResult(interp,storage,(char *)NULL);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_AddhostCmd --
 *
 *	This procedure is invoked to process the "addhost" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_AddhostCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error;
    char storage[16];

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], "\" host", (char *) NULL);
	return TCL_ERROR;
    }
    argv+=1; argc-=1;
    while (argc--){
	if (pvm_addhosts(argv,1,&error)<1 || error<0 ) {
	    if (error>=0) error = PvmBadParam;
	    return Pvm_ReturnError(interp,error);
	}
	argv++;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_HaltCmd --
 *
 *	This procedure is invoked to process the "halt" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_HaltCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error;
    char storage[16];

    if (argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    signal(SIGTERM, SIG_IGN);
    pvm_halt();
    Tcl_Exit(0);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_ConfCmd --
 *
 *	This procedure is invoked to process the "conf" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_ConfCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int no_of_hosts;
    int arch_info;
    struct pvmhostinfo *host_info;
    int i;
    char buffer[20];

    if (argc != 1) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!pvm_config(&no_of_hosts, &arch_info, &host_info)) {
	for (i = 0; i < no_of_hosts; i++) {
	    sprintf(buffer," {0x%x",host_info[i].hi_tid);
	    Tcl_AppendResult(interp, buffer + (i==0), (char *) NULL);
	    Tcl_AppendElement(interp, host_info[i].hi_name);
	    Tcl_AppendElement(interp, host_info[i].hi_arch);
	    sprintf(buffer," %d}",host_info[i].hi_speed);
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	}
    }
    return (TCL_OK);
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_DelhostCmd --
 *
 *	This procedure is invoked to process the "delhost" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_DelhostCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error;
    char storage[16];

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], "\" host", (char *) NULL);
	return TCL_ERROR;
    }
    pvm_delhosts(argv+1,argc-1,NULL);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_JoingroupCmd --
 *
 *	This procedure is invoked to process the "joingroup" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_JoingroupCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error;
    char storage[16];

    if (argc!=2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			    " groupname\"",(char *) NULL);
	return TCL_ERROR;
    }
    if ((error = pvm_joingroup(argv[1]))>=0) {
	sprintf(storage,"%d",error);
	Tcl_AppendResult(interp,storage,(char *)NULL);
	return TCL_OK;
    }
    return Pvm_ReturnError(interp,error);
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_LeavegroupCmd --
 *
 *	This procedure is invoked to process the "leavegroup" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_LeavegroupCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error;
    char storage[16];

    if (argc!=2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
				" groupname\"",(char *) NULL);
	return TCL_ERROR;
    }
    return Pvm_ReturnError(interp,pvm_lvgroup(argv[1]));
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_ParentCmd --
 *
 *	This procedure is invoked to process the "parent" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_ParentCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int error, tid;
    char storage[16];

    if (argc<2) {
	tid = pvm_parent();
	if (tid==PvmSysErr) {
	    return Pvm_ReturnError(interp,tid);
	}
	if (tid!=PvmNoParent) {
	    sprintf(storage,"0x%x",tid);
	    Tcl_AppendResult(interp,storage,(char *)NULL);
	}
	return TCL_OK;
    }
    if (argc<3) {
	struct pvmtaskinfo *taskp;
	int i,ntask;
	if ((error = pvm_tasks(0, &ntask, &taskp))<0)
	    return Pvm_ReturnError(interp,error);
	if (Tcl_GetInt(interp, argv[1],&tid) != TCL_OK) {
	    return TCL_ERROR;
	}
	i=0; while(i<ntask && taskp[i].ti_tid!=tid) i++;
	if (i<ntask && taskp[i].ti_ptid) {
	    sprintf(storage,"0x%x",taskp[i].ti_ptid);
	    Tcl_AppendResult(interp, storage,(char*)NULL);
	}
	return TCL_OK;	    
    } else {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			    " tid\"",(char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PvmSetInitScript --
 *
 *	This procedure sets the initialization script used in Pvm_Init().
 *	It will be used instead of the file "pvm.tcl" in all future calls
 *	to Pvm_Init. If the argument is NULL, pvm.tcl will be used again.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

void
PvmSetInitScript(script)
    char  **script;		/* Script to be executed. */
{
    initScripts = script;
}

/*
 *----------------------------------------------------------------------
 *
 * DllEntryPoint --
 *
 *	This wrapper function is used by Windows to invoke the
 *	initialization code for the DLL.  If we are compiling
 *	with Visual C++, this routine will be renamed to DllMain.
 *	routine.
 *
 * Results:
 *	Returns TRUE;
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

#ifdef __WIN32__
BOOL APIENTRY
DllEntryPoint(hInst, reason, reserved)
    HINSTANCE hInst;		/* Library instance handle. */
    DWORD reason;		/* Reason this function is being called. */
    LPVOID reserved;		/* Not used. */
{
    return TRUE;
}
#endif

/*
 *--------------------------------------------------------------
 *
 * Pvm_Init , Pvm_SafeInit--
 *	Create pvm commands.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
EXPORT(int,Pvm_Init)(interp)
    Tcl_Interp *interp;
{
    char buf[10];
    register char *libDir;
    register char *p;
    register TkpvmCmd *cmdPtr;
#ifdef ITCL_NAMESPACES
    Itcl_Namespace spaceId;	/* Token for "pvm" namespace created, used
				 * to delete the namespace on errors. */
    char cmdPath[200];
#endif
    int isSafe;

    if (Tcl_PkgProvide(interp, "Pvm", TKPVM_VERSION) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Set variables for the intepreter.
     */

    if (!PvmMytid) {
	if (Pvm_StartDaemon(interp)<=0) {
	    Tcl_AppendResult(interp,"Cannot start pvm daemon",(char *)NULL);
	    return TCL_ERROR;
	}
	if (!PvmInterp) {
	    PvmInterp = interp;
	}
    }

    isSafe = Tcl_IsSafe(interp);

#ifdef ITCL_NAMESPACES
    if (Itcl_CreateNamesp(interp, "pvm", (ClientData)0, (Itcl_DeleteProc *)0,
		  &spaceId) != TCL_OK) {
	return TCL_ERROR;
    }
    for (cmdPtr = commands; cmdPtr->name != NULL; cmdPtr++) {
	if ((cmdPtr->flags & 2) && ((!isSafe) || (cmdPtr->flags & 1))) {
	    sprintf(cmdPath, "pvm::%s", cmdPtr->name);
	    Tcl_CreateCommand(interp, cmdPath, cmdPtr->cmdProc,
		    (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
	}
    }
#else
    for (cmdPtr = commands; cmdPtr->name != NULL; cmdPtr++) {
	if ((cmdPtr->flags & 2) && ((!isSafe) || (cmdPtr->flags & 1))) {
	    Tcl_CreateCommand(interp, cmdPtr->name, cmdPtr->cmdProc,
		    (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
	}
    }
#endif

    libDir = Tcl_GetVar(interp,"pvm_library",TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	Tcl_SetVar(interp, "pvm_library", defaultLibraryDir,	TCL_GLOBAL_ONLY);
    }

    Tcl_SetVar(interp, "pvm_version",     pvm_version(),	TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tkpvm_patchLevel",TKPVM_PATCHLEVEL,	TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tkpvm_version",   TKPVM_VERSION,	TCL_GLOBAL_ONLY);

    if (initScripts != NULL) {
	char **p = initScripts;
	Tcl_DString data;

	Tcl_SetVar(interp, "pvm_library", "", TCL_GLOBAL_ONLY);
	Tcl_DStringInit(&data);
	while(*p) {
	    /* Copy the constant into a dynamic string. This */
	    /* is necessary because Tcl doesn't accept    */
	    /* constants as an argument to Tcl_Eval()        */
	    Tcl_DStringSetLength(&data,0);
	    Tcl_DStringAppend(&data,*p++,-1);
	    if(Tcl_Eval(interp,Tcl_DStringValue(&data)) == TCL_ERROR) {
		Tcl_DStringFree(&data);
#ifdef ITCL_NAMESPACES
		Itcl_DeleteNamesp(spaceId);
#endif
		return TCL_ERROR;
	    }
	}
	Tcl_DStringFree(&data);
	return TCL_OK;
    }
    if (Tcl_Eval(interp, initScript) != TCL_OK) {
#ifdef ITCL_NAMESPACES
	Itcl_DeleteNamesp(spaceId);
#endif
	return TCL_ERROR;
    }
    return TCL_OK;
}

EXPORT(int,Pvm_SafeInit)(interp)
    Tcl_Interp *interp;
{
    return Pvm_Init(interp);
}
