#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#ifndef NO_STRING_H
#   include <string.h>
#else
#   include <strings.h>
#endif

#include <pwd.h>
#include "tkpvmInt.h"

/*
 * The data structure below is used by the "pvm_bind" command to remember
 * the command to be executed later.
 */

typedef struct TkpvmBindInfo {
    Tcl_Interp *interp;		/* Interpreter in which to execute command. */
    char *command;		/* Command to execute.  Malloc'ed, so must
				 * be freed when structure is deallocated. */
    int tid;			/* task identifier (or -1 if none) */
    int msgtag;			/* message identifier (or -1 if none) */
    struct TkpvmBindInfo *nextPtr;/* Next in list of all "pvm bind" commands for
				 * the application. */
} TkpvmBindInfo;

static TkpvmBindInfo *firstPvmBindPtr = NULL;
				/* First in list of all pending "pvm bind"
				 * commands. */


static void EvalCommand _ANSI_ARGS_((int tid, int msgtag, int number,
	ClientData clientData));

void PvmCleanup (tid)
int tid;
{
    TkpvmBindInfo *prevPtr, *Ptr;
	prevPtr=NULL; Ptr = firstPvmBindPtr;
	while (Ptr) {
	    if (Ptr->tid==tid) {
		if (prevPtr) {
		    prevPtr->nextPtr = Ptr->nextPtr;
		} else {
		    firstPvmBindPtr = Ptr->nextPtr;
		}
	    } else {
		prevPtr = Ptr;
	    }
	    Ptr = Ptr->nextPtr;
	}
}


/*
 *----------------------------------------------------------------------
 *
 * Pvm_BindCmd --
 *
 *	This procedure is invoked to process the "bind" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_BindCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int tid, pvmerror, msgtag;
    char storage[16];
    char *command;
    TkpvmBindInfo *prev_pevPtr, *pevPtr;

    if (!PvmMytid) {
	if (!Pvm_StartDaemon(interp))
	    return TCL_ERROR;
    }
    if (argc<3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			    " tid msgtag ?command?\"",(char *) NULL);
	return Pvm_ReturnError(interp,pvmerror);
    }
    if (!strcmp(argv[1],"parent")) {
	tid = pvm_parent();
	if (tid==PvmSysErr||tid==PvmNoParent) {
	    pvmerror = tid;
	    return Pvm_ReturnError(interp,pvmerror);
	}
    } else if (!strcmp(argv[1],"id") || !strcmp(argv[1],"tid")) {
	tid = PvmMytid;
    } else if (!strcmp(argv[1],"any")) {
	tid = -1;
    } else if (Tcl_GetInt(interp, argv[1], &tid) != TCL_OK) {
	return Pvm_ReturnError(interp,pvmerror);
    } else if (tid<0x40000) {
	pvmerror = PvmBadParam;
	return Pvm_ReturnError(interp,pvmerror);
    }
    if (argc>2 && *argv[2] && (tid != -1)) {
	PvmNotifyTid(interp, tid, -1, (Tcl_Channel) NULL);
    }
    pvmerror = -1;
    if (!strcmp(argv[2],"any"))
	msgtag = -1;
    else if (!strcmp(argv[2],"kill"))
	msgtag = -2;
    else if (Tcl_GetInt(interp, argv[2], &msgtag) != TCL_OK) {
	return Pvm_ReturnError(interp,pvmerror);
    } else if (msgtag<0&&msgtag>-3) {
	Tcl_AppendResult(interp,"The value \"",argv[2],
			     "\" is reserved for internal use",(char *) NULL);
	return Pvm_ReturnError(interp,pvmerror);
    }
    if (argc < 4) {
	command = (char *)NULL;
    } else if (argc == 4) {
	command = (char *) ckalloc((unsigned)(strlen(argv[3])+1));
	strcpy(command,argv[3]);
    } else {
	command = Tcl_Concat(argc-3, argv+3);
    }

	/*
	 * Locate an existing pvm handler for this file, if one exists,
	 * and make a new one if none currently exists.
	 */
	prev_pevPtr = NULL;
	for (pevPtr = firstPvmBindPtr; ; prev_pevPtr = pevPtr, pevPtr = pevPtr->nextPtr) {
	    if (pevPtr == NULL) {
		if (argc < 4) {
		    return TCL_OK;
		}
		pevPtr = (TkpvmBindInfo *) ckalloc(sizeof(TkpvmBindInfo));
		pevPtr->interp = interp;
		pevPtr->tid = tid;
		pevPtr->msgtag = msgtag;
		pevPtr->command = NULL;
		pevPtr->nextPtr = firstPvmBindPtr;
		prev_pevPtr = NULL;
		firstPvmBindPtr = pevPtr;
		break;
	    }
	    if (pevPtr->tid == tid && pevPtr->msgtag == msgtag) {
		break;
	    }
	}
	if (command==NULL) {
	    Tcl_AppendResult(interp, pevPtr->command, (char *)NULL);
	    return TCL_OK;
	} else if (*command=='\0') {
	    Pvm_DeleteEventHandler(tid ,msgtag);
	    if (pevPtr->command) {
		ckfree(pevPtr->command);
	    }
	    if (prev_pevPtr == NULL) {
		firstPvmBindPtr = firstPvmBindPtr->nextPtr;
	    } else {
		prev_pevPtr->nextPtr = pevPtr->nextPtr;
	    }
	    ckfree(pevPtr);
	} else {
	    Pvm_CreateEventHandler(tid ,msgtag ,EvalCommand
						,(ClientData)pevPtr);
	    if (pevPtr->command) {
		ckfree(pevPtr->command);
	    }
	    pevPtr->command = command;

	}
        pvmerror = 0;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Pvm_SpawnCmd --
 *
 *	This procedure is invoked to process the "spawn" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_SpawnCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int tid, pvmerror, i;
    char storage[16];
    int flag=0,ntask=1,*tids,*t;
    char *where = NULL,*process= NULL; Tcl_Channel channel = (Tcl_Channel) NULL; int msgtag = -1;

    if (argc<2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			    " ?options? process\"",(char *) NULL);
	return Pvm_ReturnError(interp,pvmerror);
    }
    argc-=1; argv+=1;
    while(argc>1 && **argv=='-') {
	if (!strcmp(*argv,"-ntask")) {
	    if (Tcl_GetInt(interp, argv[1],&ntask) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else if (!strcmp(*argv,"-host")) {
	    if (!*argv[1]&&!strcmp(argv[1],"any")) {
		flag &= ~PvmTaskHost;
	    } else {
		where = argv[1]; flag = (flag&(~PvmTaskArch))|PvmTaskHost;
		if (*argv[1]=='!'||*argv[1]=='~') {
		    flag |= PvmHostCompl; where++;
		} else {
		    flag &= ~PvmHostCompl;
		}
	    }
	} else if (!strcmp(*argv,"-output")) {
	    char *p;
	    msgtag = -1; channel = (Tcl_Channel) NULL;
	    if (Tcl_GetInt(interp, argv[1],&msgtag) != TCL_OK) {
		int mode;
		Tcl_ResetResult(interp);
		channel = Tcl_GetChannel(interp, argv[1], &mode);
		if (channel == (Tcl_Channel) NULL) {
		    return TCL_ERROR;
		}
		if ((mode & TCL_WRITABLE) == 0) {
		    Tcl_AppendResult(interp, "channel \"", argv[1],
	    		"\" wasn't opened for writing", (char *) NULL);
		        return TCL_ERROR;
    }
	    }
	} else if (!strcmp(*argv,"-arch")&& !(flag&PvmTaskHost)) {
	    if (!*argv[1]||!strcmp(argv[1],"any")) {
		flag &= ~PvmTaskArch;
	    } else {
		where = argv[1]; flag |= PvmTaskArch;
		if (*argv[1]=='!'||*argv[1]=='~') {
		    flag &= ~PvmHostCompl; where++;
		} else {
		    flag |= PvmHostCompl;
		}
	    }
	} else if (!strcmp(*argv,"-debug")) {
	    if (Tcl_GetBoolean(interp, argv[1],&i) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (i) flag |= PvmTaskDebug; else flag &= ~PvmTaskDebug;
	} else if (!strcmp(*argv,"-trace")) {
	    if (Tcl_GetBoolean(interp, argv[1],&i) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (i) flag |= PvmTaskTrace; else flag &= ~PvmTaskTrace;
	} else {
	    Tcl_AppendResult(interp, "invalid option \"",*argv,"\"",(char *)NULL);
	    pvmerror = -1; return Pvm_ReturnError(interp,pvmerror);
	}
	argc-=2; argv+=2;
    }
    if (!argc) {
	Tcl_AppendResult(interp,"task to be spawned missing",(char *)NULL);
	pvmerror = -1; return Pvm_ReturnError(interp,pvmerror);
    }
    if (ntask>0) {
	int oldtid, oldmsgtag;
	int n;
	tids = (int *)ckalloc(ntask*sizeof(int));
	if (channel || msgtag!=-1) {
	    oldtid = pvm_setopt(PvmOutputTid,PvmMytid);
	    oldmsgtag = pvm_setopt(PvmOutputCode,REDIRECT_MSGTAG);
	}
	pvmerror = ntask = pvm_spawn(argv[0],argv+1,flag, where, ntask, tids);
	if (channel || msgtag!=-1) {
	    pvm_setopt(PvmOutputTid,oldtid);
	    pvm_setopt(PvmOutputCode,oldmsgtag);
	}
	if (pvmerror==0) {
	    Tcl_AppendResult(interp,"Cannot spawn task \"",argv[0],"\"",(char *)NULL);
	    return TCL_ERROR;
	}
	if (pvmerror<0) return Pvm_ReturnError(interp,pvmerror);
	for(t=tids;ntask>0;t++) {
	    if (t!=tids) Tcl_AppendResult(interp," ",(char *)NULL);
	    sprintf(storage,"0x%x",*t);
	    Tcl_AppendResult(interp,storage,(char *)NULL);
	    PvmNotifyTid(interp, *t, msgtag, channel);
	    ntask--;
	}
	ckfree(tids);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Pvm_KillCmd --
 *
 *	This procedure is invoked to process the "kill" Tkpvm command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Pvm_KillCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int tid, error;

    if (!PvmMytid) {
	if (!Pvm_StartDaemon(interp))
	    return TCL_ERROR;
    }
    if (argc<2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" process\"",(char *) NULL);
	return TCL_ERROR;
    }
    argc-=1; argv+=1;
    error = 0;
    while(argc>0) {
	if (!strcmp(*argv,"parent")) {
	    tid = pvm_parent();
	    if (tid==PvmSysErr||tid==PvmNoParent) {
		    return Pvm_ReturnError(interp,tid);
	    }
	} else if (!strcmp(*argv,"id")) {
	    tid = PvmMytid;
	} else if (Tcl_GetInt(interp, *argv,&tid) != TCL_OK) {
	    return TCL_ERROR;
	}
	error = pvm_kill(tid);
	if (error<0) {
 	    return Pvm_ReturnError(interp,error);
	}
	argc--; argv++;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * EvalCommand --
 *
 *	This is executed every time a PVM-packet arrives.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The registered procedure will be executed.
 *
 *--------------------------------------------------------------
 */


static void EvalCommand(tid, msgtag, number, clientData)
int tid;
int msgtag;
int number;
ClientData clientData;
{
    register char *s;
    TkpvmBindInfo *pevPtr = (TkpvmBindInfo *) clientData;
    char storage[15];
    char *before = pevPtr->command;
    Tcl_DString ds;
    /*
     *    expand %t and %m
     */
    Tcl_DStringInit(&ds);
    while(1) {
	/* search for first '%'-sign */
	for (s = before ; (*s!=0)&&(*s!='%'); s++) ;
	if (s!=before) {
	    Tcl_DStringAppend(&ds, before, s-before);
	    before = s;
	}
	if (*before == 0) {
	    break;
	}
	/* There's a percent sequence here. Process it. */
	switch (before[1]) {
	    case 't':
		sprintf(storage,"0x%x",tid);
		break;
	    case 'm':
		if (msgtag==-2) {
		    sprintf(storage,"kill");
		} else {
		    sprintf(storage,"%d",msgtag);
		}
		break;
	    case 'n':
		sprintf(storage,"%d",number);
		break;
	    default:
		sprintf(storage,"%%%c",before[1]);
		break;
	}
	before+=2;
	Tcl_DStringAppend(&ds,storage,strlen(storage));
    }
    if (Tcl_GlobalEval(pevPtr->interp,Tcl_DStringValue(&ds))!=TCL_OK) {
	Tcl_AddErrorInfo(pevPtr->interp,"\n    (command bound to PVM)");
	Tcl_BackgroundError(pevPtr->interp);
    }
    Tcl_DStringFree(&ds);
}
