/*
 *	imgObj.tcl
 */

#include "imgInt.h"
#include <string.h>
#include <stdlib.h>

#if TK_MAJOR_VERSION < 8

/*
 * Procedure types defined by Tcl:
 */

typedef void (Tcl_FreeInternalRepProc) _ANSI_ARGS_((struct Tcl_Obj *objPtr));
typedef void (Tcl_DupInternalRepProc) _ANSI_ARGS_((struct Tcl_Obj *srcPtr, 
        struct Tcl_Obj *dupPtr));
typedef void (Tcl_UpdateStringProc) _ANSI_ARGS_((struct Tcl_Obj *objPtr));
typedef int (Tcl_SetFromAnyProc) _ANSI_ARGS_((Tcl_Interp *interp,
	struct Tcl_Obj *objPtr));
typedef int (Tcl_ObjCmdProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int objc, struct Tcl_Obj *CONST objv[]));

/*
 * The following structure represents a type of object, which is a
 * particular internal representation for an object plus a set of
 * procedures that provide standard operations on objects of that type.
 */

typedef struct Tcl_ObjType {
    char *name;			/* Name of the type, e.g. "int". */
    Tcl_FreeInternalRepProc *freeIntRepProc;
				/* Called to free any storage for the type's
				 * internal rep. NULL if the internal rep
				 * does not need freeing. */
    Tcl_DupInternalRepProc *dupIntRepProc;
    				/* Called to create a new object as a copy
				 * of an existing object. */
    Tcl_UpdateStringProc *updateStringProc;
    				/* Called to update the string rep from the
				 * type's internal representation. */
    Tcl_SetFromAnyProc *setFromAnyProc;
    				/* Called to convert the object's internal
				 * rep to this type. Frees the internal rep
				 * of the old type. Returns TCL_ERROR on
				 * failure. */
} Tcl_ObjType;

/*
 * One of the following structures exists for each object in the Tcl
 * system.  An object stores a value as either a string, some internal
 * representation, or both.
 */

typedef struct Tcl_Obj {
    int refCount;		/* When 0 the object will be freed. */
    char *bytes;		/* This points to the first byte of the
				 * object's string representation. The
				 * array must be followed by a null byte
				 * (i.e., at offset length) but may also
				 * contain embedded null characters. The
				 * array's storage is allocated by
				 * ckalloc. NULL indicates the string
				 * rep is empty or invalid and must be
				 * regenerated from the internal rep.
				 * Clients should use Tcl_GetStringFromObj
				 * to get a pointer to the byte array
				 * as a readonly value.  */
    int length;			/* The number of bytes at *bytes, not
				 * including the terminating null. */
    Tcl_ObjType *typePtr;	/* Denotes the object's type. Always
				 * corresponds to the type of the object's
				 * internal rep. NULL indicates the object
				 * has no internal rep (has no type). */
    union {			/* The internal representation: */
	long longValue;		/*   - an long integer value */
	double doubleValue;	/*   - a double-precision floating value */
	VOID *otherValuePtr;	/*   - another, type-specific value */
	struct {		/*   - internal rep as two pointers */
	    VOID *ptr1;
	    VOID *ptr2;
	} twoPtrValue;
    } internalRep;
} Tcl_Obj;

#endif

/*
 * The variable "initialized" contains flags indicating which
 * version of Tcl or Perl we are running:
 *
 *      IMG_PERL	perl
 *	IMG_TCL		Tcl
 *	IMG_CHAN	using Tcl_Chan in stead of FILE *
 *	IMG_OBJS	using Tcl_Obj in stead of char *
 *
 * These flags will be determined at runtime (except the IMG_PERL
 * flag, for now), so we can use the same dynamic library for all
 * Tcl/Tk versions (and for Perl/Tk in the future).
 */

static int initialized = 0;
static Tcl_ObjType* byteArrayType = 0;

int
ImgObjInit(interp, version)
    Tcl_Interp *interp;
    char *version;
{
#ifdef _LANG
    return (initialized = IMG_PERL);
#else
    initialized = IMG_TCL;
    if (version[0] == '8') {
	struct CmdInfo {
	    int isNativeObjectProc;
	    Tcl_ObjCmdProc *objProc;
	    ClientData objClientData;
	    VOID *dummy[10]; /* worst case space that could be written
			      * by Tcl_GetCommandInfo() */
	} cmdInfo;
	initialized |= IMG_CHAN;
	if (!Tcl_GetCommandInfo(interp,"image", (Tcl_CmdInfo *) &cmdInfo)) {
	    Tcl_AppendResult(interp, "cannot find the \"image\" command",
		    (char *) NULL);
	    initialized = 0;
	    return TCL_ERROR;
	}
	if (cmdInfo.isNativeObjectProc == 1) {
	    initialized |= IMG_OBJS; /* we use objects */
	}
    }
    return initialized;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * ImgGetStringFromObj --
 *
 *	Returns the string representation's byte array pointer and length
 *	for an object.
 *
 * Results:
 *	Returns a pointer to the string representation of objPtr.  If
 *	lengthPtr isn't NULL, the length of the string representation is
 *	stored at *lengthPtr. The byte array referenced by the returned
 *	pointer must not be modified by the caller. Furthermore, the
 *	caller must copy the bytes if they need to retain them since the
 *	object's string rep can change as a result of other operations.
 *      REMARK: This function reacts a little bit different than
 *	Tcl_GetStringFromObj():
 *	- objPtr is allowed to be NULL. In that case the NULL pointer
 *	  will be returned, and the length will be reported to be 0;
 *	In the Img code there is never a distinction between en empty
 *	string and a NULL pointer, while the latter is easier to check
 *	for. That's the reason for this difference.
 *
 * Side effects:
 *	May call the object's updateStringProc to update the string
 *	representation from the internal representation.
 *
 *----------------------------------------------------------------------
 */

char *
ImgGetStringFromObj(objPtr, lengthPtr)
    register Tcl_Obj *objPtr;	/* Object whose string rep byte pointer
				 * should be returned, or NULL */
    register int *lengthPtr;	/* If non-NULL, the location where the
				 * string rep's byte array length should be
				 * stored. If NULL, no length is stored. */
{
    if (!objPtr) {
	if (lengthPtr != NULL) {
	    *lengthPtr = 0;
	}
	return (char *) NULL;
    } else
#ifdef _LANG
    {
	char *string = LangString((Arg) objPtr);
	if (lengthPtr != NULL) {
	    *lengthPtr = string ? strlen(string) : 0;
	}
	return string;
    }
#else /* _LANG */
    if (initialized & IMG_OBJS) {
	if (objPtr->bytes != NULL) {
	    if (lengthPtr != NULL) {
		*lengthPtr = objPtr->length;
	    }
	    return (objPtr->length) ? objPtr->bytes : (char *) NULL;
	}

	if (objPtr->typePtr == NULL) {
	    if (lengthPtr != NULL) {
		*lengthPtr = 0;
	    }
	    return "";
	}

	objPtr->typePtr->updateStringProc(objPtr);
	if (lengthPtr != NULL) {
	    *lengthPtr = objPtr->length;
	}
	return (objPtr->length) ? objPtr->bytes : (char *) NULL;
    } else {
	char *string =  (char *) objPtr;
	if (lengthPtr != NULL) {
	    *lengthPtr = string ? strlen(string) : 0;
	}
	return string;
    }
#endif /* _LANG */
}

/*
 * The following structure is the internal rep for a ByteArray object.
 * Keeps track of how much memory has been used and how much has been
 * allocated for the byte array to enable growing and shrinking of the
 * ByteArray object with fewer mallocs.  The ByteArray is also guaranteed
 * to have a terminating 0 byte at the end of the used length.
 */

typedef struct ByteArray {
    int used;			/* The number of bytes used in the byte
				 * array. */
    int allocated;		/* The amount of space actually allocated
				 * minus 1 byte. */
    unsigned char bytes[4];	/* The array of bytes.  The actual size of
				 * this field depends on the 'allocated' field
				 * above. */
} ByteArray;

/*
 *----------------------------------------------------------------------
 *
 * ImgGetByteArrayFromObj --
 *
 *	Returns the string representation's byte array pointer and length
 *	for an object.
 *
 * Results:
 *	Returns a pointer to the string representation of objPtr.  If
 *	lengthPtr isn't NULL, the length of the string representation is
 *	stored at *lengthPtr. The byte array referenced by the returned
 *	pointer must not be modified by the caller. Furthermore, the
 *	caller must copy the bytes if they need to retain them since the
 *	object's string rep can change as a result of other operations.
 *      REMARK: This function reacts a little bit different than
 *	Tcl_GetStringFromObj():
 *	- objPtr is allowed to be NULL. In that case the NULL pointer
 *	  will be returned, and the length will be reported to be 0;
 *	In the Img code there is never a distinction between en empty
 *	string and a NULL pointer, while the latter is easier to check
 *	for. That's the reason for this difference.
 *
 * Side effects:
 *	May call the object's updateStringProc to update the string
 *	representation from the internal representation.
 *
 *----------------------------------------------------------------------
 */

char *
ImgGetByteArrayFromObj(objPtr, lengthPtr)
    register Tcl_Obj *objPtr;	/* Object whose string rep byte pointer
				 * should be returned, or NULL */
    register int *lengthPtr;	/* If non-NULL, the location where the
				 * string rep's byte array length should be
				 * stored. If NULL, no length is stored. */
{
    if (!objPtr) {
	if (lengthPtr != NULL) {
	    *lengthPtr = 0;
	}
	return (char *) NULL;
    } else
#ifdef _LANG
    {
	char *string = LangString((Arg) objPtr);
	if (lengthPtr != NULL) {
	    *lengthPtr = string ? strlen(string) : 0;
	}
	return string;
    }
#else /* _LANG */
    if (initialized & IMG_OBJS) {
	ByteArray *baPtr;
	if (byteArrayType) {
	    if (objPtr->typePtr != byteArrayType) {
		byteArrayType->setFromAnyProc(NULL, objPtr);
	    }
        } else if (objPtr->typePtr && !strcmp(objPtr->typePtr->name, "bytearray")) {
	    byteArrayType = objPtr->typePtr;
        } else {
	    if (objPtr->bytes != NULL) {
		if (lengthPtr != NULL) {
		    *lengthPtr = objPtr->length;
		}
		return (objPtr->length) ? objPtr->bytes : (char *) NULL;
	    }

	    if (objPtr->typePtr == NULL) {
		if (lengthPtr != NULL) {
		    *lengthPtr = 0;
		}
		return "";
	    }

	    objPtr->typePtr->updateStringProc(objPtr);
	    if (lengthPtr != NULL) {
		*lengthPtr = objPtr->length;
	    }
	    return (objPtr->length) ? objPtr->bytes : (char *) NULL;
	}
	baPtr = (ByteArray *) (objPtr)->internalRep.otherValuePtr;
	if (lengthPtr != NULL) {
	    *lengthPtr = baPtr->used;
	}
	return baPtr->bytes;
    } else {
	char *string =  (char *) objPtr;
	if (lengthPtr != NULL) {
	    *lengthPtr = string ? strlen(string) : 0;
	}
	return string;
    }
#endif /* _LANG */
}


/*
 *----------------------------------------------------------------------
 *
 * ImgListObjGetElements --
 *
 *	Splits an object into its compoments.
 *
 * Results:
 *	If objPtr is a valid list (or can be converted to one),
 *	TCL_OK will be returned. The object will be split in
 *	its components. If the list turns out to have 0 elements,
 *	argv will become NULL. The caller must free argv with
 *	ckfree() when it is no longer needed.
 *	Otherwise TCL_ERROR is returned. If interp is not a NULL
 *	pointer, an error message will be left in it as well.
 *
 * Side effects:
 *	May call the object's updateStringProc to update the string
 *	representation from the internal representation.
 *
 *----------------------------------------------------------------------
 */

int
ImgListObjGetElements(interp, objPtr, argc, argv)
    Tcl_Interp *interp;
    struct Tcl_Obj *objPtr;
    int *argc;
    char ***argv;
{
#ifdef _LANG
    /* how is this done in perl? */
#else /* _LANG */
    int result;
    char *string = ImgGetStringFromObj(objPtr, NULL);
    if (string && string[0]) {
	result = Tcl_SplitList(interp, string, argc, argv);
	/* make sure that Tcl_SplitList doesn't allocate memory
	 * when argc == 0. */
	if (*argc) {
	    /* here we could do something more later */
	} else if (*argv) {
	    ckfree((char *) *argv);
	    *argv = (char **) NULL;
	}
	return result;
    } else {
	*argc = 0;
	return TCL_OK;
    }
#endif /* _LANG */
}

/*
 *----------------------------------------------------------------------
 *
 * ImgGetIndexFromObj --
 *
 *	This procedure looks up a value in a table of strings
 *	and returns the index of the matching string, if any.
 *
 * Results:
 *
 *	If the value of objPtr is identical to or a unique abbreviation
 *	for one of the entries in tablePtr, then the return value is
 *	TCL_OK and the index of the matching entry is stored at
 *	*indexPtr.  If there isn't a proper match, then TCL_ERROR is
 *	returned and an error message is left in interp's result (unless
 *	interp is NULL).  The msg argument is used in the error
 *	message; for example, if msg has the value "option" then the
 *	error message will say something flag 'bad option "foo": must be
 *	...'
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

int
ImgGetIndexFromObj(interp, objPtr, tablePtr, msg, flags, indexPtr)
    Tcl_Interp *interp; 	/* Used for error reporting if not NULL. */
    char *objPtr;		/* string to lookup. */
    char **tablePtr;		/* Array of strings to compare against the
				 * value of objPtr; last entry must be NULL
				 * and there must not be duplicate entries. */
    char *msg;			/* Identifying word to use in error messages. */
    int flags;			/* not used */
    int *indexPtr;		/* Place to store resulting integer index. */
{
    int index, i, numAbbrev;
    char *key, *p1, *p2, **entryPtr;

    /*
     * Lookup the value of the object in the table.  Accept unique
     * abbreviations.
     */

    key = objPtr;
    index = -1;
    numAbbrev = 0;
    for (entryPtr = tablePtr, i = 0; *entryPtr != NULL; entryPtr++, i++) {
	for (p1 = key, p2 = *entryPtr; *p1 == *p2; p1++, p2++) {
	    if (*p1 == 0) {
		index = i;
		goto done;
	    }
	}
	if (*p1 == 0) {
	    /*
	     * The value is an abbreviation for this entry.  Continue
	     * checking other entries to make sure it's unique.  If we
	     * get more than one unique abbreviation, keep searching to
	     * see if there is an exact match, but remember the number
	     * of unique abbreviations and don't allow either.
	     */

	    numAbbrev++;
	    index = i;
	}
    }
    if (numAbbrev != 1) {
	goto error;
    }

    done:
    *indexPtr = index;
    return TCL_OK;

    error:
    if (interp != NULL) {
	Tcl_AppendResult(interp,
		(numAbbrev > 1) ? "ambiguous " : "bad ", msg, " \"",
		key, "\": must be ", *tablePtr, (char *) NULL);
	for (entryPtr = tablePtr+1; *entryPtr != NULL; entryPtr++) {
	    if (entryPtr[1] == NULL) {
		Tcl_AppendResult(interp, ", or ", *entryPtr,
			(char *) NULL);
	    } else {
		Tcl_AppendResult(interp, ", ", *entryPtr,
			(char *) NULL);
	    }
	}
    }
    return TCL_ERROR;
}
