/* tr_demo : sample program of the tape_reader library.
 *
 * Author: Eric Maillet, LMC-INPG, Grenoble, France
 * Date  : december 1994
 * We read the TAPE/PVM trace and display it in a readable form.
 * This program can be used as a starting point to transform
 * TAPE/PVM traces in your own formats or to write simulators 
 * based on TAPE/PVM traces.
 *
 * Note: tr_demo only uses part of all TAPE/PVM events. See TAPEPVMFMT
 * for a complete list of event formats.
 */

/* $Log$ */

#include "tape_reader.h"
#include "tape_events.h"

#include <stdio.h>
#include <math.h>

#define dabs(x)   (x)>0 ? (x) : -(x)

void main(argc, argv)
int argc;
char *argv[];
{

  TapeEvent *evt;
  TapeTrace t;
  TapeTaskList tl;
  double salpha = 0.0;  /* sum of elementary intrusion */
  double salpha2 = 0.0;
  int n=0;

  /* check command line */

  if( argc != 2 ) {
    fprintf(stderr, "usage: %s <tape-trace>\n", argv[0]);
    exit(1);
  }

  /* open a tape trace file */

  t = tape_open_trace(argv[1]);

  if( t == NULL ) {
    fprintf(stderr, "tape_reader: cannot open %s\n", argv[1]);
    exit(2);
  }

  /* browse through the trace */

  tape_init_event(&evt);

  while( tape_read_event( t, &evt ) ) 
    {

    salpha+=evt->header.alpha;  /* compute sum of elementary intrusion */
    salpha2+=pow( ((double)evt->header.alpha), 2.0 );
    n++;

    switch( evt->header.type ) 
      {
      case event_enroll:
	{
	  TapeEnrollEvent *e = (TapeEnrollEvent *) evt;
	  printf("at %u task %x enroles in pvm on machine %s (a=%s,s=%d)\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->hi_name,
		 e->hi_arch,
		 e->hi_speed
		 );
	  break;
	}

      case event_exit:
	{
	  TapeExitEvent *e = (TapeExitEvent *) evt;
	  printf("at %u task %x quits pvm\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task
		 );
	  break;
	}

      case event_send:
	{
	  TapeSendEvent *e = (TapeSendEvent *) evt;
	  printf("at %u task %x sends msg %d of len %d to task %x (ovhd=%dus)\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->msgtag,
		 e->bytes,
		 e->tid,
		 tape_get_us( e->delta_s, e->delta_us )
		 );
	  break;
	}


      case event_recv:
	{
	  TapeRecvEvent *e = (TapeRecvEvent *) evt;

	  /* check if message was present or not when calling pvm_recv */

	  if( ! e->arrived ) {

	    printf("at %u task %x blocks for %d us until recv'g msg %d of len %d from task %x\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 tape_get_us( e->delta_s, e->delta_us ),
		 e->msgtag,
		 e->bytes,
		 e->tid
		 );

	  } else {

	    printf("at %u task %x finds msg %d of len %d from task %x (ovhd=%dus)\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->msgtag,
		 e->bytes,
		 e->tid,
		 tape_get_us( e->delta_s, e->delta_us )
		 );

	  }
	  break;
	}

      case event_mcast:
	{
	  TapeMcastEvent *e = (TapeMcastEvent *) evt;
	  printf("at %u task %x mcasts msg %d of len %d to ",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->msgtag,
		 e->bytes
		 );
	  
	  tl = e->TaskList;
	  
	  while( tl ) 
	    printf("%x ", tape_get_next_task( &tl ) );
	  
	  printf(" (ovhd=%dus)\n", tape_get_us( e->delta_s, e->delta_us ) );
	  break;
	}

      case event_spawn:
	{
	  TapeTask t;
	  TapeSpawnEvent *e = (TapeSpawnEvent *) evt;
	  printf("at %u task %x spawns %d tasks and gets tids array ",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->ntask
		 );
	  
	  tl = e->TaskList;
	  
	  while( tl ) {
	    t = tape_get_next_task( &tl );
	    if( t> 0 )
	      printf("%x ", t);
	    else
	      printf("%d ", t);
	  }
	  
	  printf("\n");
	  break;
	}

      case event_joingroup:
	{
	  TapeJoingroupEvent *e = (TapeJoingroupEvent *) evt;
	  printf("at %u task %x joins group '%s'\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->group
		 );
	  break;
	}

      case event_lvgroup:
	{
	  TapeLvgroupEvent *e = (TapeLvgroupEvent *) evt;
	  printf("at %u task %x leaves group '%s'\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->group
		 );
	  break;
	}

      case event_bcast:
	{
	  TapeBcastEvent *e = (TapeBcastEvent *) evt;
	  printf("at %u task %x bcasts msg %d of len %d to group '%s' (ovhd=%dus)\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->msgtag,
		 e->bytes,
		 e->group,
		 tape_get_us( e->delta_s, e->delta_us )
		 );
	  break;
	}

      case event_open_phase:
	{
	  TapeOpenPhaseEvent *e = (TapeOpenPhaseEvent *) evt;
	  printf("at %u task %x enters phase '%s'\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->phase
		 );
	  break;
	}

      case event_close_phase:
	{
	  TapeClosePhaseEvent *e = (TapeClosePhaseEvent *) evt;
	  printf("at %u task %x exits phase '%s'\n",
		 tape_get_us( e->header.date_s, e->header.date_us ),
		 e->header.task,
		 e->phase
		 );
	  break;
	}

      }
  }   

  tape_dispose_event( &evt );

  tape_close_trace( t );

  printf("\n *** Elementary intrusion : %.6lf +/- %.6lf\n",
	 salpha/n,  sqrt(dabs(salpha2/n  -  salpha*salpha/n/n))   );
}

