/* tapediff : compares 2 TAPE/PVM traces
 *
 * Author: Eric Maillet, LMC-INPG, Grenoble, France
 * Date  : october 95
 * We read 2 TAPE/PVM traces (of a same application) and compare
 * the dates of all synchronization events. The result is a floating
 * point number which can be interpreted as the "distance in performance"
 * between the 2 traces. 
 *
 * Possible applications:
 *
 *   - determine in how far performance of successive executions of a
 *     same application is changing (due to an non-stable environment
 *     for example)
 *
 *   - determine in how far an intrusion-compensated trace (cf. tape_ic)
 *     differs from a perturbated trace
 *     
 * Note: both input traces have to be ordered chronologically 
 *       both traces have to come from the same application
 *       both traces have to be recorded with the same trace mask
 *       the application has to be SPMD or master-slave
 *
 * In order the analysis be representative, the pvm configuration
 * has to be homogeneous.
 *
 */

/* $Log$ */

#include "tape_reader.h"
#include "tape_events.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#define MAXNODES 128

#define DEFAULTTBSET FirstBarrier

#define dabs(x) (x)>0 ? (x):-(x)
#define dmin(x,y) (x)>(y) ? (y) : (x)

/* protos */
char ReferenceEvent(TapeEvent *e);
double ReferenceEventDate(TapeEvent *e);

/* globals */
double tbA, tbB;
double VA=0.0, VBA=0.0;
int nb_evts=0, nb_ref_evts=0;

/* methods to choose time base */
typedef enum {FirstEvent, FirstBarrier} TBSet;


/* list of events */
typedef struct s_eventlist {
  TapeEvent *e;
  struct s_eventlist *next;
} *EventList;

/* tables of list of events */
EventList threadsA[MAXNODES];
EventList threadsB[MAXNODES];

void DispatchEvent(TapeEvent *e, char AB);

int
Task2NodeA( task )  /* node-id "server" for trace A */
     int task;
{
  static int n = 0;  /* number of nodes */
  static int NT[MAXNODES];

  int i;

  for(i=0;i<n;i++)
    if(task==NT[i])
      break;

  if( i < n )   /* found task in i */
    return i;

  NT[i] = task; /* insert new task at i */
  n++;

  return i;
}

int
Task2NodeB( task )   /* node id server for trace B */ 
     int task;
{
  static int n = 0;  /* number of nodes */
  static int NT[MAXNODES];

  int i;

  for(i=0;i<n;i++)
    if(task==NT[i])
      break;

  if( i < n )   /* found task in i */
    return i;

  NT[i] = task; /* insert new task at i */
  n++;

  return i;
}

void FirstEventTB(TapeEvent *e,
		  int nb,
		  char *Ready,
		  double *tb)

     /* In case e is the first event in its trace, let its
      * date be the time base tb for that trace. The Ready
      * flag is set.
      */

{
  if(nb==1) {
    *tb=tape_get_s(e->header.date_s, e->header.date_us);
    *Ready=1;
  }
}

void FirstBarrierTB(TapeEvent *e,
		    char *BarrierGroup,
		    int *BarrierReached,
		    double *BarrierExit,
		    char *Ready,
		    double *tb)

     /* In case e is the last entry event of the first
      * barrier in group BarrierGroup, the time base
      * tb is set to the min of the barrier exit dates.
      * The Ready flag is set. Else, if e is not the last
      * of the entry events, its exit date is used to 
      * update the min of the exit dates BarrierExit
      */

{
  if(e->header.type==event_barrier) {
    TapeBarrierEvent *be = (TapeBarrierEvent*)e;

    /* set default barrier group */
    if(strlen(BarrierGroup)==0)
      strcpy(BarrierGroup,be->group);
    
    if(strcmp(BarrierGroup,be->group)==0) {
      (*BarrierReached)++;
      if(*BarrierReached==1)
	*BarrierExit=ReferenceEventDate(e);
      else
	*BarrierExit=dmin(ReferenceEventDate(e),*BarrierExit);
      if(*BarrierReached==be->count) {
	*tb=*BarrierExit;
	*Ready=1;
      }
    }
  }
}


void main(argc, argv)
int argc;
char *argv[];
{

  TapeEvent *evtA, *evtB;
  TapeTrace tA, tB;
  int nbA=0, nbB=0;
  int i;
  char ReadyA=0, ReadyB=0; /* flags indicating that diff is active */

  /* vars for barrier time base setting */

  int BarrierReachedA=0, BarrierReachedB=0;
  char BarrierGroupA[80], BarrierGroupB[80];
  double BarrierExitA, BarrierExitB;

  /* how to choose per-trace time base */

  TBSet TimeBaseSetting = DEFAULTTBSET;

  /* check command line */

  if( argc != 3 ) {
    fprintf(stderr, "usage: %s <tape-trace-1> <tape-trace-2>\n", argv[0]);
    exit(1);
  }

  /* open a tape trace file */

  tA = tape_open_trace(argv[1]);
  tB = tape_open_trace(argv[2]);

  if( tA == NULL ) {
    fprintf(stderr, "%s: cannot open %s\n", argv[0], argv[1]);
    exit(2);
  }

  if( tB == NULL ) {
    fprintf(stderr, "%s: cannot open %s\n", argv[0], argv[2]);
    exit(2);
  }

  /* init per-thread event lists */

  for(i=0;i<MAXNODES;i++) {
    threadsA[i]=NULL;
    threadsB[i]=NULL;
  }

  /* browse through the trace */

  tape_init_event(&evtA);
  tape_init_event(&evtB);

  /* If BarrierGroup is null, tapediff takes the first group
   * on which a barrier operation is encountered. If the user
   * wants to use the first barrier of a specific group to set the
   * time base, BarrierGroup can be initialized to this group
   */

  BarrierGroupA[0]='\0';
  BarrierGroupB[0]='\0';

  while(1)
    {
      int nbread=0;

      /* read and en-queue event from trace A */
      if(tape_read_event( tA, &evtA )) {
	nbA++; nbread++;

	/* set time base */
	if(!ReadyA) {
	  switch(TimeBaseSetting) {

	  case FirstEvent:
	    FirstEventTB(evtA,nbA,&ReadyA,&tbA);
	    break;
	    
	  case FirstBarrier:
	    FirstBarrierTB(evtA,
			   BarrierGroupA, 
			   &BarrierReachedA,
			   &BarrierExitA,
			   &ReadyA,
			   &tbA);
	    break;
	  }
	}

	if(ReadyA && ReferenceEvent(evtA))
	  DispatchEvent(evtA,0);
      }

      /* read and en-queue event from trace B */
      if(tape_read_event( tB, &evtB )) {
	nbB++; nbread++;

	/* set time base */
	if(!ReadyB) {
	  switch(TimeBaseSetting) {

	  case FirstEvent:
	    FirstEventTB(evtB,nbB,&ReadyB,&tbB);
	    break;
	    
	  case FirstBarrier:
	    FirstBarrierTB(evtB,
			   BarrierGroupB, 
			   &BarrierReachedB,
			   &BarrierExitB,
			   &ReadyB,
			   &tbB);
	    break;
	  }
	}
	
	if(ReadyB && ReferenceEvent(evtB))
	  DispatchEvent(evtB,1);
      }

      /* no events remain - leave the loop */
      if(!nbread)
	break;
    }

  tape_dispose_event( &evtA );
  tape_dispose_event( &evtB );

  tape_close_trace( tA );
  tape_close_trace( tB );

  printf("\n%d events in %s, %d events in %s, %d ref. events\n", 
	 nbA, argv[1], nbB, argv[2], nb_ref_evts);

  switch(TimeBaseSetting) {
  case FirstEvent:
    printf("Time bases set on first event in trace.\n\n");
    break;
  case FirstBarrier:
    printf("Time bases set on first barrier exit (group \"%s\").\n\n",
	   BarrierGroupA);
    if(strcmp(BarrierGroupA, BarrierGroupB))
      printf("Warning: barriers chosen for time bases are from different groups (%s, %s)!\n\n",
	     BarrierGroupA, BarrierGroupB);
    break;
  }

  printf("Time base for trace %s:  %.6f\n", argv[1], tbA);
  printf("Time base for trace %s:  %.6f\n", argv[2], tbB);

  printf("Dimension of measure space: %d\n", nb_ref_evts);
  printf("Distance measure: %.6f\n",
	 sqrt(VBA)/sqrt(VA));
}

/* IMPLEMENTATION */


char
ReferenceEvent(TapeEvent *e)

     /** returns 1 if e is a reference event
     /** else returns 0
      **/
{
  switch(e->header.type) {
  case event_recv:
    return 1;
  default:
    return 0;
  }
}

double
ReferenceEventDate(TapeEvent *e)

     /** return reference date (in secs) of reference
      ** event e. If e is not a known reference event
      ** its date is returned.
      **/

{
  switch(e->header.type) {

  case event_recv:  /** reference date is end of receive **/
    
    {
      double rdate;
      TapeRecvEvent *re;

      re=(TapeRecvEvent*) e;

      rdate=tape_get_s(re->header.date_s, re->header.date_us);
      rdate+=re->delta_s;
      rdate+=re->delta_us*1e-6;

      return rdate;
    }
    break;

  case event_barrier: /** reference date is barrier exit **/

    {
      double rdate;
      TapeBarrierEvent *be;

      be=(TapeBarrierEvent*) e;

      rdate=tape_get_s(be->header.date_s, be->header.date_us);
      rdate+=be->delta_s;
      rdate+=be->delta_us*1e-6;

      return rdate;
    }
    break;
      

  default:
    return tape_get_s(e->header.date_s,e->header.date_us);
  }
}


EventAnalysis(TapeEvent *evtA, TapeEvent *evtB)
{
  /** compare two reference events **/

  double dA, dB;

  /* check event compatibility */

  if( evtA->header.type!=evtB->header.type )
    {
      fprintf(stderr,"Input traces are not compatible\n");
      fprintf(stderr,"TypeA=%d, TypeB=%d\n", evtA->header.type,
	      evtB->header.type);
      exit(1);
    }

  /* compare dates */

  dA=ReferenceEventDate(evtA)-tbA;
  dB=ReferenceEventDate(evtB)-tbB;

  /* printf("%d %.6f %.6f %.6f\n",
	 evtA->header.type,
	 dA, dB, dabs(dA-dB) ); */
  nb_ref_evts++;

  /* compute distance measure */

  VA+=dA*dA;
  VBA+=(dB-dA)*(dB-dA);
}

TapeEvent*
carevent(EventList *el) {

  /**  remove head event from list el and return it
   **  if list is empty, return NULL
   **/

  if(*el==NULL)
    return NULL;
  else
    {
      EventList tmp;
      TapeEvent *e;
      tmp=*el;
      e=tmp->e;
      *el=(*el)->next;
      free(tmp);
      return e;
    }
}

void
appendevent(TapeEvent *e, EventList *l)

     /** append event e at the tail of list *l **/
{
  EventList newl, ll=*l, temp=NULL;

  newl=(EventList)malloc(sizeof(struct s_eventlist));
  newl->next=NULL;
  newl->e=e;

  while(ll!=NULL) {
    temp=ll;
    ll=ll->next;
  }

  if(temp==NULL)
    *l=newl;
  else
    temp->next=newl;
}

void EnqueueEvent(TapeEvent *e,
		  EventList *ptrthreadsA,
		  EventList *ptrthreadsB,
		  char tAB)
{
  /** enqueue event e into its corresponding thread 'ptrthreadsA'
   ** if another event is ready in the corresponding thread from
   ** the other trace, it is dequeued and compared to e
   **/

  TapeEvent *e2;

  if(*ptrthreadsA==NULL) {
    if(*ptrthreadsB==NULL)
      appendevent(e, ptrthreadsA);
    else {
      e2=carevent(ptrthreadsB);
      tAB==0 ? EventAnalysis(e,e2) : EventAnalysis(e2,e);
      free(e);
      free(e2);
    }
  } else {
    appendevent(e,ptrthreadsA);
    if(*ptrthreadsB!=NULL) {
      e=carevent(ptrthreadsA);
      e2=carevent(ptrthreadsB);
      tAB==0 ? EventAnalysis(e,e2) : EventAnalysis(e2,e);
      free(e);
      free(e2);
    }
  }
}

void
DispatchEvent(TapeEvent *evt, char tAB )

     /**  Insert evt (reference) in its thread list
      **  tAB=0 <=> evt is from trace A
      **/
{
  TapeEvent *e;
  int node;

   /** copy event from evt to e 
   /** note: possible dynamic structures part of e
   /** are NOT copied. That's why we 'free()' to
   /** discard the event, instead of tape_dispose_event()
    **/

  e=(TapeEvent*)malloc(evt->header.memsize);
  memcpy(e,evt,evt->header.memsize);

  if(tAB==0) {

    node=Task2NodeA(evt->header.task);
    EnqueueEvent(e,&threadsA[node],&threadsB[node],tAB);

  } else {

    node=Task2NodeB(evt->header.task);
    EnqueueEvent(e,&threadsB[node],&threadsA[node],tAB);
  }
}
