/* TAPE/PVM trace reader library */

/* $Log: tape_reader.c,v $
 * Revision 1.2  1994/12/21  10:47:19  maillet
 * Support for multiple open trace files
 * */

/* 25/05/95 added event_trecv
   15/06/95 added event_precv, event_psend
   15/06/95 added event_gather, event_reduce, event_scatter
            (added FMTGRPEVT7)
   31/08/95 added size field (run-time storage space for event)
            in event header 
   01/09/95 added tape_get_group_size
   31/10/95 added memsize field to event header
   21/11/95 per-trace group management
   15/12/95 first field in event now is instrumentation delay for event
            number of fields in header is now 7 (NBFHEADER)
            adapted tape_read_header and tape_wb_event
   19/03/96 added tape_wb_send_event, tape_wb_mcast_event,
            tape_wb_bcast_event
*/


#include "tape.h"

#include "tape_reader.h"          /* pvm event type definitions */
#include "ath0_tape_reader.h"     /* ath0 event type definitions */

#include "tape_events.h"          /* pvm event identifiers */
#include "ath0_tape_events.h"     /* ath0 event identifiers */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define NBFHEADER 7       /* number of fields in event header */

#define MAXEVENTLEN 8192
#define TOKENSEP    " \n"

/* Macros to extract data fields from events. Arguments are 
 * the name of the event structure followed by the field names
 * of the event structure. The order of the field names in
 * the arg list is important. Special macros are used for group
 * related events ( string type in second place ).
 */

#define MKEVT(Type)                      \
 Type *evt;                              \
 evt = (Type *) malloc(sizeof(Type));    \
 *e = (TapeEvent *) evt;                 \
 evt->header.memsize=sizeof(Type);       \
 tape_read_header(type, buffer, &(evt->header) )

#define FMTEVT1(Type,f1)                 \
 MKEVT(Type);                            \
 evt->f1 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT2(Type,f1,f2)              \
 FMTEVT1(Type,f1);                       \
 evt->f2 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT3(Type,f1,f2,f3)           \
 FMTEVT2(Type,f1,f2);                    \
 evt->f3 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT4(Type,f1,f2,f3,f4)        \
 FMTEVT3(Type,f1,f2,f3);                 \
 evt->f4 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT5(Type,f1,f2,f3,f4,f5)     \
 FMTEVT4(Type,f1,f2,f3,f4);              \
 evt->f5 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT6(Type,f1,f2,f3,f4,f5,f6)  \
 FMTEVT5(Type,f1,f2,f3,f4,f5);           \
 evt->f6 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT7(Type,f1,f2,f3,f4,f5,f6,f7)  \
 FMTEVT6(Type,f1,f2,f3,f4,f5,f6);           \
 evt->f7 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT2(Type,f1,grp)          \
 char *g;                                \
 FMTEVT1(Type,f1);                       \
 g = strchk(strtok(NULL,TOKENSEP));      \
 if(!(evt->grp = malloc(1+strlen(g))))   \
   fprintf(stderr,"FMTGRPEVT2 error\n"); \
 strcpy(evt->grp,g)

#define FMTGRPEVT3(Type,f1,grp,f3)       \
 FMTGRPEVT2(Type,f1,grp);                \
 evt->f3 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT4(Type,f1,grp,f3,f4)    \
 FMTGRPEVT3(Type,f1,grp,f3);             \
 evt->f4 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT5(Type,f1,grp,f3,f4,f5) \
 FMTGRPEVT4(Type,f1,grp,f3,f4);          \
 evt->f5 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT6(Type,f1,grp,f3,f4,f5,f6)    \
 FMTGRPEVT5(Type,f1,grp,f3,f4,f5);             \
 evt->f6 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT7(Type,f1,grp,f3,f4,f5,f6,f7) \
 FMTGRPEVT6(Type,f1,grp,f3,f4,f5,f6);          \
 evt->f7 = atoi(strchk(strtok(NULL,TOKENSEP)))

/* globals */

char buffer[MAXEVENTLEN];  /* active event read buffer */
int bufferlen;             /* length of last buffer read */
int TapeLine;              /* current line number in Tape file */
int TapeError;             /* current library error */

/* utility functions */

char* strchk( s )
     char *s;
{
  if( s != NULL )
    return s;
  else
    {
      fprintf(stderr,
	      "tape-reader, trace format error in line %d, aborting.\n",
	      TapeLine);
      exit(1);
    }
}

unsigned
tape_get_us(s, us)
     int s;
     int us;
{
  return( s*1000000 + us );
}

double 
tape_get_s( s, us )
     int s;
     int us;
{
  return ( (double) s + 1e-6 * (double) us );
}

void
tape_get_s_us(t,s,us)
     double t;
     int *s;
     int *us;
{
  *s=(int)t;
  *us=(int)((t-*s)*1e6);
}

/* group and host management functions */

TapeTask
tape_get_next_task(tl)
     TapeTaskList *tl;
{
  TapeTask thisTask;

  assert(*tl != NULL);

  thisTask = (*tl)->task;
  *tl = (*tl)->next;

  return( thisTask );
}

TapeHost
tape_get_next_host(hl)
     TapeHostList *hl;
{
  TapeHost thisHost;

  assert(*hl != NULL);

  thisHost = (*hl)->host;
  *hl = (*hl)->next;

  return( thisHost );
}

void tape_free_tasklist( ll )
     TapeTaskList ll;
{
  while( ll!=NULL ) {
    TapeTaskList tmp;
    tmp = ll->next;
    free(ll);
    ll=tmp;
  }
}

void tape_free_grouplist(setofgroups)
     TapeGroupList *setofgroups;
{
TapeGroupList g = *setofgroups, gtemp;
TapeTaskList n, ntemp;

while(g!=NULL)
  {
    n=g->tasks;
    while(n!=NULL)
      {
        ntemp=n;
        n=n->next;
        free(ntemp);
      }
    gtemp=g;
    g=g->next;
    free(gtemp);
  }
}


/* GetGroupDesc(setofgroups,groupname)
 *   return group descriptor of groupname
 *   if groupname does not exist a new descr. is created
 *   setofgoups is the per-file set of groups
 */

TapeGroupList
tape_get_group_desc(setofgroups,group)
     TapeGroupList *setofgroups;
     char *group;
{
  TapeGroupList g=*setofgroups, gl=NULL;

  while(g!=NULL)
    {
      if( strcmp(g->group_name,group) )
	{
	  gl=g;
	  g=g->next;
	}
      else
	break;
    }

  if(g==NULL)
    {
      g=(TapeGroupList)malloc(sizeof(struct s_grouplist));
      strcpy(g->group_name,group);
      g->group_size=0;
      g->tasks=NULL;
      g->next=NULL;
      if(gl==NULL)
	*setofgroups=g;
      else
	gl->next=g;
    }

  return(g);
}

/* tape_add_task_to_group(setofgroups,groupname,task)
 *   add task to groupname
 */

void
tape_add_task_to_group(setofgroups,group,t)
     TapeGroupList *setofgroups;
     char* group;
     TapeTask t;
{
  TapeGroupList g;
  TapeTaskList n, ln=NULL;

  g = tape_get_group_desc(setofgroups,group);

  g->group_size++;

  n=g->tasks;

  while(n!=NULL) {
    ln=n;
    n=n->next;
  }

  n=(TapeTaskList)malloc(sizeof(struct s_tasklist));
  n->next=NULL;
  n->task=t;

  if(ln==NULL)
    g->tasks=n;
  else
    ln->next=n;
}
    

/* tape_del_task_from(setofgroups,groupname,task)
 *   delete a task from groupname
 */

void
tape_del_task_from_group(setofgroups,group,t)
     TapeGroupList *setofgroups;
     char* group;
     TapeTask t;
{
  TapeGroupList g;
  TapeTaskList n, ln=NULL;

  g = tape_get_group_desc(setofgroups,group);

  g->group_size--;

  n=g->tasks;

  while(n!=NULL)
    {
      if(n->task!=t)
	{ 
	  ln=n;
	  n=n->next;
	}
      else
	break;
    }

  if(n==NULL)
    fprintf(stderr,"NodeDel, warning, task %x not in group %s\n",
	    t,group);
  else
    {
      if(ln==NULL)
	{
	  g->tasks=n->next;
	  free(n);
	}
      else
	{
	  ln->next=n->next;
	  free(n);
	}
    }
}


TapeTaskList
tape_get_tasks_in_group(setofgroups,group)
     TapeGroupList *setofgroups;
     char *group;
{
  TapeGroupList gl = tape_get_group_desc( setofgroups, group );

  return( gl->tasks );
}

int
tape_get_group_size(setofgroups, group)
     TapeGroupList *setofgroups;
     char *group;
{
  TapeGroupList gl = tape_get_group_desc( setofgroups, group );

  return( gl->group_size );
}

/* tracefile management functions */

TapeTrace
tape_open_trace( file )
     char *file;
{
  TapeTrace tt;
  FILE *f;

  TapeLine = 0;

  if( ! ( tt = (TapeTrace) malloc(sizeof(struct s_trace)) ) )
    {
      TapeError = TapeMallocError;
      return( NULL );
    }

  if( file != NULL  && strcmp(file, "-")) {
    if( (f = fopen(file,"r")) == NULL )
      {
	TapeError = TapeFileError;
	free( (char *) tt );
	return( NULL );
      }
  } else
    f = stdin;

  tt->line = 0;
  tt->file = f;
  tt->version = 0;
  tt->groups = NULL;

  TapeError = TapeOK;

  return( tt );
}

void
tape_close_trace( t )
     TapeTrace t;
{
  if(t->file!=stdout)
    fclose( t->file );
  if(t->groups)
    tape_free_grouplist( &t->groups );
  free( (char *) t );
  TapeLine=0;
}

/* event management functions */

void
tape_init_event( e )
     TapeEvent **e;
{
  *e = (TapeEvent *) NULL;
}

void
tape_dispose_event( e )
     TapeEvent **e;
{

  if(*e != NULL) {
    switch( (*e)->header.type )
      {
	/* free event lists */
      case event_spawn:
      case event_mcast:
      case event_delhosts:
	tape_free_tasklist(((TapeSpawnEvent *) *e)->TaskList);
	break;
	
	/* free group and phase names */
      case event_barrier:
      case event_getinst:
      case event_bcast:
      case event_gettid:
      case event_gsize:
      case event_joingroup:
      case event_lvgroup:
      case event_open_phase:
      case event_close_phase:
	{
	  TapeGsizeEvent *evt;
	  evt = (TapeGsizeEvent *) *e;
	  free(evt->group);
	}
	break;
	
	/* free machine name and archi */
      case event_enroll:
	{
	  TapeEnrollEvent *evt;
	  evt = (TapeEnrollEvent *) *e;
	  free(evt->hi_name);
	  free(evt->hi_arch);
	}
	break;
      }
    free( *e );
  }
}

int
comment(buf,version)
     char *buf;
     char *version;
{
  int i=0;
  while(buf[i]==' '||buf[i]=='\t')
    i++;
  
  if(buf[i]=='#'||buf[i]=='\n') {
    /* this is a comment line - check for version number */
    if(strstr(buf,TAPE_VERSION))
      *version=1;
    return 1;
  }
  else
    return 0;
}

void
tape_read_header(type, buf, h)
     int type;
     char *buf;
     TapeEventHeader *h;
{

  h->size = strlen(buf);
  h->alpha = atoi(strchk(strtok(buf,TOKENSEP)));
  h->type = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->task = atoi(strchk(strtok(NULL,TOKENSEP)));
  if( ATH0EVENT(h->type) ) {
    h->thread = atoi(strchk(strtok(NULL,TOKENSEP)));
    h->nickname = atoi(strchk(strtok(NULL,TOKENSEP)));
  } else {
    h->thread = -1;
    h->nickname = -1;
  }
  h->file = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->line = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->date_s = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->date_us = atoi(strchk(strtok(NULL,TOKENSEP)));

}

/********************************************************
  tape_read_event: read and format an event from trace t
 ********************************************************/

int
tape_read_event( t, e )
     TapeTrace t;
     TapeEvent **e;
{
  FILE *f;
  int alpha, type;

  tape_dispose_event( e );

  f = t->file;

  fgets(buffer, MAXEVENTLEN, f);
  bufferlen=strlen(buffer);
  t->line++;
  TapeLine=t->line;

  while( !feof(f) && comment(buffer,&t->version) ) {
    fgets(buffer, MAXEVENTLEN, f);
    bufferlen=strlen(buffer);
    t->line++;
    TapeLine=t->line;
  }

  if( feof(f) )
    return 0;

  assert( !feof(f) && !comment(buffer) );

  /* get event type - known to be second field of event */
  sscanf(buffer, "%d %d", &alpha, &type);

  switch(type) {


    /* FORMAT PVM EVENTS */

  case  event_addhosts:
    {
      FMTEVT2(TapeAddhostsEvent,ret,count);
      break;
    }
    
  case  event_advise:
    {
      FMTEVT2(TapeAdviseEvent,ret,what);
      break;
    }

  case  event_barrier:
    {
      FMTGRPEVT5(TapeBarrierEvent,ret,group,delta_s,delta_us,count);
      break;
    }

  case  event_bcast:
    {
      FMTGRPEVT6(TapeBcastEvent,ret,group,delta_s,delta_us,msgtag,bytes);
      break;
    }

  case  event_bufinfo:
    {
      FMTEVT5(TapeBufinfoEvent,ret,bufid,bytes,msgtag,tid);
      break;
    }

  case  event_config:
    {
      FMTEVT1(TapeConfigEvent,ret);
      break;
    }

  case  event_delete:
    {
      FMTEVT2(TapeDeleteEvent,ret,req);
      break;
    }

  case  event_delhosts:
    {
      TapeHostList queue;
      char *token;

      FMTEVT2(TapeDelhostsEvent,ret,nhost);

      evt->HostList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeHostList tmp;

	  tmp = (TapeHostList) malloc( sizeof(struct s_hostlist) );
	  tmp->next = NULL;
	  tmp->host = atoi(token);

	  if(evt->HostList == NULL) {
	    queue = tmp;
	    evt->HostList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_exit:
    {
      MKEVT(TapeExitEvent);
      break;
    }

  case  event_freebuf:
    {
      FMTEVT2(TapeFreebufEvent,ret,bufid);
      break;
    }

  case event_gather:
    {
      FMTGRPEVT7(TapeGatherEvent,ret,group,delta_s,delta_us,msgtag,bytes,root);
      break;
    }

  case  event_getfds:
    {
      FMTEVT1(TapeGetfdsEvent,ret);
      break;
    }

  case  event_getinst:
    {
      FMTGRPEVT3(TapeGetinstEvent,ret,group,tid);
      break;
    }

  case  event_getopt:
    {
      FMTEVT2(TapeGetoptEvent,ret,what);
      break;
    }

  case  event_getrbuf:
    {
      FMTEVT1(TapeGetrbufEvent,ret);
      break;
    }
      
  case  event_getsbuf:
    {
      FMTEVT1(TapeGetsbufEvent,ret);
      break;
    }
      
  case  event_gettid:
    {
      FMTGRPEVT3(TapeGettidEvent,ret,group,inum);
      break;
    }

  case  event_gsize:
    {
      FMTGRPEVT2(TapeGsizeEvent,ret,group);
      break;
    }

  case  event_halt:
    {
      FMTEVT1(TapeHaltEvent,ret);
      break;
    }
      
  case  event_initsend:
    {
      FMTEVT2(TapeInitsendEvent,ret,encoding);
      break;
    }

  case  event_insert:
    {
      FMTEVT3(TapeInsertEvent,ret,req,data);
      break;
    }

  case  event_joingroup:
    {
      FMTGRPEVT2(TapeJoingroupEvent,ret,group);

      if( evt->ret >= 0 )
	tape_add_task_to_group( &t->groups, evt->group, evt->header.task );

      break;
    }

  case  event_kill:
    {
      FMTEVT2(TapeKillEvent,ret,tid);
      break;
    }

  case  event_lookup:
    {
      FMTEVT2(TapeLookupEvent,ret,req);
      break;
    }

  case  event_lvgroup:
    {
      FMTGRPEVT2(TapeLvgroupEvent,ret,group);

      if( evt->ret >=0 )
	tape_del_task_from_group( &t->groups, evt->group, evt->header.task );

      break;
    }

  case event_enroll:
    {
      char *token;

      FMTEVT1(TapeEnrollEvent,ret);

      token = strchk(strtok(NULL,TOKENSEP));
      if(!(evt->hi_name=malloc(strlen(token)+1)))
	fprintf(stderr, "event_enroll, hi_name, fatal malloc error\n");
      strcpy(evt->hi_name,token);

      token = strchk(strtok(NULL,TOKENSEP));
      if(!(evt->hi_arch=malloc(strlen(token)+1)))
	fprintf(stderr, "event_enroll, hi_arch, fatal malloc error\n");
      strcpy(evt->hi_arch,token);

      token = strchk(strtok(NULL,TOKENSEP));
      evt->hi_speed = atoi(token);

      break;
    }


  case  event_mcast:
    {
      TapeTaskList queue;
      char *token;

      FMTEVT6(TapeMcastEvent,ret,delta_s,delta_us,msgtag,bytes,ntask);

      evt->TaskList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeTaskList tmp;

	  tmp = (TapeTaskList) malloc( sizeof(struct s_tasklist) );
	  tmp->next = NULL;
	  tmp->task = atoi(token);

	  if(evt->TaskList == NULL) {
	    queue = tmp;
	    evt->TaskList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_mkbuf:
    {
      FMTEVT2(TapeMkbufEvent,ret,encoding);
      break;
    }

  case  event_mstat:
    /* not supported */
    break;

  case  event_mytid:
    {
      FMTEVT1(TapeMytidEvent,ret);
      break;
    }

  case  event_notify:
    {
      FMTEVT4(TapeNotifyEvent,ret,what,msgtag,ntask);
      break;
    }

  case  event_nrecv:
    {
      FMTEVT6(TapeNrecvEvent,ret,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case  event_parent:
    {
      FMTEVT1(TapeParentEvent,ret);
      break;
    }

  case  event_perror:
    {
      FMTEVT1(TapePerrorEvent,ret);
      break;
    }

  case  event_pkbyte:
  case  event_pkcplx:
  case  event_pkdcplx:
  case  event_pkdouble:
  case  event_pkfloat:
  case  event_pkint:
  case  event_pklong:
  case  event_pkshort:
  case  event_pkuint:
  case  event_pkulong:
  case  event_pkushort:
    {
      FMTEVT5(TapePkEvent,ret,delta_s,delta_us,nitem,stride);
      break;
    }

  /* case  event_packf: */
  case  event_pkstr:
    {
      FMTEVT3(TapePk1Event,ret,delta_s,delta_us);
      break;
    }

  case  event_probe:
    {
      FMTEVT3(TapeProbeEvent,ret,tid,msgtag);
      break;
    }

  case  event_psend:
    {
      FMTEVT6(TapePsendEvent,ret,tid,delta_s,delta_us,msgtag,bytes);
      break;
    }

  case  event_pstat:
    {
      FMTEVT2(TapeKillEvent,ret,tid);
      break;
    }

  case  event_precv:
    {
      FMTEVT7(TapePrecvEvent,ret,arrived,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case event_reduce:
    {
      FMTGRPEVT7(TapeReduceEvent,ret,group,delta_s,delta_us,msgtag,bytes,root);
      break;
    }

  case  event_recv:
    {
      FMTEVT7(TapeRecvEvent,ret,arrived,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case  event_recvf:
    {
      MKEVT(TapeRecvfEvent);
      break;
    }

  case event_scatter:
    {
      FMTGRPEVT7(TapeScatterEvent,ret,group,delta_s,delta_us,msgtag,bytes,root);
      break;
    }

  case  event_send:
    {
      FMTEVT6(TapeSendEvent,ret,tid,delta_s,delta_us,msgtag,bytes);
      break;
    }

  case  event_sendsig:
    {
      FMTEVT1(TapeSendsigEvent,ret);
      break;
    }

  case  event_serror:
    {
      FMTEVT2(TapeSerrorEvent,ret,how);
      break;
    }

  case  event_setdebug:
    {
      FMTEVT2(TapeSetdebugEvent,ret,mask);
      break;
    }

  case  event_setopt:
    /* not supported */
    break;
    
  case  event_setrbuf:
    {
      FMTEVT2(TapeSetrbufEvent,ret,bufid);
      break;
    }

  case  event_setsbuf:
    {
      FMTEVT2(TapeSetsbufEvent,ret,bufid);
      break;
    }

  case  event_spawn:
    {
      TapeTaskList queue;
      char *token;

      FMTEVT3(TapeSpawnEvent,ret,flag,ntask);

      evt->TaskList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeTaskList tmp;

	  tmp = (TapeTaskList) malloc( sizeof(struct s_tasklist) );
	  tmp->next = NULL;
	  tmp->task = atoi(token);

	  if(evt->TaskList == NULL) {
	    queue = tmp;
	    evt->TaskList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_start_pvmd:
    {
      FMTEVT2(TapeStart_pvmdEvent,ret,block);
      break;
    }

  case  event_tasks:
    {
      FMTEVT2(TapeTasksEvent,ret,where);
      break;
    }

  case  event_tickle:
    {
      FMTEVT3(TapeTickleEvent,ret, narg, nresp);
      break;
    }

  case  event_tidtohost:
    {
      FMTEVT2(TapeTidtohostEvent,ret, tid);
      break;
    }

  case  event_trecv:
    {
      FMTEVT6(TapeTrecvEvent,ret,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case  event_upkbyte:
  case  event_upkcplx:
  case  event_upkdcplx:
  case  event_upkdouble:
  case  event_upkfloat:	
  case  event_upkint:
  case  event_upklong:
  case  event_upkshort:
  case  event_upkuint:
  case  event_upkulong:
  case  event_upkushort:
    {
      FMTEVT5(TapeUpkEvent,ret,delta_s,delta_us,nitem,stride);
      break;
    }

  /* case  event_unpackf: */
  case  event_upkstr:
    {
      FMTEVT3(TapeUpk1Event,ret,delta_s,delta_us);
      break;
    }

  case  event_version:
    {
      FMTEVT1(TapeVersionEvent,ret);
      break;
    }

  case  event_trace:
    {
      MKEVT(TapeTraceEvent);
      break;
    }

  case  event_open_phase:
    {
      FMTGRPEVT2(TapeOpenPhaseEvent,ret,phase);
      break;
    }

  case  event_close_phase:
    {
      FMTGRPEVT2(TapeClosePhaseEvent,ret,phase);
      break;
    }

  case  event_set_mask:
    {
      FMTGRPEVT2(TapeSetMaskEvent,ret,mask);
      break;
    }

    /* FORMAT ATH0 EVENTS */

  case event_Ath0ServiceStart:
  case event_Ath0ServiceEnd:
    {
      FMTEVT3(TapeAth0ServiceStartEvent,other_task,other_thread,bytes);
      break;
    }

  case event_Ath0Call:
  case event_Ath0DoCall:
  case event_Ath0Spawn:
  case event_Ath0DoSpawn:
  case event_Ath0WaitSpawn:
  case event_Ath0WaitSpawnRes:
    {
      FMTEVT7(TapeAth0CallEvent,ret,other_task,other_thread,other_nickname,delta_s,delta_us,bytes);
      break;
    }

  case event_Ath0TestSpawn:
  case event_Ath0Pack:
  case event_Ath0UnPack:
  case event_Ath0LSemP:
  case event_Ath0WaitLSignal:
  case event_Ath0LPass:
  case event_Ath0LPassAndReuse:
    {
      FMTEVT3(TapeAth0TestSpawnEvent,ret,delta_s,delta_us);
      break;
    }

  default:
    {
      fprintf(stderr,"tape_reader, skipping unknown event type %d in line %d.\n",
	       type, TapeLine);
      break;
    }

  } /* switch(type) */

  return 1;

}

/**************************************************************
  tape_wb_event: write-back an event to stdout from active
                 event buffer
 --------------------------------------------------------------
 Used by trace analysis tools which modify the values of event 
 header fields only, and write the modified event to stdout in the 
 Tape/Pvm trace format. This is used by tape_ic (which corrects
 the time-stamps (header.date_s,header.date_us).

 The event's header is written from e->header, while the variable
 part is copied (as is) from the active event buffer.
 --------------------------------------------------------------
 In:   e is event to write-back
 **************************************************************/

void
tape_wb_header(h)
     TapeEventHeader *h;
{
  
  printf("%d", h->alpha);
  printf(" %d",h->type);
  printf(" %d",h->task);
  printf(" %d",h->file);
  printf(" %d",h->line);
  printf(" %d",h->date_s);
  printf(" %d",h->date_us);
}

void
tape_wb_event( e )
     TapeEvent *e;
{
  int n=0;        /* field index in buffer */
  int nbfield=1;  /* field number in buffer */

  /* write-back NBFHEADER field-header from header struct */
  tape_wb_header(&e->header);

  /* write-back variable part of event */
  while(n<bufferlen) {
    if(nbfield>NBFHEADER) {
      /* now we are in variable part, which is dumped as is
	 from the active event buffer
       */
      printf(" %s",&buffer[n]);
    }
    n+=(strlen(&buffer[n])+1);
    nbfield++;
  }
  /* printf("\n"); */
}

/*************************************************************
  Event write-back functions for special events in which not
  only header fields are modified, but also fields of the 
  variable part or for events where buffer is not available
  any more (because other events have already been read).

  Such events have to be written back from their data
  structures.
 *************************************************************/

void
tape_wb_recv_event(e)
     TapeRecvEvent *e;
{
  tape_wb_header(&e->header);

  /* write-back variable part: format dependent ! Has
   * to be done in the same order as the corresponding
   * FMTEVT
   */

  printf(" %d %d %d %d %d %d %d\n",
	 e->ret,
	 e->arrived,
	 e->delta_s,
	 e->delta_us,
	 e->tid,
	 e->msgtag,
	 e->bytes);
}

void
tape_wb_send_event(e)
     TapeSendEvent *e;
{
  tape_wb_header(&e->header);

  /* write-back variable part: format dependent ! Has
   * to be done in the same order as the corresponding
   * FMTEVT
   */

  printf(" %d %d %d %d %d %d\n",
	 e->ret,
	 e->tid,
	 e->delta_s,
	 e->delta_us,
	 e->msgtag,
	 e->bytes);
}

void
tape_wb_mcast_event(e)
     TapeMcastEvent *e;
{
  TapeTaskList tl;

  tape_wb_header(&e->header);

  /* write-back variable part: format dependent ! Has
   * to be done in the same order as the corresponding
   * FMTEVT
   */

  printf(" %d %d %d %d %d %d",
	 e->ret,
	 e->delta_s,
	 e->delta_us,
	 e->msgtag,
	 e->bytes,
	 e->ntask);

  tl = e->TaskList;
  while(tl)
    printf(" %d",tape_get_next_task( &tl ));
  printf("\n");

}

void
tape_wb_bcast_event(e)
     TapeBcastEvent *e;
{
  tape_wb_header(&e->header);

  /* write-back variable part: format dependent ! Has
   * to be done in the same order as the corresponding
   * FMTEVT
   */

  printf(" %d %s %d %d %d %d\n",
	 e->ret,
	 e->group,
	 e->delta_s,
	 e->delta_us,
	 e->msgtag,
	 e->bytes);
}

void
tape_wb_barrier_event(e)
     TapeBarrierEvent *e;
{
  tape_wb_header(&e->header);

  /* write-back variable part: format dependent ! Has
   * to be done in the same order as the corresponding
   * FMTEVT
   */

  printf(" %d %s %d %d %d\n",
	 e->ret,
	 e->group,
	 e->delta_s,
	 e->delta_us,
	 e->count);
}
