/* tape_ic: deterministic intrusion compensation for Tape/Pvm
 * 
 * Input: events in input trace have to be ordered chronologically
 * (measured time scale)
 *
 * Output: events in the compensated trace are in the same order as
 * in the input trace. Their dates, however, are now expressed with
 * respect to the compensated time scale. Thus, they are likely to
 * be out of (compensated) chronological order.
 *
 * Author: Eric Maillet, LMC-INPG, Grenoble, France
 * Initial revision: august 1995
 *
 * 18/12/95: new format of 0.9pl8 includes event generation delay
 *           in first field -- adapted alpha() function, alpha field
 *           is reset to 0 before event write-back (after compensation
 *           no (direct) intrusion remains) 
 *
 * 29/01/96: added counter of tr. time simulations required in
 *           Sarukkai/Malony (nbsim_sm) to compare with our model (nbsim)
 *
 * 09/02/96: in CorrComEvt, when modeling transmission delay (trsim),
 *           taB>taS is no longer an assertion. We print a warning if
 *           violated and use taB=taS.
 *
 * 18/03/96: added FULLSIM mode -- on the user's request *ALL*
 *           communication delays are simulated with trsim, even if
 *           such simulations are not necessary. This allows performance
 *           prediction on other communication libraries or hardware.
 *           FULLSIM uses models not only for the communication time trsim,
 *           but also for message copy and extraction times, cpysim and extrsim.
 *
 *           added a number of command line options to control the parameters
 *           of the models used by FULLSIM and to (de)activate FULLSIM
 *
 * 19/03/96: added REMOVEALPHA flag -- if set (default) elementary intrusions are
 *           removed. This is usefull for studying the effect of FULLSIM with and
 *           without alpha removal.
 *
 * 20/03/96: added PKASALPHA flag
 *
 * 09/08/96: added COMTIMES file -- interesting for external off-line communication 
 *           time analysis.
 *
 */

/* $Log$ */

#include "tape_reader.h"
#include "tape_events.h"

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


/* CONSTANTS */

/* Maximum number of tasks */
#define N 64

/* Number of slots in message size space */
#define K 20

/* default message find delay (whatever the size) */
#define DEFAULT_FIND_DELAY 0.000001

/* default parameters for transmission time simulator (PVM/TCPIP Switch/SP2) */
#define DEFAULTTAU      0.688442     /* in musec/bytes */
#define DEFAULTBETA  2902.616964     /* in musec */

/* default parameters for extraction time simulator (MPI-F/SP2/async. send) */
#define DEFAULTTAU_EXTR     0.0244055     /* in musec/bytes */
#define DEFAULTBETA_EXTR           63     /* in musec */

/* default parameters for copy time simulator (MPI-F/SP2) */
#define DEFAULTTAU_CPY     0.0244055     /* in musec/bytes */
#define DEFAULTBETA_CPY           63     /* in musec */

/* tape_ic operating modes (values of SIMMODE) */
#define MINSIM  0   /* minimize number of transmission time modelings */
#define FULLSIM 1   /* model all communiations */

/* Name of info file */

#define INFOFILE "tico.out"
#define COMTIMESFILE "ticoms.out"

/* DATA TYPES */

/* Message specifier */
typedef struct {  
  int tid;        /* id of the sender */
  int tag;        /* tag of the message */
} MsgSpec;

/* Data relative to a message in reveiver's in-box */
typedef struct s_msginbox {
  MsgSpec msg;   /* message specifier */
  double tS;     /* measured expedition date */
  double taS;    /* effective expedition date */
  struct s_msginbox *next;
} MsgInBox;

/* Data relative to a message receive post */
typedef struct {
  MsgSpec msg;            /* message specifier */
  double tSR;             /* measured start of receive date */
  double taSR;            /* effective start of receive date */
  double tER;             /* measured end of receive date */
  TapeRecvEvent recvevt;  /* associated event (for write-back) */
} MsgReq;

/* Data relative to a group member's barrier entry */
typedef struct s_groupmember {
  int tid;                      /* tid of member */
  double tEntry;                /* barrier entry date */
  double taEntry;               /* effective barrier entry date */
  double tExit;                 /* barrier exit date */
  double alpha;                 /* event storage intrusion */
  TapeBarrierEvent barrierevt;  /* associated event (for write-back) */
  struct s_groupmember *next;
} *GroupMember;

/* Data relative to barrier */
typedef struct s_barrierdesc {
  char group[80];        /* name of group to barrier */
  int NbBarr;            /* nb of members arrived at barrier */
  int count;             /* total nb of members to arrive */
  GroupMember gmembers;  /* list of members arrived */
  struct s_barrierdesc *next;   /* next barrier */
} *BarrierDesc;

/* Message size slot */
typedef struct {
  double sum;      /* sum of msg find delays */
  int count;       /* number of samples */
} MsgSlot;

/* Task descriptor */
typedef struct s_taskdesc {
  int tid;
  double pacc;
  double tb;
  double tba;
  MsgReq msgreq;
  MsgInBox *msginbox;
  MsgSlot msgslot[K];
} TaskDesc;

/* GLOBALS */

/* Number of tr. delay simulations */
static int nbsim = 0;

/* Number of tr. delay simulations in Sarukkai/Malony */
static int nbsim_sm = 0;

/* number of communications */
static int nbcom = 0;
static int nbcomfound = 0;
static int nbcomfounda = 0;

/* set flag to remove elementary alpha intrusions */
static int REMOVEALPHA = 1;

/* set flag to consider packing and unpacking as intrusions */
static int PKASALPHA = 0;

/* Simulation mode (is MINSIM by default) */
static int SIMMODE = MINSIM;

/* Parameters of communication time model */
static double BETA = DEFAULTBETA; 
static double TAU  = DEFAULTTAU;

/* Parameters of message extraction time model */
static double BETA_EXTR = DEFAULTBETA_EXTR; 
static double TAU_EXTR  = DEFAULTTAU_EXTR;

/* Parameters of message copy time model */
static double BETA_CPY = DEFAULTBETA_CPY; 
static double TAU_CPY  = DEFAULTTAU_CPY;

/* residual sum of squares to check trsim estimator */
static double TRSIM_RSQ = 0.0;
static double TRSIM_RS = 0.0;
static double TRSIM_MODEL_S = 0.0;
static double TRSIM_MEASU_S = 0.0;
static double TRSIM_MODEL_MIN;
static double TRSIM_MODEL_MAX;
static double TRSIM_MEASU_MIN;
static double TRSIM_MEASU_MAX;
#define ALPHA 0.05
#define U_ALPHA ((double)1.96)
/* u_alpha=P(U<u_alpha)=1-alpha/2) */ 

static BarrierDesc BDESC=NULL;

/* Table with task descriptors */
TaskDesc TDTABLE[N];

/* File for table of measured and estimated communication times */
FILE *COMTIMES;

/* PROTOTYPES */

/* per-task perturbation accumulator */
void set_pacc(int tid, double pacc);
void inc_pacc(int tid, double alpha);
double get_pacc(int tid);

/* per-task measured time base */
void set_tb(int tid, double tb);
double get_tb(int tid);

/* per-task compensated time base */
void set_tba(int tid, double tba);
double get_tba(int tid);

/* per-task reception request */
void set_recv_post(int tid, MsgReq *msgreq);
void get_recv_post(int tid, MsgReq *msgreq);

/* per-task message-in box */
void add_msg_in(int tid, MsgInBox *msgib);
void rm_msg_in(int tid, MsgSpec *msgspec);
void get_msg_in(int tid, MsgSpec *msgspec, MsgInBox **msgib);

/* per-task message find delays */
void add_find_delay(int tid, int bytes, double d);
double estim_find_delay(int tid, int bytes);

/* elementary intrusion of event e */
double alpha(TapeEvent *e);

/* local event date correction */
double CorrEvt(double t, int tid);

/* non-blocking send, blocking receive event date correction */
void CorrComEvt(int src, int dst, int bytes,
		double tS, double taS,
		double tSR, double taSR,
		double tER, double *taER,
		char arrived, char *arriveda);

/* transmission time model (machine specific) */
double trsim(int src, int dst, int bytes);
double cpysim(int src, int dst, int bytes);
double extrsim(int src, int dst, int bytes);

void CheckTrsimModel(int src,int dst,int bytes,double tER,double tS);

/* barrier handling */
BarrierDesc get_barrier_desc(char *group);
void add_barrier_member(BarrierDesc bdesc, int tid,
			double tNB, double taNB, double tXB,
			double alpha,
			TapeBarrierEvent *be);
void free_barrier_members(BarrierDesc bdesc);
void correct_barrier_exit(BarrierDesc bdesc,
			  int tidXF,
			  double taXF,
			  double tXF);
void compute_barrier_fl(BarrierDesc bdesc,
			double *tNL, double *taNL, double *alphaNL,
			double *tXF, int *tidXF);
BarrierDesc find_task_in_barrier(int tid);

/* handle sent events on receiver's side */
void handle_send_event(MsgSpec *thismsg, int rtid, int bytes,
		       double tS, double taS);

/* MAIN PROGRAM */

void main(argc, argv)
int argc;
char *argv[];
{

  TapeEvent *evt;
  TapeTrace t;
  TapeTaskList tl;
  FILE *f;
  int k;

  double event_alpha;  /* intrusion alpha of current event */

  /* check command line */

  if (argc==1) {
    fprintf(stderr,"usage: %s [ -sf -sm -beta[extr|cpy] -tau[extr|cpy] -na ] eventfile\n", argv[0]);
    exit(1);
  }

  k=1;
  while(k<argc) {
    if( ! strcmp(argv[k],"-sm") )
      {
	SIMMODE=MINSIM;
	k++;
      }
    else if (! strcmp(argv[k],"-sf"))
      {
	SIMMODE=FULLSIM;
	k++;
      }
    else if (! strcmp(argv[k],"-na"))
      {
	REMOVEALPHA=0;
	k++;
      }
    else if (! strcmp(argv[k],"-p"))
      {
	PKASALPHA=1;
	k++;
      }
    else if (! strcmp(argv[k],"-beta"))
      {
	BETA=atof(argv[k+1]);
	if(BETA<=0)
	  fprintf(stderr,"%s: ignoring erroneous -beta paramater\n",
		  argv[0]);
	k+=2;
      }
    else if (! strcmp(argv[k],"-betaextr"))
      {
	BETA_EXTR=atof(argv[k+1]);
	if(BETA_EXTR<=0)
	  fprintf(stderr,"%s: ignoring erroneous -betaextr paramater\n",
		  argv[0]);
	k+=2;
      }
    else if (! strcmp(argv[k],"-betacpy"))
      {
	BETA_CPY=atof(argv[k+1]);
	if(BETA_CPY<=0)
	  fprintf(stderr,"%s: ignoring erroneous -betacpy paramater\n",
		  argv[0]);
	k+=2;
      }
    else if (! strcmp(argv[k],"-tau"))
      {
	TAU=atof(argv[k+1]);
	if(TAU<=0)
	  fprintf(stderr,"%s: ignoring erroneous -tau parameter\n",
		  argv[0]);
	k+=2;
      }
    else if (! strcmp(argv[k],"-tauextr"))
      {
	TAU_EXTR=atof(argv[k+1]);
	if(TAU_EXTR<=0)
	  fprintf(stderr,"%s: ignoring erroneous -tauextr parameter\n",
		  argv[0]);
	k+=2;
      }
    else if (! strcmp(argv[k],"-taucpy"))
      {
	TAU_CPY=atof(argv[k+1]);
	if(TAU_CPY<=0)
	  fprintf(stderr,"%s: ignoring erroneous -tau_cpy parameter\n",
		  argv[0]);
	k+=2;
      }
    else /* interpret this as filename to open */
      {
	if((t = tape_open_trace(argv[k]))==NULL) {
	  fprintf(stderr, "tape_reader: cannot open %s\n", argv[k]);
	  exit(2);
	}
	k++;
      }
  }

  /* init task descriptor table */
  for(k=0;k<N;k++)
    TDTABLE[k].tid=-1;

  /* open file for communication times */
  if( (COMTIMES=fopen(COMTIMESFILE,"w"))==NULL ) {
    fprintf(stderr,"%s: cannot create table with com. times\n", argv[0]);
  }

  /* browse through the trace */

  tape_init_event(&evt);

  while( tape_read_event( t, &evt ) ) 
    {
      
    event_alpha = alpha(evt); /* in secs */

    switch( evt->header.type ) 
      {
      case event_enroll:
	{
	  TapeEnrollEvent *e = (TapeEnrollEvent *) evt;

	  set_pacc(e->header.task,.0);          /* reset pacc for new task */
	  set_tb(e->header.task,.0);            /* reset tb for new task */
	  set_tba(e->header.task,.0);           /* reset tba for new task */
	  set_recv_post(e->header.task, NULL);  /* reset recv post for new task */
	  
	  /* enroll events are written-back as is */
	  if(REMOVEALPHA) evt->header.alpha=0; tape_wb_event(evt);
	  
	  break;
	}

      case event_send:
	{
	  TapeSendEvent *e = (TapeSendEvent *) evt;
	  MsgSpec  thismsg;
	  double taS, tS;

	  /* measured and effective dates of send */
	  tS  = tape_get_s(e->header.date_s, e->header.date_us);
	  taS = CorrEvt(tS, e->header.task);

	  /* write-back corrected send event :
             header modifications    : date_s and date_us,
             var. part modifications : delta_s and delta_us in FULLSIM mode only
	     Note: delta overhead in var. part is supposed to be the same
	           in MINSIM mode
           */
	  tape_get_s_us(taS,&evt->header.date_s,&evt->header.date_us);

	  if(SIMMODE==FULLSIM)
	    tape_get_s_us( cpysim(e->header.task, e->tid, e->bytes ), 
			   &e->delta_s, &e->delta_us);

	  if(REMOVEALPHA) evt->header.alpha=0; tape_wb_send_event( e );
	  

	  /* message we send here */
	  thismsg.tid = e->header.task;
	  thismsg.tag = e->msgtag;

	  handle_send_event(&thismsg,e->tid,e->bytes,tS,taS);

	  break;
	}


      case event_recv:
	{
	  TapeRecvEvent *e = (TapeRecvEvent *) evt;
	  MsgInBox *msginbox;
	  MsgSpec reqmsgspec;
	  MsgReq recvpost;
	  double tSR, taSR, tER;

	  /* coherency check: when new recv is encountered,
             recv post must not be active */
	  get_recv_post(e->header.task,&recvpost);
	  assert( recvpost.msg.tid==-1 );

	  /* measured and effective dates of recv */
	  tSR  = tape_get_s(e->header.date_s, e->header.date_us);
	  taSR = CorrEvt(tSR, e->header.task);
	  tER  = tape_get_s( e->delta_s+e->header.date_s,
			     e->delta_us+e->header.date_us );

	  /* requested message specification */
	  reqmsgspec.tid = e->tid;
	  reqmsgspec.tag = e->msgtag;

	  /* check if requested message is in my in-box */
	  get_msg_in(e->header.task,
		     &reqmsgspec,
		     &msginbox);

	  if(msginbox)
	    {
	      /* yes, the requested message is already in my in-box */
              /* (this does not mean it has arrived yet !)           */

	      char arriveda;
	      double taER;

	      CorrComEvt(reqmsgspec.tid, e->header.task, e->bytes,
			 msginbox->tS, msginbox->taS,
			 tSR, taSR,
			 tER, &taER,
			 (char)e->arrived, &arriveda);

	      /* write-back corrected recv event
		 header modifications: date_s date_us
		 var. part modifications: arrived delta_s delta_us
               */
	      tape_get_s_us(taSR,
			    &e->header.date_s,
			    &e->header.date_us);
	      e->arrived=arriveda;
	      tape_get_s_us(taER-taSR,
			    &e->delta_s,
			    &e->delta_us);
	      if(REMOVEALPHA) e->header.alpha=0; tape_wb_recv_event(e);
	      

	      /* reset time bases and pacc on recv side */
	      set_tb(e->header.task,tER);
	      set_tba(e->header.task,taER);
	      set_pacc(e->header.task,.0);

	      /* remove message from my in-box */
	      rm_msg_in(e->header.task,
			 &reqmsgspec);
	      	 
	    }
	  else
	    {
	      /* no, the requested message has not been sent yet */
	      /* post a recv request */

	      recvpost.msg.tid = e->tid;
	      recvpost.msg.tag = e->msgtag;
	      recvpost.tSR = tSR;
	      recvpost.taSR = taSR;
	      recvpost.tER = tER;
	      memcpy((char*)&recvpost.recvevt,
		     (char*)e,
		     sizeof(TapeRecvEvent));

	      set_recv_post(e->header.task, &recvpost);
	      set_pacc(e->header.task,.0);

	    }

	  break;
	}

      case event_mcast:
	{
	  TapeMcastEvent *e = (TapeMcastEvent *) evt;
	  MsgSpec thismsg;
	  double taMC, tMC;

	  /* measured and effective dates of multi-cast */
	  tMC  = tape_get_s(e->header.date_s, e->header.date_us);
	  taMC = CorrEvt(tMC, e->header.task);

	  /* write-back corrected mcast event:
	     header modifications    : date_s date_us
	     var. part modifications : delta_s delta_us in FULLSIM mode only
	     Note: delta overhead in var.part is supposed to be the same
	           in MINSIM mode
           */
	  tape_get_s_us(taMC,&evt->header.date_s,&evt->header.date_us);

	  if(SIMMODE==FULLSIM)
	    tape_get_s_us( cpysim(-1, -1, e->bytes ), 
			   &e->delta_s, &e->delta_us);

	  if(REMOVEALPHA) evt->header.alpha=0; tape_wb_mcast_event( evt );

	  /* message we mcast here */
	  thismsg.tid=e->header.task;
	  thismsg.tag=e->msgtag;

	  /* handle send event for each receiver in TaskList */
	  
	  tl = e->TaskList;
	  
	  while( tl ) 
	    handle_send_event(&thismsg,tape_get_next_task( &tl ),e->bytes,
			      tMC, taMC);
	  
	  break;
	}


      case event_bcast:
	{
	  TapeBcastEvent *e = (TapeBcastEvent *) evt;
	  MsgSpec thismsg;
	  double taBC, tBC;

	  /* measured and effective dates of broad-cast */
	  tBC  = tape_get_s(e->header.date_s, e->header.date_us);
	  taBC = CorrEvt(tBC, e->header.task);

	  /* write-back corrected bcast event:
	     header modifications    : date_s date_us
	     var. part modifications : delta_s delta_us in FULLSIM mode only
	     Note: delta overhead is var.part is supposed to be the same
	           in MINSIM mode
           */
	  tape_get_s_us(taBC,&evt->header.date_s,&evt->header.date_us);

	  if(SIMMODE==FULLSIM)
	    tape_get_s_us( cpysim(-1, -1, e->bytes ), 
			   &e->delta_s, &e->delta_us);

	  if(REMOVEALPHA) evt->header.alpha=0; tape_wb_bcast_event( evt );

	  /* message we bcast here */
	  thismsg.tid=e->header.task;
	  thismsg.tag=e->msgtag;

	  /* handle send event for each receiver in TaskList */
	  
	  tl = tape_get_tasks_in_group( &t->groups, e->group);
	  
	  while( tl ) 
	    handle_send_event(&thismsg,tape_get_next_task( &tl ),e->bytes,
			      tBC, taBC);
	  
	  break;
	}

      case event_barrier:
	{
	  TapeBarrierEvent *e = (TapeBarrierEvent *) evt;
	  BarrierDesc bdesc;

	  double taNB;  /* effective barrier entry date */
	  double tNB;   /* measured barrier entry date */
	  double tXB;   /* measured barrier exit date */

	  tNB = tape_get_s(e->header.date_s, e->header.date_us);
	  taNB = CorrEvt(tNB, e->header.task);
	  tXB = tape_get_s(e->header.date_s+e->delta_s,
			   e->header.date_us+e->delta_us);

	  bdesc = get_barrier_desc(e->group);
	  bdesc->NbBarr++;  /* new member arrived */
	  if(e->count==-1)  /* total nb of members to arrive */
	    bdesc->count=tape_get_group_size( &t->groups, e->group);
	  else
	    bdesc->count=e->count;

	  add_barrier_member(bdesc, e->header.task, tNB, taNB, tXB,
			     event_alpha, e);
	  
	  if(bdesc->count==bdesc->NbBarr) {

	    /* All (count) members have arrived at barrier.
	       We now apply Malony's barrier perturbation model
	     */

	    double tNL, taNL;    /* date of last entry */
	    double tXF, taXF;    /* date of first exit */
	    int tidXF;           /* tid which exits first */
	    double deltaBC;      /* exec time of barrier code */
	    double alphaNL;      /* perturbation for last entry event */

	    /* compute tNL, taNL, tXF, tidXF, alphaNL */
	    compute_barrier_fl(bdesc,&tNL,&taNL,&alphaNL,&tXF,&tidXF);

	    /* approximated exec time of barrier code */
	    deltaBC = tXF-tNL-alphaNL;
	    assert(deltaBC>0.0);

	    /* approximated date of first barrier exit */
	    taXF = taNL+deltaBC;

	    /* update time bases for barrier exit events */	    
	    correct_barrier_exit(bdesc,tidXF,taXF,tXF);

	    /* clean up */
	    free_barrier_members(bdesc);
	    
	  }

	  /* reset pacc */
	  set_pacc(e->header.task,.0);

	  break;
	}

      case event_trace: /* buffer flush event (obsolete since 0.9pl8) */
	{
	  TapeTraceEvent *e = (TapeTraceEvent *) evt;
	  
	  inc_pacc(evt->header.task,
		   tape_get_s(e->header.file, e->header.line));
	  break;
	}

      default: 
	/* any local event other than those explicitly handled above */
	/* in general, these are simply corrected and written back   */
	{
	  MsgReq msgreq;
	  double taEvt;

	  /* coherency check: there must not be an active receive
	                      nor an ongoing barrier rendez-vous involving
                              this task
           */

	  get_recv_post(evt->header.task, &msgreq);
	  assert( msgreq.msg.tid == -1 );
	  assert( find_task_in_barrier(evt->header.task)==NULL );

	  /* correct event */
	  taEvt=CorrEvt(tape_get_s(evt->header.date_s,evt->header.date_us), evt->header.task);
	  tape_get_s_us(taEvt,&evt->header.date_s,&evt->header.date_us);

	  /* write-back corrected event:
	     - header modifications: date_s date_us
	     - var. part modifications: none
	     - Note: 
	        if PKASALPHA is set, pacc is increased by the delta
	        overhead in the (u)pk event and the event is not written
	        back
	   */

	  switch(evt->header.type) {

	  case  event_pkbyte:
	  case  event_pkcplx:
	  case  event_pkdcplx:
	  case  event_pkdouble:
	  case  event_pkfloat:
	  case  event_pkint:
	  case  event_pklong:
	  case  event_pkshort:
	  case  event_pkuint:
	  case  event_pkulong:
	  case  event_pkushort:
	  case  event_pkstr:
	  case  event_upkbyte:
	  case  event_upkcplx:
	  case  event_upkdcplx:
	  case  event_upkdouble:
	  case  event_upkfloat:	
	  case  event_upkint:
	  case  event_upklong:
	  case  event_upkshort:
	  case  event_upkuint:
	  case  event_upkulong:
	  case  event_upkushort:
	  case  event_upkstr:
	    {
	      TapePk1Event *e = (TapePk1Event *) evt;

	      if(PKASALPHA)
		/* consider (u)pk as intrusion and do not write back */
		inc_pacc(evt->header.task,
			 tape_get_s(e->delta_s, e->delta_us));
	      else {
		/* simply write back as is */
		if(REMOVEALPHA) evt->header.alpha=0; tape_wb_event(evt);
	      }

	      break;
	    }

	  default:
	    if(REMOVEALPHA) evt->header.alpha=0; tape_wb_event(evt);
	    break;

	  } /* switch */

	  break;

	} /* default */

      } /* switch */

    /* increment perturbation accumulator*/
    if(REMOVEALPHA) inc_pacc(evt->header.task, event_alpha );
  }

  tape_dispose_event( &evt );

  tape_close_trace( t );

  /* close file with communication time statistics */
  if(COMTIMES!=NULL)
    fclose(COMTIMES);

  /* write info about intrusion compensation */

  {

    double n;
    double SIGMA;
    double k;
    double mean_res;

    if( (f=fopen(INFOFILE,"w"))==NULL ) {
      f=stdout;
      fprintf(stderr,"%s: cannot write info file, using stdout instead\n", argv[0]);
    }

    fprintf(f,"NBCOM=%d NBCOMFOUND=%d (%.0f%%) NBCOMFOUNDA=%d (%.0f%%) NBSIM=%d (%.0f%%) NBSIM_SM=%d (%.0f%%)\n\n",
	    nbcom,
	    nbcomfound,
	    nbcomfound*100.0/nbcom,
	    nbcomfounda,
	    nbcomfounda*100.0/nbcom,
	    nbsim,
	    nbsim*100.0/nbcom,
	    nbsim_sm,
	    nbsim_sm*100.0/nbcom);

    fprintf(f,"Alpha correction is ");
    if(REMOVEALPHA) fprintf(f,"on\n");
    else fprintf(f,"off\n");
    fprintf(f,"Simulation mode is ");
    if(SIMMODE==MINSIM) fprintf(f,"MINSIM\n");
    if(SIMMODE==FULLSIM) fprintf(f,"FULLSIM\n");
    fprintf(f,"Packing and unpacking delays are ");
    if(!PKASALPHA) fprintf(f,"NOT ");
    fprintf(f,"removed\n");

    /* info about automatic model checking */

    n=nbcom-nbcomfound;
    SIGMA=sqrt( TRSIM_RSQ/(n-2.0) );
    k=U_ALPHA*SIGMA/sqrt(n);
    mean_res=TRSIM_RS/n;

    fprintf(f,"\nTRSIM MODEL CHECK\n\n");
    fprintf(f,"  BETA = %.6f\n", BETA);
    fprintf(f,"  TAU = %.6f\n", TAU);
    fprintf(f,"  NBCHECK = %d\n", nbcom-nbcomfound);
    fprintf(f,"  SUM OF RESIDUALS = %.6f\n", TRSIM_RS);
    fprintf(f,"  MEAN OF RESIDUALS = %.6f\n", mean_res);
    fprintf(f,"  STDEV OF RESIDUALS = %.6f\n", SIGMA);
    fprintf(f,"  K VALUE FOR ALPHA (%.3f) = %.6f ", ALPHA, k);
    if(mean_res>k||mean_res<-k)
      fprintf(f,"[REJECT]\n");
    else
      fprintf(f,"[ACCEPT]\n");
    fprintf(f,"  MIN:MEAN:MAX OF MEASU TRANSIT = %.6f:%.6f:%.6f\n",
	    TRSIM_MEASU_MIN,TRSIM_MEASU_S/n,TRSIM_MEASU_MAX);
    fprintf(f,"  MIN:MEAN:MAX OF MODEL TRANSIT = %.6f:%.6f:%.6f\n",
	    TRSIM_MODEL_MIN,TRSIM_MODEL_S/n,TRSIM_MODEL_MAX);

    if(SIMMODE==FULLSIM) {
      fprintf(f,"\n\nEXTRSIM MODEL INFO\n\n");
      fprintf(f,"  BETAEXTR = %.6f\n", BETA_EXTR);
      fprintf(f,"  TAUEXTR = %.6f\n", TAU_EXTR);

      fprintf(f,"\n\nCPYSIM MODEL INFO\n\n");
      fprintf(f,"  BETACPY = %.6f\n", BETA_CPY);
      fprintf(f,"  TAUCPY = %.6f\n", TAU_CPY);
    }

  fclose(f);
  }
}

/* IMPLEMENTATION */


/* ================================================================= 
 * Handle compensation of send event: used in send, mcast and bcast
 * -----------------------------------------------------------------
 * In:   thismsg is <tid,tag> of sent, mcast or bcast message
 *       rtid is task id. of receiver
 *       bytes is size of the sent message
 *       tS, taS are mesured and acutal dates of send, mcast or bcast
 * ================================================================= */

void handle_send_event(MsgSpec *thismsg, int rtid, int bytes,
		       double tS, double taS)
{
  MsgReq   msgreq;
	
  /* message expected on other side ( if any ) */
  get_recv_post(rtid, &msgreq);

  if( msgreq.msg.tid == thismsg->tid && 
      msgreq.msg.tag == thismsg->tag )
    {
      /* The receiver is waiting for this message */

      char arriveda;
      double taER;
      TapeRecvEvent *recvevt;

      recvevt=&msgreq.recvevt;

      /* coherency check: message received before
	 sent in case assertion fails */
      assert(recvevt->arrived==0);

      CorrComEvt(thismsg->tid, rtid, bytes,
		 tS, taS,
		 msgreq.tSR, msgreq.taSR,
		 msgreq.tER, &taER,
		 0, &arriveda);

      /* write-back corrected receive event
	 header modifications: date_s date_us
	 var. part modifications: arrived delta_s delta_us
	 */
      tape_get_s_us(msgreq.taSR,
		    &recvevt->header.date_s,
		    &recvevt->header.date_us);
      recvevt->arrived=arriveda;
      tape_get_s_us(taER-msgreq.taSR,
		    &recvevt->delta_s,
		    &recvevt->delta_us);
      if(REMOVEALPHA) recvevt->header.alpha=0; tape_wb_recv_event(recvevt);

      /* reset time bases on recv side -- pacc has already */
      /* been set during post of req. by event_recv        */

      set_tb(rtid,msgreq.tER);
      set_tba(rtid,taER);

      /* clean message request */
      set_recv_post(rtid, NULL);
    }
  else
    {
      /* The receiver has not posted a request for this     */
      /* message yet. We put the data relative to the sent  */
      /* message in the receiver's in-box                   */

      /* create a new MsgInBox entry */
      MsgInBox msginbox;
      msginbox.msg.tid = thismsg->tid;
      msginbox.msg.tag = thismsg->tag;
      msginbox.tS = tS;
      msginbox.taS = taS;
      add_msg_in(rtid,&msginbox);
    }
}


/* ================================================================
 * Message transition time estimator. Message transmission time
 * is the delay between the begining of the send primitive and 
 * the end of the corresponding receive primitive. It includes
 * message copying (sender) and extracting (receiver) delays !
 * ----------------------------------------------------------------
 * In:  src, dst are source and destination task id. of comm.
 *      bytes is size of message
 *      src=-1 or dst=-1  <=> the caller explicitly states that src
 *      and dst are not relevant to the model
 * Ret: estimated transition delay in sec
 * ================================================================ */

double trsim(int src, int dst, int bytes)
{
  return (TAU*bytes+BETA)*1e-6;
}

/* extraction time estimator (only used by FULLSIM) */
double extrsim(int src, int dst, int bytes)
{
  return (TAU_EXTR*bytes+BETA_EXTR)*1e-6;
}

/* copy time estimator (only used by FULLSIM) */
double cpysim(int src, int dst, int bytes)
{
  return (TAU_CPY*bytes+BETA_CPY)*1e-6;
}

/* ================================================================
 * Elementary event generation overhead
 * ----------------------------------------------------------------
 * In:   evt 
 * ----------------------------------------------------------------
 * Ret:  overhead in s inferred to the task on which event occurred
 * ================================================================ */

double alpha(TapeEvent *evt)
{
  return (double)evt->header.alpha * 1e-6 ;
}

/* ================================================================
 * Blocking Receive / non-blocking send correction scheme
 * ----------------------------------------------------------------
 * In:  src, dst are source and destination task id of comm
 *      bytes is size of message
 *      tS, taS are measured and actual dates of send
 *      tSR, taSR are measured and actual dates of start receive 
 *      tER is measured date of end receive
 *      arrived <=> message arrived before tSR
 * ----------------------------------------------------------------
 * Out: taER actual date of end receive
 *      arriveda <=> compensated message arrives before taSR
 * ================================================================ */

void CorrComEvt(int src, int dst, int bytes,
		double tS, double taS,
		double tSR, double taSR,
		double tER, double *taER,
		char arrived, char *arriveda )
{
  double taB;

  /* Let tB be the arrival date of the message.
   * We suppose tB cannot be measured. We first check
   * our sufficient conditions to see if the actual
   * communication pattern is the same as the
   * measured one.
   */

  nbcom++;

  if(tS<tSR)
    nbsim_sm++;

  if( arrived ) {  /* case 1 communication pattern */

    nbcomfound++;

    /* compute message find delay */
    add_find_delay(dst,bytes,tER-tSR);

    if(SIMMODE==MINSIM) {

      /* arrived <=> tB <= tSR : do we also have taB <= taSR ? */

      if( tSR-taSR-tS+taS <= 0 )

	{ /* yes ! */

	  nbcomfounda++;
	  *taER = taSR+tER-tSR;
	  *arriveda = arrived;
	}

      else

	{ /* not necessarily ! We have to simulate transmission
	   * time to estimate taB, so we can decide if 
	   * taB <= taSR or not.
	   */

	  nbsim++;

	  taB = taS + trsim(src,dst,bytes) - (tER-tSR);
	  /* we substract the measured message extraction delay, because
	   * trsim includes that delay 
	   */

	  if(taB<taS) {
	    taB=taS+0.000001;
	    fprintf(stderr,"tape_ic: warning, trsim is smaller than message extraction delay.\n");
	  }

	  *arriveda = (taB<=taSR);
	  if(*arriveda) nbcomfounda++;
	  *taER = *arriveda ? taSR+tER-tSR : taB+tER-tSR ;
	}
    } /* SIMMODE */

  } /* arrived */

  else {  /* case 2 communication pattern */

    /* in case 2 we can measure the message transmission delay */
    /* and check it against the model in trsim */
    CheckTrsimModel(src,dst,bytes,tER,tS);

    if(SIMMODE==MINSIM) {

      /* !arrived <=> tB > tSR : do we also have taB > taSR ? */

      if( tSR-taSR-tS+taS > 0 ) 

	{ /* yes ! */

	  *taER = tER-tS+taS;
	  *arriveda = arrived;
	}

      else

	{ /* not necessarily ! We estimate tB=tER-DC, where DC
	   * is message extraction delay. DC is estimated using
	   * previous measured receive delays which found their
	   * message.
	   */

	  double DC = estim_find_delay(dst,bytes);
	  taB = /* taS+tER-DC-tS+taS */
	    tER-DC-tS+taS ;
	  *arriveda = (taB<=taSR);
	  if(*arriveda) nbcomfounda++;
	  *taER = *arriveda ? taSR+tER-taB-tS+taS : tER-tS+taS;
	}
    } /* SIMMODE */
  } /* ! arrived */


  if(SIMMODE==FULLSIM) { /* always simulate communication times */

    /* model message extraction delay DC */
    double DC = extrsim(src,dst,bytes);
    double DS = trsim(src,dst,bytes);

    nbsim++;

    if(DS<DC) {
      DC = DS - 0.000001;
      fprintf(stderr,"tape_ic: warning, trsim (%.6f) < extrsim (%.6f), using extrsim=trsim.\n",
	      DS, DC);
    }

    taB = taS + DS - DC;

    *arriveda = (taB<=taSR);
    if(*arriveda) nbcomfounda++;
    *taER = *arriveda ? taSR+DC : taB+DC ;
  }    

}

/* ================================================================
   Check communication model used by "trsim" against measured comm-
   unication delay in a case2 communication pattern.
   ----------------------------------------------------------------
   In:   src,dst are source and destination tasks
         bytes is size of the message (in bytes) exchanged
	 tER is measured date of end of receive (on task "dst")
	 tS is measured date of start of send (on task "src")
   ================================================================ */

void CheckTrsimModel(int src, int dst, int bytes,
		    double tER, double tS)
{
  double model=trsim(src,dst,bytes);
  double measu=tER-tS;
  static int count = 0;

  count++;

  if(COMTIMES!=NULL) {
    if(count==1)
      fprintf(COMTIMES,"# Message comm. times: size measu model measu-model\n\n");
    fprintf(COMTIMES,"%d\t%.6f\t%.6f\t%.6f\n", bytes, measu, model, measu-model);
  }

  TRSIM_RSQ+=pow(measu-model,2.0);
  TRSIM_RS+=measu-model;
  TRSIM_MODEL_S+=model;
  TRSIM_MEASU_S+=measu;
  if(nbcom-nbcomfound==1) {
    TRSIM_MODEL_MAX=TRSIM_MODEL_MIN=model;
    TRSIM_MEASU_MAX=TRSIM_MEASU_MIN=measu;
  } else {
    if(model>TRSIM_MODEL_MAX)
      TRSIM_MODEL_MAX=model;
    if(model<TRSIM_MODEL_MIN)
      TRSIM_MODEL_MIN=model;
    if(measu>TRSIM_MEASU_MAX)
      TRSIM_MEASU_MAX=measu;
    if(measu<TRSIM_MEASU_MIN)
      TRSIM_MEASU_MIN=measu;
  }
}


/* ================================================================
   Compute effective barrier exit dates
   ----------------------------------------------------------------
   In:   bdesc is barrier descriptor with barrier performance data
         tidXF is tid which exits barrier first
         taXF, tXF are effective and measured exit dates of tidXF
   ================================================================ */

void correct_barrier_exit(BarrierDesc bdesc,
			  int tidXF,
			  double taXF,
			  double tXF)
{
  GroupMember gm;
  double taX;    /* effective exit date */

  /* browse list of group members */
  gm=bdesc->gmembers;
  while(gm) {

    /* compute effective exit */
    taX=taXF+gm->tExit-tXF;
    
    /* set time bases */
    set_tb(gm->tid,gm->tExit);
    set_tba(gm->tid,taX);

    /* write-back barrier event:
       header modifications: date_s date_us
       var. part modifications: delta_s delta_us
     */
    tape_get_s_us(gm->taEntry,
		  &gm->barrierevt.header.date_s,
		  &gm->barrierevt.header.date_us);
    tape_get_s_us(taX-gm->taEntry,
		  &gm->barrierevt.delta_s,
		  &gm->barrierevt.delta_us);
    if(REMOVEALPHA) gm->barrierevt.header.alpha=0; tape_wb_barrier_event(&gm->barrierevt);
  
    gm=gm->next;
  }
}


/* ================================================================
   free barrier members (after rendez-vous)
   ----------------------------------------------------------------
   In:   bdesc is barrier descriptor of barrier to free
   ================================================================ */

void free_barrier_members(BarrierDesc bdesc)
{
  GroupMember gm, gm2;

  bdesc->NbBarr=0;
  bdesc->count=0;

  /* free member list */
  gm=bdesc->gmembers;
  bdesc->gmembers=NULL;
  while(gm) {
    gm2=gm->next;
    free(gm->barrierevt.group);
    free(gm);
    gm=gm2;
  }
}

/* ================================================================
   add a new member to a barrier
   ----------------------------------------------------------------
   In:   bdesc is barrier joined by the member
         tid is task identifier of the newcomer
	 tNB, taNB are barrier entry dates (measured and effective)
	 tXB is (measured) barrier exit date
	 alpha is intrusion inferred by event generation
   ================================================================ */

void add_barrier_member(BarrierDesc bdesc, int tid,
			double tNB, double taNB, double tXB,
			double alpha,
			TapeBarrierEvent *be)
{
  GroupMember gm, gm2;
  char *groupname;

  /* create new group member */
  gm=(GroupMember)malloc(sizeof(struct s_groupmember));
  groupname=(char*)malloc(strlen(be->group)+1);
  strcpy(groupname,be->group);
  gm->tid=tid;
  gm->tEntry=tNB;
  gm->taEntry=taNB;
  gm->tExit=tXB;
  gm->alpha=alpha;
  memcpy(&gm->barrierevt,be,sizeof(TapeBarrierEvent));
  gm->barrierevt.group=groupname;
  gm->next=NULL;

  /* insert at head of list */
  if(bdesc->gmembers==NULL)
    bdesc->gmembers=gm;
  else {
    gm2=bdesc->gmembers;
    bdesc->gmembers=gm;
    gm->next=gm2;
  }
}



/* ================================================================
   allocate and/or return barrier descriptor for a given group
   ----------------------------------------------------------------
   In:   group is name of the group
   ----------------------------------------------------------------
   Ret:  barrier descriptor for group
         if not existing, a new one is created with an empty list
	 of members
   ================================================================ */

BarrierDesc get_barrier_desc(char *group)
{
  BarrierDesc bdesc, bdesc2;

  /* look-up a bdesc for group */
  bdesc=bdesc2=BDESC;
  while(bdesc) {
    if(!strcmp(bdesc->group,group))
      break;
    bdesc2=bdesc;
    bdesc=bdesc->next;
  }

  if(!bdesc) {
    /* no bdesc found for group, allocate a new one */
    bdesc=(BarrierDesc)malloc(sizeof(struct s_barrierdesc));
    strcpy(bdesc->group,group);
    bdesc->NbBarr=0;
    bdesc->count=0;
    bdesc->gmembers=NULL;
    bdesc->next=NULL;
    if(!BDESC) BDESC=bdesc;
    else
      bdesc2->next=bdesc;
  }
   
  return bdesc;
  
}

/* ================================================================
 * find task in barrier
 * ----------------------------------------------------------------
 * In:  tid is task id. of task to find
 * ----------------------------------------------------------------
 * Ret: the BarrierDesc in whose members tid was found
 *      NULL if tid not found in any barrier
 *      if multiple matches the most recent barrier is returned
 * ================================================================ */

BarrierDesc find_task_in_barrier(int tid)
{
  BarrierDesc bdesc=BDESC;

  while(bdesc)
    {
      GroupMember gm=bdesc->gmembers;
      while(gm)
	{
	  if(gm->tid==tid)
	    return bdesc;
	  gm=gm->next;
	}
      bdesc=bdesc->next;
    }
  return NULL;
}

/* ================================================================
   correct local event
   ----------------------------------------------------------------
   In:   t is measured date of event
         tid is task identifier on which event occurred
   ----------------------------------------------------------------
   Ret:  approximated effective date of event
   ================================================================ */

double CorrEvt(double t, int tid)
{
  double pacc;   /* perturbation accumulation */
  double tb;     /* measured time base */
  double tba;    /* actual time base */
  
  pacc=get_pacc(tid);
  tb=get_tb(tid);
  tba=get_tba(tid);

  printf("CorrEvt: tid=%d tba=%.6f t=%.6f tb=%.6f pacc=%.6f ta=%.6f\n",
	 tid,tba,t,tb,pacc,tba+t-tb-pacc);

  return tba+t-tb-pacc;
}

/* ================================================================
   compute perf. data of first barrier entry and last barrier exit
   ----------------------------------------------------------------
   In:   bdesc is barrier descriptor
   ----------------------------------------------------------------
   Out:  tNL is date of last measured barrier entry
         taNL is date of last (effective) barrier entry
	 alphaNL is perturbation of last (measured) entry event
	 tXF is date of first (measured) barrier exit
	 tidXF is task id. of first (measured) barrier exit
   ================================================================ */

void compute_barrier_fl(BarrierDesc bdesc,
			double *tNL, double *taNL, double *alphaNL,
			double *tXF, int *tidXF)
{
  GroupMember gm;
  int gmcount;

  *tNL=*taNL=0.0;

  gm=bdesc->gmembers;
  gmcount=0;

  while(gm) {
    gmcount++;

    if(gm->tEntry > *tNL)
      { *tNL=gm->tEntry;
	*alphaNL=gm->alpha;
      }

    if(gm->taEntry > *taNL)
      *taNL=gm->taEntry;

    if(gmcount==1 || *tXF > gm->tExit) {
      *tXF=gm->tExit;
      *tidXF=gm->tid;
    }

    gm=gm->next;
  }
}

/* =================================================
   Task Descriptor Table handling
   ================================================= */

int get_td_ndx(int tid) {
  int k;

  for(k=0;k<N&&TDTABLE[k].tid!=tid&&TDTABLE[k].tid!=-1;k++);

  if(k==N) {
    fprintf(stderr,"maximum number of tasks is %d\n",N);
    exit(1);
  }

  else if(TDTABLE[k].tid==tid) {
    /* entry found at k */
    return k;
  }

  else /* TDTABLE[k].tid==-1 */ {
    int kk;
    /* allocate entry for tid at k */
    TDTABLE[k].tid=tid;
    TDTABLE[k].pacc=0.0;
    TDTABLE[k].tb=0.0;
    TDTABLE[k].tba=0.0;
    TDTABLE[k].msgreq.msg.tid=-1; /* no active request */
    TDTABLE[k].msgreq.msg.tag=-1;
    TDTABLE[k].msginbox=NULL; /* no messages in inbox */
    for(kk=0;kk<K;kk++) {
      TDTABLE[k].msgslot[kk].sum=0.0;
      TDTABLE[k].msgslot[kk].count=0;
    }
    return k;
  }
}


void set_pacc(int tid, double pacc)
{
  TDTABLE[get_td_ndx(tid)].pacc=pacc;
}

void inc_pacc(int tid, double alpha)
{
  TDTABLE[get_td_ndx(tid)].pacc+=alpha;
}

double get_pacc(int tid)
{
  return TDTABLE[get_td_ndx(tid)].pacc;
}

void set_tb(int tid, double tb)
{
  TDTABLE[get_td_ndx(tid)].tb=tb;
}

double get_tb(int tid)
{
  return TDTABLE[get_td_ndx(tid)].tb;
}

void set_tba(int tid, double tba)
{
  TDTABLE[get_td_ndx(tid)].tba=tba;
}

double get_tba(int tid)
{
  return TDTABLE[get_td_ndx(tid)].tba;
}

void add_find_delay(int tid, int bytes, double d)
{
  int k,ndx;

  /* compute slot index for bytes */
  k=(int)log((double)bytes+1)/log(2.0);
  if(k>=K)
    k=K-1;

  TDTABLE[ndx=get_td_ndx(tid)].msgslot[k].sum+=d;
  TDTABLE[ndx].msgslot[k].count++;
}

double estim_find_delay(int tid, int bytes)
{
  int ndx=get_td_ndx(tid),
      k=(int)log((double)bytes+1)/log(2.0);

  if(k>=K)
    k=K-1;

  return TDTABLE[ndx].msgslot[k].count ?
         TDTABLE[ndx].msgslot[k].sum/TDTABLE[ndx].msgslot[k].count :
	 DEFAULT_FIND_DELAY;
}

void set_recv_post(int tid, MsgReq *msgreq)
{
  if(msgreq==NULL)
    /* clean recv post on tid */
    TDTABLE[get_td_ndx(tid)].msgreq.msg.tid=-1;
  else {
    /* copy data in msgreq to table */
    memcpy((char*)&TDTABLE[get_td_ndx(tid)].msgreq,
	   (char*)msgreq,
	   sizeof(MsgReq));
  }
}

void get_recv_post(int tid, MsgReq *msgreq)
{
  /* copy data in table to msgreq */
  memcpy((char*)msgreq,
	 (char*)&TDTABLE[get_td_ndx(tid)].msgreq,
	 sizeof(MsgReq));
  
}

void add_msg_in(int tid, MsgInBox *msgib)
{
  MsgInBox *mb,*mbnew;
  int ndx;

  /* head of list */
  mb=TDTABLE[ndx=get_td_ndx(tid)].msginbox;

  /* create new MsgInBox entry */
  mbnew=(MsgInBox*)malloc(sizeof(MsgInBox));
  memcpy((char*)mbnew,
	 (char*)msgib,
	 sizeof(MsgInBox));
  mbnew->next=NULL; /* tail */

  /* insert message at tail of list -- insertion at tail
     is important in case a same tid sends us a series
     of messages with the same tag. These have to be
     extracted in correct order */
  if(mb==NULL)
    TDTABLE[ndx].msginbox=mbnew;
  else {
    MsgInBox *mb2;
    do 
      {
	mb2=mb;
	mb=mb->next;
      }
    while(mb);
    mb2->next=mbnew;
  }
}

/* =================================================================
 * get_msg_in_aux: return specified message from in-box
 * -----------------------------------------------------------------
 * In:  tid is task id
 *      msgspec is message specifier (tid,tag)
 * -----------------------------------------------------------------
 * Out: msgib is requested message (NULL if not found)
 *            if more than one message matches msgspec, the first one
 *            (list order) is returned
 *      msgib_before is message before msgib in list (NULL if none)
 *      msgib_after is message after msgib in list (NULL if none)
 * ================================================================= */

void get_msg_in_aux(int tid, MsgSpec *msgspec, 
		    MsgInBox **msgib,
		    MsgInBox **msgib_before,
		    MsgInBox **msgib_after)
{
  MsgInBox *m, *m_bef=NULL;

  m=TDTABLE[get_td_ndx(tid)].msginbox;

  while(m) {
    if(m->msg.tid==msgspec->tid &&
       m->msg.tag==msgspec->tag )
      break;
    m_bef=m;
    m=m->next;
  }

  if(m==NULL) {
    /* no message found */
    *msgib=*msgib_before=*msgib_after=NULL;
  } else {
    *msgib=m;
    *msgib_before=m_bef;
    *msgib_after=m->next;
  }
}

void get_msg_in(int tid, MsgSpec *msgspec, MsgInBox **msgib)
{
  MsgInBox *msgib_before, *msgib_after;

  get_msg_in_aux(tid,msgspec,msgib,&msgib_before,&msgib_after);
}

void rm_msg_in(int tid, MsgSpec *msgspec)
{
  MsgInBox *msgib, *msgib_before, *msgib_after;

  get_msg_in_aux(tid,msgspec,&msgib,&msgib_before,&msgib_after);

  if(msgib==NULL)
    /* not found, nothing to rm */
    return;

  free(msgib);

  if(msgib_before!=NULL)
    msgib_before->next=msgib_after;
  else
    TDTABLE[get_td_ndx(tid)].msginbox=msgib_after;
}
  
