/* tape2npicl: convert TAPE/PVM traces to new picl format
 *
 * Author: Eric Maillet, LMC-INPG, Grenoble, France
 * Date  : december 1994
 *
 * Notes : the TAPE/PVM input trace has to be sorted by date
 *         02/05/95: Added support for event_nrecv
 *         25/05/95: Nrecv generates OVHD when message not arrived
 *                   Added support for event_trecv
 *         15/06/95: Added support for event_psend,precv,gather,reduce
 *                   and scatter
 *                   Added some trace coherency checking
 *         18/06/95: Picl tracenode event is no longer generated when
 *                   event_mytid is encountered, but on the *first*
 *                   event encountered.
 *         19/06/95: Added support for special event_enroll, which
 *                   becomes the first event on each task 
 *         26/06/95: Added nodetable support
 */

/* To do: instead of repeating the same printf's I could use macros
          or functions. E.g. RECV0WAITENTRY(tag), 
                             RECV0WAITEXIT (bytes,msgtag,tid)
 */ 

/* $Log$ */

#include "tape_reader.h"
#include "tape_events.h"

#include <stdio.h>
#include <string.h>

/* file name for the nodetable */

#define NODETABLEOUT "nodetable.out"
#define NODETABLEIN  "nodetable"

/* maximal number of PICL nodes */

#define MAXNODES      512

/* PICL Record types */

#define RT_MARK       -2
#define RT_ENTRY      -3
#define RT_EXIT       -4

/* PICL Event types */

#define ET_TRACENODE    -901
#define ET_SEND0         -21
#define ET_RECV0_NOWAIT  -51
#define ET_RECV0_WAIT    -52
#define ET_IDLE         -601
#define ET_OVERHEAD     -602
#define ET_USER           99

/* PICL Event Header Generation Macro
 *
 *  rt = Record Type
 *  et = Event Type
 *  nb = Number of Data Fields
 */

#define EH_MACRO(rt,et,nb)                          \
 rt, et,                                            \
 tape_get_s( e->header.date_s, e->header.date_us ), \
 Task2Node( e->header.task ),                       \
 -1,nb

/* Add the delta delay to event date. Used to compute
 * the date of exit events
 */

#define DatePlusDelta                                       \
 e->header.date_s+=e->delta_s;                              \
 e->header.date_s+=(e->header.date_us+e->delta_us)/1000000; \
 e->header.date_us=(e->header.date_us+e->delta_us)%1000000

/* LoadNodes: load PICL node assignments */
/* node is the first, tid the second token in each line */

int
LoadNodes(NodeTable)
     int NodeTable[];
{
  FILE *f;
  char line[80];
  char *token;
  int ln=0,k=0;
  int node;
  int tid;

  if((f=fopen(NODETABLEIN,"r"))!=NULL) {
    while(fgets(line,80,f)!=NULL) {
      ln++; /* nb lines */
      if((token=strtok(line,"\t "))==NULL)
	break;
      node=atoi(token);
      if((token=strtok(NULL,"\t "))==NULL)
	break;
      tid=atoi(token);
      NodeTable[node]=tid;
      k++; /* nb assign */
    }
    return k;
  }
  else return 0;
}

/* Task2Node: convert PVM Task to PICL node (0..n-1)
 * Task2Node(tid)      returns the picl node assigned to tid  
 * Task2Node(NBNODES)  returns the number of nodes currently defined
 */

#define NBNODES -1

int
Task2Node( task )
     int task;
{
  static int n = 0;  /* number of nodes */
  static int NT[MAXNODES];

  int i;

  if(task==NBNODES)
    return n;

  if(!n) {
    /* load node table if present */
    n=LoadNodes(NT);
  }

  for(i=0;i<n;i++)
    if(task==NT[i])
      break;

  if( i < n )   /* found task in i */
    return i;

  NT[i] = task; /* insert new task at i */
  n++;

  return i;
}

/* generate idle state on a node */

void
tape_generate_idle_state(node, start_s, delta_s)
     int node;
     double start_s;
     double delta_s;
{
  printf("%d %d %.6f %d %d %d\n",
	 RT_ENTRY, ET_IDLE, start_s, node, -1, 0);
  printf("%d %d %.6f %d %d %d\n",
	 RT_EXIT, ET_IDLE, start_s+delta_s, node, -1, 0);
}

/* generate overhead state on a node */

void
tape_generate_overhead_state(node, start_s, delta_s)
     int node;
     double start_s;
     double delta_s;
{
  printf("%d %d %.6f %d %d %d\n",
	 RT_ENTRY, ET_OVERHEAD, start_s, node, -1, 0);
  printf("%d %d %.6f %d %d %d\n",
	 RT_EXIT, ET_OVERHEAD, start_s+delta_s, node, -1, 0);
}

/* main event converting loop */

void main(argc, argv)
int argc;
char *argv[];
{

  TapeEvent *evt;
  TapeTrace t;
  TapeTaskList tl;

  FILE *f;  /* node table */

  char TapeVersionOK;
  char count_mytid = 0;
  char count_exit = 0;
  int  count_trace =0;
  char flag_spawn = 0;
  char flag_kill  = 0; 

  /* check command line */

  if( argc > 2 ) {
    fprintf(stderr, "usage: %s <tape-trace>\n", argv[0]);
    exit(1);
  }

  /* open a tape trace file */

  switch(argc) {

  case 1:
    t = tape_open_trace( NULL );
    break;
    
  case 2:
    t = tape_open_trace(argv[1]);
    break;
  }

  if( t == NULL ) {
    fprintf(stderr, "tape_reader: cannot open %s\n", argv[1]);
    exit(2);
  }

  f=fopen(NODETABLEOUT,"w");

  /* browse through the trace */

  tape_init_event(&evt);

  while( tape_read_event( t, &evt ) ) 
    {

    switch( evt->header.type ) 
      {

	/* event_enroll is the first event on each task */

      case event_enroll:
	{
	  TapeEnrollEvent *e = (TapeEnrollEvent *) evt;
	  printf( "%d %d %.6f %d %d %d\n",
		 EH_MACRO(RT_ENTRY, ET_TRACENODE, 0)
		 );

	  /* generate entry in node table */
	  if(f!=NULL)
	    fprintf(f,"%3d%12d%10x   %s/%s/%d\n",
		    Task2Node(e->header.task),
		    e->header.task,
		    e->header.task,
		    e->hi_name,
		    e->hi_arch,
		    e->hi_speed);
	  
	  break;
	}

      case event_mytid:
	{
	  count_mytid++;
	  break;
	}

      case event_exit:
	{
	  TapeExitEvent *e = (TapeExitEvent *) evt;
	  printf( "%d %d %.6f %d %d %d\n",
                  EH_MACRO(RT_EXIT, ET_TRACENODE, 0)
                );
	  count_exit++;
	  break;
	}

      case event_kill:
	{
	  flag_kill = 1;
	  break;
	}

      case event_spawn:
	{
	  flag_spawn = 1;
	  break;
	}

      case event_open_phase:
      case event_close_phase:
	{
	  /* when a phase is opened or closed, we generate an
	     user mark event - this makes Paragraph stop at
	     start and end of phases ( we also put the name of 
	     the phase in the PICL trace although this information
	     is not used by Paragraph ) */

	  TapeOpenPhaseEvent *e = (TapeOpenPhaseEvent *) evt;
	  printf( "%d %d %.6f %d %d %d %d %s\n",
                  EH_MACRO(RT_MARK, ET_USER, 1),1,
		  e->phase
                );
	  break;
	}
	  

      /* event_send and event_psend have the same structure */

      case event_send:
      case event_psend:
	{
	  TapeSendEvent *e = (TapeSendEvent *) evt;

	  printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
	          EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		  e->bytes,
		  e->msgtag,
		  Task2Node(e->tid)
		);

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		  EH_MACRO(RT_EXIT, ET_SEND0, 0)
		);
	  break;
	}

      /* event_recv and event_precv have the same structure */

      case event_recv:
      case event_precv:
	{
	  int et;
	  TapeRecvEvent *e = (TapeRecvEvent *) evt;

	    /* check if msg arrived */
	  if( e->arrived )
	    et = ET_RECV0_NOWAIT;
	  else
	    et = ET_RECV0_WAIT;

	  printf( "%d %d %.6f %d %d %d %d\n",
	          EH_MACRO(RT_ENTRY, et, 1),2,
		  e->msgtag
		);

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		  EH_MACRO(RT_EXIT, et, 3),2,
		  e->bytes,
		  e->msgtag,
		  Task2Node(e->tid)
		);
	  break;
	}

      case event_nrecv:
	{
	  TapeNrecvEvent *e = (TapeNrecvEvent *) evt;

	    /* check if msg arrived */
	  if( e->ret > 0) {
	    printf( "%d %d %.6f %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_RECV0_NOWAIT, 1),2,
		   e->msgtag
		   );

	    DatePlusDelta;

	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_RECV0_NOWAIT, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node(e->tid)
		   );
	  }

          else /* message not found */

            tape_generate_overhead_state ( 
		Task2Node(e->header.task),
		tape_get_s(e->header.date_s,e->header.date_us),
		tape_get_s(e->delta_s,e->delta_us)
	    );

	  break;
	}

      case event_trecv:
	{
	  TapeTrecvEvent *e = (TapeTrecvEvent *) evt;

	    /* check if msg found */
	  if( e->ret > 0) {
	    printf( "%d %d %.6f %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_RECV0_WAIT, 1),2,
		   e->msgtag
		   );

	    DatePlusDelta;

	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_RECV0_WAIT, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node(e->tid)
		   );
	  }

          else if( e->ret < 0 ) /* error (=>ovhd) */

            tape_generate_overhead_state ( 
		Task2Node(e->header.task),
		tape_get_s(e->header.date_s,e->header.date_us),
		tape_get_s(e->delta_s,e->delta_us)
	    );

	  else /* time-out (=>idle) */

            tape_generate_idle_state ( 
		Task2Node(e->header.task),
		tape_get_s(e->header.date_s,e->header.date_us),
		tape_get_s(e->delta_s,e->delta_us)
	    );

	  break;
	}

      case event_mcast:
	{

	  /* MCAST is converted into a series of send events. One
	     of these send events takes account of the overhead
	     inferred by the MCAST call. The other send are null
	     overhead. */

	  TapeMcastEvent *e = (TapeMcastEvent *) evt;
	  int task_count = 0;
	  int k;

	  tl = e->TaskList;

	  while( tl ) /* send entry events */ {
	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( tape_get_next_task( &tl ) )
		   );
	    task_count++;
	  }

	  for( k=0; k < task_count-1; k++ ) /* send exit events */ {
	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );
	  }

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		 EH_MACRO(RT_EXIT, ET_SEND0, 0)
		 );

	  break;
	}

      case event_bcast:
	{

	  /* BCAST same as MCAST except that we get the tasklist
	     by a call to the 'tape_get_tasks_in_group' function */

	  TapeBcastEvent *e = (TapeBcastEvent *) evt;
	  TapeTaskList tl;
	  int task_count = 0;
	  int k;

	  tl = tape_get_tasks_in_group( &t->groups, e->group );

	  while( tl ) /* send entry events */ {
	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( tape_get_next_task( &tl ) )
		   );
	    task_count++;
	  }

	  for( k=0; k < task_count-1; k++ ) /* send exit events */ {
	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );
	  }

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		 EH_MACRO(RT_EXIT, ET_SEND0, 0)
		 );

	  break;
	}


	/* events_gather and event_reduce have the same structure */

      case event_gather:
      case event_reduce:
	{
	  TapeGatherEvent *e = (TapeGatherEvent *) evt;
	  TapeTaskList tl;
	  char first=1;
          int task;

	  if(e->ret<0)  /* do nothing in case of error */
	    break;

	  if(e->header.task==e->root) {

	    /* I am the root of the gather (reduce) operation: I do g-1 blocking
               receives from all the members in the group (g is the size
               of the group). Only the first receive takes account of
               the blocking delay. */

	    /* generate a receive start for the first receive */
	    printf( "%d %d %.6f %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_RECV0_WAIT, 1),2,
		   e->msgtag
		   );

	    DatePlusDelta;

	    tl = tape_get_tasks_in_group( &t->groups, e->group );

	    while( tl ) {
	      if( (task=tape_get_next_task(&tl))!=e->root ) {
		if(!first) {
		  /* generate a recv start for all but the first recv */
		  printf( "%d %d %.6f %d %d %d %d\n",
			 EH_MACRO(RT_ENTRY, ET_RECV0_WAIT, 1),2,
			 e->msgtag
			 );
		} else
		  first = 0;
		
		/* generate receive end */
		printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		       EH_MACRO(RT_EXIT, ET_RECV0_WAIT, 3),2,
		       e->bytes,
		       e->msgtag,
		       Task2Node( task )
		       );
	      }
	    }
	    break;

	  } else {

	    /* I am NOT the root of the scatter operation. I simply do a send
               to the root */

	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( e->root )
		   );

	    DatePlusDelta;

	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );
	    break;
	  }
	}	    

	/* event_scatter has the same structure as event_gather, but does
           sends where event_gather does receives and vice-versa
         */

      case event_scatter:
	{
	  TapeScatterEvent *e = (TapeScatterEvent *) evt;
	  TapeTaskList tl;
	  int task;

	  if(e->ret<0)  /* do nothing in case of error */
	    break;

	  if(e->header.task==e->root) {

	    /* I am the root of the scatter operation: I do g-1 sends
               to all the members in the group (g is the size
               of the group). Only the first send takes account of
               the sending overhead */

	    /* generate a send start for the first task */

	    tl = tape_get_tasks_in_group( &t->groups, e->group );	    
	    task=tape_get_next_task( &tl );
	    if(task==e->root)
	      task=tape_get_next_task( &tl );  /* task = first non root task */
	    
	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( task )
		   );

	    DatePlusDelta;

	    /* generate a send end for the first task */

	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );


	    /* for all non-root tasks generate send starts and ends */
            /* these are null overhead */

	    while( tl ) {
	      if( (task=tape_get_next_task(&tl))!=e->root ) {

		/* send start */
		printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		       EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		       e->bytes,
		       e->msgtag,
		       Task2Node( task )
		       );

		/* send end */
		printf( "%d %d %.6f %d %d %d\n",
		       EH_MACRO(RT_EXIT, ET_SEND0, 0)
		       );
	      }
	    }
	    break;

	  } else {

	    /* I am NOT the root of the scatter operation. I simply do a recv
               from the root */

	    printf( "%d %d %.6f %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_RECV0_WAIT, 1),2,
		   e->msgtag
		   );

	    DatePlusDelta;

	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_RECV0_WAIT, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( e->root )
		   );
	    break;
	  }
	}	    
	    
      case event_barrier:
	{
	  TapeBarrierEvent *e = (TapeBarrierEvent *) evt;

	  tape_generate_idle_state( Task2Node(e->header.task),
				    tape_get_s(e->header.date_s,
					       e->header.date_us),
				    tape_get_s(e->delta_s,
					       e->delta_us)
				   );
	  break;
	}

      case event_trace:
	{
	  TapeTraceEvent *e = (TapeTraceEvent *) evt;

	  tape_generate_overhead_state( Task2Node(e->header.task),
				        tape_get_s(e->header.date_s,
						   e->header.date_us),
				        tape_get_s(e->header.file,
						   e->header.line)
				       );
	  count_trace++;
	  break;
	}

      case  event_pkbyte:
      case  event_pkcplx:
      case  event_pkdcplx:
      case  event_pkdouble:
      case  event_pkfloat:
      case  event_pkint:
      case  event_pklong:
      case  event_pkshort:
      case  event_pkuint:
      case  event_pkulong:
      case  event_pkushort:
      case  event_pkstr:
      case  event_upkbyte:
      case  event_upkcplx:
      case  event_upkdcplx:
      case  event_upkdouble:
      case  event_upkfloat:	
      case  event_upkint:
      case  event_upklong:
      case  event_upkshort:
      case  event_upkuint:
      case  event_upkulong:
      case  event_upkushort:
      case  event_upkstr:
	{
	  /* (U)PK events generate overhead states. Because only the
	     delta fields are needed, all the data structures are mapped
	     onto the Pk1Event */

	  TapePk1Event *e = (TapePk1Event *) evt;
	  tape_generate_overhead_state( Task2Node(e->header.task),
				        tape_get_s(e->header.date_s,
						   e->header.date_us),
				        tape_get_s(e->delta_s,
						   e->delta_us)
				       );
	  break;
	}

      }
  }

  tape_dispose_event( &evt );

  TapeVersionOK = t->version;

  tape_close_trace( t );

  if(f!=NULL) fclose(f);

  /* do some (limited) coherency check */

  if( count_mytid != Task2Node(NBNODES) )
    fprintf(stderr,"tape2npicl: [info] some of your tasks have no 'pvm_mytid' event.\n");

  if( count_trace )
    fprintf(stderr,"tape2npicl: [info] found %d buffer flushes (%.1f per node).\n",
	    count_trace, count_trace/(float)Task2Node(-1) );

  if( ! TapeVersionOK )
    fprintf(stderr,"tape2npicl: [warning] version mismatch between your trace and tape_reader.\n");

  if( flag_kill )
    fprintf(stderr,"tape2npicl: [warning] uncomplete trace, detected one or more 'pvm_kill' events.\n");

  if( ! flag_spawn )
    fprintf(stderr,"tape2npicl: [warning] found no 'pvm_spawn' events in your trace.\n");
  
  if( count_exit != Task2Node(NBNODES) )
    fprintf(stderr,"tape2npicl: [warning] some of your tasks have no 'pvm_exit' event.\n");

  if( f == NULL )
    fprintf(stderr,"tape2npicl: [warning] cannot open %s, no node table generated.\n",NODETABLEOUT);
}
