/* TAPE/PVM %W% %G% */

#include <sys/time.h>

#include "pvm3.h"

#include "tape.h"
#include "tape_globals.h"
#include "tape_events.h"
#include "tape_classes.h"

/* Module with TAPE/PVM utility functions. These functions
 * have no PVM equivalent. When using these functions in
 * your PVM programs note the following points:
 *
 * - place them between _TPVM3_H_ ifdefs so that they are not
 *   taken into account when compiling the non-instrumented
 *   source
 * - when generating an instrumented source, tapepp automatically
 *   inserts the numfile, numline and masc arguments
 */

/* selection mask handling */

int tape_set_mask(int numfile, int numline, int masc, char *mask)
{
struct timeval tp;
int ret;

tape_clock(&tp);

ret = tape_sm( mask );

tape_event(event_set_mask,taskid,numfile,numline,tp.tv_sec-base_s,tp.tv_usec,
	   "%d %s",ret,mask);
return ret;
}

/* sets internal binary mask from textual mask given by the caller */
/* returns -1 if error in mask => mselect not changed              */
/* returns  1 if mask correct  => mselect updated                  */

int tape_sm( mask )
char *mask;
{
  int mtemp=0;
  int i=0;
  int rl;

   /* check module group letters */

  while(isupper(mask[i])) {
    rl=mask[i]-'A'+1;
    mtemp |= 1 << rl+nb_classe-1;
    i++;
  }

  if(!i)
    mtemp = all_modules;

  mtemp |= all_types;

  /* check event type letters */

  while(islower(mask[i])) {
    switch(mask[i]) {
    /* if bit is set we unset bit,
       if it is already unset, a letter occurred twice in the
       mask => error
     */
    case EC_INFO:
      if(mtemp & classe_info) {
	mtemp ^= classe_info;
	break; }
      else
	return -1;
    case EC_CONTROL:
      if(mtemp & classe_control) {
	mtemp ^= classe_control;
	break; }
      else
	return -1;
    case EC_MSG:
      if(mtemp & classe_msg) {
	mtemp ^= classe_msg;
	break; }
      else
	return -1;
    case EC_BUFFER:
      if(mtemp & classe_buffer) {
	mtemp ^= classe_buffer;
	break; }
      else
	return -1;
    case EC_SYST:
      if(mtemp & classe_syst) {
	mtemp ^= classe_syst;
	break; }
      else
	return -1;
    case EC_GROUP:
      if(mtemp & classe_group) {
	mtemp ^= classe_group;
	break; }
      else
	return -1;
    default:
      return -1;
    }
    i++;
  }

  if(i==strlen(mask)) {
    /* the whole mask was correct */
    mselect = mtemp;
    return 1;
  }
  else
    /* part of the mask was erroneous */
    return -1;  
}

/* return the textual selction mask in the caller provided string */

void tape_get_mask( str )
char *str;
{
  int k=0;
  char i;

  /* build module group letters */
  if(!((mselect & all_modules) == all_modules))
    for(i=0;i<32-nb_classe;i++)
      if( mselect & (1<<i+nb_classe) ) {
	str[k]='A'+i;
	k++;
      }

  /* build event type letters */
  if(! (mselect&classe_control)) {
    str[k]=EC_CONTROL;
    k++;
  }
  if(! (mselect&classe_msg)) {
    str[k]=EC_MSG;
    k++;
  }
  if(! (mselect&classe_info)) {
    str[k]=EC_INFO;
    k++;
  }
  if(! (mselect&classe_buffer)) {
    str[k]=EC_BUFFER;
    k++;
  }
  if(! (mselect&classe_syst)) {
    str[k]=EC_SYST;
    k++;
  }
  if(! (mselect&classe_group)) {
    str[k]=EC_GROUP;
    k++;
  }

  /* terminate string */
  str[k]=0;
}

/* phase handling */

int tape_close_phase(int numfile, int numline, int masc, int pid)
{
int ret;
char *name;
struct timeval tp;

ret=-1;
if( pid>=0 && pid<max_phase && ((name=table_phase[pid])!=NULL) )
   {
   ret=0;
   tape_clock(&tp);
   tape_event(event_close_phase,taskid,numfile,numline,tp.tv_sec-base_s,tp.tv_usec,
	      "%d %s",ret,name);
   table_phase[pid]=NULL;
   free(name);
   phase_index=pid;
   }
return ret;
}

int tape_open_phase(int numfile, int numline, int masc, char *name)
{
int ret;
struct timeval tp;

ret=-1;
if ( phase_index < max_phase && (name!=NULL) )
   {
   ret=phase_index;
   tape_clock(&tp);
   tape_event(event_open_phase,taskid,numfile,numline,tp.tv_sec-base_s,tp.tv_usec,
	      "%d %s",ret, name );
   if(!(table_phase[phase_index]=(char*)malloc(strlen(name)+1)))
     fatalMallocFailure(__FILE__,__LINE__);
   strcpy(table_phase[phase_index],name);
   phase_index=0;
   while( phase_index<max_phase && (table_phase[phase_index]!=NULL) )
     phase_index++;
 }
return ret;
}

