// This may look like C code, but it is really -*- C++ -*-
#include <vector.h>
#include <stream.h>
#include <values.h>

// error handling
void 
default_vector_T_error_handler(const char* msg) {
  cerr <<"\n\aVector: " <<msg <<endl;
  exit(1);
}

one_arg_error_handler_t vector_error_handler = default_vector_T_error_handler;

one_arg_error_handler_t 
set_vector_T_error_handler(one_arg_error_handler_t f) {
  one_arg_error_handler_t old = vector_error_handler;
  vector_error_handler = f;
  return old;
}

template<class T> ostream & 
operator << (ostream & s, vector<T> &v) {
  for(int i = 0; i < v.length(); i++)
    s <<v[i] <<" ";
  return s;
}

template<class T> void 
vector<T>::error(const char *msg){
  (*vector_error_handler)(msg);
}

template<class T> void
vector<T>::range_error() {
  (*vector_error_handler)("Index out of range.");
}

//constructors
template<class T>
vector<T>::vector(int l, T *v): len(l) {
  s = new T[l];
  T* top = &(s[len]);
  T* t = s;
  T* u = v;
  while(t < top) *t++ = *u++;
}

template<class T>
vector<T>::vector(int l): len(l) { 
  s = new T[l]; 
}

template<class T>
vector<T>::vector(vector<T> &v){
//copy constructor
  s = new T[len = v.len];
  T* top = &(s[len]);
  T* t = s;
  T* u = v.s;
  while(t < top) *t++ = *u++;
}

template<class T>
vector<T>::vector(int l, T fill_value) {
  s = new T[len = l];
  T *top = &(s[len]);
  T *t = s;
  while(t < top) *t++ = fill_value;
}

//destructor
template<class T>
vector<T>::~vector() {
  delete s;
}

template<class T> vector<T>&
vector<T>::operator +(vector<T> &right) {
  if(len != right.len)
    range_error();
    
  static vector<T> out;
  out.resize(len);

  T *top = &(s[len]);
  T *s_this = s;
  T *s_out = out.s;
  T *s_right = right.s;
  while(s_this < top) *s_out++ = *s_this++ + *s_right++; 
  return (vector<T> &)out;
}

template<class T> vector<T>&
vector<T>::operator +=(vector &right) {
  if(len != right.len)
    range_error();

  return (*this = *this + right);
}

template<class T> vector<T>&
vector<T>::operator -(vector<T> &right) {
  if(len != right.len)
    range_error();
    
  static vector<T> out;
  out.resize(len);

  T *top = &(s[len]);
  T *s_this = s;
  T *s_out = out.s;
  T *s_right = right.s;
  while(s_this < top) *s_out++ = *s_this++ - *s_right++; 
  return (vector<T> &)out;
}

template<class T> vector<T>&
vector<T>::operator -=(vector &right) {
  if(len != right.len)
    range_error();

  return (*this = *this - right);
}

template<class T> T
vector<T>::operator *(vector &right) {
  if(len != right.len)
    range_error();
  
  T out = 0;
  T *top = &(s[len]);
  T *s_this = s;
  T *s_right = right.s;
  while(s_this < top) out += *s_this++ * *s_right++;
  return out;
}

template<class T> vector<T>&
vector<T>::operator *(T right) {
  T *top = &(s[len]);
  T *s_this = s;
  while(s_this < top) *s_this++ *= right;
  return *this;
}

template<class T> vector<T>&
vector<T>::operator /(T right) {
  T *top = &(s[len]);
  T *s_this = s;
  while(s_this < top) *s_this++ /= right;
  return *this;
}

template<class T> int 
vector<T>::operator ==(vector<T> &b) {
  if(len != b.len)
    return 0;
  T *top = &(s[len]);
  T *t = s;
  T *u = b.s;
  while(t < top) if (*t++ != *u++) return 0;
  return 1;
}

template<class T> vector<T>& 
vector<T>::operator =(vector<T> &v) {
  if(this != &v) {
    delete []s;
    s = new T[len = v.len];
    T* top = &(s[len]);
    T* t = s;
    T* u = v.s;
    while(t < top) *t++ = *u++;
  }
  return *this;
}

template<class T> T& 
vector<T>::operator[](int n) {
  if(n >= len)
    range_error();
  return s[n];
}

template<class T> void 
vector<T>::apply(void (*f)(T&)) {
  T* top = &(s[len]);
  T* t = s;
  while(t < top) (*f)(*t++);
}

template<class T> T
vector<T>::min() {
  T min = s[0];
  T *top = &(s[len]);
  for(T *t = &(s[1]); t < top; t++)
    if(*t <= min) min = *t;
  return min;
}

template<class T> T
vector<T>::max() {
  T max = s[0];
  T *top = &(s[len]);
  for(T *t = &(s[1]); t < top; t++)
    if(*t >= max) max = *t;
  return max;
}

template<class T> T
vector<T>::sum(void) {
  T add = 0;
  T* top = &(s[len]);
  T* t = s;
  while(t < top) add += *t++;
  return add;
}

template<class T> void 
vector<T>::resize(int newl, T null) {
  s = (T*)realloc((void *)s, (size_t)newl * sizeof(T));
  if(newl > len) {  //zero-padding
    T* p = &(s[len]);
    T* top = &(s[newl]);
    while(p < top) *p++ = null;
  }
  len = newl;
}

template<class T> vector<T> 
concat(vector<T> &a, vector<T> &b) {
  int newl = a.len + b.len;
  T* news = new T[newl];
  T* p = news;
  T* top = &(a.s[a.len]);
  T* t = a.s;
  while(t < top) *p++ = *t++;
  top = &(b.s[b.len]);
  t = b.s;
  while(t < top) *p++ = *t++;
  return vector<T>(newl, news);
}

template<class T> vector<T> 
combine(T (*f)(T&, T&), vector<T> &a, vector<T> &b) {
  int newl = (a.len < b.len)? a.len : b.len;
  T* news = new T[newl];
  T* p = news;
  T* top = &(a.s[newl]);
  T* t = a.s;
  T* u = b.s;
  while(t < top) *p++ = (*f)(*t++, *u++);
  return vector<T>(newl, news);
}

template<class T> T 
reduce(T (*f)(T&, T&), T &base) { 
  T r = base;
  T* top = &(s[len]);
  T* t = s;
  while(t < top) r = (*f)(r, *t++);
  return r;
}

template<class T> void 
vector<T>::reverse() {
  if(len != 0) {
    T* lo = s;
    T* hi = &(s[len - 1]);
    while(lo < hi) {
      T tmp = *lo;
      *lo++ = *hi;
      *hi-- = tmp;
    }
  }
}

template<class T> int
vector<T>::index(T targ) {
  for(int i = 0; i < len; ++i) 
    if(targ == s[i]) 
      return i;

  return ~0;
}

template<class T> vector<T>& 
vector<T>::vindex(T targ) {
  vector<T> &v_ind = *new vector<T>(len);
  int count = 0;

  for(int i = 0; i < len; ++i) 
    if(targ == s[i]) 
      v_ind.fill(i, count++, 1);

  v_ind.resize(count);
  return v_ind;
}

template<class T> vector<T> 
map(T (*f)(T&), vector<T> &a) { //MAPPER
  T* news = new T[a.len];
  T* p = news;
  T* top = &(a.s[a.len]);
  T* t = a.s;
  while(t < top) *p++ = (*f)(*t++);
  return vector<T>(a.len, news);
}

template<class T> void 
vector<T>::fill(T val, int from, int n){
  int to;
  if(n < 0)
    to = len - 1;
  else
    to = from + n - 1;
  if((unsigned)from > (unsigned)to || from > len)
    range_error();
  T* t = &(s[from]);
  T* top = &(s[to]);
  while (t <= top) *t++ = val;
}

template<class T> vector<T> 
vector<T>::at(int from, int n) {
  int to;
  if(n < 0) {
    n = len - from;
    to = len - 1;
  }
  else
    to = from + n - 1;
  if((unsigned)from > (unsigned)to)
    range_error();
  T* news = new T[n];
  T* p = news;
  T* t = &(s[from]);
  T* top = &(s[to]);
  while(t <= top) *p++ = *t++;
  return vector<T>(n, news);
}
template<class T> vector<T> 
merge(vector<T> &a, vector<T> &b, int (*f)(T&, T&)) {  //COMPARATOR
  int newl = a.len + b.len;
  T* news = new T[newl];
  T* p = news;
  T* topa = &(a.s[a.len]);
  T* as = a.s;
  T* topb = &(b.s[b.len]);
  T* bs = b.s;

  for(;;) {
    if (as >= topa) {
      while(bs < topb) *p++ = *bs++;
      break;
    }
    else if (bs >= topb) {
      while (as < topa) *p++ = *as++;
      break;
    }
    else if((*f)(*as, *bs) <= 0)
      *p++ = *as++;
    else
      *p++ = *bs++;
  }
  return vector<T>(newl, news);
}

template<class T> void 
vector<T>::sort(int (*cmp)(T&, T&)) {
  gsort(s, len, cmp);
}

// An adaptation of Schmidt's new quicksort
template<class T> static inline void 
SWAP(T *A, T *B) {
  T tmp = *A; *A = *B; *B = tmp;
}

/* This should be replaced by a standard ANSI macro. */
#define BYTES_PER_WORD 8
#define BYTES_PER_LONG 4

/* The next 4 #defines implement a very fast in-line stack abstraction. */

#define STACK_SIZE (BYTES_PER_WORD * BYTES_PER_LONG)
#define PUSH(LOW,HIGH) do {top->lo = LOW;top++->hi = HIGH;} while (0)
#define POP(LOW,HIGH)  do {LOW = (--top)->lo;HIGH = top->hi;} while (0)
#define STACK_NOT_EMPTY (stack < top)                

/* Discontinue quicksort algorithm when partition gets below this size.
   This particular magic number was chosen to work best on a Sun 4/260. */
#define MAX_THRESH 4


/* Order size using quicksort.  This implementation incorporates
   four optimizations discussed in Sedgewick:
   
   1. Non-recursive, using an explicit stack of pointer that
      store the next array partition to sort.  To save time, this
      maximum amount of space required to store an array of
      MAX_INT is allocated on the stack.  Assuming a 32-bit integer,
      this needs only 32 * sizeof (stack_node) == 136 bits.  Pretty
      cheap, actually.

   2. Chose the pivot element using a median-of-three decision tree.
      This reduces the probability of selecting a bad pivot value and 
      eliminates certain extraneous comparisons.

   3. Only quicksorts TOTAL_ELEMS / MAX_THRESH partitions, leaving
      insertion sort to order the MAX_THRESH items within each partition.  
      This is a big win, since insertion sort is faster for small, mostly
      sorted array segements.
   
   4. The larger of the two sub-partitions is always pushed onto the
      stack first, with the algorithm then concentrating on the
      smaller partition.  This *guarantees* no more than log (n)
      stack size is needed! */

template<class T> int 
vector<T>::gsort(T *base_ptr, int total_elems, int (*cmp)(T&, T&)) { //COMPARATOR
/* Stack node declarations used to store unfulfilled partition obligations. */
  struct stack_node {  T *lo;  T *hi; };
  T pivot_buffer;
  int max_thresh = MAX_THRESH;

  if(total_elems > MAX_THRESH) {
    T *lo = base_ptr;
    T *hi = lo + (total_elems - 1);
    T *left_ptr;
    T *right_ptr;
    stack_node stack[STACK_SIZE]; /* Largest size needed for 32-bit int!!! */
    stack_node *top = stack + 1;

    while(STACK_NOT_EMPTY) {
      {
	T *pivot = &pivot_buffer;
	{
	  /* Select median value from among LO, MID, and HI. Rearrange
	     LO and HI so the three values are sorted. This lowers the 
	     probability of picking a pathological pivot value and 
	     skips a comparison for both the LEFT_PTR and RIGHT_PTR. */
	  
	  T *mid = lo + ((hi - lo) >> 1);

	  if((*cmp) (*mid, *lo) < 0)
	    SWAP(mid, lo);
	  if((*cmp) (*hi, *mid) < 0) {
	    SWAP(mid, hi);
	    if((*cmp) (*mid, *lo) < 0)
	      SWAP(mid, lo);
	  }
	  *pivot = *mid;
	  pivot = &pivot_buffer;
	}
	left_ptr  = lo + 1;
	right_ptr = hi - 1; 

	/* Here's the famous ``collapse the walls'' section of quicksort.  
	   Gotta like those tight inner loops!  They are the main reason 
	   that this algorithm runs much faster than others. */
	do { 
	  while((*cmp) (*left_ptr, *pivot) < 0)
	    left_ptr += 1;

	  while((*cmp) (*pivot, *right_ptr) < 0)
	    right_ptr -= 1;

	  if(left_ptr < right_ptr) {
	    SWAP(left_ptr, right_ptr);
	    left_ptr += 1;
	    right_ptr -= 1;
	  }
	  else if(left_ptr == right_ptr) {
	    left_ptr += 1;
	    right_ptr -= 1;
	    break;
	  }
	} 
	while(left_ptr <= right_ptr);
      }

      /* Set up pointers for next iteration.  First determine whether
	 left and right partitions are below the threshold size. If so, 
	 ignore one or both.  Otherwise, push the larger partition's
	 bounds on the stack and continue sorting the smaller one. */

      if((right_ptr - lo) <= max_thresh) {
	if((hi - left_ptr) <= max_thresh) /* Ignore both small partitions. */
	  POP(lo, hi); 
	else              /* Ignore small left partition. */  
	  lo = left_ptr;
      }
      else if ((hi - left_ptr) <= max_thresh) /* Ignore small right partition. */
	hi = right_ptr;
      else if ((right_ptr - lo) > (hi - left_ptr)) /* Push larger left partition indices. */
	{                   
	  PUSH (lo, right_ptr);
	  lo = left_ptr;
	}
      else                  /* Push larger right partition indices. */
	{                   
	  PUSH (left_ptr, hi);
	  hi = right_ptr;
	}
    }
  }

  /* Once the BASE_PTR array is partially sorted by quicksort the rest
     is completely sorted using insertion sort, since this is efficient 
     for partitions below MAX_THRESH size. BASE_PTR points to the beginning 
     of the array to sort, and END_PTR points at the very last element in
     the array (*not* one beyond it!). */

  {
    T *end_ptr = base_ptr + 1 * (total_elems - 1);
    T *run_ptr;
    T *tmp_ptr = base_ptr;
    T *thresh  = (end_ptr < (base_ptr + max_thresh))? 
      end_ptr : (base_ptr + max_thresh);

    /* Find smallest element in first threshold and place it at the
       array's beginning.  This is the smallest array element,
       and the operation speeds up insertion sort's inner loop. */

    for (run_ptr = tmp_ptr + 1; run_ptr <= thresh; run_ptr += 1)
      if ((*cmp) (*run_ptr, *tmp_ptr) < 0)
        tmp_ptr = run_ptr;

    if (tmp_ptr != base_ptr)
      SWAP (tmp_ptr, base_ptr);

    /* Insertion sort, running from left-hand-side up to `right-hand-side.' 
       Pretty much straight out of the original GNU qsort routine. */

    for (run_ptr = base_ptr + 1; (tmp_ptr = run_ptr += 1) <= end_ptr; )
      {

        while ((*cmp) (*run_ptr, *(tmp_ptr -= 1)) < 0)
          ;

        if ((tmp_ptr += 1) != run_ptr)
          {
            T *trav;

            for (trav = run_ptr + 1; --trav >= run_ptr;)
              {
                T c = *trav;
                T *hi, *lo;

                for (hi = lo = trav; (lo -= 1) >= tmp_ptr; hi = lo)
                  *hi = *lo;
                *hi = c;
              }
          }
      }
  }
  return 1;
}

template<class T> ofstream &
vector<T>::OpenVectorFile(String file, String comment) {
  static ofstream *OutFile = NULL;
  OutFile = new ofstream((char *)file);
  if(!OutFile)
    error("\n\aVector: Unable to write file!");

  int NbCommentLines = comment.freq('\n') + 1;
  String CutComments[NbCommentLines];
  NbCommentLines = split(comment, CutComments, NbCommentLines, String("\n"));  
  for(int i = 0; i < NbCommentLines; i++)
    *OutFile <<CutComments[i] <<endl;

  //write length
  *OutFile <<"% " <<len <<endl;
  return (ofstream &)*OutFile;
}

template<class T> void
vector<T>::ASCII_File(String file, String comment) {
  ofstream OutFile = OpenVectorFile(file, comment);
  //write data
  for(T *t = s; t < &(s[len]);)
    OutFile <<*t++ <<endl;
}

template<class T>
vector<T>::vector(String FileName) {
  char err_out[256];
  ifstream InFile(FileName);
  if(!InFile)
    error("Unable to read file!");

  //pass over comment if any, read length, but may not exist!
  const int MaxHeaderLength = 1024;
  char ReadString[MaxHeaderLength];
  InFile.getline(ReadString, MaxHeaderLength, '%');

  if(strlen(ReadString) == MaxHeaderLength - 1 || InFile.eof()) {
    sprintf(err_out, "Can't find pattern '%% size' in file %s", (char *)FileName);
    error(err_out);
  }
  else
    InFile >>len;

  //read data
  s = new T[len];
  T* top = &(s[len]);
  T* t = s;
  
  while(t < top && !InFile.eof()) 
    InFile >>*t++;
}
