#include <iostream.h>
#include <pvm++.h>

//static members
int PVM_Task::signal_exist = false;
int PVM_Task::recv_from_tid[NSIG];
int PVM_Task::msgtag[NSIG];
PVM_SignalHandler_t PVM_Task::hdlr[NSIG];
PDU* PVM_Task::pdu[NSIG];

void
PVM_Task::Update_Children() {
  int nb_process;
  struct pvmtaskinfo *taskp;
  pvm_tasks(0, &nb_process, &taskp);
  ctid.resize(nb_process);

  for(int i = 0, nb_children = 0; i < nb_process; i++) 
    if(taskp[i].ti_ptid == tid)
      ctid.fill(taskp[i].ti_tid, nb_children++, 1);

  ctid.resize(nb_children);
}

vector<int> & 
PVM_Task::children() {
  vector<int> &v_out = ctid;
  return v_out;
}

void
PVM_Task::KillAllTasks(bool flag) {
  if(flag)
    cout <<"\n\aSending KILL signal to all my children " <<flush;

  Update_Children();
  for(int i = 0; i < nb_children(); i++) {
    pvm_kill(ctid[i]);
    if(flag)
      cout <<"." <<flush;
  }
  if(flag)
    cout <<"OK" <<endl;

  ctid.resize(0);
}

void
PVM_Task::KillTasks(vector<int> &tids) {
  Update_Children();
  for(int i = 0; i < tids.length(); i++ )
    if(ctid.index(tids[i]) != -1)  //verify that this tid is one of my children
      pvm_kill(tids[i]);

  Update_Children();
}

void 
PVM_Task::error(const char *message) {
  KillAllTasks(false);
  pvm_perror((char *)message);
  pvm_exit();
  exit(1);
}

PVM_Task::PVM_Task(): tid(pvm_mytid()), ptid(pvm_parent()), \
                      ctid(*new vector<int>) {
  if(tid < 0)
    exit(1);

  if(!PVM_Task::signal_exist)  //first object PVM_Task of the process ?
    for(unsigned i = 0; i < NSIG; i++)
      PVM_Task::hdlr[i] = NULL;
}

PVM_Task::~PVM_Task() {
  KillAllTasks(false);
  delete &ctid;
  pvm_exit();
}

void
PVM_Task::context() {
  int nb_host, nb_arch;
  struct pvmhostinfo *host;
  pvm_config(&nb_host, &nb_arch, &host);
  
  cout <<"\n   HOST    PVMD_TID    ARCH    SPEED" <<endl;
  for(int i = 0; i < nb_host; i++) {
    cout <<i+1 <<"- " <<host[i].hi_name <<"     "  <<host[i].hi_tid;
    cout <<"     " <<host[i].hi_arch <<"     " <<host[i].hi_speed <<endl;
  }
}

int
PVM_Task::spawn(char *task, char **argv, int flag, char *where,\
		vector<int> &v_tids) {
  int nb_task = v_tids.length();
  if(nb_task < 1)
    error("\aAttempting to spawn less than ONE thread!");

  int tids[nb_task];
  int out = pvm_spawn(task, argv, flag, where, nb_task, tids);
  v_tids = vector<int>(nb_task, tids);

  //verify if all threads have been well spawned
  for(int i = 0; i < nb_task; i++)
    if(tids[i] < 0) 
      error("\aSpawn Failed!");

  Update_Children();
  return out;
}

int 
PVM_Task::mkbuf(int encoding, PDU &pdu) {
  int new_buff = pvm_mkbuf(encoding);
  if(new_buff < 0)
    error("\aMkbuf failed!");

  pvm_setsbuf(new_buff);
  pdu.pack();  //pack the derived class of PDU (pure virtual function)
  return new_buff;
}

int 
PVM_Task::mcast(int encoding, vector<int> &v_tids, int msgtag, PDU &pdu) {
  int old_buff = pvm_getsbuf();
  int new_buff = mkbuf(encoding, pdu);

  int out = pvm_mcast((int *)v_tids, v_tids.length(), msgtag);
  if(new_buff < 0 || out < 0)
    error("\aMcast failed!");

  pvm_freebuf(new_buff);
  pvm_setsbuf(old_buff);  //restore old receive buffer
  return out;
}

int
PVM_Task::bcast(int encoding, char *group, int msgtag, PDU &pdu) {
  int old_buff = pvm_getsbuf();
  int new_buff = mkbuf(encoding, pdu);

  int out = pvm_bcast(group, msgtag);
  if(new_buff < 0 || out < 0)
    error("\aBcast failed!");
  
  pvm_freebuf(new_buff);
  pvm_setsbuf(old_buff);  //restore old receive buffer
  return out;
}

int
PVM_Task::send(int encoding, int tid, int msgtag, PDU &pdu) {
  int old_buff = pvm_getsbuf();
  int new_buff = mkbuf(encoding, pdu);

  int out = pvm_send(tid, msgtag);
  if(new_buff < 0 || out < 0)
    error("\aSend failed!");

  pvm_freebuf(new_buff);
  pvm_setsbuf(old_buff);  //restore old receive buffer
  return out;
}

int 
PVM_Task::recv(int tid, int msgtag, PDU &pdu) {
  int out = pvm_recv(tid, msgtag);
  if(out < 0)
    error("\aRecv failed!");
  pdu.unpack();
  return out;
}

void
PVM_Task::PVM_handler(int signal) {
  if(signal < 1 || signal > NSIG) 
    return;

  if(hdlr[signal - 1] == NULL) {
    cerr <<"\aWarning: Signal #" <<signal <<" not registered by\n"
	   <<"         async_recv function!" <<endl;
    return;
  }

  /* block the signal as this procedure can be quite 
     long, depending on the user call */
  int old_mask = sigblock(sigmask(signal));
  switch(signal) {

  case SIGALRM:
    //PVM_Task::hdlr[signal - 1](*(PVM_Task::pdu[signal - 1]));
    PDU_FOO foo;
    PVM_Task::hdlr[signal - 1](foo);
      break;

  default:
      int from_tid = PVM_Task::recv_from_tid[signal - 1];
      int mask = PVM_Task::msgtag[signal - 1];
      while(pvm_nrecv(from_tid, mask) > 0) {  
	PVM_Task::pdu[signal - 1]->unpack();  //virtual !!
      
	//call the predefined user signal_handler with the unpacked pdu
	PVM_Task::hdlr[signal - 1](*(PVM_Task::pdu[signal - 1]));
      }
      break;
    }

  //reset the signal handler
  setsignal(signal);
  sigsetmask(old_mask);
  return;
}

void
PVM_Task::setsignal(int sig) {
  if(signal(sig, (void(*)(int))(PVM_Task::PVM_handler)) == SIG_IGN)
    signal(sig, SIG_IGN);
}

void 
PVM_Task::TimeOut(long seconds, void(*handler)(void)) {
//register the signal
  if(hdlr == NULL)
    error("\aAlarm failed: NULL handler defined!");
  
  PVM_Task::signal_exist = true;
  PVM_Task::hdlr[SIGALRM - 1] = (PVM_SignalHandler_t)handler;
  setsignal(SIGALRM);
  alarm(seconds);
}

void
PVM_Task::async_recv(int from_tid, int mask, int signal, \
		     PVM_SignalHandler_t handler, PDU *pdu) {
//register the signal
  if(hdlr == NULL)
    error("\aAsync_recv failed: NULL handler defined!");

  int index_sig = signal - 1;
  PVM_Task::signal_exist = true;
  PVM_Task::recv_from_tid[index_sig] = from_tid;
  PVM_Task::msgtag[index_sig] = mask; 
  PVM_Task::hdlr[index_sig] = handler;
  PVM_Task::pdu[index_sig] = pdu;
  setsignal(signal);
}
