#include <unistd.h>
#include <stdlib.h>
#include <iostream.h>
#include <getopt.h>
#include <pvm++.h>
#include "pdu_random.h"

//GLOBAL
#define SLAVE "slave"
#define NACK 1
#define ACK 0

int nb_slaves = 1;
PVM_Task *master;
vector<int> *acknowlegements;
vector<int> *slaves_tids;

void timeout();

void
usage(char *call) {
  cerr <<"\aUsage: " <<call <<" [-w nb_workers (default: 1)]\n" <<endl;
  exit(1);
}

void //signal handler called by the first signal received
SlaveAnswer(PDU_answer &ans) {
//read all pdu_answer in the buffer
  if(acknowlegements->elem(ans.from) == ACK)
    return;  //PDU already received, abort

  acknowlegements->fill(ACK, ans.from, 1);
  cout <<ans <<"  new acks: " <<(vector<int>&)*acknowlegements <<endl <<endl;

  //kill slave
  pvm_kill(slaves_tids->elem(ans.from));

  if(!acknowlegements->sum()) {
    cout <<"All PDUs have correctly been received!" <<endl;
    delete slaves_tids;
    delete acknowlegements;
    delete master;
    exit(0);
  }
}

main(int argc, char *argv[]) {
  //parse arguments
  int option;
  while((option = getopt(argc, argv, "w:h")) != -1) {
    switch(option) {
    case '?':  
    case ':':
    case 'h':
      usage(argv[0]);

    case 'w':
      nb_slaves = atoi(optarg);
      break;
    }
  }
  if(argc < 1)
    usage(argv[0]);

  //register pvm
  master = new PVM_Task();
  pvm_setopt(PvmRoute, PvmRouteDirect);
  master->context();

  acknowlegements = new vector<int>(nb_slaves, NACK);
  //set the handler before launching slaves
  master->async_recv(-1, RESULT, SIGUSR1, (PVM_SignalHandler_t)SlaveAnswer,\
		     new PDU_answer);

  slaves_tids = new vector<int>(nb_slaves);
  master->spawn(SLAVE, NULL, PvmTaskDefault, NULL, *slaves_tids);

  //ask threads
  float mean_var[] = {0.0, 1.0};
  vector<float> v_mean_var(2, mean_var);
  PDU_request req(*slaves_tids, v_mean_var, (int)1e4);

  cout <<"\nSpawning " <<nb_slaves <<" slave tasks to compute real mean"
       <<"\n  (Asked mean = " <<v_mean_var[0] <<", Asked variance = " 
       <<v_mean_var[1] <<")\n" <<endl;

  //broadcast
  master->bcast(PvmDataDefault, SLAVE_GROUP, COMPUTE, req);

  //set the alarm with respect to the number of unacknowlegemnts
  master->TimeOut(acknowlegements->sum(), timeout);
  while(true);
}

void 
timeout() {
  vector<int> slave_nack = acknowlegements->vindex(NACK);
  cout <<"   --> slave #" <<slave_nack <<" are asked to retransmit" <<endl;

  for(int i = 0; i < slave_nack.length(); i ++) {
    int who_tid = slaves_tids->elem(slave_nack[i]);
    PDU_Retransmit ret(who_tid);
    master->send(PvmDataDefault, who_tid, RETRANSMIT, ret); 
  }
  //reset timeout ONE second
  master->TimeOut(1, timeout);
}

/*
  //synchronous receives
  PDU_answer ans;
  for(int i = 0; i < nb_slaves; i++) {
  master.recv(-1, RESULT, ans);
  cout <<ans;
  }
*/

