/*
 * PGPVM version 1.1: ParaGraph extensions for PVM
 *
 * Emory University, Atlanta GA 
 * Authors: Brad B. Topol, V. S. Sunderam, and Anders Alund
 * (C) 1996 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither Emory University nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 */


#ifdef MAN
NAME
	pgtracelib - Augmented versions of the pvm library to produce tracefiles

SYNOPSIS
	int pg_tids(tids, ntids)
	int*	tids	I:	Tids array of the processes
	int	ntids	I:	Number of tids in array

	SUBROUTINE pgftids(tids, ntids, info)
	INTEGER	tids(ntids)	I:	Tids array of the processes
	INTEGER ntids		I:	Number of tids in array
	INTEGER info		O:	Return code

	All the other routines maintain the pvm_ calling sequence. Those
	available are:

DESCRIPTION
    pgtracelib augments the pvm system by having nodes 
    produce tracefiles for use with ParaGraph.

IMPLEMENTATION
	Straight forward. Processes must register tids-array of those it
	wish to communicate with.

RETURN VALUES
	For pg_tids() 
	PvmOk if OK, < 0 otherwise.

	For pgftids() 
	PvmOk OK, < 0 otherwise.

	The other routines return the same error codes as their
	corresponding pvm_ routines.

SEE ALSO
	PVM 3.3 users manual

#endif
/************************************************************************/
/**/
/* Include files */
/**/
#include <stdio.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <stdlib.h>
#include <errno.h>
#include "pvm3.h"
/************************************************************************/
/**/
/* useful Macros */
/**/

#define T_BYTE (sizeof(char))
#define T_CPLX (2*sizeof(float))
#define T_DCPLX (2*sizeof(double))
#define T_DOUBLE (sizeof(double))
#define T_FLOAT (sizeof(float))
#define T_INT (sizeof(int))
#define T_LONG (sizeof(long))
#define T_SHORT (sizeof(short))
#define TRACEBUFSIZ 1400
#define TRACETAG 0x22cb173b
#define TRACEBARRTAG 0x22cb173c

/************************************************************************/
/**/
/* Static variables */
/**/
static int me=0;		/* My process number			*/
static int mytid=0;		/* My tid				*/
static int *tids=NULL;		/* Tids for all processes to monitor	*/
static int np=0;		/* Number of processes			*/
static FILE *logfile=stderr;	/* Log file name			*/
static struct timeval t0;	/* Reference timestamp			*/
static struct timeval tt;	/* Temp time variable			*/
static struct timeval tt1;	/* Temp time variable			*/
static struct timezone *tz=NULL;/* Temp timezone variable		*/
static struct timeval st0;	/* Time when last initsend was called	*/
static int initialized=0;	/* Set once pg_tids is called		*/
static int barriertotal=0;	/* count early bird barrier msgs*/
static int barrierflag=1;   /* default to using barriers */
static char logname[256];
static char *outfilenm=NULL;     /*optional filename*/
static char *prefixfilenm=NULL;  /*optional prefix filename*/
static int called_exit=0;
/************************************************************************/
static void timestamp(t)
    struct timeval* t;
{
	struct timeval tx;
	double m1, m2;
	gettimeofday(&tx,tz);
	if (tx.tv_usec >= t0.tv_usec) {
	    t->tv_usec = tx.tv_usec-t0.tv_usec;
	    t->tv_sec = tx.tv_sec-t0.tv_sec;
    }
	else {
	   /* borrow a second */
	    t->tv_usec = (tx.tv_usec+1000000)-t0.tv_usec;
	    t->tv_sec = (tx.tv_sec-1)-t0.tv_sec;
	}

}
/************************************************************************/
static int tid2me(tid)
    int tid;
{
	int i;
	int m = -1;           /* Assume host process if not found */
	for (i=0; i<np; i++) {
		if(tids[i]==tid) {
			m=i;
			break;
		}
	}
/**/
	return m;
}
/************************************************************************/
/**********************************************************************/
static int datasize(dtype)
     int dtype;
{

switch(dtype) {
case PVM_STR:
     return(T_BYTE);
case PVM_BYTE:
     return(T_BYTE);
case PVM_USHORT:
case PVM_SHORT:
     return(T_SHORT);
case PVM_UINT:
case PVM_INT:
     return(T_INT);
case PVM_ULONG:
case PVM_LONG:
     return(T_LONG);
case PVM_FLOAT:
     return(T_FLOAT);
case PVM_DOUBLE:
     return(T_DOUBLE);
case PVM_CPLX:
     return(T_CPLX);
case PVM_DCPLX:
     return(T_DCPLX);


}


}
/**********************************************************************/

int pg_mytid()
{
	int t;
/**/
	mytid=pvm_mytid();
/**/
	return mytid;
}
/************************************************************************/
int pg_tids(ttids, tnp)
    int *ttids;
    int tnp;
{
	char hostname[100];
	register int i;
	FILE *fd;

	mytid=pvm_mytid();
	if (mytid==0)
		return (0);
/**/
    tids = (int *) malloc(sizeof(int) * tnp);
	for (i=0;i<tnp;i++)
	     tids[i] = ttids[i];
	np   = tnp;
	me   = tid2me(mytid);
	if (me == -1) {
		printf("Warning: a process called pg_tids and is not in tids array\n");
	    return PvmOk;
    }
	if (!initialized) {
	  gettimeofday(&t0,tz);
	  gethostname(hostname,sizeof(hostname));
	  if (me > 0) {
		 if (prefixfilenm) {
	          sprintf(logname, "/tmp/%s.pgfile.%d.%s.%d",prefixfilenm,getuid(),hostname,getpid());
         }
		 else {
	          sprintf(logname, "/tmp/pgfile.%d.%s.%d",getuid(),hostname,getpid());
		 }
	  }
      else if (me == 0) {
		  if (prefixfilenm) {
	          sprintf(logname, "/tmp/%s.pgfile.%d",prefixfilenm,getuid());
          }
		  else {
	          sprintf(logname, "/tmp/pgfile.%d",getuid());
		  }
      }	
	  fd = fopen (logname,  "w+");
	  if (fd!=0) {
		logfile=fd;
	    setvbuf(logfile, NULL, _IOFBF, 10240);
      }
	  else {
		printf("pg_tids:unable to open logfile\n");
      }	

	  initialized=1;
	  timestamp(&tt);
/**/
/* PG: trace_start */
/**/
	  fprintf (logfile, "1 %d %d %d 0 0 0\n",tt.tv_sec, tt.tv_usec, me);
/**/
	  printf("pg_tids: initialization completed  \n");
	}
	return PvmOk;
}
int pg_chprefix(nme)
char *nme;
{

	if(!initialized) {
	   barrierflag=0;
	   prefixfilenm = malloc(strlen(nme) + 1);
	   strcpy (prefixfilenm,nme);
    }

}

int pg_outfile(nme)
char *nme;
{

	if(initialized && (me == 0)) {
	   outfilenm = malloc(strlen(nme) + 1);
	   strcpy (outfilenm,nme);
    }

}
/************************************************************************/
static int filecp( f1, f2) 
char *f1,*f2;
{
int fd1, fd2,n;
char buf[TRACEBUFSIZ];
		 if ((fd1 = open(f1,O_RDONLY,0)) == -1) {
			  printf("filecp: can't reopen tracefile");
			  return -1;
         }
		 if ((fd2 = open(f2,O_WRONLY | O_CREAT | O_TRUNC,0644)) == -1) {
			  printf("filecp: can't create outfile");
			  close(fd1);
			  return -1;
         }
		 while((n= read(fd1,buf,TRACEBUFSIZ)) >0)
			  if (write(fd2,buf,n) != n) {
			  printf("filecp: write error");
			  close(fd1);
			  close(fd2);
			  return -1;
         }
				  

			  close(fd1);
			  close(fd2);
   return(0);
}
int pg_exit()
{
    int inum,retval;
	int f1,f2,n,i;
	char buf[TRACEBUFSIZ];
	if (called_exit) return pvm_exit();
	called_exit=1;
	if(initialized) {
	  timestamp(&tt);
/**/
/* PG: trace_end */
/**/
	  fprintf (logfile, "19 %d %d %d 0\n", tt.tv_sec, tt.tv_usec, me);
/**/
/*  barrier  */
  if (barrierflag) {
      /* barrier to wait for everyone to enter pg_exit */
		 /* perform a barrier */
		 if (me == 0) {  
		     for (i=1;i<(np-barriertotal);i++) 
				  pvm_recv(-1, TRACEBARRTAG);
			 pvm_initsend(PvmDataDefault);
			 pvm_mcast(&tids[1], np-1, TRACEBARRTAG);
		 }
		 else {
		     pvm_initsend(PvmDataDefault);
		     pvm_send(tids[0], TRACEBARRTAG);
		     pvm_recv(tids[0],TRACEBARRTAG);
         }
		 /* end of barrier */
  }
	  fclose(logfile);

  if (barrierflag) {
	  if (me > 0) {
		 /* If not tids[0], send my tracefile to tids[0]*/
		 if ((f1 = open(logname,O_RDONLY,0)) == -1) {
			  printf("pg_exit: can't reopen tracefile");
         }
		 else {
			  unlink(logname);
			  while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					  pvm_initsend(PvmDataDefault);
					  pvm_pkint(&n, 1, 1);
					  pvm_pkbyte(buf, n, 1);
					  pvm_send(tids[0], TRACETAG);
              } 
			  n= -1;
			  pvm_initsend(PvmDataDefault);
		      pvm_pkint(&n, 1, 1);
			  pvm_send(tids[0], TRACETAG);
			  close(f1);
         }
      }
      else if (me==0) {
		 /* if tids[0],i.e.  me==0, collect others tracefiles and */
		 /* concatenate */
		 if (outfilenm != NULL) {
			 if (filecp(logname,outfilenm) != 0) {
				 printf("pg_exit:filecp failed");
             }
			 else {
			 unlink(logname);
			 strcpy(logname,outfilenm);
			 }
         }
		 if ((f2 = open(logname,O_WRONLY | O_APPEND, 0644)) == -1) {
			  printf("pg_exit: can't reopen MAIN tracefile");
         }
		 else {
		     i=1;
		     while (i< np) {
                 pvm_recv (tids[i], TRACETAG);
		         pvm_upkint(&n, 1, 1);
			     if (n != -1) {
		             pvm_upkbyte(buf, n, 1);
				     if (write(f2, buf, n) != n) {
					     printf("pg_exit: write error \n while writing %d\n",i);
                     }
                 }
			     else {
				     i++;
                 }
            }
			close(f2);
        }
     }
 }
    printf("pg_exit: calling pvm_exit\n");
	}
	return pvm_exit();
}
/************************************************************************/
int pg_close()
{
    int inum,retval;
	int f1,f2,n,i;
	char buf[TRACEBUFSIZ];
	if (called_exit) return;
	called_exit=1;
	if(initialized) {
	  timestamp(&tt);
/**/
/* PG: trace_end */
/**/
	  fprintf (logfile, "19 %d %d %d 0\n", tt.tv_sec, tt.tv_usec, me);
/**/
      /* barrier to wait for everyone to enter pg_exit */
/* Comment out if no barrier  */
if (barrierflag) {
      /* barrier to wait for everyone to enter pg_exit */
		 /* perform a barrier */
		 if (me == 0) {  
		     for (i=1;i<(np-barriertotal);i++) 
				  pvm_recv(-1, TRACEBARRTAG);
			 pvm_initsend(PvmDataDefault);
			 pvm_mcast(&tids[1], np-1, TRACEBARRTAG);
		 }
		 else {
		     pvm_initsend(PvmDataDefault);
		     pvm_send(tids[0], TRACEBARRTAG);
		     pvm_recv(tids[0],TRACEBARRTAG);
         }
		 /* end of barrier */
}
	  fclose(logfile);

if (barrierflag) {
	  if (me > 0) {
		 /* If not tids[0], send my tracefile to tids[0]*/
		 if ((f1 = open(logname,O_RDONLY,0)) == -1) {
			  printf("pg_exit: can't reopen tracefile");
         }
		 else {
			  unlink(logname);
			  while ((n = read(f1, buf, TRACEBUFSIZ)) > 0) {
					  pvm_initsend(PvmDataDefault);
					  pvm_pkint(&n, 1, 1);
					  pvm_pkbyte(buf, n, 1);
					  pvm_send(tids[0], TRACETAG);
              } 
			  n= -1;
			  pvm_initsend(PvmDataDefault);
		      pvm_pkint(&n, 1, 1);
			  pvm_send(tids[0], TRACETAG);
			  close(f1);
         }
      }
      else if (me==0) {
		 /* if tids[0],i.e.  me==0, collect others tracefiles and */
		 /* concatenate */
		 if (outfilenm != NULL) {
			 if (filecp(logname,outfilenm) != 0) {
				 printf("pg_close:  filecp failed");
             }
			 else {
			 unlink(logname);
			 strcpy(logname,outfilenm);
			 }
         }
		 if ((f2 = open(logname,O_WRONLY | O_APPEND, 0644)) == -1) {
			  printf("pg_close: can't reopen MAIN tracefile");
         }
		 else {
		     i=1;
		     while (i< np) {
                 pvm_recv (tids[i], TRACETAG);
		         pvm_upkint(&n, 1, 1);
			     if (n != -1) {
		             pvm_upkbyte(buf, n, 1);
				     if (write(f2, buf, n) != n) {
					     printf("pg_exit: write error \n while writing %d\n",i);
                     }
                 }
			     else {
				     i++;
                 }
            }
			close(f2);
        }
     }

  }
    printf("pg_close: closing\n");
	}
     initialized = 0;
}

int pg_recv(tid, msgtag)
    int tid;
    int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who;
	if(initialized) {
	  timestamp(&tt);
	}
	info = bufid = pvm_recv(tid, msgtag);
	if(bufid>0) {
	  if(initialized) {
	    timestamp(&tt1);
	    (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
  if (barrierflag) {
        while(tag==TRACEBARRTAG) {
			  barriertotal++;
	          info = bufid = pvm_recv(tid, msgtag);
			  if (info <= 0) return info;
	          (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
		}
  }
	    who=tid2me(rid);
		if (who != -1) {
            /**/
            /* PG: recv_blocking */
            /**/
	        fprintf (logfile, "7 %d %d %d %d\n", 
		       tt.tv_sec, tt.tv_usec, me, msgtag);
            /**/
            /* PG: recv_waking */
            /**/
	        fprintf (logfile, "8 %d %d %d %d %d %d\n",
		     tt1.tv_sec, tt1.tv_usec, me, who, tag, nb);
        }
	  }
	}
/**/
	return info;
}
/************************************************************************/
int pg_nrecv(tid, msgtag)
    int tid;
    int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who;
	timestamp(&tt);
	info = bufid = pvm_nrecv(tid, msgtag);
	if(bufid>0) {
	  if(initialized) {
	    timestamp(&tt1);
	    (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
  if (barrierflag) {
        while(tag==TRACEBARRTAG) {
			  barriertotal++;
	          info = bufid = pvm_nrecv(tid, msgtag);
			  if (info <= 0) return info;
	          (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
		}
  }
	    who=tid2me(rid);
		if (who != -1) {
            /**/
            /* PG: recv_blocking */
            /**/
	         fprintf (logfile, "7 %d %d %d %d\n", 
		     tt.tv_sec, tt.tv_usec, me, msgtag);
            /**/
            /* PG: recv_waking */
            /**/
	         fprintf (logfile, "8 %d %d %d %d %d %d\n",
		     tt1.tv_sec, tt1.tv_usec, me, who, tag, nb);
        }
	  }
	}
/**/
	return info;
}
/************************************************************************/
/************************************************************************/
int pg_trecv(tid, msgtag, tmout)
    int tid;
    int msgtag;
	struct timeval *tmout;
{
	int info;
	int nb, rid, tag, bufid, who;
	timestamp(&tt);
	info = bufid = pvm_trecv(tid, msgtag, tmout);
	if(bufid>0) {
	  if(initialized) {
	    timestamp(&tt1);
	    (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
  if (barrierflag) {
        while(tag==TRACEBARRTAG) {
			  barriertotal++;
	          info = bufid = pvm_trecv(tid, msgtag, tmout);
			  if (info <= 0) return info;
	          (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
		}
  }
	    who=tid2me(rid);
		if (who != -1) {
            /**/
            /* PG: recv_blocking */
            /**/
	         fprintf (logfile, "7 %d %d %d %d\n", 
		     tt.tv_sec, tt.tv_usec, me, msgtag);
            /**/
            /* PG: recv_waking */
            /**/
	         fprintf (logfile, "8 %d %d %d %d %d %d\n",
		     tt1.tv_sec, tt1.tv_usec, me, who, tag, nb);
         }
	  }
	}
/**/
	return info;
}
/************************************************************************/
int pg_mcast (ttids, ntid, msgtag)
    int *ttids, ntid, msgtag;
{
	int i, info, bufid, nb, idum, who,found_one=0;
/**/
	if(initialized) {
	  timestamp(&tt);
	}
	info = pvm_mcast(ttids, ntid, msgtag);
	if(initialized) {
	  bufid=pvm_getsbuf();
	  (void)pvm_bufinfo(bufid, &nb, &idum, &idum);
	  for (i=0;i<ntid;i++) {
	    who = tid2me(ttids[i]);
		if (who != -1) {
			found_one = 1;
            /**/
            /* PG: send */
            /**/
	         fprintf (logfile, "4 %d %d %d %d %d %d\n",
		     tt.tv_sec, tt.tv_usec, me, who, msgtag, nb);
        }
	  }
	  if (found_one) {
          /**/
          /* PG: computation_stat */
          /**/
	      fprintf (logfile, "11 %d %d %d %d %d\n",
		         tt.tv_sec, tt.tv_usec, me, 0, 0);
	      timestamp(&tt);
	      fprintf (logfile, "11 %d %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, 0, 0);
      }
	}
	return info;
}
/************************************************************************/
int pg_send (tid, msgtag)
    int tid, msgtag;
{
	int info, who, bufid, nb, idum1, idum2;
/**/
	if(initialized) {
	  timestamp(&tt);
	}
	info = pvm_send(tid, msgtag);
	if(initialized) {
	  who = tid2me(tid);
	  bufid=pvm_getsbuf();
	  (void)pvm_bufinfo(bufid, &nb, &idum1, &idum2);
	  if (who != -1) {
          /**/
          /* PG: send */
          /**/
	      fprintf (logfile, "4 %d %d %d %d %d %d\n",
		      tt.tv_sec, tt.tv_usec, me, who, msgtag, nb);
          /**/
          /* PG: computation_stat */
          /**/
	      fprintf (logfile, "11 %d %d %d %d %d\n",
		        tt.tv_sec, tt.tv_usec, me, 0, 0);
	      timestamp(&tt);
	      fprintf (logfile, "11 %d %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, 0, 0);
      }
	}
/**/
	return info;
}
/************************************************************************/
int pg_initsend (type)
    int type;
{
	timestamp(&st0);
	return pvm_initsend(type);
}
/************************************************************************/
int pg_psend (tid, msgtag, buf,  nitems, dtype)
    int tid, msgtag, dtype, nitems;
    void* buf;
{
	int info, who;
	int nb = datasize(dtype)*nitems;
/**/
	if(initialized) {
	  who = tid2me(tid);
	  timestamp(&tt);
	}
	info = pvm_psend(tid, msgtag, buf, nitems ,dtype);
	if(initialized && (who != -1)) {
      /**/
      /* PG: send */
      /**/
	  fprintf (logfile, "4 %d %d %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, who, msgtag, nb);
      /**/
      /* PG: computation_stat */
      /**/
	  fprintf (logfile, "11 %d %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, 0, 0);
	  timestamp(&tt);
	  fprintf (logfile, "11 %d %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, 0, 0);
	}
/**/
	return info;
}
/************************************************************************/
int pg_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt)
    int tid, msgtag,nitems,dtype,*rtid,*rtag,*rcnt;
    void* buf;
{
	int info, nb, who;
	if(initialized) {
	  timestamp(&tt);
	}
    info = pvm_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt);
	if(initialized) {
  if (barrierflag) {
        while(*rtag==TRACEBARRTAG) {
			  barriertotal++;
              info = pvm_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt);
			  if (info < 0) return info;
		}
  }
	  who=tid2me(*rtid);
	  nb = (datasize(dtype))*(*rcnt);
	  if (who != -1) {
          /**/
          /* PG: recv_blocking */
          /**/
	      fprintf (logfile, "7 %d %d %d %d\n",
		   tt.tv_sec, tt.tv_usec, me, msgtag);
	      timestamp(&tt1);
          /**/
          /* PG: recv_waking */
          /**/
	       fprintf (logfile, "8 %d %d %d %d %d %d\n",
		   tt1.tv_sec, tt1.tv_usec, me, who, *rtag, nb);
      }
	}
/**/
	return info;
}
/************************************************************************/
/*									*/
/*			 FORTRAN INTERFACES				*/
/*									*/
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfmytid (tid)
#else
void
pgfmytid_ (tid)
#endif
    int *tid;
{
	*tid = pg_mytid();
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgftids(ttids, tnp, info)
#else
void
pgftids_(ttids, tnp, info)
#endif
    int *ttids;
    int *tnp;
    int *info;
{
	*info = pg_tids(ttids, *tnp);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfexit (info)
#else
void
pgfexit_ (info)
#endif
    int *info;
{
	*info = pg_exit();
}

#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfchprefix (nme)
#else
void
pgfchprefix_ (nme)
#endif
   char *nme; 
{
	pg_chprefix(nme);
}

#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfoutfile (nme)
#else
void
pgfoutfile_ (nme)
#endif
   char *nme; 
{
	pg_outfile(nme);
}

#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfclose (info)
#else
void
pgfclose_ (info)
#endif
    int *info;
{
	pg_close();
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfrecv (tid, msgtag, info)
#else
void
pgfrecv_ (tid, msgtag, info)
#endif
    int *tid, *msgtag, *info;
{
	*info = pg_recv(*tid, *msgtag);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfnrecv (tid, msgtag, info)
#else
void
pgfnrecv_ (tid, msgtag, info)
#endif
    int *tid, *msgtag, *info;
{
	*info = pg_nrecv(*tid, *msgtag);
}
/************************************************************************/
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgftrecv (tid, msgtag, sec, usec, info)
#else
void
pgftrecv_ (tid, msgtag, sec, usec, info)
#endif
    int *tid, *msgtag, *sec, *usec, *info;
{
    struct timeval temp;
	temp.tv_sec = *sec;
	temp.tv_usec = *usec;
	*info = pg_trecv(*tid, *msgtag, &temp);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfmcast (ntid, ttids, msgtag, info)
#else
void
pgfmcast_ (ntid, ttids, msgtag, info)
#endif
    int *ntid, *ttids, *msgtag, *info;
{
	*info = pg_mcast(ttids, *ntid, *msgtag);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfsend (tid, msgtag, info)
#else
void
pgfsend_ (tid, msgtag, info)
#endif
    int *tid, *msgtag, *info;
{
	*info = pg_send(*tid, *msgtag);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfinitsend (type, info)
#else
void
pgfinitsend_ (type, info)
#endif
    int *type, *info;
{
	*info = pg_initsend(*type);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfpsend(tid, msgtag,  buf, nitems, dtype,  info)
#else
void
pgfpsend_(tid, msgtag, buf, nitems, dtype, info)
#endif
    int* tid;
    int* msgtag;
    void* buf;
    int* nitems;
    int* dtype;
    int* info;
{
	*info = pg_psend(*tid, *msgtag, buf, *nitems, *dtype);
}
/************************************************************************/
#if(defined(IMA_HPPA) || defined(IMA_RS6K))
void
pgfprecv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt, info)
#else
void
pgfprecv_(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt, info)
#endif
    int* tid;
    int* msgtag;
    void* buf;
    int* nitems;
    int* dtype;
	int* rtid;
	int* rtag;
	int* rcnt;
    int* info;
{
   *info =  pg_precv(*tid, *msgtag, buf, *nitems, *dtype, rtid, rtag, rcnt);
}
/************************************************************************/
