/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
    
                        Author:  William W. Y. Liang

           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW

                   Copyright (C) 1996 All Rights Reserved

                                  NOTICE

      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.

      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

// --------------------------------------------------------------------
// Timing Usage:
//   start()   : start the timer
//   stop()    : stop the timer
//   restart() : restart the timer
//   time()    : get the timer at the calling time
// General sequence is start()<->stop(), which must appear in pairs,
// and can be nested. Timing can be directly used in by ostream.
// Example:
//	Timing t();
//	t.start();
//	...
//	t.stop();
//	cout<<"time="<<t<<endl;
// If a name is given on construct time, time will be printed out on
// destruct time on standard error.
// If Launch is also specified on construct time, timer will start 
// immediately.
// --------------------------------------------------------------------

#ifndef __ADSMUTIL_H__
#define __ADSMUTIL_H__

#include <stdio.h>
#include <iostream.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>

#include <unistd.h>
#include <sys/time.h>
// extern "C" int gettimeofday(struct timeval *tp,struct timezone *tzp);

class Timing {

  friend ostream& operator<<(ostream& s, Timing& t);

protected:

  char *name;

  timeval t_last; // ,t_all;

  double alltime;

  int level;		// nest level

  void cumulate();	// cumulate time from last start

public:

  enum Control { Launch, Ready };

  Timing(char *n=0,Control ctrl=Ready);
  ~Timing();

  void start();
  void stop();

  void restart();

  double time();
};

// -- Methods of Timing --

inline Timing::Timing(char *n,Control ctrl) : name(0), level(0) { 
  t_last.tv_sec=t_last.tv_usec=0; // t_all.tv_sec=t_all.tv_usec=0;
  alltime=0.0;
  if (n) name=strdup(n);
  if (ctrl==Launch) {
    gettimeofday(&t_last,0);
    level=1;
  }
}

inline void Timing::cumulate() { 
  struct timeval t_now;
  gettimeofday(&t_now,0);

  // t_all.tv_sec +=t_now.tv_sec -t_last.tv_sec;
  // t_all.tv_usec+=t_now.tv_usec-t_last.tv_usec;
  alltime+=(t_now.tv_sec+t_now.tv_usec*1e-6)-
           (t_last.tv_sec+t_last.tv_usec*1e-6);
}

inline void Timing::restart() {
  // t_all.tv_sec=t_all.tv_usec=0;	// reset cumulation
  alltime=0.0;
  gettimeofday(&t_last,0);
  level=1;
}

// force start/restart timing
inline void Timing::start() {
  if (level==0) gettimeofday(&t_last,0);
  level++;
}

// stop timing, cumulate time from start to stop
inline void Timing::stop() {
  level--;
  if (level==0) 
    cumulate();
  else if (level<0) 
    cerr<<"Error: Timing::stop(): no corresponding start()"<<endl; 
}

inline double Timing::time() { 
  if (level>0) {	// nest not finished, cumulate and reset t_last
    cumulate();
    gettimeofday(&t_last,0);
  }

  return alltime; // t_all.tv_sec+t_all.tv_usec*1e-6; 
}

inline Timing::~Timing() { 
  if (name) {
    cerr<<"Time "<<name<<" = "<<time()<<" sec"<<endl;
    free(name); 
  }
}

inline ostream& operator<<(ostream& s, Timing& t) { 
  if (t.name) s<<"Time "<<t.name<<" = "; 
  s<<t.time();
  if (t.name) s<<" sec"; 
  return s;
}

#endif // __ADSMUTIL_H__
