/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved
 
                                  NOTICE
 
      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.
 
      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

/* Notes:
   - pvm_psend can be used instead of pvm_send, but it has compatibility
     problem, pvm_precv is better to be used at the same time (man pvm_psend)
   - PvmDataInPlace will not work on Shared Memory machines (man pvm_initsend)
   - pvm_mcast do not obey the message ordering under PvmDirectRoute
*/

#ifndef __ADSMPACK_H__
#define __ADSMPACK_H__

#include <iostream.h>
#include "common.h"
#include "pvm3.h"
#include "modset.h"

#define PACK_LEN	8000	// packet growing size
#define MAXPKTSIZE	7500	// max packet size 

#ifdef ADSMCTRL
// Timing
extern unsigned long _adsm_timestamp;
extern Timing _adsm_pkt_time;
extern Timing _adsm_idle_time;
extern unsigned long _adsm_ctrlflags;
// timing is not necessary in private methods, since they are called interally
// there should be no return between start() and stop()
#endif

// -- Send Packet --

class AdsmSendPacket {
private:

  int	packlen;	// buffer size
  char	*modbuf;		// buffer
  ModifySet modset;	// modified set

#ifdef ADSMCTRL
  // statistics
  int	msgnum;
  int	msgsize;
  int	last_msgnum;
  int	last_msgsiz;
#endif

  void grow(int num=PACK_LEN);	// realloc modbuf
  void need(int num);		// need more num space to use
  int  ready2send();

public:

  AdsmSendPacket();

  ~AdsmSendPacket() { delete [] modbuf; } 

  void reset();	// packet reset

  // packing
  AdsmSendPacket& pkbyte(char *c,int num);
  AdsmSendPacket& operator<<(char c);
  AdsmSendPacket& operator<<(int val); 
  AdsmSendPacket& operator<<(long val); 
  AdsmSendPacket& operator<<(unsigned long val); 
  AdsmSendPacket& operator<<(short val); 
  AdsmSendPacket& operator<<(char *s);
  AdsmSendPacket& pkint(int *vals,int num);
  AdsmSendPacket& pklong(long *vals,int num);
  AdsmSendPacket& pkulong(unsigned long *vals,int num);
  AdsmSendPacket& pkshort(short *vals,int num);

  // for multiple write protocol: modified set pack
  int pkmodset(char *olddata,char *newdata,int datasize,int upgrade=1);
  AdsmSendPacket& pkmodset(ModifySet *ms);

  // send 
  int send(int tid,int msgtag);
  int mcast(int *tids,int num,int msgtag);

  // information

#ifdef ADSMCTRL
  int get_msgnum() { return msgnum; } 
  int get_msgsize() { return msgsize; } 
  void classify_message(int &msgnum,int &msgsiz) { 
    msgnum+=last_msgnum;
    msgsiz+=last_msgsiz;
  }
#endif

  int suggestsend() { return 0; } 
};

// -- Receive Packet --

class AdsmRecvPacket {
private:

  int	packlen;
  char	*modbuf;	
  ModifySet modset;

#ifdef ADSMCTRL
  // statistics
  int	msgnum;
  int	msgsize;
#endif

  // set packet size
  void set_packsize(int size);
  int setpacket(int rbuf,int *stid=NULL);

public:

  AdsmRecvPacket() : packlen(0), modbuf(NULL) {} 
  ~AdsmRecvPacket() { delete [] modbuf; }

  // unpacking

  AdsmRecvPacket& upkbyte(char *c,int num);
  AdsmRecvPacket& operator>>(char &c);
  AdsmRecvPacket& operator>>(char *s);
  AdsmRecvPacket& operator>>(int &i);
  AdsmRecvPacket& operator>>(long &i);
  AdsmRecvPacket& operator>>(unsigned long &i);
  AdsmRecvPacket& operator>>(short &i);
  AdsmRecvPacket& upkint(int *i,int num);
  AdsmRecvPacket& upklong(long *i,int num);
  AdsmRecvPacket& upkulong(unsigned long *i,int num);
  AdsmRecvPacket& upkshort(short *i,int num);

  // for multiple write protocol: modified set unpack
  
  AdsmRecvPacket& upkmodset(char *olddata,char *newdata=NULL);
  ModifySet *getmodset() { return &modset; } // retrieval modified set

  // recv

  int recv(int tid,int msgtag,int *stid=NULL);
  int nrecv(int tid,int msgtag,int *stid=NULL);

#ifdef ADSMCTRL
  // information
  int get_msgnum() { return msgnum; } 
  int get_msgsize() { return msgsize; } 
  int add_msgnum(int delta) { return msgnum+=delta; } 
  int add_msgsize(int delta) { return msgsize+=delta; } 
#endif

};

// -- Methods of send packet --

inline AdsmSendPacket::AdsmSendPacket() : packlen(PACK_LEN) { 
#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  modbuf=new char[packlen]; 

#ifdef ADSMCTRL
  msgnum=msgsize=0;

  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif
} 

inline void AdsmSendPacket::need(int num) { 
  if (num>packlen) {
    delete [] modbuf;
    modbuf=new char[num];
  }
}

inline AdsmSendPacket& AdsmSendPacket::pkbyte(char *c,int num) { 
   if (!c||num<=0) return *this;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  pvm_pkbyte(c,num,1);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return *this;
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(char c) { 
  return pkbyte(&c,sizeof(char));
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(int val) { 
  return pkbyte((char*)&val,sizeof(int));
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(long val) { 
  return pkbyte((char*)&val,sizeof(long));
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(unsigned long val) { 
  return pkbyte((char*)&val,sizeof(unsigned long));
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(short val) { 
  return pkbyte((char*)&val,sizeof(short));
};

inline AdsmSendPacket& AdsmSendPacket::operator<<(char *s) { 
  int len=strlen(s);
  (*this)<<len;
  return pkbyte(s,len);
};

inline AdsmSendPacket& AdsmSendPacket::pkint(int *vals,int num) { 
  return pkbyte((char*)vals,sizeof(int)*num);
};

inline AdsmSendPacket& AdsmSendPacket::pklong(long *vals,int num) { 
  return pkbyte((char*)vals,sizeof(long)*num);
};

inline AdsmSendPacket& AdsmSendPacket::pkulong(unsigned long *vals,int num) { 
  return pkbyte((char*)vals,sizeof(unsigned long)*num);
};

inline AdsmSendPacket& AdsmSendPacket::pkshort(short *vals,int num) { 
  return pkbyte((char*)vals,sizeof(short)*num);
};

inline AdsmSendPacket& AdsmSendPacket::pkmodset(ModifySet *ms) {
  char fourbyte[4];
  int2fourbyte(ms->size(),fourbyte);
  pkbyte(fourbyte,4);
  return pkbyte(ms->content(),ms->size());
};

// make and pack the modified set, upgrade the new data optionally
inline int AdsmSendPacket::pkmodset(char *olddata,char *newdata,
                                    int datasize,int upgrade) 
{
#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  int maxsize=modset.max_size(datasize);	// required modify set size

  need(4+maxsize);	// prepare buffer needed by the modify set

  modset.set_buffer(modbuf+4,maxsize);	// set buffer of modset

  int setsize=modset.make(olddata,newdata,datasize);
  // Note: future using of modset.make in the same packing must be avoided

  int2fourbyte(setsize,modbuf);	// fill 4 byte setsize in sizeptr

  pkbyte(modbuf,setsize+4); // pack into send buffer

  if (upgrade) modset.apply(olddata); // upgrade old data

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return setsize;
};

// reset the buffer
inline void AdsmSendPacket::reset() { 
  pvm_initsend(PvmDataRaw);	// initialize pvm buffer
  // PvmDataInPlace can not be used in this version because 
  // operator<<(char*) has local (volatile) data and 
  // modify set will share the space modbuf

#ifdef ADSMCTRL
  (*this)<<_adsm_timestamp;

  last_msgnum=last_msgsiz=0;
#endif
}

// pack data into send buffer
inline int AdsmSendPacket::ready2send() {
  int len;
  pvm_bufinfo(pvm_getsbuf(),&len,NULL,NULL);
  return len;
}

inline int AdsmSendPacket::send(int tid,int msgtag) {
#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  int size=ready2send();
  int ret=pvm_send(tid,msgtag);

#ifdef ADSMDEBUG
  if (ret<0) {
    cerr<<"AdsmSendPacket::send() : pvm buffer preparing fail"<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

#ifdef ADSMCTRL
  if (!is_local(tid)) {
    msgnum++; 
    msgsize+=size; 
    last_msgnum=1;
    last_msgsiz=size;
  }

  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return ret>=0?size:ret;
}

inline int AdsmSendPacket::mcast(int *tids,int num,int msgtag) {
#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  int size=ready2send();

  int ret=0;
  int totalsize=0,remote_msgnum=0,remote_msgsiz=0;
  for (int i=0; i<num; i++) {
    int code=pvm_send(tids[i],msgtag);
    if (ret>=0) ret=code;
    totalsize+=size;
#ifdef ADSMCTRL
    if (!is_local(tids[i])) {
      remote_msgnum++;
      remote_msgsiz+=size;
    }
#endif
  }

#ifdef ADSMDEBUG
  if (ret<0) {
    cerr<<"AdsmSendPacket::mcast() : pvm buffer preparing fail"<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

#ifdef ADSMCTRL
  msgnum+=remote_msgnum;
  msgsize+=remote_msgsiz; 
  last_msgnum=remote_msgnum;
  last_msgsiz=remote_msgsiz;

  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return ret>=0?totalsize:ret;
}
  
// -- Methods of receive packet --

inline AdsmRecvPacket& AdsmRecvPacket::upkbyte(char *c,int num) { 
   if (num<=0) return *this;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  if (c) 
    pvm_upkbyte(c,num,1);
  else { 
    char tmpbuf[100];
    for (; num>=100; num-=100) pvm_upkbyte(tmpbuf,100,1);
    if (num>0) pvm_upkbyte(tmpbuf,num,1);
  }

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return *this;
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(char &c) { 
  return upkbyte(&c,sizeof(char));
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(int &i) { 
  return upkbyte((char*)&i,sizeof(int));
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(char *s) { 
  int len;
  (*this)>>len;
  upkbyte(s,len);
  s[len]=0;
  return *this;
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(long &i) { 
  return upkbyte((char*)&i,sizeof(long));
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(unsigned long &i) { 
  return upkbyte((char*)&i,sizeof(unsigned long));
};

inline AdsmRecvPacket& AdsmRecvPacket::operator>>(short &i) { 
  return upkbyte((char*)&i,sizeof(short));
};

inline AdsmRecvPacket& AdsmRecvPacket::upkint(int *i,int num) { 
  return upkbyte((char*)i,sizeof(int)*num);
};

inline AdsmRecvPacket& AdsmRecvPacket::upklong(long *i,int num) { 
  return upkbyte((char*)i,sizeof(long)*num);
};

inline AdsmRecvPacket& AdsmRecvPacket::upkulong(unsigned long *i,int num) { 
  return upkbyte((char*)i,sizeof(unsigned long)*num);
};

inline AdsmRecvPacket& AdsmRecvPacket::upkshort(short *i,int num) { 
  return upkbyte((char*)i,sizeof(short)*num);
};

// make and pack the modified set, upgrade the new data optionally
inline void AdsmRecvPacket::set_packsize(int size) {
  if (packlen<size) {
    delete [] modbuf;
    modbuf=new char[packlen=size];
  }
}

// unpack and apply to old data and new data (optional)
inline AdsmRecvPacket& AdsmRecvPacket::upkmodset(char *olddata,char *newdata) {
#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

  char fourbyte[4];
  upkbyte(fourbyte,4);

  int size=fourbyte2int(fourbyte); // retrive 4 byte as size of modset
  set_packsize(size); // set size for modify set

  upkbyte(modbuf,size);
  modset.set_buffer(modbuf,size);	

  modset.set_setsize(size);	// set setsize of modset as 'size'
  // Note: future setting of modset in the same unpacking must be avoided

  modset.apply(newdata);
  modset.apply(olddata);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return *this;
};

inline int AdsmRecvPacket::setpacket(int rbuf,int *stid) {
  int size,tid;
  pvm_bufinfo(rbuf,&size,NULL,&tid); // read in buffer

  if (stid) *stid=tid;

#ifdef ADSMCTRL
  if (!is_local(tid)) {
    msgnum++; 
    msgsize+=size; 
  }
#endif

#ifdef ADSMCTRL
  // set timestamp
  unsigned long ts;
  (*this)>>ts;
  if (ts>=_adsm_timestamp) _adsm_timestamp=ts+1;
#endif

  return rbuf;
}

inline int AdsmRecvPacket::recv(int tid,int msgtag,int *stid) {

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimIdl) _adsm_idle_time.start();
#endif

  int rbuf=pvm_recv(tid,msgtag);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimIdl) _adsm_idle_time.stop();
#endif

#ifdef ADSMDEBUG
  if (rbuf<0) {
    cerr<<"AdsmRecvPacket::recv(): fail to recv"<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  int ret=setpacket(rbuf,stid);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return ret;
}

inline int AdsmRecvPacket::nrecv(int tid,int msgtag,int *stid) {

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.start();
#endif

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimIdl) _adsm_idle_time.start();
#endif

  int rbuf=pvm_nrecv(tid,msgtag);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimIdl) _adsm_idle_time.stop();
#endif

#ifdef ADSMDEBUG
  if (rbuf<0) {
    cerr<<"AdsmRecvPacket::nrecv(): fail to recv"<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  int ret=0;
  if (rbuf!=0) ret=setpacket(rbuf,stid);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimPkt) _adsm_pkt_time.stop();
#endif

  return ret;
}

#endif // __ADSMPACK_H__
