/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved
 
                                  NOTICE
 
      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.
 
      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#ifndef __ADSMMODSET_H__
#define __ADSMMODSET_H__

#include "common.h"
#include "timing.h"

#ifdef ADSMCTRL
// Timing
extern Timing _adsm_modset_time;
extern unsigned long _adsm_ctrlflags;
// timing is not necessary in private methods, since they are called interally
// there should be no return between start() and stop()
#endif

// modified set class

class ModifySet {

  friend ostream& operator<<(ostream& s, ModifySet& t);

private:
  char *buffer;
  int buffersize;	// size of the buffer
  int setsize;		// size of the modify set

  char myownbuf;	// buffer is self-generated or provided by others

  void nextseg(char* &segdata,char* &segbegin,unsigned char* &segsize);
  char *set_set(int size);		// set set size, buffer is self prepare

public:
  ModifySet() : buffer(NULL), buffersize(0), setsize(0), myownbuf(0) {}
  ~ModifySet() { if (myownbuf) delete [] buffer; }

  // retrieval modified set 
  char *content() { return buffer; }
  int size() { return setsize;}
  int max_size(int datasize);

  // settings
  void set_buffer(char *b,int s);	// set buffer provided by caller
  void set_setsize(int s) { setsize=s; }	// set setsize

  // modify set
  int make(char *olddata,char *newdata,int datasize);	// made from two data
  void dup(ModifySet *ms);	// made by duplicating other modify set

  // apply modified set to the data
  void apply(char *data);	
};

// -- Methods of ModifySet --

// set for pointers of next segment
inline void ModifySet::nextseg(char* &segdata,char* &segbegin,
                               unsigned char* &segsize) 
{
  segbegin=segdata;
  segsize=(unsigned char*)(segbegin+4);
  segdata=(char*)segsize+1;
}

// set the buffer to at least size of 'size', the set size
inline char *ModifySet::set_set(int size) {
  if (!myownbuf || buffersize<size) {
    if (myownbuf) delete [] buffer;
    buffer=new char[buffersize=size];
    myownbuf=1;
  }
  setsize=size;
  return buffer;
}

// set the buffer to a prepared space b, of size s
inline void ModifySet::set_buffer(char *b,int s) {
  if (myownbuf) delete [] buffer;
  buffer=b;
  buffersize=s;
  setsize=0;
  myownbuf=0;
}

// get the maximal buffer size required for data of size 'datasize'
inline int ModifySet::max_size(int datasize) { 
  // The size required by ModifySet is calculated as follows:
  //   Let S = segment num, L = data size, S <= L/2
  //   max size = S(4+1+(L-S)/S) = 5S+(L-S) = 4S+L <= 3L
  return datasize*3;
}

int ModifySet::make(char *olddata,char *newdata,int datasize) {

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.start();
#endif

#ifdef ADSMDEBUG
  int maxsize=max_size(datasize);
#endif

  // set_buffersize(maxsize); // prepare buffer
  //+++// assume buffer is already prepared

  // setup the first modified record
  char *segdata=buffer,*segbegin;
  unsigned char *segsize;
  int recording=0;

  // fill the segment
  setsize=0;
  // nextseg(segdata,segbegin,segsize);
  for (int index=0; index<datasize; index++) {
    if (newdata[index]!=olddata[index]) { // find modified byte
      if (!recording) { // first mismatch, set up this segment header

#ifdef ADSMDEBUG
        if (setsize>=maxsize) {
          cerr<<"make_modset(): buffer full"<<endl;
          Exit(__FILE__,__LINE__);
        }
#endif
        nextseg(segdata,segbegin,segsize);
        int2fourbyte(index,segbegin); // *(int*)segbegin=index;
        *segsize=0;
        setsize=segdata-buffer;
        recording=1;
      }
      *segdata=newdata[index];
      segdata++;
      setsize++;
      if (++(*segsize)>=255) // 255 is the max val of a u_char
        recording=0;
    } else // find matched byte
      if (recording) // prepare next segment
        recording=0;
  }

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.stop();
#endif
  
  return setsize;
}

void ModifySet::apply(char *data) {
  if (data==NULL) return;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.start();
#endif

  char *lastdata=buffer+setsize;
  char *segdata=buffer,*segbegin;
  unsigned char *segsize;

  while (segdata<lastdata) {
    nextseg(segdata,segbegin,segsize); // setup the apply record
    char *ptr=&data[fourbyte2int(segbegin)];
    for (int i=0; i<(*segsize); i++,segdata++)
      ptr[i]=*segdata;
  }

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.stop();
#endif
  
}

// duplicate set from other modify set
void ModifySet::dup(ModifySet *ms) { 

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.start();
#endif

  set_set(ms->size());	// buffer is prepared by myself
  memcpy(buffer,ms->content(),ms->size());

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlTimMod) _adsm_modset_time.stop();
#endif
  
}

inline ostream& operator<<(ostream& s, ModifySet& m) {
  if (m.setsize<=0) return s<<"{}";

  char *lastdata=m.buffer+m.setsize;
  char *segdata=m.buffer,*segbegin;
  unsigned char *segsize;

  s<<'{';
  while (segdata<lastdata) {
    m.nextseg(segdata,segbegin,segsize); // setup the apply record
    s<<'['<<fourbyte2int(segbegin)<<','<<(unsigned)*segsize<<','<<hex;
    for (int i=0; i<(*segsize); i++,segdata++) s<<(unsigned)*segdata;
    s<<dec<<']';
  }
  s<<'}';

  return s;
}

#endif // __ADSMMODSET_H__
