/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
    
                        Author:  William W. Y. Liang

           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW

                   Copyright (C) 1996 All Rights Reserved

                                  NOTICE

      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.

      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <new.h>
#include "pvm3.h"
#include "adsm.h"
#include "common.h"
#include "libadsm.h"
#include "dynarr.h"
#include "hash.h"
#include "pack.h"
#include "cohver.h"
#include "unoffic.h"
#include "sysdep.h"

// Control Flags
unsigned long _adsm_ctrlflags=0;

Timing _adsm_total_time;

#ifdef ADSMCTRL

unsigned long _adsm_timestamp=0;

Timing _adsm_modset_time;
Timing _adsm_pkt_time;
Timing _adsm_idle_time;

int _adsm_msgnum_system=0; 
int _adsm_msgsiz_system=0; 
int _adsm_msgnum_consist=0; 
int _adsm_msgsiz_consist=0; 
int _adsm_msgnum_malloc=0;
int _adsm_msgsiz_malloc=0;
int _adsm_msgsiz_access=0;      
int _adsm_msgnum_access=0;      
int _adsm_msgsiz_sync=0;            
int _adsm_msgnum_sync=0;            
#endif

// system packet for send & recv
static AdsmRecvPacket rpkt;
static AdsmSendPacket spkt; 

// system object
static Adsmith adsm; 

// Message Head
static MsgHead msghead;

// --- other functions ---

static ostream& operator<<(ostream& s, MsgHead& m) {
   m.useless++;

#ifdef ADSMCTRL
   s<<_adsm_timestamp<<":\t";
#endif

   return s<<'P'<<adsm.procno<<"/H"<<adsm.hostno<<":\t";
}

static void shutdown() { adsm.shutdown(); }

// --- inline methods of class Adsmith ---

inline AdsmObj *Adsmith::get_sctrl(void *data) {	// get share conrotl
  AdsmObj *sctrl=(AdsmObj*)((char*)data-ALIGNED_AdsmObj_SIZE);
  if (sctrl->data!=data) {
    cerr<<msghead<<"Warning: wrong pointer "<<data<<" to a shared object"<<endl;
    // return NULL;
    shutdown();
  }
  return sctrl;
}

// --- inline methods of class AdsmObj ---

inline void AdsmObj::validate(int force) {
  // reset reading bit, all previous prefetch will be ignored
  status&=~AdsmObjStatusPrefetching;
  status&=~AdsmObjStatusBulkRefresh;

  int ever_cached=status&AdsmObjStatusValid;

  // set valid bit on Cache
  switch (force) {
    case -1: // unchanged
      break;
    case 0:  // force invalidated
      status&=~AdsmObjStatusValid; 
      break;
    case 1:  // force validated
      status|=AdsmObjStatusValid; 
      break;
    default: // default
      if (hint&AdsmDataCache)
        status|=AdsmObjStatusValid; 
      else
        status&=~AdsmObjStatusValid; 
      break;
  }

  // record home in home-list if data should be, but not previously, cached 
  if (status&AdsmObjStatusValid && !ever_cached)
      /* eliminate handshaking with local host, but the situation on
         nodes of MP is still unknown; because the handshaking is used to 
         ensure the arrival of coherence messages. 
      && home!=adsm.hostno) */
  {
#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"add "<<home<<" to homelist"<<endl;
#endif
    adsm.homelist.uniq_insert(home);
  }
}

// Ordinary load: allow nonblocking refresh the contains, prefetch
inline void AdsmObj::prefresh() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  // check if valid
  if (status&AdsmObjStatusWriting||status&AdsmObjStatusValid) return;

  if (!(status&AdsmObjStatusPrefetching)) {
    AdsmObj *that=this;
    adsm.readlist[home].insert(that);
    status|=AdsmObjStatusPrefetching;
  }

  if (adsm.bulk_prefresh) return; // aggregate prefresh

  adsm.bulkprefresh();

  // clean prefetch message when outgoing mesg num > CLEAN_OUTMSG_NUM
  if (adsm.readnum>CLEAN_OUTMSG_NUM) {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"clean outgoing prefetch requests"<<endl;
#endif

    adsm.donerefresh(gindex);
  }
}

// Ordinary load: wait till prefresh done
inline void AdsmObj::refresh() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  // check if valid
  if (status&AdsmObjStatusWriting||status&AdsmObjStatusValid) return;

  if (!(status&AdsmObjStatusPrefetching)) prefresh();

  // aggregate refresh
  if (adsm.bulk_refresh) {
    if (!(status&AdsmObjStatusBulkRefresh)) {
      adsm.gndxlist.insert(gindex);
      status|=AdsmObjStatusBulkRefresh;
    }
    return;
  }

  adsm.donerefresh(gindex); // wait till #gindex arrive
}

// Ordinary store: allow pipeline store to home
inline void AdsmObj::flush() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  // deffer flush to release time
  if (!(status&AdsmObjStatusWriting)) { // make it be done once
    AdsmObj *that=this;
    adsm.writelist[home].insert(that);
    status|=AdsmObjStatusWriting;
  }

  if (hint&AdsmDataMWriter) 	
    validate(-1); // unchanged	
  else                     	
    validate();
}

// --- non-inline methods of class Adsmith ---

// flag setting
unsigned long Adsmith::flag_setting() {
  char *flags=getenv("ADSM_FLAGS");

  if (flags==NULL) return 0;

  for (int i=0; flags[i]!=0; i+=2) {
    int ok=1;
    switch (flags[i]) {
      case 'D': switch (flags[i+1]) {
                  case 'a': _adsm_ctrlflags|=AdsmCtrlDbgApp;
                            cerr<<"Adsmith: Da: Debug Applications"<<endl; 
                            break;
#ifdef ADSMCTRL
                  case 'd': _adsm_ctrlflags|=AdsmCtrlDbgDmn;
                            cerr<<"Adsmith: Dd: Debug Daemons"<<endl; 
                            break;

                  case 't': _adsm_ctrlflags|=AdsmCtrlDbgTrc;
                            cerr<<"Adsmith: Dt: Trace all events"<<endl; 
                            break;
                  case 'p': _adsm_ctrlflags|=AdsmCtrlDbgDat;
                            cerr<<"Adsmith: Dp: Trace with dumping data"<<endl; 
                            break;
#endif

                  default : ok=0;
                }
                break;

      case 'T': switch (flags[i+1]) {
                  case 'a': _adsm_ctrlflags|=AdsmCtrlTimTtl;
                            cerr<<"Adsmith: Ta: Timing On Total Run"<<endl; 
                            break;
#ifdef ADSMCTRL
                  case 'p': _adsm_ctrlflags|=AdsmCtrlTimPkt;
                            cerr<<"Adsmith: Tm: Timing On (un)Packing"<<endl; 
                            break;
                  case 'm': _adsm_ctrlflags|=AdsmCtrlTimMod;
                            cerr<<"Adsmith: Tm: Timing On ModifySet"<<endl; 
                            break;
                  case 'i': _adsm_ctrlflags|=AdsmCtrlTimIdl;
                            cerr<<"Adsmith: Ti: Timing Of Idleness"<<endl; 
                            break;
#endif
                  default : ok=0;
                }
                break;
      case 'M': switch (flags[i+1]) {
#ifdef ADSMCTRL
                  case 'a': _adsm_ctrlflags|=AdsmCtrlMsgTtl;
                            cerr<<"Adsmith: Ma: Total Message Statistics"
                                <<endl; 
                            break;
                  case 'c': _adsm_ctrlflags|=AdsmCtrlMsgCls;
                            cerr<<"Adsmith: Mc: Classified Message Statistics"
                                <<endl; 
                            break;
#endif
                  default : ok=0;
                }
                break;
      case 'S': switch (flags[i+1]) {
                  case 'o': _adsm_ctrlflags|=AdsmCtrlSttObj;
                            cerr<<"Adsmith: So: Statistics of object number"
                                <<endl; 
                            break;
#ifdef ADSMCTRL
                  case 'h': _adsm_ctrlflags|=AdsmCtrlSttHsh;
                            cerr<<"Adsmith: Sh: Statistics of hash table"
                                <<endl; 
                            break;
                  case 'r': _adsm_ctrlflags|=AdsmCtrlSttRst;
                            cerr<<"Adsmith: Sr: Statistics of rest info"
                                <<endl; 
                            break;
#endif
                  default : ok=0;
                }
                break;
      case 'O': switch (flags[i+1]) {
#ifdef ADSMCTRL
                  case 'h': _adsm_ctrlflags|=AdsmCtrlOptHsk;
                            cerr<<"Adsmith: Oh: Turn On Handshake"<<endl; 
                            break;
                  case 'a': _adsm_ctrlflags|=AdsmCtrlOptAck;
                            _adsm_ctrlflags|=AdsmCtrlOptHsk; // Ack => Hsk
                            cerr<<"Adsmith: Oa: Turn On Acknowledge"<<endl; 
                            break;
#endif
                  case 'm': _adsm_ctrlflags|=AdsmCtrlOptMix;
                            cerr<<"Adsmith: Om: Turn On Heterogeneous Mode"<<endl; 
                            break;
                  case 'r': _adsm_ctrlflags|=AdsmCtrlOptPDR;
                            cerr<<"Adsmith: Or: Turn On Pvmd Routing"<<endl; 
                            break;
                  default : ok=0;
                }
                break;
      case 'I': switch (flags[i+1]) {
                  case 'v': _adsm_ctrlflags|=AdsmCtrlInfVer;
                            cerr<<"Adsmith: Iv: Print Version Info"<<endl; 
                            break;
                  default : ok=0;
                }
                break;
      case 'H': cerr<<"Adsmith Control Flags"<<endl; 
                cerr<<"  Da: Debug Applications"<<endl; 
#ifdef ADSMCTRL
                cerr<<"  Dd: Debug Daemons"<<endl; 
                cerr<<"  Dt: Trace all events"<<endl; 
                cerr<<"  Dp: Trace with dumping data"<<endl; 
                cerr<<"  Ma: Total Message Statistics"<<endl;
                cerr<<"  Mc: Classified Message Statistics"<<endl;
                cerr<<"  Sh: Statistics of hash table"<<endl;
                cerr<<"  Sr: Statistics of rest info"<<endl;
#endif
                cerr<<"  So: Statistics of object number"<<endl;
                cerr<<"  Ta: Timing On Total Run"<<endl; 
#ifdef ADSMCTRL
                cerr<<"  Tm: Timing On ModifySet"<<endl; 
                cerr<<"  Tp: Timing On (un)Packing"<<endl; 
                cerr<<"  Ti: Timing Of Idleness"<<endl; 
                cerr<<"  Oh: Turn On Handshake"<<endl; 
                cerr<<"  Oa: Turn On Acknowledge"<<endl; 
#endif
                cerr<<"  Or: Turn Off Pvmd Routing"<<endl; 
                cerr<<"  Om: Turn On Heterogeneous Mode"<<endl; 
                cerr<<"  Iv: Print Version Info"<<endl; 
                sysdep_exit(0); 
      default : ok=0;
    }
    
    if (!ok) {
      cerr<<"Adsmith: unknown ADSM_FLAGS "<<flags[i]<<flags[i+1]<<endl;
      sysdep_exit(-1); 
    }
  }

  return _adsm_ctrlflags;
}

// startup adsm daemons, return mytid, else -1 
void Adsmith::startup() {
  // get configuration
  int nh,narch;
  struct pvmhostinfo  *hinfo;
  if (pvm_config(&nh,&narch,&hinfo)<0) {
    cerr<<"Adsmith::startup(): pvm config retrival error"<<endl;
    sysdep_exit(-1); 
  }
  nhost=nh;

  // check homogeniouty
  if (narch>1 && !(_adsm_ctrlflags&AdsmCtrlOptMix)) { 
    cerr<<"Adsmith::startup(): current implementation only allow "
        <<"homogeneous enviroment"<<endl;
    sysdep_exit(-1); 
  }

  arch=my_strdup(hinfo[0].hi_arch);

  // allocate adsm daemon host table
  htab_dtids=new int[nhost];

  /* startup daemons */
 
  // spawn daemons on each hosts
  for (int i=0; i<nhost; i++)
    if (pvm_spawn("adsmd",NULL,
                  PvmTaskHost|(_adsm_ctrlflags&AdsmCtrlDbgDmn?PvmTaskDebug:0),
                  hinfo[i].hi_name,1,&htab_dtids[i])<0 || htab_dtids[i]<0) 
    {
      cerr<<"Adsmith::startup(): adsmith daemon on "<<hinfo[i].hi_name
          <<" spawned error"<<endl;
      sysdep_exit(-1);
    }


  // talk with each daemon : send adsm daemon host table to each daemon
  spkt.reset();
  spkt<<(char)CHECKCODE<<(short)-1<<(char)ADSMD_CMD_INIT<<nhost
      <<_adsm_ctrlflags;
  spkt.pkint(htab_dtids,nhost);
  spkt.mcast(htab_dtids,nhost,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  mdtid=htab_dtids[0];
}

Adsmith::Adsmith() {
  sysdep_set_new_handler(); // set new exception handler

  // try to startup a pvmd if it does not exist
  pvm_setopt(PvmAutoErr,0); 
  char *param[1];
  param[0]=getenv("ADSM_HOSTFILE");
  if (param[0]&&param[0][0]) pvm_start_pvmd(1,param,1);
  pvm_setopt(PvmAutoErr,1);

  if ((mytid=pvm_mytid())<0) {
    cerr<<"Adsmith::Adsmith(): Adsmith can not enroll into PVM"<<endl;
    sysdep_exit(-1); 
  }
  ptid=pvm_parent(); // retrieve parent daemon
  pdtid=pvm_tidtohost(mytid);  // retrieve my pvm daemon

  // startup system

#ifdef __ADSM_EXEC__
  is_starter=1;
#else
  is_starter=(ptid==PvmNoParent); 
#endif

  if (is_starter) { // system starter
    flag_setting();
    pvm_catchout(stdout); // direct output of children to first process
    pvm_setopt(PvmShowTids,0);
    startup(); // startup daemons
    // pvm_setopt(PvmShowTids,1);
    // cerr<<"Adsmith Startupped"<<endl;
  } else { // child processes, receive system info from spawner
    rpkt.recv(ptid,ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
    char cmd;
    rpkt>>cmd;
    if (cmd!=ADSMD_CMD_SPAWN) {
      cerr<<"Adsmith::Adsmith(): wrong message #"<<(int)cmd<<endl;
      Exit(__FILE__,__LINE__,0);
    }
#endif

    rpkt>>nhost;
    htab_dtids=new int[nhost]; // allocate daemon host table
    rpkt.upkint(htab_dtids,nhost);

    // retrieve system options
    unsigned long ac;
    rpkt>>ac;
    _adsm_ctrlflags=ac;

    mdtid=htab_dtids[0]; // set master dtid
  }

  // set to direct route
  if (!(_adsm_ctrlflags&AdsmCtrlOptPDR)) 
    pvm_setopt(PvmRoute,PvmRouteDirect); 

  if (_adsm_ctrlflags&AdsmCtrlTimTtl) _adsm_total_time.start();

  int i;
  // search for my local adsm dtid
  for (i=0; i<nhost; i++) { // i=0 is master
    if (pvm_tidtohost(htab_dtids[i])==pdtid) {
      ldtid=htab_dtids[i];
      hostno=i;
      break;
    }
  }

  if (i==nhost+1) { // no local daemon exists
    cerr<<msghead<<"Error: Adsmith startup: local daemon does not exist"<<endl;
    Exit(__FILE__,__LINE__);
  }

  // enroll to master
  spkt.reset();
  spkt<<(char)CHECKCODE<<(short)-1<<(char)ADSMD_CMD_ENROLL<<hostno<<VERSION;
  spkt.send(mdtid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  if (!is_starter) {
    // reply to spawner to prevent some problem
    spkt.reset();
#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_SPAWN;
#endif
    spkt.send(ptid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
  }

  // get process no
  rpkt.recv(mdtid,ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_ENROLL) {
    cerr<<"Adsmith::Adsmith(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt>>procno>>nproc;
  proc_tids=new int[nproc];
  proc_hostnos=new short[nproc];
  for (i=0; i<nproc; i++) rpkt>>proc_tids[i]>>proc_hostnos[i];

  // enroll to other daemons
  spkt.reset();
  spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_ENROLL<<hostno<<VERSION;
  spkt.mcast(htab_dtids+1,nhost-1,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  // initialize AdsmObj list
  maxobjnum=GLB_TABLE_SIZE;
  objlist=new AdsmObj*[maxobjnum];
  for (i=0; i<maxobjnum; i++) objlist[i]=NULL;
  objnum=0;

  // other initialization

  cohvec=new unsigned long[nhost];
  for (i=0; i<nhost; i++) cohvec[i]=0; // set my cohvec to 0's

  readlist=new DynArray<AdsmObj*>[nhost];
  // for (i=0; i<nhost; i++) readlist[i].set_inc(100);

  writelist=new DynArray<AdsmObj*>[nhost];
  // for (i=0; i<nhost; i++) writelist[i].set_inc(100);

  alloclist=new DynArray<AdsmObj*>[nhost];
  // for (i=0; i<nhost; i++) alloclist[i].set_inc(100);

  // gndxlist.set_inc(100);

  cohvertab.set_nhost(nhost);
  cohvertab.set_nproc(nproc);

  writenum=readnum=allocnum=0;
  bulk_refresh=bulk_prefresh=bulk_malloc=0;
  active=1;
  unique_turn=hostno;

  if (_adsm_ctrlflags&AdsmCtrlInfVer) 
    cerr<<msghead<<"Adsmith version: "<<VERSION<<endl;

}

void Adsmith::getproctids() {
  // request process list from local daemon
  spkt.reset();
  spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_PROCTIDS;
  spkt.send(ldtid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  // receive the result
  rpkt.recv(ldtid,ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_PROCTIDS) {
    cerr<<"Adsmith::procno2tid(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt>>nproc;
  
  delete [] proc_tids;
  delete [] proc_hostnos;
  proc_tids=new int[nproc];
  proc_hostnos=new short[nproc];

  for (int i=0; i<nproc; i++) rpkt>>proc_tids[i]>>proc_hostnos[i];
}

// translate proceno to hostno
short Adsmith::procno2hostno(short procno) {
  if (procno<nproc) return proc_hostnos[procno];

  getproctids();	// load proc_hostnos from local daemon

  if (procno>=nproc) {
    cerr<<msghead<<"Warning: adsm_hostno(): given procno ("<<procno
        <<") exceeds the current max process no. ("<<nproc-1<<')'<<endl;
    return -1;
  }

  return proc_hostnos[procno];

  // note: procno given by user is type 'int', but we use 'short' internally
}

// translate proceno to tid
int Adsmith::procno2tid(short procno) {
  if (procno<nproc) return proc_tids[procno];

  getproctids();	// load proc_tids from local daemon

  if (procno>=nproc) {
    cerr<<msghead<<"Warning: procno2tid(): given procno ("<<procno
        <<") exceeds the current max process no. ("<<nproc-1<<')'<<endl;
    return -1;
  }

  return proc_tids[procno];

  // note: procno given by user is type 'int', but we use 'short' internally
}

// translate tid to proceno
short Adsmith::tid2procno(int tid) {
  short i;

  // search in proc_tids first
  for (i=0; i<nproc; i++) if (proc_tids[i]==tid) return i;

  // load proc_tids from local daemon
  getproctids();	

  // search again from i+1
  for (; i<nproc; i++) if (proc_tids[i]==tid) return i;

  cerr<<msghead<<"Warning: tid2procno(): given tid ("<<tid
      <<") is not found."<<endl;
  return -1;
}

// check if size of objlist need to enlarge or not
void Adsmith::check_maxobjnum(int gtsize) {
  int i;
  if (gtsize>maxobjnum) { // enlarge objlist size
    AdsmObj **new_objlist=new AdsmObj*[gtsize];
    for (i=0; i<maxobjnum; i++) new_objlist[i]=objlist[i];
    for (; i<gtsize; i++) new_objlist[i]=NULL;
    delete [] objlist;
    objlist=new_objlist;
    maxobjnum=gtsize;
  }
}

// aggreate allocate
void *Adsmith::allocate(char *id,int size,short hint,void *init,short home) {
  if (id) {
    if (strlen(id)>=MAX_NAME_LEN)
      cerr<<msghead<<"Warning: adsm_malloc(): length of identifier("<<id
          <<") must be less than or equal to "<<MAX_NAME_LEN<<endl;
    if (unique(id))
      cerr<<msghead<<"Warning: adsm_malloc(): declaration of "<<id
          <<": identifiers leading with '#' are reserved by Adsmith"<<endl;
  } else
    id=UNIQUE_OBJ;

  // if (hint&AdsmDataMovable) hint&=~AdsmDataUpdate;
  if (hint&AdsmDataUpdate) hint|=AdsmDataCache;

  //1.8.0d// check if exist
  short hindex;
  VarChainEntry *the_entry=NULL;
  AdsmObj *objptr=NULL;

  if (!unique(id)) {
    the_entry=hash_tab.insert(id,hindex);
    objptr=(AdsmObj*)the_entry->get_objptr();
    if (objptr) { // allocated locally, check if match
      if (objptr->size!=size)
        cerr<<msghead<<"Warning: adsm_malloc(): size of "<<id<<'('<<size
            <<") mismatch with the existing one ("<<objptr->size//1.8.0d//
            <<')'<<endl;

      objptr->count++;
      return (void*)objptr->data;
    } 
  } // else means UNIQUE allocation

  // allocate locally and insert request to queue, 
  // gindex will be set on doneallocate()

  // allocate, leave gindex blank, home be some process tid
  objptr=(AdsmObj*)new char[size+ALIGNED_AdsmObj_SIZE];

  if (!unique(id)) the_entry->set_objptr(objptr); 
  //1.8.0d// save objptr in hash_tab

  // decide home
  if (home<0||home>=nhost) { // not indicated
    if (unique(id)) // unique object: allocated in turn
      home=(unique_turn=(unique_turn+1)%nhost);
    else {
      if (hint&AdsmDataLocal) { 
        home=hostno;
        hint&=~AdsmDataLocal;
      } else
        home=default_home(hindex,nhost);
    }
  }

  // insert in queue (default_home is the manager (not necessarily the home))
  // for unique object, default home is the actual home
  alloclist[unique(id)?home:default_home(hindex,nhost)].insert(objptr);

  // initialize the object
  new (objptr) AdsmObj(unique(id)?id:the_entry->save_name(),
                       -1,size,hint,home,init,the_entry);

  objptr->status|=AdsmObjStatusAllocating;	// tag as in allocating

  return (void*)((char*)objptr+ALIGNED_AdsmObj_SIZE); 
  // real starting pointer to access
}


// bulk allocate
void Adsmith::bulkallocate() {
  for (int hni=0; hni<nhost; hni++) { 
    int hn=(hni+hostno)%nhost; // for fairness
    int anum=alloclist[hn].get_size();
    if (anum<=0) continue;
    allocnum+=anum;
    AdsmObj **aobj=alloclist[hn].get_array();

    int i=0;
    while (i<anum) {
      int conti=1;

      spkt.reset();
      spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_REGISTER<<anum;

      while (conti==1) {
        conti=(i+1>=anum||spkt.suggestsend())?-1:1;

        spkt.pkbyte((char*)&aobj[i],sizeof(AdsmObj*)); // send pointer too
        spkt<<aobj[i]->ident<<aobj[i]->size<<aobj[i]->hint
            <<aobj[i]->home<<conti;

        if (aobj[i]->status&AdsmObjStatusInit) {
          spkt<<1;
          spkt.pkbyte((char*)aobj[i]->data,aobj[i]->size); // initialize
        } else
          spkt<<0;

        i++;
      }
      spkt.send(htab_dtids[hn],ADSM_CHANNEL_CMD); // parallel allocate
#ifdef ADSMCTRL
      spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif
    }
    alloclist[hn].reset();
  }
}

// wait the specified object been allocated
void Adsmith::doneallocate(AdsmObj *obj) {
  //1.8.0d// if (allocnum<=0 || (obj && !(obj->status&AdsmObjStatusAllocating))) return;

  if (obj && !(obj->status&AdsmObjStatusAllocating)) return;

  //1.8.0d// prevent user who uses BulkBegin but forgot to use BulkEnd 
  bulkallocate(); 

  if (allocnum<=0) return;

  int num,sz,gtsize,done=0;
  short home;
  AdsmObj *theobj;

  while (!done) {
    rpkt.recv(-1,ADSM_CHANNEL_ALLOCATE);

#ifdef ADSMDEBUG
    char cmd;
    rpkt>>cmd;
    if (cmd!=ADSMD_CMD_REGISTER) {
      cerr<<"Adsmith::doneallocate(): wrong message #"<<(int)cmd<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

    rpkt>>gtsize>>home>>num;
    check_maxobjnum(gtsize);
    for (int i=0; i<num; i++) {
      rpkt.upkbyte((char*)&theobj,sizeof(AdsmObj*));

#ifdef ADSMDEBUG
      if (theobj->data!=(char*)theobj+ALIGNED_AdsmObj_SIZE) {
        cerr<<"Adsmith::doneallocate(): wrong object pointer"<<endl;
        Exit(__FILE__,__LINE__);
      }
#endif

      rpkt>>theobj->gindex>>theobj->hint>>sz;

      if (obj && !done) done=(theobj==obj);

      // check size
      if (sz!=theobj->size)
        cerr<<msghead<<"Warning: adsm_malloc(): size of "<<theobj->ident<<" ("
            <<theobj->size<<") mismatch with the existing one ("<<sz<<')'<<endl;

      theobj->size=sz;

      theobj->home=home;

      // already exist: not allowed

#ifdef ADSMDEBUG
      if (objlist[theobj->gindex]!=NULL) {
        cerr<<"Adsmith::doneallocate(): duplicated allocation requests"<<endl;
        Exit(__FILE__,__LINE__);
      }
  
      //1.8.0d// if(hash_tab.get_entry(theobj->hashindex,theobj->chaindex)->
      //1.8.0d//    get_gindex()>=0)
      //1.8.0d// {
      //1.8.0d//   cerr<<"Adsmith::doneallocate(): gindex already set in hash table"<<endl;
      //1.8.0d//   Exit(__FILE__,__LINE__);
      //1.8.0d// }
#endif

      //1.8.0d// hash_tab.get_entry(theobj->hashindex,theobj->chaindex)
      //1.8.0d//  ->set_gindex(theobj->gindex); // set gindex 

      objlist[theobj->gindex]=theobj;
      theobj->status&=~AdsmObjStatusAllocating;
    }
    allocnum-=num;
    if (allocnum<=0) break;
  }
}

// attach given gindex into local address
void *Adsmith::attach(int gindex) {
  if (gindex<maxobjnum&&objlist[gindex]!=NULL) { // already exist
    objlist[gindex]->count++;
    return objlist[gindex]->data;
  }

  // request to for attach 
  spkt.reset();
  spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_ATTACH<<gindex;
  spkt.send(htab_dtids[gindex%nhost],ADSM_CHANNEL_CMD); // for parallel alloc
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif

  // recv gindex and home tid from home
  char name[MAX_NAME_LEN];
  int size,retcode,gtsize;
  short home,hint;

  rpkt.recv(htab_dtids[gindex%nhost],ADSM_CHANNEL_CMD); // for parallel alloc

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_ATTACH) {
    cerr<<"Adsmith::attach(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt>>gtsize>>retcode>>name>>size>>hint>>home;
  check_maxobjnum(gtsize);
  if (!retcode) {
    cerr<<msghead<<"Error: try to attach non-existing shared data of global "
        <<"index "<<gindex<<endl;
    return NULL;
  }

  //1.8.0d// short hindex;
  //1.8.0d// int cindex;
  //1.8.0d// VarChainEntry *the_entry=hash_tab.insert(name,hindex,cindex);
  VarChainEntry *the_entry=hash_tab.insert(name);
  
  AdsmObj *objptr=(AdsmObj*)the_entry->get_objptr(); //1.8.0d// check in hash table
  if (objptr) { //1.8.0d// exist, check if match
    if (objptr->size!=size)
      cerr<<msghead<<"Warning: adsm_attach(): size of "<<name<<'('<<size
          <<") mismatch with the existing one ("<<objptr->size<<')'<<endl;
#ifdef ADSMDEBUG
    if (objptr->gindex!=gindex)
      cerr<<msghead<<"Warning: adsm_attach(): gindex of "<<name<<'('<<gindex
          <<") mismatch with the existing one ("<<objptr->gindex<<')'<<endl;
#endif
    objptr->count++;
    return objptr->data; 
  }
  
  // not exist, attach to a new space
  objptr=(AdsmObj*)new char[size+ALIGNED_AdsmObj_SIZE];

  new (objptr) AdsmObj(the_entry->save_name(),gindex,size,hint,home,NULL,
  //1.8.0d//        hindex,cindex);
                    the_entry); //1.8.0d//

  //1.8.0d// the_entry->set_gindex(gindex); // set gindex 
  the_entry->set_objptr(objptr); //1.8.0d//

  return (void*)((char*)objptr+ALIGNED_AdsmObj_SIZE);
}


// process coherence messages
void Adsmith::coherent() {
#if defined(ADSMCTRL) || defined(ADSMDEBUG)
  static int times=0;
#endif

#ifdef ADSMCTRL
  int chnum=0;

  if (_adsm_ctrlflags&AdsmCtrlOptHsk) {
    short *cdno;
    if (_adsm_ctrlflags&AdsmCtrlOptHsk) {
      // handshake with each daemons in homelist
      chnum=homelist.get_size();
      cdno=homelist.get_array();
    } else if (_adsm_ctrlflags&AdsmCtrlOptAck) {
      // handshake with local daemon 
      chnum=1;
      cdno=&hostno;
    }

    int *cdtid=new int[chnum];
    for (int i=0; i<chnum; i++) cdtid[i]=adsm.htab_dtids[cdno[i]];

    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) {
      cerr<<msghead<<"handshake with ";
      for (int i=0; i<chnum; i++) cerr<<"D"<<cdno[i]<<" ";
      cerr<<"; no."<<(++times)<<endl;
    }

    spkt.reset();
    spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_HANDSHAKE<<times;
    spkt.mcast(cdtid,chnum,ADSM_CHANNEL_CMD);
    spkt.classify_message(_adsm_msgnum_consist,_adsm_msgsiz_consist);

    delete [] cdtid;
  }
#endif

  // check coherence message from local daemon
  int shakenum=0; // number of handshake done
  int lowhn=0; // lowest host no that newest version still not arrive
  while (1) {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlOptHsk) { // by handshake
      if (shakenum<chnum) // handshake not done 
        rpkt.recv(-1,ADSM_CHANNEL_COHERENCE); // at least handshake messages
      else if (rpkt.nrecv(-1,ADSM_CHANNEL_COHERENCE)<=0) // no coherence message
        break;
    } else { // by coherence version
#endif 
      // check if all coherence message with required version have arrived
      for (; lowhn<nhost; lowhn++) 
        // if (cohvec[lowhn]<newcohvec[lowhn])
        if (cohvec[lowhn]<cohvertab.get_ver(lowhn,procno)) 
          break; // exists not arrived cohmsg
      if (lowhn==nhost) return; // all newest versions have arrived
      rpkt.recv(-1,ADSM_CHANNEL_COHERENCE); // wait newer coherence message
#ifdef ADSMCTRL
    }
#endif 

    short dmno;
    rpkt>>dmno;	// damon hostno
    rpkt>>cohvec[dmno]; // version/timestamp of this following coherence msgs

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"coherence message from D"<<dmno
          <<", version "<<cohvec[dmno]<<endl;
#endif 

    // aggreagate inv/update

    int conti=1;
    
    while (conti==1) {
      int cotype;
      rpkt>>cotype;

      // one handshake done
      if (cotype==ADSM_COHERENCE_HANDSHAKE) {

#ifdef ADSMDEBUG
        char cmd;
        int ret_times;
        rpkt>>cmd>>ret_times;
        if (cmd!=ADSMD_CMD_HANDSHAKE) {
          cerr<<"Adsmith::coherent(): wrong message #"<<(int)cmd<<endl;
          Exit(__FILE__,__LINE__);
        }
        if (ret_times!=times) {
          cerr<<"Adsmith::coherent(): handshake number not match"<<endl;
          Exit(__FILE__,__LINE__);
        }
#endif

        shakenum++;
        conti=-1;
        break;
      }

      int gindex;
      rpkt>>gindex;

#ifdef ADSMDEBUG
      if (!(objlist[gindex]->hint&AdsmDataCache)) {
        cerr<<"Adsmith::coherent(): Nocache got a coherence message"<<endl;
        Exit(__FILE__,__LINE__);
      }
#endif

      rpkt>>conti;
      switch (cotype) {
        case ADSM_COHERENCE_INV:
          objlist[gindex]->validate(0); // force invalidate

#ifdef ADSMCTRL
          if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
            cerr<<msghead<<"invalidate "<<objlist[gindex]->ident;
#endif

          break;
        case ADSM_COHERENCE_UPDATE:
          // make multiple writer special
          if (objlist[gindex]->hint&AdsmDataMWriter) {
            // read modified data, and apply into 'data' & 'olddata'
            rpkt.upkmodset(objlist[gindex]->olddata,objlist[gindex]->data);
            objlist[gindex]->validate(-1);
          } else {
            rpkt.upkbyte(objlist[gindex]->data,objlist[gindex]->size);
            objlist[gindex]->validate();
          }

#ifdef ADSMCTRL
          if (_adsm_ctrlflags&AdsmCtrlDbgTrc) {
            cerr<<msghead<<"update "<<objlist[gindex]->ident;
            if (_adsm_ctrlflags&AdsmCtrlDbgDat) {
              if (objlist[gindex]->hint&AdsmDataMWriter)
                cerr<<" by ModifySet "<<rpkt.getmodset();
              cerr<<" to generate : "<<hex;
              for (int i=0; i<objlist[gindex]->size; i++)
                cerr<<(unsigned)objlist[gindex]->data[i];
              cerr<<dec;
            }
            cerr<<endl;
          }
#endif

          break;

#ifdef ADSMDEBUG
        default:
          cerr<<"Adsmith::coherent(): unknown coherence type"<<endl;
          Exit(__FILE__,__LINE__);
#endif

      }
    }
  }
}

// aggregate prefresh

void Adsmith::bulkprefresh() {
  for (int hni=0; hni<nhost; hni++) {
    int hn=(hni+hostno)%nhost;
    int rnum=readlist[hn].get_size();
    if (rnum<=0) continue;
    AdsmObj **robj=readlist[hn].get_array();

    int i=0;
    while (i<rnum) {
      int conti=1;
      spkt.reset();
      spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_REFRESH;
      while (conti==1) {
        spkt<<robj[i]->gindex;
        i++;
        conti=(i>=rnum||spkt.suggestsend())?-1:1;
        spkt<<conti;
      }
      spkt.send(htab_dtids[hn],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
      spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

      readnum++;
    }
    readlist[hn].reset();
  }
}

// wait till #gindex arrive
void Adsmith::donerefresh(int waitindex) {
  if (!(objlist[waitindex]->status&AdsmObjStatusPrefetching)) 
    return; // already done

  int gindex,done=0,conti;
  while (!done) { // continue till target has been received
    rpkt.recv(-1,ADSM_CHANNEL_REFRESH); // out of order access
    adsm.readnum--;	// thanx cwliu notify me to add this

#ifdef ADSMDEBUG
    char cmd;
    rpkt>>cmd;
    if (cmd!=ADSMD_CMD_REFRESH) {
      cerr<<"Adsmith::donerefresh(): wrong message #"<<(int)cmd<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

    do {
      rpkt>>gindex>>conti;
      if (!done && gindex==waitindex) done=1;
      if (!(objlist[gindex]->status&AdsmObjStatusPrefetching)) {
        rpkt.upkbyte(NULL,objlist[gindex]->size); // obscure this pkt
      } else {
        rpkt.upkbyte(objlist[gindex]->data,objlist[gindex]->size);
        if (objlist[gindex]->hint&AdsmDataMWriter)
          memcpy(objlist[gindex]->olddata,objlist[gindex]->data,
            objlist[gindex]->size);
        objlist[gindex]->validate();
      }
    } while (conti==1);
  } 
}

// aggregate flushs

void Adsmith::bulkflush() {
  for (int hni=0; hni<nhost; hni++) {
    int hn=(hni+hostno)%nhost;
    int wnum=writelist[hn].get_size();
    if (wnum<=0) continue;
    AdsmObj **wobj=writelist[hn].get_array();

    // pipeline store
    int i=0;
    while (i<wnum) {
      int conti=1;
      spkt.reset();
      spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_FLUSH;
      while (conti==1) {
        spkt<<wobj[i]->gindex;
        conti=(i+1>=wnum||spkt.suggestsend())?-1:1;
        spkt<<conti;
        if (wobj[i]->hint&AdsmDataMWriter) {
          // find modified set and pack it, // pack anyway
          spkt.pkmodset(wobj[i]->olddata,wobj[i]->data,wobj[i]->size); 
          wobj[i]->validate(-1); // unchanged
          // for the 1st time, leave cache occured on Update time
        } else {
          spkt.pkbyte(wobj[i]->data,wobj[i]->size);
          wobj[i]->validate(); 
        } // make caching info logged on home
        wobj[i]->status&=~AdsmObjStatusWriting; 
        i++;
      }
      spkt.send(htab_dtids[hn],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
      spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

      writenum++; // only need to record the number of write request
    }
    writelist[hn].reset();
  }
}

void Adsmith::doneflush() {
  //1.8.0d// since doneflush are used on release time, bulk allocate and done
  //1.8.0d// allocate can be performed at the same time to ensure correctness.
  bulkallocate();

  // aggregate flush
  bulkflush();

  //1.8.0d// continue the above bulkallocate
  doneallocate();

  // wait all flush done

  if (writenum==0) 
    return;

  // wait for all out going write done
  for (int i=0; i<writenum; i++) {
    rpkt.recv(-1,ADSM_CHANNEL_FLUSH); // out of order access

#ifdef ADSMDEBUG
    char cmd;
    rpkt>>cmd;
    if (cmd!=ADSMD_CMD_FLUSH) {
      cerr<<"Adsmith::doneflush(): wrong message #"<<(int)cmd<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

    // get a write info and combine it into version table

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk)) 
#endif
      cohvertab.upk_writeinfo(rpkt);

#ifdef ADSMCTRL
    int cohacknum;
    rpkt>>cohacknum;

    // wait for the coherence acks to come (only effects for OptAck)
    for (int j=0; j<cohacknum; j++) rpkt.recv(-1,ADSM_CHANNEL_ACK);
#endif
  }

  writenum=0;
}

Adsmith::~Adsmith() {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"ready to terminate"<<endl;
#endif

  if (!active) return; 

  doneflush(); // wait all write done, ~Adsmith is itself one kind of release

  adsm_wait(); // note that coherent() is in adsm_wait() and 
               // doneflush() is before this

  // inform parent my exit
  if (!is_starter) {
    spkt.reset();

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab; // send coherence version table
    spkt.send(pvm_parent(),ADSM_CHANNEL_EXIT);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif
  }

  // inform run time system my exit
  spkt.reset();
  spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_EXIT;

#ifdef ADSMCTRL
  spkt<<spkt.get_msgnum()<<spkt.get_msgsize() // msg info, for master only
      <<_adsm_msgnum_system<<_adsm_msgsiz_system
      <<_adsm_msgnum_consist<<_adsm_msgsiz_consist
      <<_adsm_msgnum_malloc<<_adsm_msgsiz_malloc
      <<_adsm_msgnum_access<<_adsm_msgsiz_access
      <<_adsm_msgnum_sync<<_adsm_msgsiz_sync;
#endif
  spkt.mcast(htab_dtids,nhost,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  // pvm_setopt(PvmShowTids,0);
  pvm_exit();

#ifdef ADSMCTRL
  // print statistics information
  if (_adsm_ctrlflags&AdsmCtrlSttRst) {
    cerr<<msghead<<"Statistics :"<<endl
        <<msghead<<"Message count: "<<spkt.get_msgnum()<<"(send) + "
                 <<rpkt.get_msgnum()<<"(recv) = "
                 <<spkt.get_msgnum()+rpkt.get_msgnum()<<" messages"<<endl
        <<msghead<<"Message size: "<<spkt.get_msgsize()<<"(send) + "
                 <<rpkt.get_msgsize()<<"(recv) = "
                 <<spkt.get_msgsize()+rpkt.get_msgsize()<<" bytes"<<endl;
  }

  // modset timing
  if (_adsm_ctrlflags&AdsmCtrlTimMod)
    cerr<<msghead<<"Time in ModifySet = "<<_adsm_modset_time<<" sec"<<endl;

  if (_adsm_ctrlflags&AdsmCtrlTimPkt)
    cerr<<msghead<<"Time in (un)Packing = "<<_adsm_pkt_time<<" sec"<<endl;

  if (_adsm_ctrlflags&AdsmCtrlTimIdl)
    cerr<<msghead<<"Time of Idleness (including time in pvm_recv()) = "
        <<_adsm_idle_time<<" sec"<<endl;
#endif

  if (_adsm_ctrlflags&AdsmCtrlTimTtl) {
    _adsm_total_time.stop();
    cerr<<msghead<<"Total execution time = "<<_adsm_total_time<<" sec"<<endl;
  }

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"terminated"<<endl;
#endif

}

void Adsmith::shutdown() {
  cerr<<msghead<<"Adsmith: try to shutdown the application..."<<endl;

  fflush(stdout);
  fflush(stderr);

  spkt.reset();
  spkt<<(char)CHECKCODE<<procno<<(char)ADSMD_CMD_SHUTDOWN;
  spkt.send(mdtid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  // pvm_setopt(PvmShowTids,0);
  pvm_exit();

  active=0;
}

// --- non-inline methods of class AdsmObj ---

// register to adsm
AdsmObj::AdsmObj(char *id,int gidx,int sz,short hnt,
                 short hm,void *init,VarChainEntry *h_entry) 
{
  ident=id;
  size=sz;
  hint=hnt;
  gindex=gidx;
  home=hm;
  status=0; 
  data=(char*)this+ALIGNED_AdsmObj_SIZE; 
  olddata=(hint&AdsmDataMWriter)?(new char[size]):(char*)NULL;

  if (init!=NULL) { // initialize
    memcpy((char*)data,(char*)init,size);
    status|=AdsmObjStatusInit; 
  } else
    memset(data,0,size);    

  if (hint&AdsmDataMWriter)
    if (init!=NULL)
      memcpy((char*)olddata,(char*)data,size); 
    else
      memset(olddata,0,size); 

  if (gindex>=0) adsm.objlist[gindex]=this; // record in objlist

  count=0;

  //1.8.0d// hashindex=hindex;
  //1.8.0d// chaindex=cindex;
  hash_entry=h_entry;

  adsm.objnum++;
}

// undeclare the shared object
AdsmObj::~AdsmObj() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  if (!unique(ident)) {
    adsm.hash_tab.remove(ident); // make gindex in hash_tab unset
    delete [] ident;
  }
  delete [] olddata; 

  if (gindex>=0) adsm.objlist[gindex]=NULL;
}

// Nsync load: immediately refresh from home
void AdsmObj::refresh_now() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_REFRESH_NOW
      <<gindex<<-1; // not continue
  spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

  int index,conti;
  rpkt.recv(adsm.htab_dtids[home],ADSM_CHANNEL_CMD); // (*2)

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_REFRESH_NOW) {
    cerr<<"AdsmObj::refresh_now(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  } 
#endif

  rpkt>>index>>conti;

#ifdef ADSMDEBUG
  if (index!=gindex) {
    cerr<<"AdsmObj::refresh_now(): refresh #"<<index
        <<" but got #"<<index<<endl;
    Exit(__FILE__,__LINE__);
  }
  if (conti!=-1) {
    cerr<<"AdsmObj::refresh_now(): wrong continue/stop flag";
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt.upkbyte(data,size);
  if (hint&AdsmDataMWriter) memcpy(olddata,data,size); 

  validate();
}

// Atomic Begin: lock and refresh from home (viewed as Nsync)
void AdsmObj::atomic_begin(char rw,short cltnum) {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  if (status&AdsmObjStatusInAtomic) { 
    cerr<<"AdsmObj::atomic_begin(): "<<ident<<" already in atomic section"
        <<endl;
    return;
  }

  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_ATOMIC_BEGIN
      <<gindex<<-1<<rw; // not continue
  spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

  int index,conti;
  rpkt.recv(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_ATOMIC_BEGIN) {
    cerr<<"AdsmObj::atomic_begin(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  } 
#endif

  rpkt>>index>>conti;

#ifdef ADSMDEBUG
  if (index!=gindex) {
    cerr<<"AdsmObj::atomic_begin(): refresh #"<<index
        <<" but got #"<<index<<endl;
    Exit(__FILE__,__LINE__);
  }
  if (conti!=-1) {
    cerr<<"AdsmObj::atomic_begin(): wrong continue/stop flag";
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt.upkbyte(data,size);
  if (hint&AdsmDataMWriter) memcpy(olddata,data,size); 

  atomic_type=rw;
  collectnum=cltnum; // for collect operation only
  status|=AdsmObjStatusInAtomic; 
  validate();
}

void AdsmObj::atomic(char *expr) {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  if (hint&AdsmDataMWriter) {
    cerr<<msghead<<"Warning: adsm_atomic(): currently not support MWriter data"
        <<" ("<<ident<<") yet "<<endl;
    return;
  }

  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_ATOMIC<<gindex;

  // pack expression
  int exprlen=strlen(expr)+1;
  spkt<<exprlen;
  spkt.pkbyte(expr,exprlen);
  spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

  rpkt.recv(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_ATOMIC) {
    cerr<<"AdsmObj::atomic(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  } 
#endif

#ifdef ADSMDEBUG
  int index;
  rpkt>>index;
  if (index!=gindex) {
    cerr<<"AdsmObj::atomic(): against #"<<index<<" but return #"<<index<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  int retcode;
  rpkt>>retcode;
  if (retcode<0) {
    cerr<<"Error::adsm_atomic(): expression "<<expr<<" is incorrect"<<endl;
    Exit(__FILE__,__LINE__);
  }

  rpkt.upkbyte(data,size);

  // write must be ensured to finish by ack before release, but return 
  // on atomic implies the finish of the write, thus ack is not needed.

  validate();


#ifdef ADSMCTRL
  // if OptAck is enabled, write-ack will be sent anyway
  if (_adsm_ctrlflags&AdsmCtrlOptAck) adsm.writenum++;
#endif
}

// Nsync store: immediately store to home directly
void AdsmObj::flush_now() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  // pipeline store
  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_FLUSH<<gindex<<-1; 
  if (hint&AdsmDataMWriter) { 
    // find modified set and pack it
    if (spkt.pkmodset(olddata,data,size)==0 && 
        (!(hint&AdsmDataCache)||(status&AdsmObjStatusValid)))
      return; // not modified, return if already valid
  } else
    spkt.pkbyte(data,size); 
  spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

  adsm.writenum++; // only need to record the number of write request

  // clean write message when outgoing mesg num > CLEAN_OUTMSG_NUM
  if (adsm.writenum>CLEAN_OUTMSG_NUM) {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"clean outgoing write requests"<<endl;
#endif

    adsm.doneflush();
  }

  if (hint&AdsmDataMWriter)	
    validate(-1); // for 1st time write, leave cache occured on Update time 
  else                     	
    validate();
}

// Atomic End: flush and unlock to home (viewed as Nsync)
void AdsmObj::atomic_end() {
  if (status&AdsmObjStatusAllocating) adsm.doneallocate(this);

  if (!(status&AdsmObjStatusInAtomic)) { 
    cerr<<"AdsmObj::atomic_end(): "<<ident<<" not in atomic section"<<endl;
    return;
  }
  status&=~AdsmObjStatusInAtomic; 

  // do doneflush() here? i.e. take atomic_end as an release. Not for now!
  // adsm.doneflush();

  if (atomic_type==AdsmAtomicWrite) {
    spkt.reset();
    spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_ATOMIC_END
        <<gindex<<-1<<atomic_type<<collectnum;
    // find modified part and pack it
    if (hint&AdsmDataMWriter) 
      spkt.pkmodset(olddata,data,size); // not same as flush, for unlock
    else
      spkt.pkbyte(data,size); 
    spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif

    // write must be ensured to finish by ack before release, but return 
    // on collect implies the finish of all write, thus ack is not needed.
    if (collectnum<2) { 
      adsm.writenum++; // only need to record the number of write request

      // clean write message when outgoing mesg num > CLEAN_OUTMSG_NUM
      if (adsm.writenum>CLEAN_OUTMSG_NUM) {

#ifdef ADSMCTRL
        if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
          cerr<<msghead<<"clean outgoing write requests"<<endl;
#endif

        adsm.doneflush();
      }
    } else { // collect is performed only on AtomicWrite
      int index,conti;
      rpkt.recv(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
      char cmd;
      rpkt>>cmd;
      if (cmd!=ADSMD_CMD_ATOMIC_END) {
        cerr<<"AdsmObj::atomic_end(): wrong message #"<<(int)cmd<<endl;
        Exit(__FILE__,__LINE__);
      } 
#endif

      rpkt>>index>>conti;

#ifdef ADSMDEBUG
      if (index!=gindex) {
        cerr<<"AdsmObj::atomic_end(): refresh #"<<index
            <<" but got #"<<index<<endl;
        Exit(__FILE__,__LINE__);
      }
      if (conti!=-1) {
        cerr<<"AdsmObj::atomic_end(): wrong continue/stop flag";
        Exit(__FILE__,__LINE__);
      }
#endif

      rpkt.upkbyte(data,size);
      if (hint&AdsmDataMWriter) memcpy(olddata,data,size); 

#ifdef ADSMCRTL
      // if OptAck is enabled, write-ack will be sent anyway
      if (_adsm_ctrlflags&AdsmCtrlOptAck) adsm.writenum++;
#endif
    }
  } else { 
    // conservative implementation of atomic_end(,AdsmAtomicRead)
    // spkt.reset();
    // spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_ATOMIC_END
    //     <<gindex<<-1<<atomic_type<<collectnum;
    //
    // spkt.send(adsm.htab_dtids[home],ADSM_CHANNEL_CMD);
    //#ifdef ADSMCTRL
    //  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
    //#endif

    /* Now we assume that in most cases, atomic section for AtomicRead
       contains at most one statment which references the target object.
       That is, atomic_end(,AtomicRead) is not intended to be used for
       synchronizing with corresponding atomic_begin(,AtomicWrite).
       Thus atomic_end(,AtomicRead) need not send messages! */
  }

  atomic_type=AdsmAtomicWrite;
  collectnum=0; // for collect operation only

  if (hint&AdsmDataMWriter)	
    validate(-1); // for 1st time write, leave cache occured on Update time
  else				
    validate();
}

// (*2): Special accesses are SC

// --- Methods of Semaphore ---

AdsmSemaphore::AdsmSemaphore(char *id,int val) {
  char ident[100];
  sprintf(ident,SEMAPHORE_OBJ":%s",id);
  value=(int*)adsm_malloc(ident,sizeof(int),&val,(short)0);
}

AdsmSemaphore::~AdsmSemaphore() {
  adsm_free(value);
}

int AdsmSemaphore::get() {
  adsm_refresh_now(value);
  return *value;
}

void AdsmSemaphore::set(int val) {
  *value=val;
  adsm_flush_now(value);
}

void AdsmSemaphore::wait() {
  AdsmObj *sctrl=adsm.get_sctrl(value);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"to acquire "<<sctrl->ident<<endl;
#endif

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_P<<sctrl->gindex;
  spkt.send(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif
 
  rpkt.recv(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_P) {
    cerr<<"AdsmSemaphore::p(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>adsm.cohvertab;	// receive coherence version table here

  // RC: receive update/invalidate
  adsm.coherent();

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<sctrl->ident<<" acquired"<<endl;
#endif

}

void AdsmSemaphore::signal() {

  // RC: wait all previous flush done
  adsm.doneflush();
 
  // nonblocking release
  AdsmObj *sctrl=adsm.get_sctrl(value);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"to release "<<sctrl->ident<<endl;
#endif

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

  // relase
  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_V<<sctrl->gindex;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
  spkt<<adsm.cohvertab;	// send coherence version table
  spkt.send(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<sctrl->ident<<" released"<<endl;
#endif

}

// --- Methods of Mutex ---

AdsmMutex::AdsmMutex(char *id) {
  char ident[100];
  sprintf(ident,MUTEX_OBJ":%s",id);
  tag=(int*)adsm_malloc(ident,sizeof(int),(short)0);
}

AdsmMutex::~AdsmMutex() {
  adsm_free(tag);
}

void AdsmMutex::lock() {
  AdsmObj *sctrl=adsm.get_sctrl(tag);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"to acquire "<<sctrl->ident<<endl;
#endif

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_LOCK<<sctrl->gindex;
  spkt.send(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif
 
  // wait for lock grant
  rpkt.recv(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_LOCK) {
    cerr<<"AdsmMutex::lock(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>adsm.cohvertab;	// receive coherence version table here

  // RC: receive update/invalidate
  adsm.coherent();

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<sctrl->ident<<" acquired"<<endl;
#endif

}

void AdsmMutex::unlock() {
  // RC: wait all previous flush done
  adsm.doneflush();
 
  AdsmObj *sctrl=adsm.get_sctrl(tag);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"to release "<<sctrl->ident<<endl;
#endif

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

  // nonblocking release
  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_UNLOCK<<sctrl->gindex;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
  spkt<<adsm.cohvertab;	// send coherence version table here
  spkt.send(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<sctrl->ident<<" released"<<endl;
#endif

}

// --- Methods of Barrier ---

AdsmBarrier::AdsmBarrier(char *id) {
  char ident[100];
  sprintf(ident,BARRIER_OBJ":%s",id);
  tag=(int*)adsm_malloc(ident,sizeof(int)*2,(short)0);
}

AdsmBarrier::~AdsmBarrier() {
  adsm_free(tag);
}

int AdsmBarrier::barrier(int count) {
  if (count<=0) return 1;

  // RC: wait all previous flush done
  adsm.doneflush();

  AdsmObj *sctrl=adsm.get_sctrl(tag);

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
    cerr<<msghead<<"arrive "<<sctrl->ident<<endl;
#endif

  // release
  spkt.reset();
  spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_BARRIER
      <<sctrl->gindex<<(short)count;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
  spkt<<adsm.cohvertab;	// send coherence version table here
  spkt.send(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif
 
  // acquire
  short total;
  rpkt.recv(adsm.htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
  char cmd;
  rpkt>>cmd;
  if (cmd!=ADSMD_CMD_BARRIER) {
    cerr<<"AdsmBarrier::lock(): wrong message #"<<(int)cmd<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  rpkt>>total;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>adsm.cohvertab;	// receive coherence version table here

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
    cerr<<msghead<<"leave "<<sctrl->ident<<endl;
#endif

  // RC: receive update/invalidate
  adsm.coherent();

  return total==count;
}

// --- Interfaces ---

// spawn and initialize, option: allow tids to be NULL
// return spawned task number, -1 on fail
int adsm_spawn(char *file,char **argv,int flags,char *where,
               int count,int *tids)
{
  int i;

  if (count<1) return 0;
  if ((flags&PvmTaskArch)&&strcmp(where,adsm.arch)!=0) {
    cerr<<"spawn(): can not spawn processes on '"<<where
        <<"' in homogeneous environment of "<<adsm.arch<<endl;
    return -1;
  }

  // RC: wait all previous flush done
  adsm.doneflush();
 
  int	tmptid=0;
  if (tids==NULL) {
    tids=new int[count];
    tmptid=1;
  }

  // int oldset=pvm_setopt(PvmShowTids,0);

#ifndef __ADSM_EXEC__
    int ntask=pvm_spawn(file,argv,
                        flags|(_adsm_ctrlflags&AdsmCtrlDbgApp?PvmTaskDebug:0),
                        where,count,tids);
#else
    int ntask=pvm_spawn(file,argv,flags,where,count,tids);
#endif

  // pvm_setopt(PvmShowTids,oldset);

  // send master tid to children
  spkt.reset();

#ifdef ADSMDEBUG
  spkt<<(char)ADSMD_CMD_SPAWN;
#endif

  spkt<<adsm.nhost;
  spkt.pkint(adsm.htab_dtids,adsm.nhost);

  spkt<<_adsm_ctrlflags;

  // need not send coherence version table to child
  // since they have no cache yet
  spkt.mcast(tids,ntask,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

  for (i=0; i<ntask; i++)
    adsm.childlist.insert(tids[i]);

  for (i=0; i<ntask; i++) {
    rpkt.recv(tids[i],ADSM_CHANNEL_CMD);

#ifdef ADSMDEBUG
    char cmd;
    rpkt>>cmd;
    if (cmd!=ADSMD_CMD_SPAWN) {
      cerr<<"adsm_spawn(): wrong message #"<<(int)cmd<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

  }

  if (tmptid) delete [] tids;

  return ntask; 
}

int adsm_spawn(char *file,int count) {
  return adsm_spawn(file,NULL,PvmTaskDefault,NULL,count,NULL);
}

// wait children done
int adsm_wait(int tid) {
#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"wait t"<<hex<<tid<<dec<<" to finish"<<endl;
#endif

  rpkt.recv(tid,ADSM_CHANNEL_EXIT);

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>adsm.cohvertab;	// receive coherence table from a child

  adsm.childlist.remove(tid);

  adsm.coherent(); // wait is like an acquire
  return 1;
}

int adsm_wait(int *tids,int num) {
  int i;
  for (i=0; i<num; i++) {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc) 
      cerr<<msghead<<"wait t"<<hex<<tids[i]<<dec<<" to finish"<<endl;
#endif

    rpkt.recv(tids[i],ADSM_CHANNEL_EXIT);

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
      rpkt>>adsm.cohvertab;	// receive coherence table from a child

    adsm.childlist.remove(tids[i]);
  }

  adsm.coherent(); // wait is like an acquire
  return i;
}

int adsm_wait() {
  int cnum=adsm.childlist.get_size();
  int *ctid=adsm.childlist.get_array();
  return adsm_wait(ctid,cnum);
}

// aggregate malloc
void adsm_malloc(AdsmBulkType type,int num) {
/* All malloc's are now performed using bulk-transfer implicitly
  switch (type) {
    case AdsmBulkBegin:
      if (num>0) { // number of objects in this bulk transfer
        for (int i=0; i<adsm.nhost; i++)
          adsm.alloclist[i].addsize((num+adsm.nhost-1)/adsm.nhost);
      }
      adsm.bulk_malloc++;
      break;
    case AdsmBulkEnd: // bulk flush
      adsm.bulk_malloc--;
      if (adsm.bulk_malloc<=0)
        adsm.bulkallocate();
      // if (num<=0)
      //   adsm.doneallocate(); // temp // wait all allocation done
      break;
    default:
      cerr<<"adsm_malloc(): Unknown bulk transfer type"<<endl;
  }
*/
}

void adsm_malloc() {
  adsm.doneallocate();
}

void *adsm_malloc(char *id,int size,void *initdata,int home,short hint) {
  if (home>MAXSHORT)
    cerr<<msghead<<"adsm_malloc(): Warning: assigned home node number "
        <<"exceeds the number of participating nodes "<<adsm.nhost<<endl;

  void *obj=adsm.allocate(id,size,hint,initdata,(short)home);
  //0d// if (!adsm.bulk_malloc) adsm.bulkallocate();
  return obj;
}

void *adsm_malloc(char *id,int size,void *initdata,short hint) {
  return adsm_malloc(id,size,initdata,-1,hint);
}

void *adsm_malloc(char *id,int size,short hint) {
  return adsm_malloc(id,size,NULL,-1,hint);
}

// array: allow arrange setting, inital data (all elmts the same)
void adsm_malloc_array(char *id,int size,int begin,int num,void *varrptr,
                       void *initdata,int *distarr,short hint) 
{
  int i,j;
  void **arrptr=(void**)varrptr;
  char name[MAX_NAME_LEN];
  adsm_malloc(AdsmBulkBegin,num);
  for (i=begin,j=0; j<num; i++,j++) {
    sprintf(name,"%s[%d]",id,i);
    short home=(distarr==NULL?-1:(short)distarr[j]);
    arrptr[j]=adsm_malloc(name,size,initdata,home,hint);
  }
  adsm_malloc(AdsmBulkEnd);
}

// array: start from 0
void adsm_malloc_array(char *id,int size,int num,void *varrptr,short hint) {
  adsm_malloc_array(id,size,0,num,varrptr,NULL,NULL,hint);
}

// array: given data distribution pattern
void adsm_malloc_array(char *id,int size,int num,void *varrptr,
                       int *distarr,short hint) 
{
  adsm_malloc_array(id,size,0,num,varrptr,NULL,distarr,hint);
}

void adsm_free(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);
  if (sctrl->count==0) {

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<sctrl->ident<<" is freed"<<endl;
#endif

    // undeclare in LSML
    // spkt.reset();
    // spkt<<(char)CHECKCODE<<adsm.procno<<(char)ADSMD_CMD_UNDECL
    //     <<sctrl->gindex;
    // spkt.send(htab_dtids[sctrl->home],ADSM_CHANNEL_CMD);
    //#ifdef ADSMCTRL
    //  spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
    //#endif
    //
    sctrl->~AdsmObj();
    delete [] (char*)sctrl; 
  } else
    sctrl->count--;
}

// aggregate prefetch 
inline void inline_adsm_prefresh(AdsmBulkType type) {
  switch (type) {
    case AdsmBulkBegin:
      for (int i=0; i<adsm.nhost; i++)
        adsm.readlist[i].addsize((adsm.objnum+adsm.nhost-1)/adsm.nhost);
      adsm.bulk_prefresh++;
      break;
    case AdsmBulkEnd:
      adsm.bulk_prefresh--;
      if (adsm.bulk_prefresh<=0)
        adsm.bulkprefresh();
      break;
    default:
      cerr<<"adsm_prefresh(): Unknown bulk transfer type"<<endl;
  }
}

// aggregate ordinary load 
inline void inline_adsm_refresh(AdsmBulkType type) {
  int rnum,*rgindex,i;
  switch (type) {
    case AdsmBulkBegin:
      for (i=0; i<adsm.nhost; i++)
        adsm.readlist[i].addsize((adsm.objnum+adsm.nhost-1)/adsm.nhost);
      adsm.gndxlist.addsize(adsm.objnum);
      adsm.bulk_prefresh=adsm.bulk_refresh=1;
      break;
    case AdsmBulkEnd:
      adsm.bulk_prefresh--;
      adsm.bulk_refresh--;

      // bulk prefetching
      if (adsm.bulk_prefresh<=0)
        adsm.bulkprefresh();

      // finish reading
      if (adsm.bulk_refresh<=0) {
        rnum=adsm.gndxlist.get_size();
        rgindex=adsm.gndxlist.get_array();
        for (i=0; i<rnum; i++) adsm.donerefresh(rgindex[i]);
        adsm.gndxlist.reset();
      }

      break;
    default:
      cerr<<"adsm_prefresh(): Unknown bulk transfer type"<<endl;
  }
}

// aggregate ordinary store 
inline void inline_adsm_flush(AdsmBulkType type) {
  switch (type) {
    case AdsmBulkBegin:
      for (int i=0; i<adsm.nhost; i++)
        adsm.writelist[i].addsize((adsm.objnum+adsm.nhost-1)/adsm.nhost);
      break;
    case AdsmBulkEnd: // bulk flush
      //1.8.0d// adsm.doneflush();
      adsm.bulkflush(); //1.8.0d//
      break;
    default:
      cerr<<"adsm_prefresh(): Unknown bulk transfer type"<<endl;
  }
}

inline void inline_adsm_prefresh(void *data) {
  adsm.get_sctrl(data)->prefresh();
}

inline void inline_adsm_refresh(void *data) {
  adsm.get_sctrl(data)->refresh();
}

inline void inline_adsm_flush(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);
  sctrl->flush();
}

// non-inline version

void adsm_prefresh(AdsmBulkType type) { inline_adsm_prefresh(type); }

void adsm_refresh(AdsmBulkType type) { inline_adsm_refresh(type); }

void adsm_prefresh(void *data) { inline_adsm_prefresh(data); }

void adsm_refresh(void *data) { inline_adsm_refresh(data); }

void adsm_flush(AdsmBulkType type) { inline_adsm_flush(type); }

void adsm_flush(void *data) { inline_adsm_flush(data); }

void adsm_prefresh_array(void *data,int len) {
  inline_adsm_prefresh(AdsmBulkBegin);
  for (int i=0; i<len; i++) inline_adsm_prefresh(((void**)data)[i]);
  inline_adsm_prefresh(AdsmBulkEnd);
}

void adsm_refresh_array(void *data,int len) {
  inline_adsm_refresh(AdsmBulkBegin);
  for (int i=0; i<len; i++) inline_adsm_refresh(((void**)data)[i]);
  inline_adsm_refresh(AdsmBulkEnd);
}

void adsm_flush_array(void *data,int len) {
  inline_adsm_flush(AdsmBulkBegin);
  for (int i=0; i<len; i++) inline_adsm_flush(((void**)data)[i]);
  inline_adsm_flush(AdsmBulkEnd);
}

void adsm_refresh_now(void *data) {
  adsm.get_sctrl(data)->refresh_now();
}

void adsm_flush_now(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);
  sctrl->flush_now();
}

void adsm_atomic_begin(void *data,int rw) {
  adsm.get_sctrl(data)->atomic_begin((char)rw);
}

void adsm_atomic_end(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);
  sctrl->atomic_end();
}

void adsm_atomic(void *data,char *expr) {
  adsm.get_sctrl(data)->atomic(expr);
}

void adsm_collect_begin(void *data,int num) {
  adsm.get_sctrl(data)->atomic_begin(AdsmAtomicWrite,num);
}

void adsm_collect_end(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);
  sctrl->atomic_end();
}

int adsm_gid(void *data) {
  AdsmObj *sctrl=adsm.get_sctrl(data);

  if (sctrl->status&AdsmObjStatusAllocating) adsm.doneallocate(sctrl);

  return sctrl->gindex;
}

// attach gindex into local address
void *adsm_attach(int gindex) {
  return adsm.attach(gindex); 
}

// misc functions

void adsm_shutdown() { adsm.shutdown(); }

void adsm_showtid(int onoff) { pvm_setopt(PvmShowTids,onoff); }

const char *adsm_version() { return VERSION; }

int adsm_hostno(int procno) { 
  if (procno<0) return adsm.hostno; 
  return adsm.procno2hostno(procno);
}

int adsm_procno() { return adsm.procno; }

int adsm_procno2tid(int procno) { return adsm.procno2tid(procno); }

int adsm_tid2procno(int tid)    { return adsm.tid2procno(tid); }

#ifdef ADSMCTRL
void adsm_msgnum(int msginfo[4]) {
  msginfo[0]=spkt.get_msgnum();
  msginfo[1]=spkt.get_msgsize();
  msginfo[2]=rpkt.get_msgnum();
  msginfo[3]=rpkt.get_msgsize();
}
#endif

void adsm_enroll() { static int forceuse; forceuse=1; }

// for adsmith executor
#ifdef __ADSM_EXEC__
   main(int argc,char *argv[]) {
     if (argc>1)
       adsm_spawn(argv[1],argv+2);
   }
#endif

