/*
        Adsmith 1.0 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved
 
                                  NOTICE
 
      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.
 
      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#ifndef __ADSMHASH_H__
#define __ADSMHASH_H__

#include <string.h>
#include <stdlib.h>
#include <values.h>
#include "dynarr.h"
#include "common.h"

// Hash Chain Entry
class VarChainEntry {
private:

  char	code;
  char	*name;
  //1.8.0d// int	gindex;
  union { //1.8.0d// AdsmObj*(libadsm.c) or gindex(adsmd.c)
    void  *objptr; 
    int   gindex;  // assume sizeof(gindex) <= sizeof(void*)
  };

public:

  //1.8.0d// VarChainEntry() { code=gindex=-1; name=0; };
  VarChainEntry() { code=-1; objptr=name=0; };
  VarChainEntry(const char *n);
  ~VarChainEntry() {}  // name will be deleted by ~AdsmObj()

  inline int operator == (const VarChainEntry& e);

  int get_gindex() { return gindex; }
  void *get_objptr() { return objptr; } //1.8.0d// 
  void set_gindex(int gdx) { gindex=gdx; }
  void set_objptr(void *ptr) { objptr=ptr; } //1.8.0d//

  char *save_name() { return name=my_strdup(name); }
};

// Hash Table
class VarHashTab {
private:

  DynArray<VarChainEntry> *tab[HASH_NUM];

  short hash(const char* name);

public:

  VarHashTab() { for (int i=0; i<HASH_NUM; i++) tab[i]=NULL; }
  ~VarHashTab() { for (int i=0; i<HASH_NUM; i++) delete tab[i]; }

  VarChainEntry* insert(const char *name);
  VarChainEntry* insert(const char *name,short &hidx);
  void remove(const char *name);

  friend inline ostream& operator<<(ostream& s,VarHashTab& v);
};

// Methods for Hash Chain Entry

inline VarChainEntry::VarChainEntry(const char *n) {
  code=(char)str_encode(n);
  name=(char*)n; // note: not strdup()
  //1.8.0d// gindex=-1;
  objptr=0; //1.8.0d//
}

inline int VarChainEntry::operator == (const VarChainEntry& e) {
  return this->code==e.code && strcmp(this->name,e.name)==0;
}

// Methods for Hash Table

inline short VarHashTab::hash(const char* name) { 
  int len=0,i;
  char val[MAX_NAME_LEN];
  for (i=0; name[i]!=0; i++) {
    if (name[i]>='0'&&name[i]<='9')
      val[len++]=name[i]-'0';
    else if (name[i]>='a'&&name[i]<='z')
      val[len++]=(name[i]-'a')*10/26;
    else if (name[i]>='A'&&name[i]<='Z')
      val[len++]=(name[i]-'A')*10/26;
  }
  long key=0;
  int radix=1;
  for (i=len-1; i>=0; i--,radix*=10) {
    if (radix>MAXLONG/10) radix=1;
    key+=val[i]*radix;
  }
  return key%HASH_NUM; 
}

inline VarChainEntry* VarHashTab::insert(const char *name,short &hidx) {
  hidx=hash(name);
  if (tab[hidx]==NULL) tab[hidx]=new DynArray<VarChainEntry>;
  VarChainEntry vce(name);
  int cidx=tab[hidx]->uniq_insert(vce);
  return (VarChainEntry*)&((*tab[hidx])[cidx]);
}

inline VarChainEntry* VarHashTab::insert(const char *name) {
  short hidx;
  return insert(name,hidx);
}

inline void VarHashTab::remove(const char *name) {
  short hidx=hash(name);
  if (tab[hidx]==NULL) return; // not exist
  VarChainEntry vce(name);
  tab[hidx]->remove(vce); // remove from the chain
}

inline ostream& operator<<(ostream& s,VarHashTab& v) {
  for (int i=0; i<HASH_NUM; i++) 
    if (v.tab[i]!=NULL) s<<endl<<"  Entry "<<i<<": "<<v.tab[i]->get_size(); 
  return s;
}

#endif // __ADSMHASH_H__
