/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved
 
                                  NOTICE
 
      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.
 
      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#ifndef __ADSMDARR_H__
#define __ADSMDARR_H__

#include <iostream.h>

#define DYN_GAP 8

// dynamic array
template <class T>
class DynArray {
protected:

  T *array;
  int crnt_size;	// current buffer size
  int next_index;	// next index (array size)

  void resize(int s);	// resize array size to the specified size
  void grow();		// grow up the array size

public:

  DynArray(int s=0);
  ~DynArray() { delete [] array; } 

  int operator == (DynArray<T> &da);
  DynArray<T>& operator = (DynArray<T> &da);
  T& operator [] (int index);

  int find_index(T& elmt);	// find index of elmt

  int insert(T& elmt);		// append anyway
  int uniq_insert(T& elmt);	// insert uniquely

  int overwrite(T& elmt);	// overwrite if exists, otherwise insert

  void remove_index(int idx);	// remove the indexed element
  void remove(T& elmt);	// remove the element if exist

  void addsize(int s) { resize(get_size()+s); }	// add array (not buffer) size

  void reset();

  int is_empty() { return next_index==0; }

  int get_size() { return next_index; }	// array size
  T *get_array() { return array; }

  T *get_entry(int index) { return (index<next_index)?(array+index):(T*)NULL;
}
  void set_entry(int index,T& elmt);
};

// Methods

template<class T>
inline DynArray<T>::DynArray(int s) : crnt_size(s), next_index(0) { 
  if (crnt_size<=0) crnt_size=0;
  array=crnt_size==0?(T*)NULL:new T[crnt_size];
}

template<class T>
void DynArray<T>::resize(int s) { 
  if (s<=crnt_size) return;
  T *new_array=new T[crnt_size=s];
  for (int i=0; i<next_index; i++) new_array[i]=array[i]; // copy to new
  delete [] array; 
  array=new_array;
}

template<class T>
inline void DynArray<T>::grow() { 
  resize((crnt_size<=0)?DYN_GAP:(crnt_size+(crnt_size>>1)+1)); 
  // increase half of the original size
}

template<class T>
inline void DynArray<T>::reset() { 
  next_index=0;
  // delete [] array;
  // array=NULL;
  // crnt_size=0;
}

template<class T>
inline int DynArray<T>::operator == (DynArray<T> &da) {
  return array==da.array && crnt_size==da.crnt_size &&
         next_index==da.next_index;
}

template<class T>
DynArray<T>& DynArray<T>::operator = (DynArray<T> &da) {
  array=new T[crnt_size=da.crnt_size];
  for (int i=0; i<crnt_size; i++) array[i]=da.array[i];
  next_index=da.next_index;
  return *this;
}

template<class T>
inline int DynArray<T>::find_index(T& elmt) {
  for (int i=0; i<next_index; i++)
    if (array[i]==elmt) return i; // found, return the index of existing one
  return -1;
}

template<class T>
inline int DynArray<T>::insert(T& elmt) {
  if (next_index>=crnt_size) grow(); // grow the space in double space
  array[next_index]=elmt; // put elmt into array
  return next_index++;
}

template<class T>
inline int DynArray<T>::uniq_insert(T& elmt) {
  int index=find_index(elmt);
  return index<0?insert(elmt):index;
}

template<class T>
inline int DynArray<T>::overwrite(T& elmt) {
  int index=find_index(elmt);
  if (index<0) 
    index=insert(elmt);
  else 
    array[index]=elmt;
  return index;
}

template<class T>
inline void DynArray<T>::remove_index(int idx) {
  if (idx<next_index)
    array[idx]=array[--next_index]; // move the last to the idx-th
}

template<class T>
inline void DynArray<T>::remove(T& elmt) {
  for (int i=0; i<next_index; i++)
    if (array[i]==elmt) // remove the i-th element
      array[i]=array[--next_index];	// move the last to the i-th
}

template<class T>
inline void DynArray<T>::set_entry(int index,T& elmt) { 
  if (index>=crnt_size) grow();
  array[index]=elmt; 
  if (index>=next_index) next_index=index+1;
}

template<class T>
inline T& DynArray<T>::operator [] (int index) { 
  if (index>=next_index)
    cerr<<"DynArray<T>: warning: index out-of-bound for operator []"<<endl;
  return array[index]; 
}

#endif // __ADSMDARR_H__
